with System;

package body Ada.Text_IO is

   -----------
   -- Local --
   -----------

   type Reserved_24 is array (0 .. 23) of Boolean;
   for Reserved_24'Size use 24;
   pragma Pack (Reserved_24);

   UART_1_Data_Register_Address :
     constant System.Address := System'To_Address (16#8000_0100#);

   UART_1_Status_Register_Address :
     constant System.Address := System'To_Address (16#8000_0104#);

   type FIFO_Count is mod 64;
   for FIFO_Count'Size use 6;

   pragma Warnings (Off);
   type Parity_Kind is (Even, Odd);

   --  Mapping between bits in a 32-bit register as used in the
   --  hardware documentation and bit order as used by Ada.
   --  This makes it easier to verify correctness against the AUM.
   --  Ranges will need to be reversed, but the compiler will check this.
   Bit00 : constant := 31; Bit01 : constant := 30; Bit02 : constant := 29;
   Bit03 : constant := 28; Bit04 : constant := 27; Bit05 : constant := 26;
   Bit06 : constant := 25; Bit07 : constant := 24; Bit08 : constant := 23;
   Bit09 : constant := 22; Bit10 : constant := 21; Bit11 : constant := 20;
   Bit12 : constant := 19; Bit13 : constant := 18; Bit14 : constant := 17;
   Bit15 : constant := 16; Bit16 : constant := 15; Bit17 : constant := 14;
   Bit18 : constant := 13; Bit19 : constant := 12; Bit20 : constant := 11;
   Bit21 : constant := 10; Bit22 : constant := 09; Bit23 : constant := 08;
   Bit24 : constant :=  7; Bit25 : constant := 06; Bit26 : constant := 05;
   Bit27 : constant :=  4; Bit28 : constant := 03; Bit29 : constant := 02;
   Bit30 : constant :=  1; Bit31 : constant := 00;
   pragma Warnings (On);

   type UART_Data_Register is
      record
         FIFO     : Character;
         --  Reading and writing accesses receiver resp. transmitter FIFOs

         Reserved : Reserved_24;
         --  Not used r
      end record;

   for UART_Data_Register use
      record
         Reserved at 0 range Bit31 .. Bit08;
         FIFO     at 0 range Bit07 .. Bit00;
      end record;

   for UART_Data_Register'Size use 32;

   pragma Suppress_Initialization (UART_Data_Register);

   type Reserved_9 is mod 2**9;
   for Reserved_9'Size use 9;

   type UART_Status_Register is
      record
         Data_Ready                       : Boolean;
         Transmitter_Shift_Register_Empty : Boolean;
         Transmitter_FIFO_Empty           : Boolean;
         Break_Received                   : Boolean;
         Overrun                          : Boolean;
         Parity_Error                     : Boolean;
         Framing_Error                    : Boolean;
         Transmitter_FIFO_Half_Full       : Boolean;
         Receiver_FIFO_Half_Full          : Boolean;
         Transmitter_FIFO_Full            : Boolean;
         Receiver_FIFO_Full               : Boolean;
         Reserved                         : Reserved_9;
         Transmitter_FIFO_Count           : FIFO_Count;
         Receiver_FIFO_Count              : FIFO_Count;
      end record;

   for UART_Status_Register use
      record
         Receiver_FIFO_Count              at 0 range Bit31 .. Bit26;
         Transmitter_FIFO_Count           at 0 range Bit25 .. Bit20;
         Reserved                         at 0 range Bit19 .. Bit11;
         Receiver_FIFO_Full               at 0 range Bit10 .. Bit10;
         Transmitter_FIFO_Full            at 0 range Bit09 .. Bit09;
         Receiver_FIFO_Half_Full          at 0 range Bit08 .. Bit08;
         Transmitter_FIFO_Half_Full       at 0 range Bit07 .. Bit07;
         Framing_Error                    at 0 range Bit06 .. Bit06;
         Parity_Error                     at 0 range Bit05 .. Bit05;
         Overrun                          at 0 range Bit04 .. Bit04;
         Break_Received                   at 0 range Bit03 .. Bit03;
         Transmitter_FIFO_Empty           at 0 range Bit02 .. Bit02;
         Transmitter_Shift_Register_Empty at 0 range Bit01 .. Bit01;
         Data_Ready                       at 0 range Bit00 .. Bit00;
      end record;

   for UART_Status_Register'Size use 32;
   pragma Suppress_Initialization (UART_Status_Register);

   UART_1_Data : UART_Data_Register;
   pragma Atomic (UART_1_Data);
   for UART_1_Data'Address use UART_1_Data_Register_Address;

   UART_1_Status : UART_Status_Register;
   pragma Atomic (UART_1_Status);
   for UART_1_Status'Address use UART_1_Status_Register_Address;

   procedure Console_Send (Char : Character);

   ------------------
   -- Console_Send --
   ------------------

   procedure Console_Send (Char    : Character)
   is
      UART_Tx         : constant UART_Data_Register :=
        (FIFO => Char, Reserved => (others => False));

      UART_Status_Aux : UART_Status_Register;
   begin
      --  Send the character through the selected channel by polling
      --  Wait until the port is ready to load data

      loop
         UART_Status_Aux := UART_1_Status;
         exit when not UART_Status_Aux.Transmitter_FIFO_Full;
      end loop;

      --  Send the character once the UART is ready

      UART_1_Data := UART_Tx;

   end Console_Send;

   procedure New_Line is
   begin
      Put (ASCII.LF);
      Put (ASCII.CR);
   end New_Line;

   procedure Put (Item : Character) is
   begin
      Console_Send (Item);
   end Put;

   procedure Put (Item : String) is
   begin
      for C in Item'Range loop
         Put (Item (C));
      end loop;
   end Put;

   procedure Put_Line (Item : String) is
   begin
      Put (Item);
      New_Line;
   end Put_Line;

begin
   --  Initialize_Console;
   null;
end Ada.Text_IO;
