------------------------------------------------------------------------------
--                                                                          --
--                 S Y S T E M . C . M A L L O C . E X T                    --
--                                                                          --
--                                 B o d y
--                                                                          --
--                       Copyright (C) 2011, AdaCore                        --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with System.Storage_Elements;
with Ada.Text_IO;

package body System.C.Malloc.Ext is
   H : constant array (0 .. 15) of Character := "0123456789ABCDEF";
   --  Table of hex digits

   procedure Error (Msg : String);

   -----------
   -- Image --
   -----------

   function Image (A : Address) return Image_String is
      use System.Storage_Elements;
      S : Image_String;
      P : Natural;
      N : Integer_Address;

   begin
      S (S'Last) := '#';
      P := S'Last - 1;
      N := To_Integer (A);
      while P > 3 loop
         S (P) := H (Integer (N mod 16));
         P := P - 1;
         N := N / 16;
      end loop;

      S (1 .. 3) := "16#";
      return S;
   end Image;

   -----------
   -- Error --
   -----------

   procedure Error (Msg : String) is
      use Ada.Text_IO;
   begin
      Put ("!! Heap: ");
      Put_Line (Msg);
   end Error;

   ----------------
   -- Disp_Stats --
   ----------------

   procedure Disp_Stats is
      use Ada.Text_IO;
      Cell  : Cell_Acc;
      Fcell : Free_Cell_Acc;
      Count : Natural;

   begin
      Cell := Last_Cell;
      Count := 0;
      while Cell /= null loop
         Count := Count + 1;
         Cell := Cell.Prev;
      end loop;

      Put_Line ("# blocks: " & Natural'Image (Count));

      Fcell := Free_List;
      Count := 0;
      while Fcell /= null loop
         Count := Count + 1;

         if not Fcell.Cell.Free then
            Error ("corrupted block " & Image (Fcell.all'Address));
         end if;

         Fcell := Fcell.Next_Free;
      end loop;

      Put_Line ("# free blocks: " & Natural'Image (Count));
   end Disp_Stats;

   -----------
   -- Check --
   -----------

   procedure Check is
      use Ada.Text_IO;

   begin
      if Last_Cell = null then
         if Free_List /= null then
            Error ("free_list not empty");
         end if;

         --  Heap is empty

         return;
      end if;

      --  Check chain

      declare
         Prev, Cur : Cell_Acc;

      begin
         Cur := Get_First_Cell;
         Prev := null;
         loop
            if Cur.Prev /= Prev then
               Error ("broken back-link for cell " & Image (Cur.all'Address));
            end if;

            exit when Cur = Last_Cell;
            Prev := Cur;
            Cur := Get_Next_Cell (Cur);
         end loop;
      end;

      --  Free blocks

      declare
         Prev, Cur : Free_Cell_Acc;

      begin
         Prev := null;
         Cur := Free_List;
         while Cur /= null loop
            if Cur.Prev_Free /= Prev then
               Error ("broken free back link for cell "
                        & Image (Cur.all'Address));
            end if;

            if Cur.Prev_Free /= null
              and then Cur.Prev_Free.Next_Free /= Cur
            then
               Error ("broken free next link for cell "
                        & Image (Cur.Prev_Free.all'Address));
            end if;

            if Cur.Prev_Free /= null
              and then Cur.Prev_Free.Cell.Size > Cur.Cell.Size
            then
               Error ("broken order for cell "
                        & Image (Cur.all'Address));
            end if;

            Prev := Cur;
            Cur := Prev.Next_Free;
         end loop;
      end;
   end Check;

   ---------------
   -- Disp_Heap --
   ---------------

   function To_Free_Cell_Acc is new Ada.Unchecked_Conversion
     (Cell_Acc, Free_Cell_Acc);
   function To_Address is new Ada.Unchecked_Conversion
     (Cell_Acc, Address);
   function To_Address is new Ada.Unchecked_Conversion
     (Free_Cell_Acc, Address);

   procedure Disp_Heap is
      use Ada.Text_IO;
      Cell : Cell_Acc;

   begin
      Put_Line ("last cell: " & Image (To_Address (Last_Cell)));
      Put_Line ("first free: " & Image (To_Address (Free_List)));

      if Last_Cell = null then
         Put_Line ("Heap in empty");

      else
         Cell := Get_First_Cell;
         loop
            Put ("block " & Image (To_Address (Cell)));
            Put (", prev=" & Image (To_Address (Cell.Prev)));
            Put (", size=" & size_t'Image (Cell.Size));

            if Cell.Free then
               Put (", free");
               New_Line;
               Put ("  next_free: " & Image
                      (To_Address (To_Free_Cell_Acc (Cell).Next_Free)));
               Put ("  prev_free: " & Image
                      (To_Address (To_Free_Cell_Acc (Cell).Prev_Free)));
            else
               Put (", busy");
            end if;

            New_Line;
            exit when Cell = Last_Cell;
            Cell := Get_Next_Cell (Cell);
         end loop;
      end if;
   end Disp_Heap;
end System.C.Malloc.Ext;
