#!/bin/sh

# You can use ln for debugging (option --link).
CP=cp

verbose=
objdir=install
debug=0
crossdir=cross-dir
libm=""

set -e

copy()
{
    if [ "$verbose" = "y" ]; then
      echo "Copying $1"
    fi
    $CP $1 $2
}

while [ $# -ne 0 ]; do
  case $1 in
      -v) verbose=y ;;
      -d) debug=`expr $debug + 1`;;
      --output=*) objdir=`echo $1 | sed -e "s/^--output=//"`;;
      --cross-dir=*) crossdir=`echo $1 | sed -e "s/^--cross-dir=//"`;;
      --objdir=*) objdir=`echo $1 | sed -e "s/^--objdir=//"`;;
      --libm=*) libm=`echo $1 | sed -e "s/^--libm=//"`;;
      --link) CP="ln -s";;
      -*) echo "unknown option $1"; exit 1;;
      *) break
  esac
  shift
done

if [ $# -ne 2 ]; then
  echo "Usage: $0 [-v] [-o dir] config gnat-src"
  exit 1
fi

config=$1
gnatsrc=$2

if [ -d $objdir ]; then
  echo "Object dir \"$objdir\" already exists"
  exit 1
fi

# Create directories.
mkdir $objdir
mkdir $objdir/arch
mkdir $objdir/common
mkdir $objdir/obj
mkdir $objdir/lib

discarded_sources="none"

prep_arch_files="powerpc/prep/qemu-rom.ld
		 powerpc/prep/qemu-ram.ld
		 powerpc/prep/start-rom.S
		 powerpc/prep/start-ram.S
		 powerpc/prep/setup.S
		 powerpc/prep/s-ioport.ads powerpc/prep/s-ioport.adb"

prep_raven_files="$prep_arch_files
                  powerpc/6xx/s-bbcpsp.ads powerpc/6xx/s-bbcpsp.adb
		  powerpc/6xx/handler.S powerpc/6xx/context_switch.S"

p2020_arch_files="powerpc/p2020/start-ram.S
		  powerpc/p2020/setup.S
		  powerpc/p2020/p2020.ld"

p2020_raven_files="$p2020_arch_files
                   powerpc/spe/s-bbcpsp.ads powerpc/spe/s-bbcpsp.adb
                   powerpc/spe/handler.S powerpc/spe/context_switch.S"

leon_arch_files="$crossdir/bare_board/libbareboard/leon-elf/leon.ld
		 $crossdir/bare_board/libbareboard/leon-elf/crt0.S"

extra_target_pairs=
extra_rts_files=


# Sources for flexible run-times:
# mem_src: memcpy, memset, memcmp, memmove, last_chance_handler
# img_src: s-imgint, s-imglli, s-imgboo, memory
mem_src="memory_set.ads memory_set.adb
	 memory_copy.ads memory_copy.adb
	 memory_compare.ads memory_compare.adb"
lch_src="last_chance_handler.ads last_chance_handler.adb"
img_src="s-imgint.ads s-imgint.adb
	 s-imglli.ads s-imglli.adb
	 s-imgboo.ads s-imgboo.adb"
#	 s-memory.ads s-memory.adb"
textio_src="a-textio.ads a-textio.adb text_io.ads"
textio_pairs="a-textio.ads:a-textio-zfp.ads a-textio.adb:a-textio-zfp.adb
              g-io-put.adb:g-io-put-stextio.adb"

# The RTS which is based on.
# Used to copy file using Makefile.hie
gnat_rts=

# Project file for zfp-support configs
zfp_support_prj_files="$PWD/support-prj/runtime.gpr
        $PWD/support-prj/runtime_build.gpr
        $PWD/support-prj/support.gpr"

case $config in
    "zfp-support/p55-prep")
        gnat_rts=
        # FIXME: use powerpc/prep/a-textio.adb ?
	arch_files="../powerpc-eabispe/qemu/start.s
	           ../powerpc-elf/qemu/c_io.c
		   ../powerpc-elf/qemu/ppc_qemu.ld"
        mem_src="$mem_src $lch_src
                 secondary_stack.ads secondary_stack.adb"
        for f in $zfp_support_prj_files; do copy $f $objdir/; done
	;;
    "zfp/p2020")
        gnat_rts=zfp
        gnat_target=powerpc-eabispe
        arch_files=$p2020_arch_files
        override_path="powerpc"
        mem_src="$mem_src $lch_src
		 s-memory.ads s-memory.adb
                 secondary_stack.ads secondary_stack.adb"
        extra_rts_files="$textio_src s-textio.ads s-textio.adb
                         s-bb.ads
                         s-bbpara.ads
                         s-macres.ads s-macres.adb"
        extra_target_pairs="s-textio.adb:s-textio-p2020.adb
                            s-textio.ads:s-textio-zfp.ads
                            s-bbpara.ads:s-bbpara-p2020.ads
                            s-macres.adb:s-macres-p2020.adb
                            $textio_pairs"
        copy $PWD/powerpc/p2020/runtime.xml $objdir/runtime.xml
        copy $PWD/powerpc/p2020/zfp.gpr $objdir/runtime.gpr
        copy $PWD/powerpc/p2020/zfp_build.gpr $objdir/runtime_build.gpr
        ;;
    "ravenscar-sfp/p2020" )
        gnat_rts=ravenscar-sfp
        gnat_target=powerpc-eabispe
        sed -e "s/zfp/sfp/g" \
            < powerpc/p2020/runtime.xml > $objdir/runtime.xml
        sed -e "s/zfp/sfp/g" \
            < powerpc/p2020/zfp.gpr > $objdir/runtime.gpr
        sed -e "s/zfp/sfp/g" \
            < powerpc/p2020/zfp_build.gpr > $objdir/runtime_build.gpr
        override_path="powerpc/p2020 powerpc src"
        arch_files=$p2020_raven_files
        discarded_sources="s-sssita.ads s-sssita.adb"
        extra_target_pairs="s-textio.adb:s-textio-p2020.adb
                            $textio_pairs
                            s-macres.adb:s-macres-p2020.adb
                            s-multip.adb:s-multip-raven-default.adb
                            s-bbbosu.adb:s-bbbosu-p2020.adb
                            a-intnam.ads:a-intnam-xi-p2020.ads
                            system.ads:system-xi-e500v2-sfp.ads"
        mem_src="$mem_src $lch_src
		 s-memory.ads s-memory.adb"
        extra_rts_files="$textio_src
                         s-macres.ads s-macres.adb"
        ;;
    "ravenscar-full/p2020" )
        gnat_rts=ravenscar
        gnat_target=powerpc-eabispe
        # Ravenscar depends on -lm (for math routines), which also
        # depends on ravenscar (for errno).
        sed -e 's@(\(.*\)/libzfp.a"@(\1/libravenscar.a",\
                \1/libm.a", \1/libravenscar.a"@' \
            < powerpc/p2020/zfp.gpr > $objdir/runtime.gpr
        sed -e 's@(\(.*\)/libzfp.a"@(\1/libravenscar.a",\
                \1/libm.a", \1/libravenscar.a"@' \
            < powerpc/p2020/runtime.xml > $objdir/runtime.xml
        sed -e "s/zfp/ravenscar/g" \
            < powerpc/p2020/zfp_build.gpr > $objdir/runtime_build.gpr
        override_path="powerpc/p2020 powerpc"
        arch_files="$p2020_raven_files
                    src/s-c.ads
                    src/s-cerrno.ads src/s-cerrno.adb
		    src/s-cmallo.ads src/s-cmallo.adb"
        extra_target_pairs="s-textio.adb:s-textio-p2020.adb
                            $textio_pairs
                            s-macres.adb:s-macres-p2020.adb
                            s-multip.adb:s-multip-raven-default.adb
                            s-bbbosu.adb:s-bbbosu-p2020.adb
                            a-intnam.ads:a-intnam-xi-p2020.ads
                            system.ads:system-xi-e500v2-full.ads"
        img_src="" # No need to build img_src as they are already in the RTS.
        extra_rts_files="$textio_src
                         s-macres.ads s-macres.adb"
        if [ "$libm" = "" ]; then
            echo "installation of libm is required (use --libm=)"
            exit 1
        fi
        ;;
    "zfp/prep")
        gnat_rts=zfp
        gnat_target=powerpc-elf
        arch_files=$prep_arch_files
        override_path="powerpc"
        mem_src="$mem_src $lch_src
		 s-memory.ads s-memory.adb
                 secondary_stack.ads secondary_stack.adb"
        extra_rts_files="$textio_src s-textio.ads s-textio.adb
                         s-macres.ads s-macres.adb"
        extra_target_pairs="s-textio.adb:s-textio-prep.adb
                            s-textio.ads:s-textio-zfp.ads
                            s-macres.adb:s-macres-prep.adb
                            $textio_pairs"
        copy $PWD/powerpc/prep/runtime.xml $objdir/runtime.xml
        copy $PWD/powerpc/prep/zfp.gpr $objdir/runtime.gpr
        copy $PWD/powerpc/prep/zfp_build.gpr $objdir/runtime_build.gpr
        ;;
    "ravenscar-sfp/prep" \
  | "ravenscar-sfp/prep-nofpu" )
        gnat_rts=ravenscar-sfp
        gnat_target=powerpc-elf
        if [ "$config" = "ravenscar-sfp/prep" ]; then
          sed -e "s/zfp/sfp/g" \
              < powerpc/prep/runtime.xml > $objdir/runtime.xml
          sed -e "s/zfp/sfp/g" \
              < powerpc/prep/zfp.gpr > $objdir/runtime.gpr
          sed -e "s/zfp/sfp/g" \
              < powerpc/prep/zfp_build.gpr > $objdir/runtime_build.gpr
        else
          sed -e "s/zfp/sfp/g" -e "s/-mhard-float/-msoft-float/g" \
              < powerpc/prep/runtime.xml > $objdir/runtime.xml
          sed -e "s/zfp/sfp/g" -e "s/-mhard-float/-msoft-float/g" \
            < powerpc/prep/zfp.gpr > $objdir/runtime.gpr
          copy $PWD/powerpc/prep/ravenscar_sfp_nofpu_build.gpr \
	     $objdir/runtime_build.gpr
        fi
        override_path="powerpc/sfp powerpc/prep powerpc src"
        arch_files=$prep_raven_files
        discarded_sources="s-sssita.ads s-sssita.adb"
        extra_target_pairs="s-textio.adb:s-textio-prep.adb
                            $textio_pairs
                            s-macres.adb:s-macres-prep.adb
	                    s-multip.adb:s-multip-raven-default.adb
                            a-intnam.ads:a-intnam-xi-ppc.ads
                            s-bbbosu.adb:s-bbbosu-prep.adb"
        mem_src="$mem_src $lch_src
		 s-memory.ads s-memory.adb"
        extra_rts_files="$textio_src
                         s-macres.ads s-macres.adb"
        ;;
    "ravenscar-full/prep" )
        gnat_rts=ravenscar
        gnat_target=powerpc-elf
        # Ravenscar depends on -lm (for math routines), which also
        # depends on ravenscar (for errno).
        sed -e 's@(\(.*\)/libzfp.a"@(\1/libravenscar.a",\
                \1/libm.a", \1/libravenscar.a"@' \
            < powerpc/prep/zfp.gpr > $objdir/runtime.gpr
        sed -e 's@(\(.*\)/libzfp.a"@(\1/libravenscar.a",\
                \1/libm.a", \1/libravenscar.a"@' \
            < powerpc/prep/runtime.xml > $objdir/runtime.xml
        sed -e "s/zfp/ravenscar/g" \
            < powerpc/prep/zfp_build.gpr > $objdir/runtime_build.gpr
        override_path="powerpc/full powerpc/prep powerpc"
        arch_files="$prep_raven_files
                    src/s-c.ads
                    src/s-cerrno.ads src/s-cerrno.adb
		    src/s-cmallo.ads src/s-cmallo.adb"
        extra_target_pairs="s-textio.adb:s-textio-prep.adb
                            $textio_pairs
                            s-macres.adb:s-macres-prep.adb
                            s-multip.adb:s-multip-raven-default.adb
                            a-intnam.ads:a-intnam-xi-ppc.ads
                            s-bbbosu.adb:s-bbbosu-prep.adb"
        img_src="" # No need to build img_src as they are already in the RTS.
        extra_rts_files="$textio_src
                         s-macres.ads s-macres.adb"
        if [ "$libm" = "" ]; then
            echo "installation of libm is required (use --libm=)"
            exit 1
        fi
        ;;
    "ravenscar-sfp/erc32")
        gnat_rts=ravenscar-sfp
        gnat_target=erc32-elf
        arch_files="
		$crossdir/bare_board/libbareboard/erc32-elf/erc32.ld
		$crossdir/bare_board/libbareboard/erc32-elf/sparc.h
		$crossdir/bare_board/libbareboard/erc32-elf/crt0.S"
        mem_src="$mem_src $lch_src"
        copy $PWD/sparc/leon/ravenscar_sfp.gpr $objdir/runtime.gpr
        copy $PWD/sparc/leon/ravenscar_sfp_build.gpr $objdir/runtime_build.gpr
        ;;
    "ravenscar-sfp/leon")
        gnat_rts=ravenscar-sfp
        gnat_target=leon-elf
        arch_files="$leon_arch_files"
        mem_src="$mem_src $lch_src"
        copy $PWD/sparc/leon/ravenscar_sfp.gpr $objdir/runtime.gpr
        copy $PWD/sparc/leon/ravenscar_sfp_build.gpr $objdir/runtime_build.gpr
        ;;
    "ravenscar-full/leon")
        gnat_rts=ravenscar
        gnat_target=leon-elf
        arch_files="$leon_arch_files"
        img_src=""  # No need to build img_src as they are already in the RTS.
        copy $PWD/sparc/leon/ravenscar_sfp.gpr $objdir/runtime.gpr
        copy $PWD/sparc/leon/ravenscar_sfp_build.gpr $objdir/runtime_build.gpr
        ;;
    "ravenscar-sfp/leon3")
        gnat_rts=ravenscar-sfp
        gnat_target=leon3-elf
        arch_files="
		$crossdir/bare_board/libbareboard/leon3-elf/leon.ld
		$crossdir/bare_board/libbareboard/leon-elf/crt0.S"
        mem_src="$mem_src $lch_src"
        copy $PWD/sparc/leon/ravenscar_sfp.gpr $objdir/runtime.gpr
        copy $PWD/sparc/leon/ravenscar_sfp_build.gpr $objdir/runtime_build.gpr
        ;;
    "ravenscar-xtratum/leon3")
        gnat_rts=ravenscar-sfp
        # Using target LEON2 because it should work on both LEON2 and LEON3,
        # and because I would get multiprocessor support when using LEON3
        # which is not what we want on XtratuM where multiprocessors would
        # be handled by the hypervisor.
        gnat_target=leon-elf
        arch_files="
		$crossdir/XtratuM/boot.S
                $crossdir/XtratuM/traps.c
                $crossdir/XtratuM/leon_xtratum.ld"
        discarded_sources="s-bbcaco.adb
                           s-bbcaco.ads
                           s-bbsule.ads
                           s-sssita.ads
                           s-sssita.adb"
        extra_target_pairs="s-textio.adb:s-textio-xtratum.adb
                            s-bbpara.ads:s-bbpara-xtratum-sparc.ads
                            s-bcpiha.adb:s-bcpiha-xtratum-sparc.adb
	                    s-multip.adb:s-multip-raven-default.adb
                            a-intnam.ads:a-intnam-xi-leon3.ads
                            s-bbbosu.adb:s-bbbosu-xtratum.adb
                            g-io-put.adb:g-io-put-xtratum.adb"
        mem_src="" # We want this run time to have the same functionality
        img_src="" # as the Ravenscar SFP one.
        copy $PWD/sparc/xtratum/runtime.xml $objdir/runtime.xml
        copy $PWD/sparc/xtratum/ravenscar_sfp.gpr $objdir/runtime.gpr
        copy $PWD/sparc/xtratum/ravenscar_sfp_build.gpr $objdir/runtime_build.gpr
        ;;
    *)
        echo "error: unknown config \"$config\""
        exit 1
        ;;
esac

# Build list of sources.
if [ "$gnat_rts" != "" ]; then
  defs=`make -f $gnatsrc/Makefile.hie RTS=$gnat_rts TARGET=$gnat_target GNAT_SRC_DIR=$gnatsrc show-sources`

  # Get them.
  eval $defs

  if [ $debug -ge 1 ]; then
    echo "Defs:"
    echo $defs
  fi

  set -e

  sedcmd=""
  for i in $extra_target_pairs $TARGET_PAIRS; do
    sedcmd="$sedcmd -e s:$i:"
  done

  # Create the list of source files.
  all_sources="$LIBGNAT_SOURCES $LIBGNARL_SOURCES $LIBGNAT_NON_COMPILABLE_SOURCES"

  for f in $discarded_sources; do
    all_sources=`echo $all_sources | sed -e "s/$f//"`
  done

  if [ "$verbose" = "y" ]; then
    echo "Sources:"
    echo $all_sources
  fi

  # Copy sources.
  for f in $all_sources; do
    dest=$objdir/common/$f

    override=
    for p in $override_path; do
      if [ -f $p/$f ]; then
          override=$PWD/$p/$f
          dest=$objdir/arch/$f
          break
      fi
    done

    case $f in
       system.ads | s-bbbosu.adb | a-intnam.ads | s-textio.adb)
         dest=$objdir/arch/$f ;;
    esac

    if [ "$override" = "" ]; then
#      if [ "$f" = "s-secsta.ads" ]; then
#          sed -e "/Default_Secondary_Stack_Size : /s/ := .*;/ := 512;/" \
#              < $gnatsrc/$tf > $objdir/$f
#      fi
      override=$gnatsrc/`echo $f | sed $sedcmd`
    fi

    copy $override $dest
  done
fi

# Copy some zfp sources
for f in $mem_src; do
   copy $PWD/../zfp-src/$f $objdir/common/$f
done

for f in $img_src $extra_rts_files; do
   copy $gnatsrc/`echo $f | sed $sedcmd` $objdir/common/$f
done

# Copy arch files
for f in $arch_files; do
   case $f in
       /*) copy $f $objdir/arch/`basename $f`;;
        *) copy $PWD/$f $objdir/arch/`basename $f`;;
   esac
done

# Create rts path files
if [ "$gnat_rts" != "" ]; then
    echo lib > $objdir/ada_object_path
    cat > $objdir/ada_source_path <<EOF
common
arch
EOF
fi

# Copy libm.
if [ "$libm" != "" ]; then
  copy $libm $objdir/lib
fi

exit 0
