#!/bin/sh

set -e

GNAT_SRC_DIR=~/Repositories/adacore/gnat
CROSS_DIR=~/Repositories/adacore/cross
TESTSUITE_DIR=~/Repositories/adacore/gnatmail/testsuites/ravenscar
ACATS_DIR=~/Repositories/adacore/gnatmail/testsuites/acats
LIBM_PATH=~/work/newlib-ppc/powerpc-elf/newlib/libm.a

build_opts=""
src_opts="--link"
flag_tests=n
flag_clean=y
flag_verbose=

build_rts()
{
    prefix=$1
    config=$2
    shift 2
    
    opts=$*

    if [ "$flag_clean" = "y" ]; then
        rm -rf install
        ./build-rts.sh $src_opts ${flag_verbose:+-v} $opts $config $GNAT_SRC_DIR
    fi
    gprbuild --target=$prefix -Pinstall/runtime_build.gpr $build_opts
    chmod a-w install/lib/*.ali
}

build_zfp_example()
{
    prefix=$1

    cd examples/hello-zfp
    rm -f *.ali *.o
    gprbuild --target=${prefix} --RTS=../../install -Phello
    cd ../..
}

build_raven_example()
{
    prefix=$1

    cd examples/tasks-sfp
    rm -f *.ali *.o
    ${prefix}-gnatmake -Phello_sfp
    cd ../..
}

raven_testsuite()
{
    config=$1

    cmd="./run-tests.sh $config $TESTSUITE_DIR"

    if [ "$flag_verbose" != "" ]; then
        echo $cmd
    fi
    eval $cmd
}

do_zfp()
{
    prefix=$1
    config=$2
    shift 2
    
    opts=$*

    build_rts $prefix $config $opts

    build_zfp_example $prefix
}

do_ravenscar()
{
    prefix=$1
    config=$2
    shift 2
    
    opts=$*

    build_rts $prefix $config $opts

    build_zfp_example $prefix
    build_raven_example $prefix

    if [ "$flag_tests" = "y" ]; then
        raven_testsuite $config
    fi
}

build_sfp_prep_nofpu()
{
    do_ravenscar powerpc-elf ravenscar-sfp/prep-nofpu
}

build_sfp_prep()
{
    do_ravenscar powerpc-elf ravenscar-sfp/prep
}

build_full_prep()
{
    do_ravenscar powerpc-elf ravenscar-full/prep --libm=$LIBM_PATH
}

build_zfp_prep()
{
    do_zfp powerpc-elf zfp/prep
}

build_zfp_p2020()
{
    do_zfp powerpc-eabispe zfp/p2020
}

build_sfp_p2020()
{
    do_ravenscar powerpc-eabispe ravenscar-sfp/p2020
}

build_full_p2020()
{
    do_ravenscar powerpc-eabispe ravenscar-full/p2020 --libm=$LIBM_PATH
}

build_sfp_leon()
{
    do_ravenscar sparc-elf ravenscar-sfp/leon --cross-dir=$CROSS_DIR
}

build_full_leon()
{
    do_ravenscar sparc-elf ravenscar-full/leon --cross-dir=$CROSS_DIR
}

build_zfp_support_prep()
{
    do_zfp powerpc-elf zfp-support/prep
}

build_stub()
{
  prefix=$1

  rm -rf stub
  mkdir stub
  cd stub
  ln -s ../powerpc/gdbstub/*.ad? .
  ln -s ../powerpc/gdbstub/*.S .
  cat > stub.gpr <<EOF
project Stub extends "../install/runtime" is
  for Languages use ("Ada", "Asm_Cpp");
  for Source_Dirs use (".");
  package Compiler is
     for Switches ("Ada") use Runtime.Compiler'Default_Switches ("Ada")
         & ("-gnatyg", "-gnatwn", "-g");
  end Compiler;

end Stub;
EOF
  gprbuild --target=$prefix -Pstub stub
}

run_acats_full()
{
    rm -rf acats
    mkdir acats
    cd acats
    $ACATS_DIR/run_acats_test.py -d stack,ravenscar --rts=ravenscar --target=ppc-elf,,qemu --project=$PWD/../install/runtime.gpr -a $ACATS_DIR
}

if [ $# -eq 0 ]; then
  #build_leon_sfp
  #build_sfp_prep_nofpu
  #build_zfp_prep
  #build_stub powerpc-elf

  #build_sfp_prep
  #build_zfp_support_prep
  build_full_prep
else
  target=prep

  # Parse options.
  while [ $# -gt 0 ]; do
      case $1 in
          -g) build_opts="$build_opts -g";;
          -t) flag_tests=y;;
          -v) flag_verbose="-v";;
          --no-clean) flag_clean=n;;
          --target=*) target=`echo $1 | sed -e s/--target=//` ;;
          -*) echo "unhandled option $1"; exit 2;;
          *) break;;
      esac
      shift
  done

  # Parse commands
  for opt; do
      case $target in
          prep)
              case $opt in
                  full) build_full_prep ;;
                  sfp)  build_sfp_prep ;;
                  zfp)  build_zfp_prep ;;
                  stub) build_stub powerpc-elf ;;
                  raven-sfp) raven_testsuite ravenscar-sfp/prep ;;
                  raven-full) raven_testsuite ravenscar-full/prep ;;
                  acats) run_acats_full ;;
                  gdbstub) build_stub powerpc-elf ;;
                  *) echo "Unknown command $opt for target $target"; exit 2;;
              esac
              ;;
          p2020)
              case $opt in
                  sfp)  build_sfp_p2020 ;;
                  full)  build_full_p2020 ;;
                  zfp)  build_zfp_p2020 ;;
                  raven-sfp) raven_testsuite ravenscar-sfp/p2020 ;;
                  raven-full) raven_testsuite ravenscar-full/p2020 ;;
                  *) echo "Unknown command $opt for target $target"; exit 2;;
              esac
              ;;
          leon)
              case $opt in
                  sfp) build_sfp_leon ;;
                  full) build_full_leon ;;
                  raven-sfp) raven_testsuite ravenscar-sfp/leon ;;
                  *) echo "Unknown command $opt for target $target"; exit 2;;
              esac
              ;;
          *)
              echo "Unknown target $target (command $opt)"; exit 2
              ;;
      esac
  done
fi
