------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--                         S Y S T E M . B B . T I M E                      --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--        Copyright (C) 1999-2002 Universidad Politecnica de Madrid         --
--             Copyright (C) 2003-2005 The European Space Agency            --
--                     Copyright (C) 2003-2011, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
-- The porting of GNARL to bare board  targets was initially  developed  by --
-- the Real-Time Systems Group at the Technical University of Madrid.       --
--                                                                          --
------------------------------------------------------------------------------

pragma Restrictions (No_Elaboration_Code);

with System.BB.Interrupts;
with System.BB.Peripherals;
with System.BB.Protection;
with System.BB.Threads.Queues;
with System.BB.Timing_Events;
with System.BB.CPU_Primitives;
with System.Machine_Code; use System.Machine_Code;

package body System.BB.Time is

   use Peripherals;
   use System.Multiprocessors;

   --  We use two timers with the same frequency:
   --     A Periodic Timer for the clock
   --     An Alarm Timer for delays

   -----------------------
   -- Local definitions --
   -----------------------

   type Unsigned_32 is mod 2 ** 32;
   for Unsigned_32'Size use 32;
   --  Values of this type represent number of times that the clock finishes
   --  its countdown. This type should allow atomic reads and updates.

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Alarm_Handler (Interrupt : Interrupts.Interrupt_ID);
   --  Handler for the alarm interrupt

   procedure Set_DEC (Ticks : Unsigned_32);
   pragma Inline (Set_DEC);
   --  Set the decrementer register

   function Read_TBL return Unsigned_32;
   pragma Inline (Read_TBL);
   --  Read the Time Base Lower word.

   function Read_TBU return Unsigned_32;
   pragma Inline (Read_TBU);
   --  Read the Time Base Upper word.

   --------------
   -- Read_TBL --
   --------------

   function Read_TBL return Unsigned_32 is
      Res : Unsigned_32;
   begin
      Asm ("mftbl %0",
        Outputs => Unsigned_32'Asm_Output ("=r", Res),
        Volatile => True);

      return Res;
   end Read_TBL;

   --------------
   -- Read_TBU --
   --------------

   function Read_TBU return Unsigned_32 is
      Res : Unsigned_32;
   begin
      Asm ("mftbu %0",
        Outputs => Unsigned_32'Asm_Output ("=r", Res),
        Volatile => True);

      return Res;
   end Read_TBU;

   ---------------
   -- Set_Alarm --
   ---------------

   procedure Set_DEC (Ticks : Unsigned_32) is
   begin
      Asm ("mtdec %0",
        Inputs => Unsigned_32'Asm_Input ("r", Ticks),
        Volatile => True);
   end Set_DEC;

   procedure Update_Alarm (Alarm : Time) is
      Max_Timer_Interval : constant Unsigned_32 := 16#FFFF_FFFF#;
      --  The maximum value that can be set in the DEC register.

      Diff : constant Time := Time'Max (0, Alarm - Clock);
      --  Alarms in the past are triggered now.

      Dec : Unsigned_32;
   begin

      --  Check whether the alarm time is within the DEC period

      if Diff <= Time (Max_Timer_Interval) then
         Dec := Unsigned_32 (Diff);
      else
         Dec := Max_Timer_Interval;
      end if;

      Set_DEC (Dec);
   end Update_Alarm;

   -------------------
   -- Alarm_Handler --
   -------------------

   procedure Alarm_Handler (Interrupt : Interrupts.Interrupt_ID) is
      pragma Unreferenced (Interrupt);

      Now             : Time;
      Wakeup_Thread   : Threads.Thread_Id;

      use type Threads.Thread_States;

   begin
      Now := Clock;

      --  The access to the queues must be protected

      Protection.Enter_Kernel;

      --  Execute expired events of the current CPU

      Timing_Events.Execute_Expired_Timing_Events (Now);

      --  Extract all the threads whose delay has expired

      while Threads.Queues.Get_Next_Alarm_Time (CPU'First) <= Now loop

         --  Extract the task(s) that was waiting in the alarm queue and insert
         --  it in the ready queue.

         Wakeup_Thread := Threads.Queues.Extract_First_Alarm;

         --  We can only awake tasks that are delay statement

         pragma Assert (Wakeup_Thread.State = Threads.Delayed);

         Wakeup_Thread.State := Threads.Runnable;

         Threads.Queues.Insert (Wakeup_Thread);
      end loop;

      Update_Alarm (Get_Next_Timeout (CPU'First));

      --  We have finished the modifications to the queues

      Protection.Leave_Kernel;
   end Alarm_Handler;

   -----------
   -- Clock --
   -----------

   function Clock return Time is
      Lo : Unsigned_32;
      Hi, Hi1 : Unsigned_32;
   begin
      --  We can't atomically read the 64-bits counter.  So check that the
      --  32 MSB don't change.

      Hi := Read_TBU;
      loop
         Lo := Read_TBL;
         Hi1 := Read_TBU;
         exit when Hi = Hi1;
         Hi := Hi1;
      end loop;

      return (Time (Hi) * 2 ** 32) + Time (Lo);
   end Clock;

   -----------------
   -- Delay_Until --
   -----------------

   procedure Delay_Until (T : Time) is
      Now               : Time;
      Self              : Threads.Thread_Id;
      Inserted_As_First : Boolean;
   begin
      Protection.Enter_Kernel;

      Now := Clock;

      Self := Threads.Thread_Self;

      --  Test if the alarm time is in the future

      if T > Now then

         --  Extract the thread from the ready queue. When a thread wants to
         --  wait for an alarm it becomes blocked.

         Self.State := Threads.Delayed;

         Threads.Queues.Extract (Self);

         --  Insert Thread_Id in the alarm queue (ordered by time) and if it
         --  was inserted at head then check if Alarm Time is closer than the
         --  next clock interrupt.

         Threads.Queues.Insert_Alarm (T, Self, Inserted_As_First);

         if Inserted_As_First then
            Update_Alarm (T);
         end if;

      else
         --  If alarm time is not in the future, the thread must yield the CPU

         Threads.Queues.Yield (Self);
      end if;

      Protection.Leave_Kernel;
   end Delay_Until;

   ----------------------
   -- Get_Next_Timeout --
   ----------------------

   function Get_Next_Timeout (CPU_Id : CPU) return Time is
      Alarm_Time : constant Time :=
        Threads.Queues.Get_Next_Alarm_Time (CPU_Id);
      Event_Time : constant Time := Timing_Events.Get_Next_Timeout (CPU_Id);
   begin

      if Alarm_Time <= Event_Time then
         return Alarm_Time;
      else
         return Event_Time;
      end if;
   end Get_Next_Timeout;

   -----------------------
   -- Initialize_Timers --
   -----------------------

   procedure Initialize_Timers is
   begin
      --  Install alarm handler

      CPU_Primitives.Install_Exception_Handler
        (Alarm_Handler'Address,
         CPU_Primitives.Decrementer_Excp);
   end Initialize_Timers;

end System.BB.Time;
