------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--               S Y S T E M . B B . C P U _ P R I M I T I V E S            --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--        Copyright (C) 1999-2002 Universidad Politecnica de Madrid         --
--             Copyright (C) 2003-2004 The European Space Agency            --
--                     Copyright (C) 2003-2010, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains the primitives which are dependent on the
--  underlying processor.

pragma Restrictions (No_Elaboration_Code);

with System;
with System.BB.Parameters;

package System.BB.CPU_Primitives is
   pragma Preelaborate;

   ------------------------
   -- Context management --
   ------------------------

   --  The context buffer is an abstract type that holds all values, indexed
   --  by Context_Id,  that makes up a thread's state and is not otherwise
   --  stored in main memory. This typically includes all user-visible
   --  registers, and possibly some other status as well.

   --  In case different contexts have different amounts of state (for example,
   --  due to absence of a floating-point unit in a particular configuration,
   --  or just the FPU not being used), it is expected that these details
   --  are handled in the implementation, which  should ignore updates of
   --  unsupported state and return a default value for queries of such state.

   type Context_Buffer is private;

   procedure Context_Switch;
   pragma Inline (Context_Switch);
   --  Perform the context switch between the running_thread and the
   --  first_thread. The value of running_thread will be updated.

   procedure Initialize_Context
     (Buffer          : not null access Context_Buffer;
      Program_Counter : System.Address;
      Argument        : System.Address;
      Stack_Pointer   : System.Address);
   pragma Inline (Initialize_Context);
   --  Initialize_Context inserts inside the context buffer the
   --  default values for each register. The values for the stack
   --  pointer, the program counter, and argument to be passed
   --  are provided as arguments.

   ---------------------------------
   -- Interrupt and trap handling --
   ---------------------------------

   type Vector_Id is range 0 .. 16#2fff#;

   External_Interrupt_Excp : constant Vector_Id := 16#500#;
   Decrementer_Excp : constant Vector_Id := 16#900#;

   procedure Install_Error_Handlers;
   pragma Inline (Install_Error_Handlers);
   --  Called at system initialization time to install a CPU specific
   --  trap handler, GNAT_Error_Handler, that converts synchronous traps
   --  to appropriate exceptions.

   procedure Install_Exception_Handler
     (Service_Routine : System.Address;
      Vector          : Vector_Id);
   --  Install a new handler in the exception table.

   procedure Disable_Interrupts;
   pragma Inline (Disable_Interrupts);
   --  All external interrupts (asynchronous traps) are disabled

   procedure Enable_Interrupts (Level : System.BB.Parameters.Interrupt_Level);
   pragma Inline (Enable_Interrupts);
   --  Interrupts are enabled if they are above the value given by Level

   procedure Initialize_Floating_Point;
   pragma Inline (Initialize_Floating_Point);
   --  Install the floating point trap handler in charge of performing
   --  floating point context switches.

private
   type Context_Buffer is record
      --  Only callee-saved registers need to be saved.
      R1  : System.Address;
      R2  : System.Address;
      R13 : System.Address;
      R14 : System.Address;
      R15 : System.Address;
      R16 : System.Address;
      R17 : System.Address;
      R18 : System.Address;
      R19 : System.Address;
      R20 : System.Address;
      R21 : System.Address;
      R22 : System.Address;
      R23 : System.Address;
      R24 : System.Address;
      R25 : System.Address;
      R26 : System.Address;
      R27 : System.Address;
      R28 : System.Address;
      R29 : System.Address;
      R30 : System.Address;
      R31 : System.Address;

      CR  : System.Address;
      LR  : System.Address;

      F14 : Long_Float;
      F15 : Long_Float;
      F16 : Long_Float;
      F17 : Long_Float;
      F18 : Long_Float;
      F19 : Long_Float;
      F20 : Long_Float;
      F21 : Long_Float;
      F22 : Long_Float;
      F23 : Long_Float;
      F24 : Long_Float;
      F25 : Long_Float;
      F26 : Long_Float;
      F27 : Long_Float;
      F28 : Long_Float;
      F29 : Long_Float;
      F30 : Long_Float;
      F31 : Long_Float;
      FPSCR : Long_Long_Integer;
   end record;

   for Context_Buffer'Alignment use 8;
   --  This array contains all the registers that the thread needs to save
   --  within its thread descriptor. Using double word boundaries allows us
   --  to use double word loads and stores safely in the context switch.

end System.BB.CPU_Primitives;
