------------------------------------------------------------------------------
--                        S Y S T E M .  I 8 2 5 9                          --
--                                                                          --
--                    Copyright (C) 2010-2011, AdaCore                      --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
------------------------------------------------------------------------------

--  PC/AT-style I8259 interrupt controler (PIC)

with System.IOPorts; use System.IOPorts;
with Interfaces; use Interfaces;

package body System.I8259 is

   Master_Mask : Unsigned_8;
   Slave_Mask : Unsigned_8;

   Master_Pic_Port : constant Port_Id := 16#20#;
   Slave_Pic_Port  : constant Port_Id := 16#a0#;

   procedure Initialize is
   begin
      --  Master PIC.

      --  ICW1: ICW4 needed, cascade mode, level-triggered.
      Outb (Master_Pic_Port + 0, 2#0001_1001#);
      --  ICW2: Vector 0-7
      Outb (Master_Pic_Port + 1, 16#00#);
      --  ICW3: (master): slave on int 2
      Outb (Master_Pic_Port + 1, 2#0000_0100#);
      --  ICW4: 8086 mode, normal EOI, buffered mode/master, not special mode
      Outb (Master_Pic_Port + 1, 2#0000_1100#);

      --  Slave PIC.

      --  ICW1: ICW4 needed, cascade mode, level-triggered.
      Outb (Slave_Pic_Port + 0, 2#0001_1001#);
      --  ICW2: Vector 8-15
      Outb (Slave_Pic_Port + 1, 16#08#);
      --  ICW3: (slave): slave id 2.
      Outb (Slave_Pic_Port + 1, 16#02#);
      --  ICW4: 8086 mode, normal EOI, buffered mode/slave, not special mode
      Outb (Slave_Pic_Port + 1, 2#0000_1000#);

      --  Mask all interrupts.
      Master_Mask := 2#1111_1011#;
      Slave_Mask := 2#1111_1111#;
      Outb (Master_Pic_Port + 1, Master_Mask);
      Outb (Slave_Pic_Port + 1, Slave_Mask);
   end Initialize;

   procedure Mask (Id : Int_Id) is
   begin
      case Id is
         when Slave_Int_Id =>
            Slave_Mask :=
              Slave_Mask or (2 ** Natural (Id - Slave_Int_Id'First));
            Outb (Slave_Pic_Port + 1, Slave_Mask);
         when Master_Int_Id =>
            Master_Mask :=
              Master_Mask or (2 ** Natural (Id - Master_Int_Id'First));
            Outb (Master_Pic_Port + 1, Master_Mask);
      end case;
   end Mask;

   procedure Unmask (Id : Int_Id) is
   begin
      case Id is
         when Slave_Int_Id =>
            Slave_Mask :=
              Slave_Mask and not (2 ** Natural (Id - Slave_Int_Id'First));
            Outb (Slave_Pic_Port + 1, Slave_Mask);
         when Master_Int_Id =>
            Master_Mask :=
              Master_Mask and not (2 ** Natural (Id - Master_Int_Id'First));
            Outb (Master_Pic_Port + 1, Master_Mask);
      end case;
   end Unmask;

   function Get_Int_Id return Int_Id is
      Intack          : Unsigned_8;
      for Intack'Address use System'To_Address (16#BFFFFFF0#);
      pragma Atomic (Intack);
      pragma Import (Ada, Intack);
   begin
      return Int_Id (Intack);
   end Get_Int_Id;

   procedure EOI (Id : Int_Id) is
   begin
      if Id in Slave_Int_Id then
         Outb (Slave_Pic_Port + 1, 2#0010_0000#);
      end if;
      Outb (Master_Pic_Port + 1, 2#0010_0000#);
   end EOI;
end System.I8259;
