------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--                   S Y S T E M . B B . I N T E R R U P T S                --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--        Copyright (C) 1999-2002 Universidad Politecnica de Madrid         --
--             Copyright (C) 2003-2005 The European Space Agency            --
--                     Copyright (C) 2003-2011, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
-- The porting of GNARL to bare board  targets was initially  developed  by --
-- the Real-Time Systems Group at the Technical University of Madrid.       --
--                                                                          --
------------------------------------------------------------------------------

pragma Restrictions (No_Elaboration_Code);

with System.Storage_Elements;
with System.BB.CPU_Primitives;
with System.BB.Threads;
with System.BB.Threads.Queues;
with System.BB.Peripherals;
with System.I8259;

package body System.BB.Interrupts is

   External_Interrupt : constant Interrupt_ID := 1;
   --  Same as Ada.Interrupts.Names.External_Interrupt

   External_Handler : Interrupt_Handler;
   --  Handler for the external interrupt

   Interrupt_Being_Handled : Interrupt_ID := No_Interrupt;
   pragma Atomic (Interrupt_Being_Handled);
   --  Interrupt_Being_Handled contains the interrupt currently being
   --  handled if any. It is equal to No_Interrupt when no interrupt
   --  is handled. Its value is updated by the trap handler.

   procedure Interrupt_Wrapper;
   --  Low level interrupt handler

   --------------------
   -- Attach_Handler --
   --------------------

   procedure Attach_Handler (Handler : Interrupt_Handler; Id : Interrupt_ID) is
   begin
      --  Check that we are attaching to a real interrupt

      pragma Assert (Id = External_Interrupt);

      External_Handler := Handler;

      --  Transform the interrupt level to the place in the interrupt vector
      --  table. Then insert the wrapper for the interrupt handlers in the
      --  underlying vector table.

      CPU_Primitives.Install_Exception_Handler
        (Interrupt_Wrapper'Address, CPU_Primitives.External_Interrupt_Excp);
   end Attach_Handler;

   -----------------------
   -- Current_Interrupt --
   -----------------------

   function Current_Interrupt return Interrupt_ID is
   begin
      return Interrupt_Being_Handled;
   end Current_Interrupt;

   -----------------------
   -- Interrupt_Wrapper --
   -----------------------

   procedure Interrupt_Wrapper
   is
      use System.I8259;

      Irq_Num         : Int_Id;
      Self_Id         : constant Threads.Thread_Id := Threads.Thread_Self;
      Caller_Priority : constant Any_Priority :=
        Threads.Get_Priority (Self_Id);
      Interrupt       : constant Interrupt_ID := External_Interrupt;
      Previous_Int    : constant Interrupt_ID := Interrupt_Being_Handled;
   begin
      --  Store the interrupt being handled

      Interrupt_Being_Handled := Interrupt;

      Irq_Num := Get_Int_Id;

      --  Then, we must set the appropriate software priority corresponding
      --  to the interrupt being handled. It comprises also the appropriate
      --  interrupt masking.

      Threads.Queues.Change_Priority
        (Self_Id, Priority_Of_Interrupt (Interrupt));

      CPU_Primitives.Enable_Interrupts (Interrupt);

      --  Call the user handler

      External_Handler.all (Interrupt);

      CPU_Primitives.Disable_Interrupts;

      --  Restore the software priority to the state before the interrupt
      --  happened. Interrupt unmasking is not done here (it will be done
      --  later by the interrupt epilogue).

      Threads.Queues.Change_Priority (Self_Id, Caller_Priority);

      --  Restore the interrupt that was being handled previously (if any)

      Interrupt_Being_Handled := Previous_Int;

      EOI (Irq_Num);
   end Interrupt_Wrapper;

   ---------------------------
   -- Priority_Of_Interrupt --
   ---------------------------

   function Priority_Of_Interrupt
     (Level : System.BB.Parameters.Interrupt_Level) return System.Any_Priority
   is
   begin
      --  Assert that it is a real interrupt

      pragma Assert (Level /= 0);

      return (Any_Priority (Level) + Interrupt_Priority'First - 1);
   end Priority_Of_Interrupt;

   ----------------------------
   -- Within_Interrupt_Stack --
   ----------------------------

   function Within_Interrupt_Stack
     (Stack_Address : System.Address) return Boolean
   is
      pragma Unreferenced (Stack_Address);
   begin
      --  There is no dedicated interrupt stack.
      return False;
   end Within_Interrupt_Stack;

   ---------------------------
   -- Initialize_Interrupts --
   ---------------------------

   procedure Initialize_Interrupts is
   begin
      null;
   end Initialize_Interrupts;

end System.BB.Interrupts;
