------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--                  S Y S T E M . B B . P E R I P H E R A L S               --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--        Copyright (C) 1999-2002 Universidad Politecnica de Madrid         --
--             Copyright (C) 2003-2005 The European Space Agency            --
--                     Copyright (C) 2003-2010, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
-- The porting of GNARL to bare board  targets was initially  developed  by --
-- the Real-Time Systems Group at the Technical University of Madrid.       --
--                                                                          --
------------------------------------------------------------------------------

with System.BB.Peripherals.MPC860; use System.BB.Peripherals.MPC860;
with System.BB.Time;
with System.Storage_Elements;      use System.Storage_Elements;
with System.Machine_Code;          use System.Machine_Code;

with Ada.Unchecked_Conversion;

package body System.BB.Peripherals is
   use CPU_Primitives;

   --  Constants defining to which level an external interrupt is programmed
   --  There are the levels 0 to 7.

   --  If possible avoid level 7 since it is the default value of
   --  SIVEC[INTC] and could lead to spurious interrupt detection.

   External_Level_Of_Periodic_Clock :
     constant SBP.External_Interrupt_Level := 3;

   External_Level_Of_UART_Interrupt :
     constant SBP.External_Interrupt_Level := 4;

   --  Constants defining the external interrupt levels

   --  The decrementer is set above the external interrupts
   --  but IRQ0, therefore it gets the ID 1.

   --  Interrupt_ID    := 1 + Interrupt_Number
   --  Interrup_Number := 1 + 2 * External_Interrupt_Level

   --  Interr_ID        type
   --  ---------        ----

   --    0              IRQ0
   --    1              Decrementer
   --    2              External_Level0
   --    3              IRQ1
   --    4              External_Level1
   --    5              IRQ2
   --    6              External_Level2
   --    7              IRQ3
   --    8              External_Level3
   --    9              IRQ4
   --   10              External_Level4
   --   11              IRQ5
   --   12              External_Level5
   --
   --   ...             ...
   --
   --   15              IRQ7
   --   16              External_Level7
   --
   --   17-32           Reserved

   --  If something here is changed other functions have to be adapted
   --  like To_Interrupt!

   Decrementer : constant SBP.Interrupt_Level := 1;

   Periodic_Clock : constant SBP.Interrupt_Level :=
      1 + 1 + 2 * External_Level_Of_Periodic_Clock;

   External_Interrupt : constant SBP.Interrupt_Level :=
      1 + 1 + 2 * External_Level_Of_UART_Interrupt;

   ----------
   -- UART --
   ----------

   UART_Tx_Buffer : array (0 .. 0) of Character;
   for UART_Tx_Buffer'Address use 16#2300#;
   --  Transmission buffer

   UART_Tx_BD_Base : constant := 16#2200#;
   UART_Rx_BD_Base : constant := 16#2210#;
   --  Base Address from IMMR (= 0xFF00_0000) for the transmission
   --  and reception buffer descriptors

   UART_Tx_Buffer_Descriptor        : TxBD;
   for UART_Tx_Buffer_Descriptor'Address
     use System'To_Address (IMMR + UART_Tx_BD_Base);
   --  Buffer Descriptors for transmission and reception

   UART_Rx_Buffer_Descriptor        : RxBD;
   for UART_Rx_Buffer_Descriptor'Address
     use System'To_Address (IMMR + UART_Rx_BD_Base);

   ----------------------
   -- Local Procedures --
   ----------------------

   procedure Stop_Watch_Dog;
   pragma Inline (Stop_Watch_Dog);
   --  Stop the watch dog timer

   procedure Initialize_Memory;
   pragma Inline (Initialize_Memory);
   --  Initialize the memory on the board

   procedure Initialize_Clock;
   --  Perform all the initialization related to the clock

   --  procedure Initialize_Machine_Status_Register;
   --  Sets the MSR with default values

   procedure Initialize_Timebase;
   --  Enable the timebase and enable stopping timebase when freeze

   procedure Initialize_UART;
   --  Procedure to initialize the UART port

   procedure UART_Send (Char : Character);
   --  Procedure to send characters to the first UART channel

   ------------------------
   -- Alarm_Interrupt_ID --
   ------------------------

   function Alarm_Interrupt_ID return Interrupts.Interrupt_ID is
   begin
      return Decrementer;
   end Alarm_Interrupt_ID;

   --------------------------
   -- Cancel_And_Set_Alarm --
   --------------------------

   procedure Cancel_And_Set_Alarm (Ticks : Timer_Interval) renames Set_Alarm;
   --  This procedure cancel a previous alarm and set a new one.
   --  Setting a new alarm cancel the previous one in this target
   --  So Cancel_And_Set_Alarm and Set_Alarm are identical.

   ---------------------------
   -- Clear_Alarm_Interrupt --
   ---------------------------

   procedure Clear_Alarm_Interrupt is
   begin
      --  Interrupts are cleared automatically when they are acknowledged

      null;
   end Clear_Alarm_Interrupt;

   ---------------------------
   -- Clear_Clock_Interrupt --
   ---------------------------

   procedure Clear_Clock_Interrupt is
   begin
      --  Interrupts are cleared automatically when they are acknowledged

      null;
   end Clear_Clock_Interrupt;

   ------------------------
   -- Clock_Interrupt_ID --
   ------------------------

   function Clock_Interrupt_ID return Interrupts.Interrupt_ID is
   begin
      return Periodic_Clock;
   end Clock_Interrupt_ID;

   ----------------------
   -- Initialize_Board --
   ----------------------

   procedure Initialize_Board is
   begin
      Stop_Watch_Dog;
      Initialize_Memory;
      Initialize_Clock;
      Initialize_Timebase;
   end Initialize_Board;

   -------------------------
   -- Initialize_Timebase --
   -------------------------

   procedure Initialize_Timebase is
   begin
      --  Set clock source for timebase

      SCCR.TBS := True;

      --  Enable freeze

      TBSCR.Timebase_Freeze_Enable := True;

      --  Enable timebase

      TBSCR.Timebase_Enable        := True;
   end Initialize_Timebase;

   ----------------------
   -- Initialize_Clock --
   ----------------------

   procedure Initialize_Clock is
   begin
      null;
   end Initialize_Clock;

   ------------------------
   -- Initialize_Console --
   ------------------------

   procedure Initialize_Console is
   begin
      --  Initialize the UART as output console

      Initialize_UART;
   end Initialize_Console;

   -----------------------
   -- Initialize_Memory --
   -----------------------

   procedure Initialize_Memory is
   begin
      --  Nothing to be done

      null;
   end Initialize_Memory;

   ---------------------
   -- Initialize_UART --
   ---------------------

   procedure Initialize_UART is
   begin
      --  For now the UART is fixed to 9600 baud, 8 data bits, no parity and
      --  1 stop bit.

      --  Configure Port B.
      --  Enable pins SMTXD1 and SMRXD1.

      PBDIR_Mirror := PBDIR;
      PBDIR_Mirror (24) := False;
      PBDIR_Mirror (25) := False;
      PBDIR := PBDIR_Mirror;

      PBPAR_Mirror := PBPAR;
      PBPAR_Mirror (24) := True;
      PBPAR_Mirror (25) := True;
      PBPAR := PBPAR_Mirror;

      PBDAT_Mirror := PBDAT;
      PBDAT_Mirror (14) := True;
      PBDAT_Mirror (15) := True;
      PBDAT_Mirror (16) := True;
      PBDAT_Mirror (17) := True;
      PBDAT_Mirror (18) := True;
      PBDAT_Mirror (22) := True;
      PBDAT_Mirror (23) := True;
      PBDAT_Mirror (24) := True;
      PBDAT_Mirror (25) := True;
      PBDAT_Mirror (26) := True;
      PBDAT_Mirror (27) := True;
      PBDAT_Mirror (28) := True;
      PBDAT_Mirror (29) := True;
      PBDAT_Mirror (30) := True;
      PBDAT_Mirror (31) := True;
      PBDAT := PBDAT_Mirror;

      PBODR_Mirror := PBODR;
      PBODR_Mirror (24) := False;
      PBODR_Mirror (25) := False;
      PBODR := PBODR_Mirror;

      --  Configure BRG1.
      --  The DIV16 bit is not used and the divider is 325.
      --  The resulting BRG1 clock is 16x the preferred bit rate.

      BRGC1_Mirror := BRGC1;

      BRGC1_Mirror.Reset            := False;
      BRGC1_Mirror.Enable           := True;
      BRGC1_Mirror.External_Clk_Src := 2#00#;
      BRGC1_Mirror.Autobaud         := False;
      BRGC1_Mirror.Clock_Divider    := 325;
      BRGC1_Mirror.DIV16            := False;

      BRGC1 := BRGC1_Mirror;

      --  Connect BRG1 to SMC1 using SI.
      --  Clear SIMODE[SMC1, SMC1CS].

      SIMODE_Mirror := SIMODE;
      SIMODE_Mirror.SMC1   := False;  --  enable NMSI mode
      SIMODE_Mirror.SMC1CS := 000;    --  Clock source = brg1
      SIMODE_Mirror.SMC2   := False;  --  enable NMSI mode
      SIMODE_Mirror.SMC2CS := 000;    --  Clock source = brg1
      SIMODE_Mirror.RFSDa := 2#00#;   --  no delay
      SIMODE_Mirror.RFSDb := 2#00#;   --  no delay
      SIMODE := SIMODE_Mirror;

      --  Write CPCR to execute the Restart Tx

      --  This command updates RBPTR and TBPTR of the serial channel with the
      --  new values of RBASE and TBASE.

      --  SMC1 Parameters

      --  disable Tx and Rx before changing SMC1 parameters

      SMCMR1_Mirror := SMCMR1;
      SMCMR1_Mirror.SMC_Transmit_Enable := False;
      SMCMR1_Mirror.SMC_Receive_Enable := False;
      SMCMR1 := SMCMR1_Mirror;

      SMC1_Parameter_Memory_Map.RBASE := Natural (UART_Rx_BD_Base);
      SMC1_Parameter_Memory_Map.TBASE := Natural (UART_Tx_BD_Base);

      --  When issuing a command we need to set FLG and wait for
      --  the CP to clear it before we continue.

      CPCR_Mirror        := CPCR;
      CPCR_Mirror.RST    := False;
      CPCR_Mirror.OPCODE := 2#0000#;  --  Init Rx and Tx Params
      CPCR_Mirror.FLG    := True;
      CPCR_Mirror.CH_NUM := 2#1001#;  --  SMC1
      CPCR := CPCR_Mirror;

      --  Spin until flag is cleared

      while CPCR.FLG loop
         null;
      end loop;

      --  SMC1 Parameters

      SMC1_Parameter_Memory_Map.RFCR.Byte_Ordering := 2#10#;
      SMC1_Parameter_Memory_Map.RFCR.Address_Type  := 0;
      SMC1_Parameter_Memory_Map.TFCR.Byte_Ordering := 2#10#;
      SMC1_Parameter_Memory_Map.TFCR.Address_Type  := 0;
      SMC1_Parameter_Memory_Map.MRBLR              := 01;

      --  UART specific SMC1 parameters

      SMC1_UART_Parameter_Memory_Map.MAX_IDL := 0;
      SMC1_UART_Parameter_Memory_Map.BRKLN   := 0;
      SMC1_UART_Parameter_Memory_Map.BRKEC   := 0;
      SMC1_UART_Parameter_Memory_Map.BRKCR   := 1;

      --  Init Tx BD

      UART_Tx_Buffer_Descriptor.Ready             := False;
      UART_Tx_Buffer_Descriptor.Wrap              := True;
      UART_Tx_Buffer_Descriptor.Interrupt         := False;

      UART_Tx_Buffer_Descriptor.Continuous_Mode   := False;
      UART_Tx_Buffer_Descriptor.Preamble          := False;

      UART_Tx_Buffer_Descriptor.Tx_Buffer_Pointer := UART_Tx_Buffer'Address;
      UART_Tx_Buffer_Descriptor.Data_Length       := UART_Tx_Buffer'Length;

      --  Clear all previous events

      SMCE1 := To_SMCEM (16#FF#);

      --  Enable all SMC interrupts

      SMCM1 := To_SMCEM (16#17#);

      --  Write 0x0000_0010 to CIMR so the SMC1 can generate a system
      --  interrupt.

      CIMR_Mirror := CIMR;
      CIMR_Mirror.SMC1 := True;
      CIMR := CIMR_Mirror;

      --  Init CICR (defines priorities for interrupts) but for SCCs!
      --  Is this necessary here???

      CICR_Mirror := CICR;
      CICR_Mirror.SCdP := 2#11#;
      CICR_Mirror.SCcP := 2#10#;
      CICR_Mirror.SCbP := 2#01#;
      CICR_Mirror.SCaP := 2#00#;

      --  Interrupt level of UART interrupt

      CICR_Mirror.IRL  := External_Level_Of_UART_Interrupt;
      CICR_Mirror.HP   := 2#11111#;
      CICR_Mirror.IEN  := True;
      CICR := CICR_Mirror;

      --  Write 0x4820 to SMCMR to configure normal operation, 8-bit
      --  characters, no parity, 1 stop bit

      SMCMR1 := To_SMCMR (16#4820#);

      --   Enable SMC1 (tx and rx)

      SMCMR1 := To_SMCMR (16#4823#);

      --  Initialization of serial port is complete for tx (not for rx!)

   end Initialize_UART;

   ------------------------
   -- Max_Timer_Interval --
   ------------------------

   function Max_Timer_Interval return Timer_Interval is
   begin
      return Timer_Interval'Last;
   end Max_Timer_Interval;

   ---------------------------
   -- Priority_Of_Interrupt --
   ---------------------------

   function Priority_Of_Interrupt
     (Level : SBP.Interrupt_Level) return System.Any_Priority
   is
   begin
      --  Assert that it is a real interrupt

      pragma Assert (Level /= 0);

      return Interrupt_Priority'Last - Level;
   end Priority_Of_Interrupt;

   ----------------
   -- Read_Clock --
   ----------------

   function Read_Clock return Timer_Interval is
      Res : Timer_Interval;
   begin
      --  ??? Untested code: Assuming this counts up?
      --  Note that this system seems to have a 64-bit clock (tbl and tbu).
      --  However, we just use the lower 32 bits to be able to share a
      --  common implementation of System.BB.Timers.

      Asm ("mftbl %0",
        Outputs => Timer_Interval'Asm_Output ("=r", Res),
        Volatile => True);

      return Res;
   end Read_Clock;

   ---------------
   -- Set_Alarm --
   ---------------

   procedure Set_Alarm (Ticks : Timer_Interval) is
   begin
      Asm ("mtdec %0",
        Inputs => Timer_Interval'Asm_Input ("r", Ticks),
        Volatile => True);
   end Set_Alarm;

   --------------------
   -- Stop_Watch_Dog --
   --------------------

   procedure Stop_Watch_Dog is
   begin
      --  Nothing to be done

      null;
   end Stop_Watch_Dog;

   ----------------------
   -- Ticks_Per_Second --
   ----------------------

   function Ticks_Per_Second return Natural is
   begin
      return System.BB.Parameters.Clock_Frequency;
   end Ticks_Per_Second;

   ---------------------------
   -- Get_Interrupt_Request --
   ---------------------------

   function Get_Interrupt_Request
     (Vector : CPU_Primitives.Vector_Id) return SBP.Interrupt_Level
   is
      Interrupt : SBP.Interrupt_Level;

   begin
      --  Source external interrupt?

      if Vector = 5 then
         Interrupt := SIVEC.Interrupt_Code / 4 + 1;

         if Interrupt = External_Interrupt then
            CIVR.Interrupt_Ack := True;
         end if;

         --  Check the source of the external interrupt

         return Interrupt;

      elsif Vector = 9 then
         return Decrementer;

      else
         raise Program_Error;
      end if;
   end Get_Interrupt_Request;

   ----------------
   -- Get_Vector --
   ----------------

   function Get_Vector
     (Interrupt : SBP.Interrupt_Level) return CPU_Primitives.Vector_Id
   is
   begin
      --  Either the decrementer is used or the external interrupt

      case Interrupt is
         when Decrementer => return 9;
         when others      => return 5;
      end case;
   end Get_Vector;

   ---------------
   -- UART_Send --
   ---------------

   procedure UART_Send (Char : Character) is
   begin
      --  Load Char into the tx buffer

      UART_Tx_Buffer (0) := Char;

      --  Indicate that we have something to send

      UART_Tx_Buffer_Descriptor.Ready := True;

      --  Spin until char has been sent out

      while UART_Tx_Buffer_Descriptor.Ready loop
         null;
      end loop;
   end UART_Send;

   ------------------
   -- Console_Send --
   ------------------

   procedure Console_Send (Char : Character) is
   begin
      UART_Send (Char);
   end Console_Send;

end System.BB.Peripherals;
