------------------------------------------------------------------------------
--                                                                          --
--         S Y S T E M . B B . P E R I P H E R A L S . M P C 8 6 0          --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                     Copyright (C) 2005-2010, AdaCore                     --
--                                                                          --
------------------------------------------------------------------------------

--  This file defines a memory map for memory mapped registers of the MPC860

with System.Storage_Elements;

with Ada.Unchecked_Conversion;

package System.BB.Peripherals.MPC860 is
   pragma Preelaborate;

   type Bool16 is array (0 .. 15) of Boolean;
   for Bool16'Size use 16;
   pragma Pack (Bool16);

   type Bool32 is array (0 .. 31) of Boolean;
   for Bool32'Size use 32;
   pragma Pack (Bool32);

   function To_Bool32 is new Ada.Unchecked_Conversion (Integer, Bool32);

   type Int16 is range 0 .. 16#FFFF#;
   type Int32 is mod 2 ** 32;

   package SSE renames System.Storage_Elements;

   type Reserved_1 is array (0 .. 0) of Boolean;
   for Reserved_1'Size use 1;
   pragma Pack (Reserved_1);

   type Reserved_2 is array (0 .. 1) of Boolean;
   for Reserved_2'Size use 2;
   pragma Pack (Reserved_2);

   type Reserved_3 is array (0 .. 2) of Boolean;
   for Reserved_3'Size use 3;
   pragma Pack (Reserved_3);

   type Reserved_4 is array (0 .. 3) of Boolean;
   for Reserved_4'Size use 4;
   pragma Pack (Reserved_4);

   type Reserved_5 is array (0 .. 4) of Boolean;
   for Reserved_5'Size use 5;
   pragma Pack (Reserved_5);

   type Reserved_6 is array (0 .. 5) of Boolean;
   for Reserved_6'Size use 6;
   pragma Pack (Reserved_6);

   type Reserved_8 is array (0 .. 5) of Boolean;
   for Reserved_8'Size use 8;
   pragma Pack (Reserved_8);

   type Reserved_10 is array (0 .. 9) of Boolean;
   for Reserved_10'Size use 10;
   pragma Pack (Reserved_10);

   type Reserved_13 is array (0 .. 12) of Boolean;
   for Reserved_13'Size use 13;
   pragma Pack (Reserved_13);

   type Reserved_14 is array (0 .. 13) of Boolean;
   for Reserved_14'Size use 14;
   pragma Pack (Reserved_14);

   type Reserved_15 is array (0 .. 14) of Boolean;
   for Reserved_15'Size use 15;
   pragma Pack (Reserved_15);

   type Reserved_16 is array (0 .. 15) of Boolean;
   for Reserved_16'Size use 16;
   pragma Pack (Reserved_16);

   type Reserved_17 is array (0 .. 16) of Boolean;
   for Reserved_17'Size use 17;
   pragma Pack (Reserved_17);

   type Reserved_24 is array (0 .. 23) of Boolean;
   for Reserved_24'Size use 24;
   pragma Pack (Reserved_24);

   type Reserved_64 is array (0 .. 63) of Boolean;
   for Reserved_64'Size use 64;
   pragma Pack (Reserved_64);

   ----------------------
   --  Memory Offsets  --
   ----------------------
   --  see table 18-9
   SCC1_Base : constant := 16#3C00#;
   SCC2_Base : constant := 16#3D00#;
   SCC3_Base : constant := 16#3E00#;
   SCC4_Base : constant := 16#3F00#;

   SMC1_Base : constant := 16#3E80#;
   SMC2_Base : constant := 16#3F80#;

   IMMR      : constant := 16#FF00_0000#;

   KEY_REGISTER_UNLOCK_VALUE : constant Int32  := 16#55CC_AA33#;

   ----------------------------------------
   --             Key Registers          --
   ----------------------------------------
   --  System Integration Timers Keys
   TBSCRK : Int32;
   for TBSCRK'Address  use System'To_Address (IMMR + 16#300#);

   TBREFAK : Int32;
   for TBREFAK'Address use System'To_Address (IMMR + 16#304#);

   TBREFBK : Int32;
   for TBREFBK'Address use System'To_Address (IMMR + 16#308#);

   TBK : Int32;
   for TBK'Address     use System'To_Address (IMMR + 16#30C#);

   RTCSCK : Int32;
   for RTCSCK'Address  use System'To_Address (IMMR + 16#324#);

   RTCK : Int32;
   for RTCK'Address    use System'To_Address (IMMR + 16#324#);

   RTSECK : Int32;
   for RTSECK'Address  use System'To_Address (IMMR + 16#328#);

   RTCALK : Int32;
   for RTCALK'Address  use System'To_Address (IMMR + 16#32C#);

   PISCRK : Int32;
   for PISCRK'Address  use System'To_Address (IMMR + 16#340#);

   PITCK : Int32;
   for PITCK'Address   use System'To_Address (IMMR + 16#344#);

   --  Clocks and Reset Keys
   SCCRK : Int32;
   for SCCRK'Address   use System'To_Address (IMMR + 16#380#);

   PLPRCRK : Int32;
   for PLPRCRK'Address use System'To_Address (IMMR + 16#384#);

   RSRK : Int32;
   for RSRK'Address    use System'To_Address (IMMR + 16#384#);

   -----------------------------------------------------
   --  Time Base Status and Control Register (TBSCR)  --
   -----------------------------------------------------

   type Access_TBSCR is record
      --  see below for possible values of interrupt requests
      Timebase_Interrupt_Request   : Natural range 0 .. 16#FF#;
      Reference_Interrupt_Status_A : Boolean;
      Reference_Interrupt_Status_B : Boolean;
      Reserved2                    : Reserved_2;
      Reference_Interrupt_A_Enable : Boolean;
      Reference_Interrupt_B_Enable : Boolean;
      Timebase_Freeze_Enable       : Boolean;
      Timebase_Enable              : Boolean;
   end record;

   for Access_TBSCR use record
      Timebase_Interrupt_Request   at 0 range  0 ..  7;
      Reference_Interrupt_Status_A at 0 range  8 ..  8;
      Reference_Interrupt_Status_B at 0 range  9 ..  9;
      Reserved2                    at 0 range 10 .. 11;
      Reference_Interrupt_A_Enable at 0 range 12 .. 12;
      Reference_Interrupt_B_Enable at 0 range 13 .. 13;
      Timebase_Freeze_Enable       at 0 range 14 .. 14;
      Timebase_Enable              at 0 range 15 .. 15;
   end record;

   for Access_TBSCR'Size use 16;

   TBSCR        : Access_TBSCR;
   TBSCR_Mirror : Access_TBSCR;
   for TBSCR'Address use System'To_Address (IMMR + 16#200#);
   function To_TBSCR is new Ada.Unchecked_Conversion (Int16, Access_TBSCR);

   --  possible values for Timebase_Interrupt_Request
   TBIRQ_Level1 : constant := 64;
   TBIRQ_Level2 : constant := 32;
   TBIRQ_Level3 : constant := 16;
   TBIRQ_Level4 : constant :=  8;
   TBIRQ_Level5 : constant :=  4;
   TBIRQ_Level6 : constant :=  2;
   TBIRQ_Level7 : constant :=  1;

   -----------------------------------------------------
   --  Time Base Reference Registers (TBREFA/TBREFB)  --
   -----------------------------------------------------

   TBREFA : Int32;
   for TBREFA'Address use System'To_Address (IMMR + 16#204#);

   TBREFB : Int32;
   for TBREFB'Address use System'To_Address (IMMR + 16#208#);

   ----------------------------------------------------
   -- System Clock and Reset Control Register (SCCR) --
   ----------------------------------------------------
   type Access_SCCR is record
      Reserved1  : Reserved_1;
      COM        : Natural range 0 .. 3;
      Reserved2a : Reserved_2;
      TBS        : Boolean;
      RTDIV      : Boolean;
      RTSEL      : Boolean;
      CRQEN      : Boolean;
      PRQEN      : Boolean;
      Reserved2b : Reserved_2;
      EBDF       : Boolean;
      Reserved2c : Reserved_2;
      DFSYNC     : Natural range 0 .. 3;
      DFBRG      : Natural range 0 .. 3;
      DFNL       : Natural range 0 .. 7;
      DFNH       : Natural range 0 .. 7;
      Reserved5  : Reserved_5;
   end record;

   for Access_SCCR use record
      Reserved1  at 0 range  0 ..  0;
      COM        at 0 range  1 ..  2;
      Reserved2a at 0 range  3 ..  5;
      TBS        at 0 range  6 ..  6;
      RTDIV      at 0 range  7 ..  7;
      RTSEL      at 0 range  8 ..  8;
      CRQEN      at 0 range  9 ..  9;
      PRQEN      at 0 range 10 .. 10;
      Reserved2b at 0 range 11 .. 12;
      EBDF       at 0 range 13 .. 14;
      Reserved2c at 0 range 15 .. 16;
      DFSYNC     at 0 range 17 .. 18;
      DFBRG      at 0 range 19 .. 20;
      DFNL       at 0 range 21 .. 23;
      DFNH       at 0 range 24 .. 26;
      Reserved5  at 0 range 27 .. 31;
   end record;

   for Access_SCCR'Size use 32;

   SCCR : Access_SCCR;
   for SCCR'Address use System'To_Address (IMMR + 16#280#);

   ------------------------------------------------------------
   -- Periodic Interrupt Status and Control Register (PISCR) --
   ------------------------------------------------------------
   type Access_PISCR is record
      PIRQ      : Natural range 0 .. 2 ** 7;
      PS        : Boolean;
      Reserved4 : Reserved_4;
      PIE       : Boolean;
      PITF      : Boolean;
      PTE       : Boolean;
   end record;

   for Access_PISCR use record
      PIRQ      at 0 range  0 ..  7;
      PS        at 0 range  8 ..  8;
      Reserved4 at 0 range  9 .. 12;
      PIE       at 0 range 13 .. 13;
      PITF      at 0 range 14 .. 14;
      PTE       at 0 range 15 .. 15;
   end record;

   for Access_PISCR'Size use 16;

   PISCR : Access_PISCR;
   for PISCR'Address use System'To_Address (IMMR + 16#240#);

   ------------------------
   -- PIT Count Register --
   ------------------------
   PITC : Int16 := 100;
   for PITC'Address use System'To_Address (IMMR + 16#244#);

   ---------------------
   --   P O R T   A   --
   ---------------------
   --  see MPC860UM.pdf chapter 33.2.1.3
   --  Port A Data Direction Register
   subtype Access_PADIR is Bool16;
   PADIR        : Access_PADIR := (others => False);
   PADIR_Mirror : Access_PADIR;
   for PADIR'Address use System'To_Address (IMMR + 16#950#);
   function To_PADIR is new Ada.Unchecked_Conversion (Int16, Access_PADIR);

   --  see MPC860UM.pdf chapter 33.2.1.3
   --  Port A Data Register
   subtype Access_PADAT is Bool16;
   PADAT        : Access_PADAT := (others => False);
   PADAT_Mirror : Access_PADAT;
   for PADAT'Address use System'To_Address (IMMR + 16#956#);
   function To_PADAT is new Ada.Unchecked_Conversion (Int16, Access_PADAT);

   --  see MPC860UM.pdf chapter 33.2.1.4
   --  Port A Pin Assignment Register
   subtype Access_PAPAR is Bool16;
   PAPAR        : Access_PAPAR := (others => False);
   PAPAR_Mirror : Access_PAPAR;
   for PAPAR'Address use System'To_Address (IMMR + 16#952#);
   function To_PAPAR is new Ada.Unchecked_Conversion (Int16, Access_PAPAR);

   --  see MPC860UM.pdf chapter 33.2.1.1
   --  Port A Open-Drain Register
   subtype Access_PAODR is Bool16;
   PAODR        : Access_PAODR := (others => False);
   PAODR_Mirror : Access_PAODR;
   for PAODR'Address use System'To_Address (IMMR + 16#954#);
   function To_PAODR is new Ada.Unchecked_Conversion (Int16, Access_PAODR);

   ---------------------
   --   P O R T   B   --
   ---------------------
   --  see MPC860UM.pdf chapter 33.3.2.1
   --  Port B Open-Drain Register
   subtype Access_PBODR is Bool32;
   PBODR        : Access_PBODR := (others => False);
   PBODR_Mirror : Access_PBODR;
   for PBODR'Address use System'To_Address (IMMR + 16#AC0#);
   function To_PBODR is new Ada.Unchecked_Conversion (Int32, Access_PBODR);

   --  see MPC860UM.pdf chapter 33.3.2.2
   --  Port B Data Register
   subtype Access_PBDAT is Bool32;
   PBDAT        : Access_PBDAT := (others => False);
   PBDAT_Mirror : Access_PBDAT;
   for PBDAT'Address use System'To_Address (IMMR + 16#AC4#);
   function To_PBDAT is new Ada.Unchecked_Conversion (Int32, Access_PBDAT);

   --  see MPC860UM.pdf chapter 33.3.2.3
   --  Port B Data Direction Register
   subtype Access_PBDIR is Bool32;
   PBDIR        : Access_PBDIR := (others => False);
   PBDIR_Mirror : Access_PBDIR;
   for PBDIR'Address use System'To_Address (IMMR + 16#AB8#);
   function To_PBDIR is new Ada.Unchecked_Conversion (Int32, Access_PBDIR);

   --  see MPC860UM.pdf chapter 33.3.2.4
   --  Port B Pin Assignment Register
   subtype Access_PBPAR is Bool32;
   PBPAR        : Access_PBPAR := (others => False);
   PBPAR_Mirror : Access_PBPAR;
   for PBPAR'Address use System'To_Address (IMMR + 16#ABC#);
   function To_PBPAR is new Ada.Unchecked_Conversion (Int32, Access_PBPAR);

   ---------------------
   --   P O R T   C   --
   ---------------------
   --  see MPC860UM.pdf chapter 33.4.2.2
   --  Port C Data Direction Register
   subtype Access_PCDIR is Bool16;
   PCDIR        : Access_PCDIR := (others => False);
   PCDIR_Mirror : Access_PCDIR;
   for PCDIR'Address use System'To_Address (IMMR + 16#960#);
   function To_PCDIR is new Ada.Unchecked_Conversion (Int16, Access_PCDIR);

   --  see MPC860UM.pdf chapter 33.4.2.3
   --  Port C Pin Assignment Register
   subtype Access_PCPAR is Bool16;
   PCPAR        : Access_PCPAR := (others => False);
   PCPAR_Mirror : Access_PCPAR;
   for PCPAR'Address use System'To_Address (IMMR + 16#962#);
   function To_PCPAR is new Ada.Unchecked_Conversion (Int16, Access_PCPAR);
   function To_PCPAR is new Ada.Unchecked_Conversion (Bool16, Access_PCPAR);
   function To_Int16 is new Ada.Unchecked_Conversion (Access_PCPAR, Int16);

   --  see MPC860UM.pdf chapter 33.4.2.4
   --  Port C Special Options Register
   subtype Access_PCSO is Bool16;
   PCSO        : Access_PCSO := (others => False);
   PCSO_Mirror : Access_PCSO;
   for PCSO'Address use System'To_Address (IMMR + 16#964#);
   function To_PCSO is new Ada.Unchecked_Conversion (Int16, Access_PCSO);

   --  see MPC860UM.pdf chapter 33.4.2.1
   --  Port C Data Register
   subtype Access_PCDAT is Bool16;
   PCDAT        : Access_PCDAT := (others => False);
   PCDAT_Mirror : Access_PCDAT;
   for PCDAT'Address use System'To_Address (IMMR + 16#966#);
   function To_PCDAT is new Ada.Unchecked_Conversion (Int16, Access_PCDAT);

   ---------------------
   --   P O R T   D   --
   ---------------------
   --  see MPC860UM.pdf chapter 33.5.1.1
   --  Port D Data Register
   type Access_PDDAT is array (0 .. 15) of Boolean;
   for Access_PDDAT'Size use 16;
   pragma Pack (Access_PDDAT);
   PDDAT        : Access_PDDAT := (others => False);
   PDDAT_Mirror : Access_PDDAT;
   for PDDAT'Address use System'To_Address (IMMR + 16#976#);
   function To_PDDAT is new Ada.Unchecked_Conversion (Int16, Access_PDDAT);

   --  see MPC860UM.pdf chapter 33.5.1.2
   --  Port D Data Direction Register
   type Access_PDDIR is array (0 .. 15) of Boolean;
   for Access_PDDIR'Size use 16;
   pragma Pack (Access_PDDIR);
   PDDIR        : Access_PDDIR := (others => False);
   PDDIR_Mirror : Access_PDDIR;
   for PDDIR'Address use System'To_Address (IMMR + 16#970#);
   function To_PDDIR is new Ada.Unchecked_Conversion (Int16, Access_PDDIR);

   --  see MPC860UM.pdf chapter 33.5.2
   --  Port D Pin Assignment Register
   type Access_PDPAR is array (0 .. 15) of Boolean;
   for Access_PDPAR'Size use 16;
   pragma Pack (Access_PDPAR);
   PDPAR        : Access_PDPAR := (others => False);
   PDPAR_Mirror : Access_PDPAR;
   for PDPAR'Address use System'To_Address (IMMR + 16#972#);
   function To_PDPAR is new Ada.Unchecked_Conversion (Int16, Access_PDPAR);

   -----------------------------------------------------
   --   B A U D   R A T E   G E N E N E R A T O R S   --
   -----------------------------------------------------
   --  see MPC860UM.pdf Table 2-1
   type Access_BRGC is record
      Reserved14       : Reserved_14;
      Reset            : Boolean;
      Enable           : Boolean;
      External_Clk_Src : Natural range 0 .. 3;
      Autobaud         : Boolean;
      Clock_Divider    : Natural range 0 .. 16#FFF#;
      DIV16            : Boolean;
   end record;

   for Access_BRGC use
      record
         Reserved14       at 0 range  0 .. 13;
         Reset            at 0 range 14 .. 14;
         Enable           at 0 range 15 .. 15;
         External_Clk_Src at 0 range 16 .. 17;
         Autobaud         at 0 range 18 .. 18;
         Clock_Divider    at 0 range 19 .. 30;
         DIV16            at 0 range 31 .. 31;
      end record;
   for Access_BRGC'Size use 32;

   function To_BRGC is new Ada.Unchecked_Conversion (Int32, Access_BRGC);

   --  BRG1 Configuration Register
   BRGC1        : Access_BRGC;
   BRGC1_Mirror : Access_BRGC;
   for BRGC1'Address use System'To_Address (IMMR + 16#9F0#);

   -------------------------------
   --  SI Clock Route Register  --
   -------------------------------

   --  see MPC860UM.pdf chapter 20.2.4.3
   type Access_SICR is record
      GR4  : Boolean;
      SC4  : Boolean;
      R4CS : Natural range 0 .. 7;
      T4CS : Natural range 0 .. 7;

      GR3  : Boolean;
      SC3  : Boolean;
      R3CS : Natural range 0 .. 7;
      T3CS : Natural range 0 .. 7;

      GR2  : Boolean;
      SC2  : Boolean;
      R2CS : Natural range 0 .. 7;
      T2CS : Natural range 0 .. 7;

      GR1  : Boolean;
      SC1  : Boolean;
      R1CS : Natural range 0 .. 7;
      T1CS : Natural range 0 .. 7;
   end record;

   for Access_SICR use
      record
         GR4  at 0 range 0 .. 0;
         SC4  at 0 range 1 .. 1;
         R4CS at 0 range 2 .. 4;
         T4CS at 0 range 5 .. 7;

         GR3  at 1 range 0 .. 0;
         SC3  at 1 range 1 .. 1;
         R3CS at 1 range 2 .. 4;
         T3CS at 1 range 5 .. 7;

         GR2  at 2 range 0 .. 0;
         SC2  at 2 range 1 .. 1;
         R2CS at 2 range 2 .. 4;
         T2CS at 2 range 5 .. 7;

         GR1  at 3 range 0 .. 0;
         SC1  at 3 range 1 .. 1;
         R1CS at 3 range 2 .. 4;
         T1CS at 3 range 5 .. 7;
      end record;

   for Access_SICR'Size use 32;

   SICR        : Access_SICR;
   SICR_Mirror : Access_SICR;

   for SICR'Address use System'To_Address (IMMR + 16#AEC#);
   function To_SICR is new Ada.Unchecked_Conversion (Int32, Access_SICR);

   -------------------------------------------------
   --  SIU Module Configuration Register (SIUMCR) --
   -------------------------------------------------
   type Access_SIUMCR is record
      External_Arbitration                  : Boolean;
      External_Arbitration_Request_Priority : Natural range 0 .. 7;
      Reserved4a                            : Reserved_4;
      Data_Show_Cycles                      : Boolean;
      Debug_Pin_Configuration               : Natural range 0 .. 3;
      Debug_Port_Pins_Configuration         : Natural range 0 .. 3;
      Reserved1                             : Reserved_1;
      FRZ_Pin_Configuration                 : Boolean;
      Debug_Register_Lock                   : Boolean;
      Odd_Parity                            : Boolean;
      Parity_Enable_For_Nonmemory_Regions   : Boolean;
      Data_Parity_Pins_Configuration        : Boolean;
      Multi_Processors_Reservation_Enable   : Boolean;
      Multi_Level_Reservation_Control       : Boolean;
      Asynchronous_External_Master_Enable   : Boolean;
      Synchronous_External_Master_Enable    : Boolean;
      BSC                                   : Boolean;
      GB5E                                  : Boolean;
      B2DD                                  : Boolean;
      B3DD                                  : Boolean;
      Reserved4b                            : Reserved_4;
   end record;

   for Access_SIUMCR use record
      External_Arbitration                  at 0 range  0 ..  0;
      External_Arbitration_Request_Priority at 0 range  1 ..  3;
      Reserved4a                            at 0 range  4 ..  7;
      Data_Show_Cycles                      at 0 range  8 ..  8;
      Debug_Pin_Configuration               at 0 range  9 .. 10;
      Debug_Port_Pins_Configuration         at 0 range 11 .. 12;
      Reserved1                             at 0 range 13 .. 13;
      FRZ_Pin_Configuration                 at 0 range 14 .. 14;
      Debug_Register_Lock                   at 0 range 15 .. 15;
      Odd_Parity                            at 0 range 16 .. 16;
      Parity_Enable_For_Nonmemory_Regions   at 0 range 17 .. 17;
      Data_Parity_Pins_Configuration        at 0 range 18 .. 18;
      Multi_Processors_Reservation_Enable   at 0 range 19 .. 19;
      Multi_Level_Reservation_Control       at 0 range 20 .. 21;
      Asynchronous_External_Master_Enable   at 0 range 22 .. 22;
      Synchronous_External_Master_Enable    at 0 range 23 .. 23;
      BSC                                   at 0 range 24 .. 24;
      GB5E                                  at 0 range 25 .. 25;
      B2DD                                  at 0 range 26 .. 26;
      B3DD                                  at 0 range 27 .. 27;
      Reserved4b                            at 0 range 28 .. 31;
   end record;

   SIUMCR        : Access_SIUMCR;
   SIUMCR_Mirror : Access_SIUMCR;

   for SIUMCR'Address use System'To_Address (IMMR + 16#000#);

   ----------------------------------
   --  SI Interrupt Mask Register  --
   ----------------------------------
   type Access_SIMASK is record
      IRM0       : Boolean;
      LVM0       : Boolean;
      IRM1       : Boolean;
      LVM1       : Boolean;
      IRM2       : Boolean;
      LVM2       : Boolean;
      IRM3       : Boolean;
      LVM3       : Boolean;
      IRM4       : Boolean;
      LVM4       : Boolean;
      IRM5       : Boolean;
      LVM5       : Boolean;
      IRM6       : Boolean;
      LVM6       : Boolean;
      IRM7       : Boolean;
      LVM7       : Boolean;
      Reserved16 : Reserved_16;
   end record;

   for Access_SIMASK use record
      IRM0       at 0 range  0 ..  0;
      LVM0       at 0 range  1 ..  1;
      IRM1       at 0 range  2 ..  2;
      LVM1       at 0 range  3 ..  3;
      IRM2       at 0 range  4 ..  4;
      LVM2       at 0 range  5 ..  5;
      IRM3       at 0 range  6 ..  6;
      LVM3       at 0 range  7 ..  7;
      IRM4       at 0 range  8 ..  8;
      LVM4       at 0 range  9 ..  9;
      IRM5       at 0 range 10 .. 10;
      LVM5       at 0 range 11 .. 11;
      IRM6       at 0 range 12 .. 12;
      LVM6       at 0 range 13 .. 13;
      IRM7       at 0 range 14 .. 14;
      LVM7       at 0 range 15 .. 15;
      Reserved16 at 0 range 16 .. 31;
   end record;

   for Access_SIMASK'Size use 32;

   SIMASK : Access_SIMASK;
   for SIMASK'Address use System'To_Address (IMMR + 16#014#);
   function To_SIMASK is new Ada.Unchecked_Conversion (Int32, Access_SIMASK);
   function To_SIMASK is new Ada.Unchecked_Conversion (Bool32, Access_SIMASK);

   -------------------------------------
   --  SI Interrupt Pending Register  --
   -------------------------------------

   type Access_SIPEND is record
      IRQ0       : Boolean;
      LVL0       : Boolean;
      IRQ1       : Boolean;
      LVL1       : Boolean;
      IRQ2       : Boolean;
      LVL2       : Boolean;
      IRQ3       : Boolean;
      LVL3       : Boolean;
      IRQ4       : Boolean;
      LVL4       : Boolean;
      IRQ5       : Boolean;
      LVL5       : Boolean;
      IRQ6       : Boolean;
      LVL6       : Boolean;
      IRQ7       : Boolean;
      LVL7       : Boolean;
      Reserved16 : Reserved_16;
   end record;

   for Access_SIPEND use record
      IRQ0       at 0 range  0 ..  0;
      LVL0       at 0 range  1 ..  1;
      IRQ1       at 0 range  2 ..  2;
      LVL1       at 0 range  3 ..  3;
      IRQ2       at 0 range  4 ..  4;
      LVL2       at 0 range  5 ..  5;
      IRQ3       at 0 range  6 ..  6;
      LVL3       at 0 range  7 ..  7;
      IRQ4       at 0 range  8 ..  8;
      LVL4       at 0 range  9 ..  9;
      IRQ5       at 0 range 10 .. 10;
      LVL5       at 0 range 11 .. 11;
      IRQ6       at 0 range 12 .. 12;
      LVL6       at 0 range 13 .. 13;
      IRQ7       at 0 range 14 .. 14;
      LVL7       at 0 range 15 .. 15;
      Reserved16 at 0 range 16 .. 31;
   end record;

   for Access_SIPEND'Size use 32;

   SIPEND : Access_SIPEND;
   for SIPEND'Address use System'To_Address (IMMR + 16#010#);

   ---------------------------------------------
   --  SIU Interrupt Vector Register (SIVEC)  --
   ---------------------------------------------

   type Access_SIVEC is record
      Interrupt_Code : Natural range 0 .. 16#FF#;
      Reserved24     : Reserved_24;
   end record;

   for Access_SIVEC use record
      Interrupt_Code at 0 range 0 ..  7;
      Reserved24     at 0 range 8 .. 31;
   end record;

   for Access_SIVEC'Size use 32;

   SIVEC : Access_SIVEC;
   for SIVEC'Address use System'To_Address (IMMR + 16#01C#);

   ------------------------
   --  SI Mode Register  --
   ------------------------
   --  see 20.2.4.2
   type Access_SIMODE is record
      SMC2   : Boolean;
      SMC2CS : Natural range 0 .. 7;
      SDMb   : Natural range 0 .. 3;
      RFSDb  : Natural range 0 .. 3;
      DSCb   : Boolean;
      CRTb   : Boolean;
      STZb   : Boolean;
      CEb    : Boolean;
      FEb    : Boolean;
      GMb    : Boolean;
      TFSDb  : Natural range 0 .. 3;
      SMC1   : Boolean;
      SMC1CS : Natural range 0 .. 7;
      SDMa   : Natural range 0 .. 3;
      RFSDa  : Natural range 0 .. 3;
      DSCa   : Boolean;
      CRTa   : Boolean;
      STZa   : Boolean;
      CEa    : Boolean;
      FEa    : Boolean;
      GMa    : Boolean;
      TFSDa  : Natural range 0 .. 3;
   end record;

   for Access_SIMODE use
      record
         SMC2   at  0 range  0 ..  0;
         SMC2CS at  0 range  1 ..  3;
         SDMb   at  0 range  4 ..  5;
         RFSDb  at  0 range  6 ..  7;
         DSCb   at  0 range  8 ..  8;
         CRTb   at  0 range  9 ..  9;
         STZb   at  0 range 10 .. 10;
         CEb    at  0 range 11 .. 11;
         FEb    at  0 range 12 .. 12;
         GMb    at  0 range 13 .. 13;
         TFSDb  at  0 range 14 .. 15;
         SMC1   at  0 range 16 .. 16;
         SMC1CS at  0 range 17 .. 19;
         SDMa   at  0 range 20 .. 21;
         RFSDa  at  0 range 22 .. 23;
         DSCa   at  0 range 24 .. 24;
         CRTa   at  0 range 25 .. 25;
         STZa   at  0 range 26 .. 26;
         CEa    at  0 range 27 .. 27;
         FEa    at  0 range 28 .. 28;
         GMa    at  0 range 29 .. 29;
         TFSDa  at  0 range 30 .. 31;
      end record;

   for Access_SIMODE'Size use 32;

   SIMODE        : Access_SIMODE;
   SIMODE_Mirror : Access_SIMODE;

   for SIMODE'Address use System'To_Address (IMMR + 16#AE0#);

   -----------------------------------
   --  SDMA Configuration Register  --
   -----------------------------------
   --  see MPC860UM.pdf chapter 19.2.1
   type Access_SDCR is record
      Reserved17 : Reserved_17;
      FRZ        : Boolean;
      Reserved10 : Reserved_10;
      FAID       : Natural range 0 .. 3;
      RAID       : Natural range 0 .. 3;
   end record;

   for Access_SDCR use
      record
         Reserved17 at 0 range  0 .. 16;
         FRZ        at 0 range 17 .. 17;
         Reserved10 at 0 range 18 .. 27;
         FAID       at 0 range 28 .. 29;
         RAID       at 0 range 30 .. 31;
      end record;

   for Access_SDCR'Size use 32;

   SDCR        : Access_SDCR;
   SDCR_Mirror : Access_SDCR;

   for SDCR'Address use System'To_Address (IMMR + 16#30#);
   function To_SDCR is new Ada.Unchecked_Conversion (Integer, Access_SDCR);

   ------------------------------------
   --  CP Command Register Register  --
   ------------------------------------
   --  see MPC860UM.pdf chapter 18.5.2
   type Access_CPCR is record
      RST       : Boolean;
      Reserved3 : Reserved_3;
      OPCODE    : Natural range 0 .. 15;
      CH_NUM    : Natural range 0 .. 15;
      Reserved  : Reserved_3;
      FLG       : Boolean;
   end record;

   for Access_CPCR use
      record
         RST       at 0 range  0 ..  0;
         Reserved3 at 0 range  1 ..  3;
         OPCODE    at 0 range  4 ..  7;
         CH_NUM    at 0 range  8 .. 11;
         Reserved  at 0 range 12 .. 14;
         FLG       at 0 range 15 .. 15;
      end record;

   for Access_CPCR'Size use 16;

   CPCR        : Access_CPCR;
   CPCR_Mirror : Access_CPCR;

   for CPCR'Address use System'To_Address (IMMR + 16#9C0#);

   function To_CPCR is new Ada.Unchecked_Conversion (Int16, Access_CPCR);

   -----------------------------------------------
   --  SCC Parameter RAM Map for all protocols  --
   -----------------------------------------------

   --  Funtion Code Registers  --
   --  see MPC860UM.pdf chapter 21.4.1
   type Access_FCR is record -- for RFCR and TFCR
      Reserved3     : Reserved_3;
      Byte_Ordering : Natural range 0 .. 3;
      Address_Type  : Natural range 0 .. 7;
   end record;

   for Access_FCR use
      record
         Reserved3     at 0 range 0 .. 2;
         Byte_Ordering at 0 range 3 .. 4;
         Address_Type  at 0 range 5 .. 7;
      end record;
   for Access_FCR'Size use 8;

   --  see MPC860UM.pdf table 21-4
   type Access_SCC_Parameter_Memory_Map is
      record
         RBASE  : Natural range 0 .. 16#FFFF#;
         TBASE  : Natural range 0 .. 16#FFFF#;
         RFCR   : Access_FCR;
         TFCR   : Access_FCR;
         MRBLR  : Natural range 0 .. 16#FFFF#;
         RSTATE : Natural;
         RIP    : Natural;
         RBPTR  : Natural range 0 .. 16#FFFF#;
         RCOUNT : Natural range 0 .. 16#FFFF#;
         RTEMP  : Natural;
         TSTATE : Natural;
         TIP    : Natural;
         TBPTR  : Natural range 0 .. 16#FFFF#;
         TCOUNT : Natural range 0 .. 16#FFFF#;
         TTEMP  : Natural;
         RCRC   : Natural;
         TCRC   : Natural;
      end record;

   for Access_SCC_Parameter_Memory_Map use
      record
         RBASE  at  0 range  0 .. 15;
         TBASE  at  2 range  0 .. 15;
         RFCR   at  4 range  0 ..  7;
         TFCR   at  5 range  0 ..  7;
         MRBLR  at  6 range  0 .. 15;
         RSTATE at  8 range  0 .. 31;
         RIP    at 12 range  0 .. 31;
         RBPTR  at 16 range  0 .. 15;
         RCOUNT at 18 range  0 .. 15;
         RTEMP  at 20 range  0 .. 31;
         TSTATE at 24 range  0 .. 31;
         TIP    at 28 range  0 .. 31;
         TBPTR  at 32 range  0 .. 15;
         TCOUNT at 34 range  0 .. 15;
         TTEMP  at 36 range  0 .. 31;
         RCRC   at 40 range  0 .. 31;
         TCRC   at 44 range  0 .. 31;
      end record;

   for Access_SCC_Parameter_Memory_Map'Size use 48 * 8;

   SCC1_Parameter_Memory_Map        : Access_SCC_Parameter_Memory_Map;
   SCC1_Parameter_Memory_Map_Mirror : Access_SCC_Parameter_Memory_Map;
   for SCC1_Parameter_Memory_Map'Address use
      System'To_Address (IMMR + SCC1_Base);

   SCC2_Parameter_Memory_Map        : Access_SCC_Parameter_Memory_Map;
   SCC2_Parameter_Memory_Map_Mirror : Access_SCC_Parameter_Memory_Map;
   for SCC2_Parameter_Memory_Map'Address use
      System'To_Address (IMMR + SCC2_Base);

   SMC1_Parameter_Memory_Map        : Access_SCC_Parameter_Memory_Map;
   SMC1_Parameter_Memory_Map_Mirror : Access_SCC_Parameter_Memory_Map;
   for SMC1_Parameter_Memory_Map'Address use
      System'To_Address (IMMR + SMC1_Base);

   --------------------------------------------------
   --  SMC UART-Specific Parameter RAM Memory Map  --
   --------------------------------------------------
   --  see MPC860UM.pdf table 29-4
   type Access_SMC_UART_Parameter_Memory_Map is
      record
         MAX_IDL    : Natural range 0 .. 16#FFFF#;
         IDLC       : Natural range 0 .. 16#FFFF#;
         BRKLN      : Natural range 0 .. 16#FFFF#;
         BRKEC      : Natural range 0 .. 16#FFFF#;
         BRKCR      : Natural range 0 .. 16#FFFF#;
         R_MASK     : Natural range 0 .. 16#FFFF#;
      end record;

   for Access_SMC_UART_Parameter_Memory_Map use
      record
         MAX_IDL    at  0 range 0 .. 15;
         IDLC       at  2 range 0 .. 15;
         BRKLN      at  4 range 0 .. 15;
         BRKEC      at  6 range 0 .. 15;
         BRKCR      at  8 range 0 .. 15;
         R_MASK     at 10 range 0 .. 15;
      end record;
   for Access_SMC_UART_Parameter_Memory_Map'Size use 6 * 16;

   SMC1_UART_Parameter_Memory_Map        :
     Access_SMC_UART_Parameter_Memory_Map;
   SMC1_UART_Parameter_Memory_Map_Mirror :
     Access_SMC_UART_Parameter_Memory_Map;

   for SMC1_UART_Parameter_Memory_Map'Address use
      System'To_Address (IMMR + SMC1_Base + 16#28#);

   -----------------------------------
   --  Transmit on demand register  --
   -----------------------------------
   type Access_TODR is record
      Transmit_On_Demand : Boolean;
      Reserved15         : Reserved_15;
   end record;

   for Access_TODR use
      record
         Transmit_On_Demand at 0 range 0 ..  0;
         Reserved15         at 0 range 1 .. 15;
      end record;

   for Access_TODR'Size use 16;

   TODR1 : Access_TODR;
   for TODR1'Address use System'To_Address (IMMR + 16#A0C#);

   TODR2 : Access_TODR;
   for TODR2'Address use System'To_Address (IMMR + 16#A2C#);

   TODR3 : Access_TODR;
   for TODR3'Address use System'To_Address (IMMR + 16#A4C#);

      TODR4 : Access_TODR;
   for TODR4'Address use System'To_Address (IMMR + 16#A6C#);

   -----------------------------------------------------------
   --  SMC UART Event Register (SMCE)/Mask Register (SMCM)  --
   -----------------------------------------------------------
   type Access_SMCEM is
      record
         Reserced1a     : Boolean;
         Break_End      : Boolean;
         Reserved1b     : Boolean;
         Break_Received : Boolean;
         Reserved1c     : Boolean;
         Busy           : Boolean;
         Tx_Buffer      : Boolean;
         Rx_Buffer      : Boolean;
      end record;

   for Access_SMCEM use
      record
         Reserced1a     at 0 range 0 .. 0;
         Break_End      at 0 range 1 .. 1;
         Reserved1b     at 0 range 2 .. 2;
         Break_Received at 0 range 3 .. 3;
         Reserved1c     at 0 range 4 .. 4;
         Busy           at 0 range 5 .. 5;
         Tx_Buffer      at 0 range 6 .. 6;
         Rx_Buffer      at 0 range 7 .. 7;
      end record;

   for Access_SMCEM'Size use 8;

   type Int8 is range 0 .. 16#FF#;

   function To_SMCEM is new Ada.Unchecked_Conversion (Int8, Access_SMCEM);

   SMCE1        : Access_SMCEM;
   SMCE1_Mirror : Access_SMCEM;
   for SMCE1'Address use System'To_Address (IMMR + 16#A86#);
   SMCM1        : Access_SMCEM;
   SMCM1_Mirror : Access_SMCEM;
   for SMCM1'Address use System'To_Address (IMMR + 16#A8A#);

   SMCE2        : Access_SMCEM;
   SMCE2_Mirror : Access_SMCEM;
   for SMCE2'Address use System'To_Address (IMMR + 16#A96#);
   SMCM2        : Access_SMCEM;
   SMCM2_Mirror : Access_SMCEM;
   for SMCM2'Address use System'To_Address (IMMR + 16#A9A#);

   ------------------------------------------------------
   --  CPM Interrupt Pending/Mask/In-Service Register  --
   ------------------------------------------------------
   type Access_CPM_Interrupts is
      record
         PC15   : Boolean;
         Scc1   : Boolean;
         SCC2   : Boolean;
         SCC3   : Boolean;
         SCC4   : Boolean;
         PC14   : Boolean;
         TIMER1 : Boolean;
         PC13   : Boolean;
         PC12   : Boolean;
         SDMA   : Boolean;
         IDMA1  : Boolean;
         IDMA2  : Boolean;
         Res1a  : Boolean;
         TIMER2 : Boolean;
         RTT    : Boolean;
         I2C    : Boolean;
         PC11   : Boolean;
         PC10   : Boolean;
         Res1b  : Boolean;
         TIMER3 : Boolean;
         PC9    : Boolean;
         PC8    : Boolean;
         PC7    : Boolean;
         Res1c  : Boolean;
         TIMER4 : Boolean;
         PC6    : Boolean;
         SPI    : Boolean;
         SMC1   : Boolean;
         SMC2   : Boolean;
         PC5    : Boolean;
         PC4    : Boolean;
         Res1d  : Boolean;
      end record;

   for Access_CPM_Interrupts use
      record
         PC15   at 0 range  0 ..  0;
         SCC1   at 0 range  1 ..  1;
         SCC2   at 0 range  2 ..  2;
         SCC3   at 0 range  3 ..  3;
         SCC4   at 0 range  4 ..  4;
         PC14   at 0 range  5 ..  5;
         TIMER1 at 0 range  6 ..  6;
         PC13   at 0 range  7 ..  7;
         PC12   at 0 range  8 ..  8;
         SDMA   at 0 range  9 ..  9;
         IDMA1  at 0 range 10 .. 10;
         IDMA2  at 0 range 11 .. 11;
         Res1a  at 0 range 12 .. 12;
         TIMER2 at 0 range 13 .. 13;
         RTT    at 0 range 14 .. 14;
         I2C    at 0 range 15 .. 15;
         PC11   at 0 range 16 .. 16;
         PC10   at 0 range 17 .. 17;
         Res1b  at 0 range 18 .. 18;
         TIMER3 at 0 range 19 .. 19;
         PC9    at 0 range 20 .. 20;
         PC8    at 0 range 21 .. 21;
         PC7    at 0 range 22 .. 22;
         Res1c  at 0 range 23 .. 23;
         TIMER4 at 0 range 24 .. 24;
         PC6    at 0 range 25 .. 25;
         SPI    at 0 range 26 .. 26;
         SMC1   at 0 range 27 .. 27;
         SMC2   at 0 range 28 .. 28;
         PC5    at 0 range 29 .. 29;
         PC4    at 0 range 30 .. 30;
         Res1d  at 0 range 31 .. 31;
      end record;

   for Access_CPM_Interrupts'Size use 32;

   --------------------------------------------
   --  CPM Interrupt Configuration Register  --
   --------------------------------------------
   CIPR        : Access_CPM_Interrupts;
   CIPR_Mirror : Access_CPM_Interrupts;
   for CIPR'Address use System'To_Address (IMMR + 16#944#);

   -----------------------------------
   --  CPM Interrupt Mask Register  --
   -----------------------------------
   CIMR        : Access_CPM_Interrupts;
   CIMR_Mirror : Access_CPM_Interrupts;
   for CIMR'Address use System'To_Address (IMMR + 16#948#);

   -------------------------------
   --  CPM In-Service Register  --
   -------------------------------
   CISR        : Access_CPM_Interrupts;
   CISR_Mirror : Access_CPM_Interrupts;
   for CISR'Address use System'To_Address (IMMR + 16#094C#);

   --------------------------------------------
   --  CPM Interrupt Vector Register (CIVR)  --
   --------------------------------------------
   type Access_CIVR is record
      Vector_Number : Natural range 0 .. 31;
      Reserved10    : Reserved_10;
      Interrupt_Ack : Boolean;
   end record;

   for Access_CIVR use
      record
         Vector_Number at 0 range  0 ..  4;
         Reserved10    at 0 range  5 .. 14;
         Interrupt_Ack at 0 range 15 .. 15;
      end record;

   for Access_CIVR'Size use 16;

   CIVR        : Access_CIVR;
   CIVR_Mirror : Access_CIVR;
   for CIVR'Address use System'To_Address (IMMR + 16#930#);

   --------------------------------------------
   --  CPM Interrupt Configuration Register  --
   --------------------------------------------
   type Access_CICR is
      record
         Reserved8 : Reserved_8;
         SCdP      : Natural range 0 ..  3;
         SCcP      : Natural range 0 ..  3;
         SCbP      : Natural range 0 ..  3;
         SCaP      : Natural range 0 ..  3;
         IRL       : Natural range 0 ..  7;
         HP        : Natural range 0 .. 31;
         IEN       : Boolean;
         Reserved6 : Reserved_6;
         SPS       : Boolean;
      end record;

   for Access_CICR use
      record
         Reserved8 at 0 range  0 ..  7;
         SCdP      at 0 range  8 ..  9;
         SCcP      at 0 range 10 .. 11;
         SCbP      at 0 range 12 .. 13;
         SCaP      at 0 range 14 .. 15;
         IRL       at 0 range 16 .. 18;
         HP        at 0 range 19 .. 23;
         IEN       at 0 range 24 .. 24;
         Reserved6 at 0 range 25 .. 30;
         SPS       at 0 range 31 .. 31;
      end record;

   for Access_CICR'Size use 32;

   CICR        : Access_CICR;
   CICR_Mirror : Access_CICR;

   for CICR'Address use System'To_Address (IMMR + 16#940#);
   function To_CICR is new Ada.Unchecked_Conversion (Int32, Access_CICR);

   ----------------------------------------------
   --  General SCC Mode Register (High Order)  --
   ----------------------------------------------
   type Access_GSMR_H is
      record
         Reserved13           : Reserved_13;
         Infrared_Rx_Polarity : Boolean;
         Reserved1            : Reserved_1;
         Glitch_Detect_Enable : Boolean;
         Transparent_CRC      : Natural range 0 .. 3;
         Reverse_Data         : Boolean;
         Transparent_Rx       : Boolean;
         Transparent_Tx       : Boolean;
         CD_Pulse             : Boolean;
         CTS_Pulse            : Boolean;
         CD_Sampling          : Boolean;
         CTS_Sampling         : Boolean;
         Transmit_FIFO_Length : Boolean;
         Rx_FIFO_Width        : Boolean;
         Transm_Synch_To_Rvc  : Boolean;
         Sync_Length          : Natural range 0 .. 3;
         RTS_Mode             : Boolean;
         Receive_Sync_Timing  : Boolean;
      end record;

   for Access_GSMR_H use
      record
         Reserved13           at 0 range  0 .. 12;
         Infrared_Rx_Polarity at 0 range 13 .. 13;
         Reserved1            at 0 range 14 .. 14;
         Glitch_Detect_Enable at 0 range 15 .. 15;
         Transparent_CRC      at 0 range 16 .. 17;
         Reverse_Data         at 0 range 18 .. 18;
         Transparent_Rx       at 0 range 19 .. 19;
         Transparent_Tx       at 0 range 20 .. 20;
         CD_Pulse             at 0 range 21 .. 21;
         CTS_Pulse            at 0 range 22 .. 22;
         CD_Sampling          at 0 range 23 .. 23;
         CTS_Sampling         at 0 range 24 .. 24;
         Transmit_FIFO_Length at 0 range 25 .. 25;
         Rx_FIFO_Width        at 0 range 26 .. 26;
         Transm_Synch_To_Rvc  at 0 range 27 .. 27;
         Sync_Length          at 0 range 28 .. 29;
         RTS_Mode             at 0 range 30 .. 30;
         Receive_Sync_Timing  at 0 range 31 .. 31;
      end record;

   GSMR_H1        : Access_GSMR_H;
   GSMR_H1_Mirror : Access_GSMR_H;
   for GSMR_H1'Address use
      System'To_Address (IMMR + 16#A04#);

   GSMR_H2        : Access_GSMR_H;
   GSMR_H2_Mirror : Access_GSMR_H;
   for GSMR_H2'Address use
      System'To_Address (IMMR + 16#A24#);

   ---------------------------------------------
   --  General SCC Mode Register (Low Order)  --
   ---------------------------------------------
   type Access_GSMR_L is
      record
         Serial_Infrared_Encoding : Boolean;
         Clock_Edge               : Natural range 0 .. 3;
         Transmit_Clock_Invert    : Boolean;
         Transmit_Sense           : Natural range 0 .. 3;
         Rx_Input_Invert          : Boolean;
         Tx_Input_Invert          : Boolean;
         Tx_Preamble_Length       : Natural range 0 .. 7;
         Tx_Preamble_Pattern      : Natural range 0 .. 3;
         Transmitter_Frame_Ending : Boolean;
         Transm_DPLL_Clock_Rate   : Natural range 0 .. 3;
         Rcv_DPLL_Clock_Rate      : Natural range 0 .. 3;
         Rcv_Encoding_Method      : Natural range 0 .. 7;
         Trans_Encoding_Method    : Natural range 0 .. 7;
         Diagnostic_Mode          : Natural range 0 .. 3;
         Enable_Receive           : Boolean;
         Enable_Transmit          : Boolean;
         Channel_Protocol_Mode    : Natural range 0 .. 15;
      end record;

   for Access_GSMR_L use
      record
         Serial_Infrared_Encoding  at 0 range  0 ..  0;
         Clock_Edge at 0 range  1 ..  2;
         Transmit_Clock_Invert at 0 range  3 ..  3;
         Transmit_Sense  at 0 range  4 ..  5;
         Rx_Input_Invert at 0 range  6 ..  6;
         Tx_Input_Invert at 0 range  7 ..  7;
         Tx_Preamble_Length at 0 range  8 .. 10;
         Tx_Preamble_Pattern at 0 range 11 .. 12;
         Transmitter_Frame_Ending at 0 range 13 .. 13;
         Transm_DPLL_Clock_Rate at 0 range 14 .. 15;
         Rcv_DPLL_Clock_Rate at 0 range 16 .. 17;
         Rcv_Encoding_Method at 0 range 18 .. 20;
         Trans_Encoding_Method at 0 range 21 .. 23;
         Diagnostic_Mode at 0 range 24 .. 25;
         Enable_Receive  at 0 range 26 .. 26;
         Enable_Transmit  at 0 range 27 .. 27;
         Channel_Protocol_Mode at 0 range 28 .. 31;
      end record;
   for Access_GSMR_L'Size use 32;

   GSMR_L1        : Access_GSMR_L;
   GSMR_L1_Mirror : Access_GSMR_L;
   for GSMR_L1'Address use
      System'To_Address (IMMR + 16#A00#);

   GSMR_L2        : Access_GSMR_L;
   GSMR_L2_Mirror : Access_GSMR_L;
   for GSMR_L2'Address use
      System'To_Address (IMMR + 16#A20#);

   ----------------------------------
   --  SMC Mode Register (SMCMRn)  --
   ----------------------------------
   type Access_SMCMR is
      record
         Reserved1           : Reserved_1;
         Character_Length    : Natural range 0 .. 15;
         Stop_Length         : Boolean;
         Parity_Enable       : Boolean;
         Parity_Mode         : Boolean;
         Reserved2           : Reserved_2;
         SMC_Mode            : Natural range 0 .. 3;
         Diagnostic_Mode     : Natural range 0 .. 3;
         SMC_Transmit_Enable : Boolean;
         SMC_Receive_Enable  : Boolean;
      end record;

   for Access_SMCMR use
      record
         Reserved1           at 0 range  0 ..  0;
         Character_Length    at 0 range  1 ..  4;
         Stop_Length         at 0 range  5 ..  5;
         Parity_Enable       at 0 range  6 ..  6;
         Parity_Mode         at 0 range  7 ..  7;
         Reserved2           at 0 range  8 ..  9;
         SMC_Mode            at 0 range 10 .. 11;
         Diagnostic_Mode     at 0 range 12 .. 13;
         SMC_Transmit_Enable at 0 range 14 .. 14;
         SMC_Receive_Enable  at 0 range 15 .. 15;
      end record;

   for Access_SMCMR'Size use 16;

   function To_SMCMR is new Ada.Unchecked_Conversion (Int16, Access_SMCMR);

   SMCMR1        : Access_SMCMR;
   SMCMR1_Mirror : Access_SMCMR;
   for SMCMR1'Address use System'To_Address (IMMR + 16#A82#);

   SMCMR2        : Access_SMCMR;
   SMCMR2_Mirror : Access_SMCMR;
   for SMCMR2'Address use System'To_Address (IMMR + 16#A92#);

   ---------------------------------------
   --  Protocol Specific Mode Register  --
   ---------------------------------------
   --  see MP860UM.pdf chapter 22.16
   type Access_PSMR_UART is
      record
         FLC      : Boolean;
         SL       : Boolean;
         CL       : Natural range 0 .. 3;
         UM       : Natural range 0 .. 3;
         FRZ      : Boolean;
         RZS      : Boolean;
         SYN      : Boolean;
         DRT      : Boolean;
         Reserved : Boolean;
         PEN      : Boolean;
         RPM      : Natural range 0 .. 3;
         TPM      : Natural range 0 .. 3;
      end record;

   for Access_PSMR_UART use
      record
         FLC      at 0 range  0 ..  0;
         SL       at 0 range  1 ..  1;
         CL       at 0 range  2 ..  3;
         UM       at 0 range  4 ..  5;
         FRZ      at 0 range  6 ..  6;
         RZS      at 0 range  7 ..  7;
         SYN      at 0 range  8 ..  8;
         DRT      at 0 range  9 ..  9;
         Reserved at 0 range 10 .. 10;
         PEN      at 0 range 11 .. 11;
         RPM      at 0 range 12 .. 13;
         TPM      at 0 range 14 .. 15;
      end record;

   for Access_PSMR_UART'Size use 16;

   PSMR1        : Access_PSMR_UART;
   PSMR1_Mirror : Access_PSMR_UART;
   for PSMR1'Address use System'To_Address (IMMR + 16#A08#);

   PSMR2        : Access_PSMR_UART;
   PSMR2_Mirror : Access_PSMR_UART;
   for PSMR2'Address use System'To_Address (IMMR + 16#A28#);

   ---------------------------------
   --  Receive Buffer Descriptor  --
   ---------------------------------
   type RxBD is
      record
         Empty               : Boolean;
         Reserved1a          : Boolean;
         Wrap                : Boolean;
         Interrupt           : Boolean;
         Control_Character   : Boolean;
         Address             : Boolean;
         Continuous_Mode     : Boolean;
         Reception_Of_Idles  : Boolean;
         Address_Match       : Boolean;
         Reserved1b          : Boolean;
         Break_Received      : Boolean;
         Framing_Error       : Boolean;
         Parity_Error        : Boolean;
         Reserved1c          : Boolean;
         Overrun             : Boolean;
         Carrier_Detect_Lost : Boolean;
         Data_Length         : Natural range 0 .. 16#FFFF#;
         Rx_Buffer_Pointer   : System.Address;
      end record;

   for RxBD use
      record
         Empty               at 0 range  0 ..  0;
         Reserved1a          at 0 range  1 ..  1;
         Wrap                at 0 range  2 ..  2;
         Interrupt           at 0 range  3 ..  3;
         Control_Character   at 0 range  4 ..  4;
         Address             at 0 range  5 ..  5;
         Continuous_Mode     at 0 range  6 ..  6;
         Reception_Of_Idles  at 0 range  7 ..  7;
         Address_Match       at 0 range  8 ..  8;
         Reserved1b          at 0 range  9 ..  9;
         Break_Received      at 0 range 10 .. 10;
         Framing_Error       at 0 range 11 .. 11;
         Parity_Error        at 0 range 12 .. 12;
         Reserved1c          at 0 range 13 .. 13;
         Overrun             at 0 range 14 .. 14;
         Carrier_Detect_Lost at 0 range 15 .. 15;
         Data_Length         at 0 range 16 .. 31;
         Rx_Buffer_Pointer   at 0 range 32 .. 63;
      end record;

   for RxBD'Size use 64;
   --  Address for RxBD is dynamically assigned

   ----------------------------------
   --  Transmit Buffer Descriptor  --
   ----------------------------------
   type TxBD is
      record
         Ready                : Boolean;
         Reserved1            : Reserved_1;
         Wrap                 : Boolean;
         Interrupt            : Boolean;
         Reserved2            : Reserved_2;
         Continuous_Mode      : Boolean;
         Preamble             : Boolean;
         Reserved8            : Reserved_6;
         Data_Length          : Natural range 0 .. 16#FFFF#;
         Tx_Buffer_Pointer    : System.Address;
      end record;

   for TxBD use
      record
         Ready                at 0 range  0 ..  0;
         Reserved1            at 0 range  1 ..  1;
         Wrap                 at 0 range  2 ..  2;
         Interrupt            at 0 range  3 ..  3;
         Reserved2            at 0 range  4 ..  5;
         Continuous_Mode      at 0 range  6 ..  6;
         Preamble             at 0 range  7 ..  7;
         Reserved8            at 0 range  8 .. 15;
         Data_Length          at 0 range 16 .. 31;
         Tx_Buffer_Pointer    at 0 range 32 .. 63;
      end record;

   for TxBD'Size use 64;
   --  Address for TxBD is dynamically assigned

   --------------------------
   --  SCC Event Register  --
   --------------------------

   type Access_SCCEM is
      record
         Reserved3                          : Reserved_3;
         Glitch_On_Rx                       : Boolean;
         Glitch_On_Tx                       : Boolean;
         Reserved1a                         : Reserved_1;
         Autobaud                           : Boolean;
         Idle_Sequence_Status_Changed       : Boolean;
         Gracefull_Stop_Complete            : Boolean;
         Break_End                          : Boolean;
         Break_Start                        : Boolean;
         Reserved1b                         : Boolean;
         Control_Char_Received_And_Rejected : Boolean;
         Busy                               : Boolean;
         Tx_Event                           : Boolean;
         Rx_Event                           : Boolean;
      end record;

   for Access_SCCEM use
      record
         Reserved3                          at 0 range  0 ..  2;
         Glitch_On_Rx                       at 0 range  3 ..  3;
         Glitch_On_Tx                       at 0 range  4 ..  4;
         Reserved1a                         at 0 range  5 ..  5;
         Autobaud                           at 0 range  6 ..  6;
         Idle_Sequence_Status_Changed       at 0 range  7 ..  7;
         Gracefull_Stop_Complete            at 0 range  8 ..  8;
         Break_End                          at 0 range  9 ..  9;
         Break_Start                        at 0 range 10 .. 10;
         Reserved1b                         at 0 range 11 .. 11;
         Control_Char_Received_And_Rejected at 0 range 12 .. 12;
         Busy                               at 0 range 13 .. 13;
         Tx_Event                           at 0 range 14 .. 14;
         Rx_Event                           at 0 range 15 .. 15;
      end record;

   for Access_SCCEM'Size use 16;

   function To_SCCEM is new Ada.Unchecked_Conversion (Int16, Access_SCCEM);

   SCCE2        : Access_SCCEM;
   SCCE2_Mirror : Access_SCCEM;

   for SCCE2'Address use System'To_Address (IMMR + 16#A30#);

   -------------------------
   --  SCC Mask Register  --
   -------------------------
   SCCM1        : Access_SCCEM;
   SCCM1_Mirror : Access_SCCEM;
   for SCCM1'Address use System'To_Address (IMMR + 16#A14#);

   SCCM2        : Access_SCCEM;
   SCCM2_Mirror : Access_SCCEM;
   for SCCM2'Address use System'To_Address (IMMR + 16#A34#);

end System.BB.Peripherals.MPC860;
