------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--               S Y S T E M . B B . C P U _ P R I M I T I V E S            --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--        Copyright (C) 1999-2002 Universidad Politecnica de Madrid         --
--             Copyright (C) 2003-2005 The European Space Agency            --
--                     Copyright (C) 2003-2010, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
-- The porting of GNARL to bare board  targets was initially  developed  by --
-- the Real-Time Systems Group at the Technical University of Madrid.       --
--                                                                          --
------------------------------------------------------------------------------

--  This package implements PowerPC architecture specific support for the GNAT
--  Ravenscar run time.

with Interfaces.C;

with System.Storage_Elements;
with System.BB.Interrupts;
with System.BB.Threads.Queues;
with System.BB.Protection;
with System.BB.Peripherals;

with System.BB.Peripherals.MPC860; use System.BB.Peripherals.MPC860;

package body System.BB.CPU_Primitives is

   package SSE renames System.Storage_Elements;
   use type SSE.Integer_Address;
   use type SSE.Storage_Offset;

   -------------
   -- Context --
   -------------

   SP       : constant Context_Id :=  1; --  see context_switch_ppc.s
   PC       : constant Context_Id :=  0;
   Arg_Reg3 : constant Context_Id :=  132 / 4;

   --  The following seems backwards: the context switch procedure should
   --  be written in assembly, the rest should be in Ada with machine code
   --  insertions where necessary ???

   procedure Context_Switch_Asm;
   pragma Export (Asm, Context_Switch_Asm, "context_switch_asm");

   ------------------------
   -- Context_Switch_Asm --
   ------------------------

   procedure Context_Switch_Asm is
   begin
      Context_Switch;
   end Context_Switch_Asm;

   -----------------
   -- Get_Context --
   -----------------

   function Get_Context
     (Context : Context_Buffer;
      Index   : Context_Id) return Word
   is
   begin
      return Word (Context (Index));
   end Get_Context;

   ------------------------
   -- Initialize_Context --
   ------------------------

   procedure Initialize_Context
     (Buffer          : not null access Context_Buffer;
      Program_Counter : System.Address;
      Argument        : System.Address;
      Stack_Pointer   : System.Address)
   is
      use Interfaces.C;

      type Context_Ptr is access all System.BB.CPU_Primitives.Context_Buffer;
      procedure Save_Context (env : Context_Ptr);
      pragma Import (C, Save_Context, "save_context");

      use System.Storage_Elements;

      Minimum_Stack_Size_In_Bytes : constant
        System.Storage_Elements.Integer_Address := Standard'Maximum_Alignment;

   begin
      --  Store Registers to Buffer

      Save_Context (Context_Ptr (Buffer));

      --  Overwrite Stack Pointer and Program Counter with values that have
      --  been passed as arguments. The Stack Pointer of the task is 2 words
      --  below Stack_Pointer. These two words correspond to the header of the
      --  new stack. This header contains the LR_Save_Word and Back_Chain_Word.
      --  Program_Counter points to the task_wrapper procedure.

      --  We create a new stack pointer with a size of at least 8 which are
      --  reserved for the header, but we also have to make sure that the stack
      --  is aligned with Standard'Maximum_Alignment

      Buffer (SP) := To_Address (To_Integer (Stack_Pointer) -
                                   Minimum_Stack_Size_In_Bytes);
      Buffer (PC) := Program_Counter;

      --  safe Argument into the buffer so we can restore it later in the call
      --  Restore_Context. This is a bit tricky! See procedure Context_Switch
      --  (below) and Restore_Context (context_switch.s) for a better
      --  understanding!

      Buffer (Arg_Reg3) := Argument;

      declare
         LR_Save_Word : System.Address;
         for LR_Save_Word'Address use
            To_Address (To_Integer (Stack_Pointer) -
                        Minimum_Stack_Size_In_Bytes + 4);

         Back_Chain_Word : System.Address;
         for Back_Chain_Word'Address use
            To_Address (To_Integer (Stack_Pointer) -
                        Minimum_Stack_Size_In_Bytes);

      begin
         --  Put Null to these two values since the task is not returning

         LR_Save_Word    := Null_Address;
         Back_Chain_Word := Null_Address;
      end;
   end Initialize_Context;

   -------------------------------
   -- Initialize_Floating_Point --
   -------------------------------

   procedure Initialize_Floating_Point is
   begin
      --  There is no floating point unit and therefore we have a null body

      null;
   end Initialize_Floating_Point;

   ----------------------------
   -- Install_Error_Handlers --
   ----------------------------

   procedure Install_Error_Handlers is
   begin
      --  To be implemented ???
      null;
   end Install_Error_Handlers;

   ---------------------
   -- Install_Handler --
   ---------------------

   procedure Install_Handler
     (Service_Routine : System.Address;
      Vector          : Vector_Id;
      Synchronous     : Boolean := False)
   is
      pragma Unreferenced (Synchronous);

      procedure Copy_Handler
        (Service_Routine : System.Address;
         Vector          : Vector_Id;
         Id              : Interrupts.Interrupt_ID);
      pragma Import (Asm, Copy_Handler, "copy_handler");
   begin
      Copy_Handler (Service_Routine, Vector, 0);
      --  Copy_Handler should not need a Interrupt_ID ???
   end Install_Handler;

   ------------------------
   -- Disable_Interrupts --
   ------------------------

   procedure Disable_Interrupts is

      procedure Disable_External_Interrupt;
      pragma Import (C, Disable_External_Interrupt,
                     "disable_external_interrupt");

   begin
      Disable_External_Interrupt;
   end Disable_Interrupts;

   -----------------------
   -- Enable_Interrupts --
   -----------------------

   procedure Enable_Interrupts
     (Level : System.BB.Parameters.Interrupt_Level)
   is
      --  Create an empty mask

      Mask : Bool32 := (others => False);

      procedure Enable_External_Interrupt;
      pragma Import (C, Enable_External_Interrupt,
                     "enable_external_interrupt");

   begin
      if Level > 1 then

         --  We are at the level of external interrupts

         --  Enable interrupts above Level

         for I in Mask'First .. Level - 2 loop
            Mask (I) := True;
         end loop;

         --  Apply mask

         SIMASK := To_SIMASK (Mask);

         --  Enable interrupts that have not been masked

         Enable_External_Interrupt;
      end if;
   end Enable_Interrupts;

   --------------------
   -- Context_Switch --
   --------------------

   procedure Context_Switch is

      use Interfaces.C;

      type Context_Ptr is access all System.BB.CPU_Primitives.Context_Buffer;

      function Save_Context (env : Context_Ptr) return int;
      pragma Import (C, Save_Context, "save_context");

      procedure Restore_Context (env : Context_Ptr; val : int);
      pragma Import (C, Restore_Context, "restore_context");

   begin
      System.BB.Protection.Enter_Kernel;

      --  Save state of running thread

      if Save_Context
        (System.BB.Threads.Queues.Running_Thread.Context'Access) = 0 then

         --  Not a return from Restore_Context since Save_Context = 0

         --  Update the new running thread

         System.BB.Threads.Queues.Running_Thread :=
           System.BB.Threads.Queues.First_Thread;

         --  Restore state of the first thread in the queue The value at
         --  Context (Arg_R3) corresponds to the argument that is passed to the
         --  Initialize_Context procedure. The function Restore_Context returns
         --  this value by placing it into the general purpose Register 3 and
         --  therefore it is passed as argument to the procedure task_wrapper.

         --  This is a bit tricky!

         Restore_Context
           (System.BB.Threads.Queues.First_Thread.Context'Access,
            int (System.BB.Threads.Queues.First_Thread.Context (Arg_Reg3)));

      end if;
   end Context_Switch;

   -----------------
   -- Set_Context --
   -----------------

   procedure Set_Context
     (Context : in out Context_Buffer;
      Index   : Context_Id;
      Value   : Word) is
   begin
      Context (Index) := Address (Value);
   end Set_Context;

end System.BB.CPU_Primitives;
