##############################################################################
#                                                                            #
#                         Copyright (C) 2005, AdaCore                        #
#                                                                            #
#                               Assembly File                                #
#                                                                            #
##############################################################################


	#####################################
	# Masks for Machine Status Register #
	#####################################

	MSR_EE=0x00008000

	# INTERRUPT HANDLER

	# When an interrupt occurs, the processor moves the address
	# of the instruction that caused the interrupt into register SRR0
	# and copies the machine state register (msr) into register SRR1.
	# The interrupt enable bit (EE) in the MSR is then set to 0
	# thereby disabling all externel interrupts. The processor begins
	# then execution of the exception handler in the exception vector
	# table at the vector offset determined by the interrupts source.
	# To return from the exception handler an rfi instruction has to be
	# executed thereby copying the stored register srr1 to msr and
	# continuing execution at *SRR0.

#ifdef NOFPU
#define FRAME_SIZE 80
#else        
#define FRAME_SIZE (80 + 15 * 8)
#endif

        .type handler_start,@function
handler_start:
	# Allocate a frame.
        stwu    %r1,-FRAME_SIZE(%r1)

        # Save volatile registers (r0, r3-r12)
        stw     %r0,8(%r1)
        stw     %r3,12(%r1)
        stw     %r4,16(%r1)
        stw     %r5,20(%r1)
        stw     %r6,24(%r1)
        stw     %r7,28(%r1)
        stw     %r8,32(%r1)
        stw     %r9,36(%r1)
        stw     %r10,40(%r1)
        stw     %r11,44(%r1)
        stw     %r12,48(%r1)
        # .. ctr, xer, lr, cr
        mfxer   %r3
        mflr    %r4
        mfctr   %r5
        mfcr    %r6
        stw     %r3,52(%r1)
        stw     %r4,56(%r1)
        stw     %r5,60(%r1)
        stw     %r6,64(%r1)

        # Save srr0 (ip) and srr1 (msr)
        mfsrr0  %r3
        mfsrr1  %r4
        stw     %r3,68(%r1)
        stw     %r4,72(%r1)
        
#ifndef NOFPU
        # First, re-enable the fpu.
        mfmsr   %r3
        ori     %r3,%r3,0x2000
        mtmsr   %r3

        # Save volatile FP regs
        stfd    %f0,80(%r1)
        stfd    %f1,88(%r1)
        stfd    %f2,96(%r1)
        stfd    %f3,104(%r1)
        stfd    %f4,112(%r1)
        stfd    %f5,120(%r1)
        stfd    %f6,128(%r1)
        stfd    %f7,136(%r1)
        stfd    %f8,144(%r1)
        stfd    %f9,152(%r1)
        stfd    %f10,160(%r1)
        stfd    %f11,168(%r1)
        stfd    %f12,176(%r1)
        stfd    %f13,184(%r1)

        # Save FPSCR
        mffs    %f0
        stfd    %f0,192(%r1)
#endif
.Lset_handler_arg:
        # Note: the 4 instructions below will be overwritten.  Do not modify
        # them!
        lis     9,0@h	# handler address
        ori     9,9,0@l
        lis     3,0@h   # handler argument
        ori     3,3,0@l

        mtctr   9

        # Set the continuation point.
       	lis      4, handler_cont@h
	ori      4, 4, handler_cont@l
	mtlr     4

        # Call user handler.
        bctr
.Lhandler_end:
        .size handler_start, . - handler_start

# Common code.
# This is not put into the handler because it would exceed the size
# and because there are two calls to the run-time.
        
        .type handler_cont, @function
handler_cont:   

	# Disable externel interrupts and machine check interrupts
	mfmsr    3
	li       4, MSR_EE@l
	andc     3, 3, 4
        mtmsr    3

	###################
	# Check if context switch is needed
	bl       context_switch_needed
        
	# context switch needed?
	cmpwi   cr7, 3, 0
	beq-    cr7, .Lno_con_sw

	# Yes, we need a context switch
	bl      context_switch

.Lno_con_sw:
#ifndef NOFPU
	lfd    %f0,192(%r1)
        mtfsf  0xff,%f0
        lfd    %f0,80(%r1)
        lfd    %f1,88(%r1)
        lfd    %f2,96(%r1)
        lfd    %f3,104(%r1)
        lfd    %f4,112(%r1)
        lfd    %f5,120(%r1)
        lfd    %f6,128(%r1)
        lfd    %f7,136(%r1)
        lfd    %f8,144(%r1)
        lfd    %f9,152(%r1)
        lfd    %f10,160(%r1)
        lfd    %f11,168(%r1)
        lfd    %f12,176(%r1)
        lfd    %f13,184(%r1)
#endif

       	# Restore srr0 and srr1
        lwz     3,68(%r1)
        lwz     4,72(%r1)
        mtsrr0  3
        mtsrr1  4

        # Restore ctr, xer, lr, cr
        lwz     3,52(1)
        lwz     4,56(1)
        lwz     5,60(1)
        lwz     6,64(1)
        mtxer   3
        mtlr    4
        mtctr   5
        mtcr    6

        # Restore r0, r3-r12
        lwz     0,8(1)
        lwz     3,12(1)
        lwz     4,16(1)
        lwz     5,20(1)
        lwz     6,24(1)
        lwz     7,28(1)
        lwz     8,32(1)
        lwz     9,36(1)
        lwz     10,40(1)
        lwz     11,44(1)
        lwz     12,48(1)

        addi    1,1,FRAME_SIZE
	rfi

        .size handler_cont, . - handler_cont


#############################
##                         ##
##       Copy_Handler      ##
##                         ##
#############################
#
# Copy the exception handler to a given address. The address
# of the handler is given in GPR3 and the destination address
# is passed in GPR4. The function copy handler copies the
# ISR function above between extern_exc_start and extern_exc_stop
# to the address given in GPR4. The ISR first executes some prologue,
# branches to the specific handler at address specified by GPR5 and
# executues the epilogue.
# The specifiec handler is passed an argument that is the interrupt ID
# that is determined by a call to To_Interrupt.
# This argument is passed via GPR3.
#
# Arguments
#         GPR3   Address of Exception Handler Routine
#         GPR4   Trap Address
#         GPR5   Exception Handler Parameter

	.global copy_handler
	.type   copy_handler, @function
copy_handler:

        # First copy handler.
        subi    8,4,4
        lis     6,handler_start@h	# r6: start of handler
        ori     6,6,handler_start@l
        lis     7,.Lhandler_end@h		# r7: end of handler
        ori     7,7,.Lhandler_end@l
.Lcopy_loop:
        cmp     0,0,6,7			# exit loop if r6 = r7
        bge     .Lcopy_end
        lwz     9,0(6)			# read word from r6
        stwu    9,4(8)			# write it to (r8+4)
        addi    6,6,4			# Add 4 to r6
        b       .Lcopy_loop

.Lcopy_end:
        # Second, write instructions that set argument and handler.
        addi    8,4,(.Lset_handler_arg - handler_start - 4)

      	srwi    6,3,16                  #  Shift right 16 bits.
       	oris    6,6,0x3d200000@h        # opcode for 'lis 9,x'
        stwu    6,4(8)

        clrlwi  6,3,16		        # clear 16 high-order bits
	oris    6,6,0x61290000@h	# opcode for 'ori 9,9,x'
        stwu    6,4(8)

        srwi    6,5,16                  #  Shift right 16 bits.
        oris    6,6,0x3c600000@h        # opcode for 'lis 3,x'
        stwu    6,4(8)

        clrlwi  6,5,16		        # clear 16 high-order bits
       	oris    6,6,0x60630000@h	# opcode for 'ori 3,3,x'
        stwu    6,4(8)

        # Flush data cache and invalidate instruction cache.
        addi    8,4,(.Lhandler_end-handler_start)
.Lflush:
        cmp     cr7,4,8
        beq     cr7,.Lflush_done
        dcbst   0,4	# Data Cache Block Store
        icbi    0,4	# Invalidate copy of storage
        addi    4,4,4
        b       .Lflush

.Lflush_done:
        sync
        blr
        .size   copy_handler, . - copy_handler
