------------------------------------------------------------------------------
--                                  wPOSIX                                  --
--                                                                          --
--                     Copyright (C) 2008-2012, AdaCore                     --
--                                                                          --
--  This library is free software;  you can redistribute it and/or modify   --
--  it under terms of the  GNU General Public License  as published by the  --
--  Free Software  Foundation;  either version 3,  or (at your  option) any --
--  later version. This library is distributed in the hope that it will be  --
--  useful, but WITHOUT ANY WARRANTY;  without even the implied warranty of --
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                    --
--                                                                          --
--  As a special exception under Section 7 of GPL version 3, you are        --
--  granted additional permissions described in the GCC Runtime Library     --
--  Exception, version 3.1, as published by the Free Software Foundation.   --
--                                                                          --
--  You should have received a copy of the GNU General Public License and   --
--  a copy of the GCC Runtime Library Exception along with this program;    --
--  see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see   --
--  <http://www.gnu.org/licenses/>.                                         --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
------------------------------------------------------------------------------

with POSIX_Win32;

package body POSIX.Process_Times is

   function Filetime_To_Tick
     (Filetime : Win32.Winbase.FILETIME) return Tick_Count;
   --  ???

   ------------------------------------
   -- Descendants_System_CPU_Time_Of --
   ------------------------------------

   function Descendants_System_CPU_Time_Of
     (Times : Process_Times) return Tick_Count is
   begin
      return Times.Children_System_Time;
   end Descendants_System_CPU_Time_Of;

   ----------------------------------
   -- Descendants_User_CPU_Time_Of --
   ----------------------------------

   function Descendants_User_CPU_Time_Of
     (Times : Process_Times) return Tick_Count is
   begin
      return Times.Children_User_Time;
   end Descendants_User_CPU_Time_Of;

   -----------------------
   -- Elapsed_Real_Time --
   -----------------------

   function Elapsed_Real_Time return Tick_Count is
   begin
      return Elapsed_Real_Time_Of (Get_Process_Times);
   end Elapsed_Real_Time;

   --------------------------
   -- Elapsed_Real_Time_Of --
   --------------------------

   function Elapsed_Real_Time_Of
     (Times : Process_Times) return Tick_Count
   is
      use type Win32.DWORD;

      Ok      : Win32.BOOL;
      Ctimes  : aliased Win32.Winbase.SYSTEMTIME;
      Now     : aliased Win32.Winbase.FILETIME;
      Elapsed : Win32.Winbase.FILETIME;
   begin
      Win32.Winbase.GetSystemTime (Ctimes'Unchecked_Access);
      Ok := Win32.Winbase.SystemTimeToFileTime
        (Ctimes'Unchecked_Access, Now'Unchecked_Access);
      POSIX_Win32.Check_Result (Ok, "SystemTimeToFileTime");

      Elapsed.dwLowDateTime  :=
        Now.dwLowDateTime  - Times.Creation_Time.dwLowDateTime;
      Elapsed.dwHighDateTime :=
        Now.dwHighDateTime - Times.Creation_Time.dwHighDateTime;

      if Now.dwLowDateTime < Times.Creation_Time.dwLowDateTime then
         Elapsed.dwHighDateTime := Elapsed.dwHighDateTime - 1;
      end if;

      return Filetime_To_Tick (Elapsed);
   end Elapsed_Real_Time_Of;

   ----------------------
   -- Filetime_To_Tick --
   ----------------------

   function Filetime_To_Tick
     (Filetime : Win32.Winbase.FILETIME) return Tick_Count
   is
      use type Win32.ULONG;
      Hundred_Nano_To_Tick : constant := 1E7 / Ticks_Per_Second;

      --  ??? change the code to handle DwHighDateTime (see code in v1.4)
   begin
      return Tick_Count (Filetime.dwLowDateTime / Hundred_Nano_To_Tick);
   end Filetime_To_Tick;

   -----------------------
   -- Get_Process_Times --
   -----------------------

   function Get_Process_Times return Process_Times is
      Result                   : Win32.BOOL;
      Creation_Time, Exit_Time : aliased Win32.Winbase.FILETIME;
      Kernel_Time, User_Time   : aliased Win32.Winbase.FILETIME;
      PT                       : Process_Times;
   begin
      Result := Win32.Winbase.GetProcessTimes
        (Win32.Winbase.GetCurrentProcess,
         Creation_Time'Unchecked_Access,
         Exit_Time'Unchecked_Access,
         Kernel_Time'Unchecked_Access,
         User_Time'Unchecked_Access);
      POSIX_Win32.Check_Result (Result, "Get_Process_Times");

      PT := (Creation_Time => Creation_Time,
             User_Time     => Filetime_To_Tick (User_Time),
             System_Time   => Filetime_To_Tick (Kernel_Time),
             Children_User_Time |
             Children_System_Time => 0);
      return PT;
   end Get_Process_Times;

   ------------------------
   -- System_CPU_Time_Of --
   ------------------------

   function System_CPU_Time_Of (Times : Process_Times) return Tick_Count is
   begin
      return Times.System_Time;
   end System_CPU_Time_Of;

   ----------------------
   -- User_CPU_Time_Of --
   ----------------------

   function User_CPU_Time_Of (Times : Process_Times) return Tick_Count is
   begin
      return Times.User_Time;
   end User_CPU_Time_Of;

end POSIX.Process_Times;
