(* This is the main file of gnatwhy3 *)

(* Gnatwhy3 does the following:
   - it reads a .mlw file that was generated by gnat2why
   - it computes the VCs
   - it runs Alt-ergo on each VC.
   - it outputs messages for each "thing to prove" in Ada, in terms that the
     Ada-programmer can understand.

   See gnat_objectives.mli for the notion of objective and goal.

   gnat_main can be seen as the "driver" for gnatwhy3. Much of the
   functionality is elsewhere.
   Specifically, this file does:
      - compute the objective that belongs to a goal/VC
      - drive the scheduling of VCs, and handling of results
      - output the messages
*)

open Why3
open Term

let search_labels =
  (* fold over the term to find the explanation *)
  let rec search_labels acc f =
    let acc =
      match Gnat_expl.extract_explanation f.t_label with
      | Gnat_expl.Expl e -> Some e
      | _ -> acc
    in
    t_fold search_labels acc f
  in
  search_labels None

let print ?(endline=true) b task expl =
   (* Print a positive or negative message for objectives *)
   if endline then
      Format.printf "%a@." (Gnat_expl.print_expl b task) expl
   else
      Format.printf "%a" (Gnat_expl.print_expl b task) expl

let rec is_trivial fml =
   (* Check wether the formula is trivial.  *)
   match fml.t_node with
   | Ttrue -> true
   | Tquant (_,tq) ->
         let _,_,t = t_open_quant tq in
         is_trivial t
   | Tlet (_,tb) ->
         let _, t = t_open_bound tb in
         is_trivial t
   | Tbinop (Timplies,_,t2) ->
         is_trivial t2
   | Tcase (_,tbl) ->
         List.for_all (fun b ->
            let _, t = t_open_branch b in
            is_trivial t) tbl
   | _ -> false

let register_goal goal =
   (* Register the goal by extracting the explanation and trace. If the goal is
    * trivial, do not register *)
   let task = Session.goal_task goal in
   let fml = Task.task_goal_fmla task in
   match is_trivial fml, search_labels fml with
   | true, None ->
         Gnat_objectives.set_not_interesting goal
   | _, None ->
         Gnat_util.abort_with_message
         "Task has no tracability label."
   | _, Some e ->
       Gnat_objectives.add_to_objective e goal

let is_digit c =
  match c with
  | '0' .. '9' -> true
  | _ -> false

let extract_steps s =
  (* extract steps from alt-ergo "valid" output; return None if output is not
     recognized, or no steps information present *)
  (* We simply search for (xxx) at the end of the first line of the  output,
     where all the xxx must be digits. *)
  let s =
    try Strings.slice s 0 (String.index s '\n' )
    with Not_found -> s
  in
  try
    let len = String.length s in
    if len = 0 then None
    else
      let i = ref (len - 1) in
      (* skip spaces *)
      while s.[!i] = ' ' do
        i := !i - 1;
      done;
      if s.[!i] = ')' then begin
        let max = !i in
        while !i > 0 && is_digit s.[!i-1] do
          i := !i - 1;
        done;
        if !i > 0 && s.[!i-1] = '(' then
          let s = Strings.slice s !i max in
          Some (int_of_string s)
        else None
      end else None
  with _ -> None

let extract_steps_fail s =
  if Strings.starts_with s "steps:" then
    try Some (int_of_string (Strings.slice s 6 (String.length s)))
    with _ -> None
  else None

let print_statistics fmt prover_result =
  (* print statistics about the proof result. *)
  let time = prover_result.Call_provers.pr_time in
  match prover_result.Call_provers.pr_answer with
  | Call_provers.Valid ->
      begin match extract_steps prover_result.Call_provers.pr_output with
      | Some steps -> Format.fprintf fmt "%.2fs - %d steps" time steps
      | None -> Format.fprintf fmt "%.2fs" time
      end
  | Call_provers.Timeout ->
      Format.fprintf fmt "Timeout: %.2fs" prover_result.Call_provers.pr_time
  | Call_provers.Failure s ->
      begin match extract_steps_fail s with
      | Some steps -> Format.fprintf fmt "%.2fs - %d steps" time steps
      | None -> Format.fprintf fmt "Failure: %s - %.2fs" s time
      end
  | _ -> ()

let rec handle_vc_result goal result prover_result =
   (* This function is called when the prover has returned from a VC.
       goal           is the VC that the prover has dealt with
       result         a boolean, true if the prover has proved the VC
       prover_result  the actual proof result, to extract statistics
   *)
   let obj, status = Gnat_objectives.register_result goal result in
   Gnat_objectives.display_progress ();
   match status with
   | Gnat_objectives.Proved ->
         begin match Gnat_config.report, prover_result with
         | Gnat_config.Fail_And_Proved, _ ->
             print true (Session.goal_task goal) obj
         | Gnat_config.Statistics, Some result ->
             print ~endline:false true (Session.goal_task goal) obj;
             Format.printf " (%a)@." print_statistics result
         | _ -> ()
         end
   | Gnat_objectives.Not_Proved ->
         if Gnat_config.proof_mode = Gnat_config.Then_Split then begin
            Gnat_objectives.Save_VCs.save_trace goal
         end;
         begin match Gnat_config.report, prover_result with
         | Gnat_config.Statistics, Some result ->
            print ~endline:false false (Session.goal_task goal)
               (Gnat_objectives.get_objective goal);
            Format.printf " (%a)@." print_statistics result
         | _ ->
            print false (Session.goal_task goal)
              (Gnat_objectives.get_objective goal)
         end
   | Gnat_objectives.Work_Left ->
         match Gnat_objectives.next obj with
         | Some g -> schedule_goal g
         | None -> ()

and interpret_result pa pas =
   (* callback function for the scheduler, here we filter if an interesting
      goal has been dealt with, and only then pass on to handle_vc_result *)
   match pas with
   | Session.Done r ->
         let goal = pa.Session.proof_parent in
         let answer = r.Call_provers.pr_answer in
         if answer = Call_provers.HighFailure then begin
            Format.eprintf "An error occured when calling alt-ergo@.";
            if Gnat_config.verbose = Gnat_config.Verbose then begin
               Format.eprintf "%s@." r.Call_provers.pr_output
            end;
         end;
         handle_vc_result goal (answer = Call_provers.Valid) (Some r)
   | _ ->
         ()


and schedule_goal (g : Gnat_objectives.goal) =
   (* schedule a goal for proof - the goal may not be scheduled actually,
      because we detect that it is not necessary. This may have several
      reasons:
         * command line given to skip proofs
         * goal already proved
         * goal already attempted with identical options
   *)

   (* first deal with command line options *)
   if Gnat_config.debug then begin
      Gnat_objectives.Save_VCs.save_vc g;
   end;
   if Gnat_config.force then
      actually_schedule_goal g
   else
      (* then implement reproving logic *)
      begin
      (* Maybe the goal is already proved *)
      if g.Session.goal_verified then begin
         handle_vc_result g true None
      (* Maybe there was a previous proof attempt with identical parameters *)
      end else if Gnat_objectives.goal_has_been_tried g then begin
         (* the proof attempt was necessarily false *)
         handle_vc_result g false None
      end else begin
         actually_schedule_goal g
      end;
   end

and actually_schedule_goal g =
   Gnat_objectives.schedule_goal interpret_result g

let normal_handle_one_subp subp =
   if Gnat_objectives.matches_subp_filter subp then begin
      Gnat_objectives.init_subp_vcs subp;
      Gnat_objectives.iter_leaf_goals ~subp register_goal;
      Gnat_objectives.stat subp;
      Gnat_objectives.iter (fun obj ->
      if Gnat_objectives.objective_status obj =
         Gnat_objectives.Proved then begin
         Format.printf "%a@." Gnat_expl.print_simple_proven obj
      end else begin
         match Gnat_objectives.next obj with
         | Some g -> schedule_goal g
         | None -> ()
      end);
      Gnat_objectives.do_scheduled_jobs ();
      Gnat_objectives.clear ()
   end

let all_split_subp subp =
   if Gnat_objectives.matches_subp_filter subp then begin
      Gnat_objectives.init_subp_vcs subp;
      Gnat_objectives.iter_leaf_goals ~subp register_goal;
      Gnat_objectives.all_split_leaf_goals ();
      Gnat_objectives.clear ()
   end

let _ =
   (* This is the main code. We read the file into the session if not already
      done, we apply the split_goal transformation when needed, and we schedule
      the first VC of all objectives. When done, we save the session.
   *)
   try
      Gnat_objectives.init ();
      match Gnat_config.proof_mode with
      | Gnat_config.Then_Split
      | Gnat_config.Path_WP
      | Gnat_config.No_Split ->
         Gnat_objectives.iter_subps normal_handle_one_subp;
         Gnat_objectives.save_session ()
      | Gnat_config.All_Split ->
         Gnat_objectives.iter_subps all_split_subp
      | Gnat_config.No_WP ->
         (* we should never get here *)
         ()
    with e ->
       Format.eprintf "Internal error:@.";
       Format.eprintf "%a.@." Exn_printer.exn_printer e;
       Gnat_util.abort_with_message ""
