-- $Id: sparksimp.adb 12509 2009-02-19 09:50:35Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


------------------------------------------------------------------------------
-- SPARKSimp                                                                --
--                                                                          --
-- This program traverses a directory and all of its subdirectories trying  --
-- to find any verification condition files that need to be simplified.     --
--                                                                          --
-- This program can only be compiled with the following compilers, since it --
-- relies on several of the GNAT.* predefined library units:                --
--                                                                          --
--   Windows : GNAT 5.04a or later                                          --
--   Solaris : GNAT 3.16a or later                                          --
--   Linux   : GNAT 5.04a or later                                          --
--   OS X    : GNAT 5.04a or later                                          --
------------------------------------------------------------------------------

with Ada.Command_Line;
with Ada.Real_Time;
with Ada.Characters.Handling;
with Ada.Characters.Latin_1;
with Ada.Exceptions;
with Ada.Integer_Text_IO;
with Ada.IO_Exceptions;
with Ada.Streams.Stream_IO;
with Ada.Strings.Fixed;
with Ada.Text_IO;

with GNAT.Command_Line;
with GNAT.Directory_Operations;
with GNAT.IO_Aux;
with GNAT.OS_Lib;
with GNAT.Regpat;
with GNAT.Table;
with GNAT.Traceback.Symbolic;

with SPARK.Expect;

with Version;

use type GNAT.OS_Lib.String_Access;
use type GNAT.OS_Lib.Argument_List;
use type SPARK.Expect.Process_Descriptor_Access;
use type SPARK.Expect.Expect_Match;


use type Ada.Real_Time.Time;

procedure SPARKSimp
is
   subtype String1_Index is Positive range 1 .. 1;
   subtype String1 is String (String1_Index);

   ---------------------------------------------------------------------------
   --  Utility subprogams                                                   --
   ---------------------------------------------------------------------------

   --  Subprograms Format_Int, Get_File_Size and Put_Message_With_Duration
   --  were previously in Safe_IO.

   function Format_Int (Item  : in Integer;
                        Width : in Integer) return String;

   function Get_File_Size
     (File : in String) return Ada.Streams.Stream_IO.Count;

   procedure Put_Message_With_Duration (Message : in String;
                                        D       : in Duration);

   --  Print Str to Standard_Output if CMD.Verbose is True
   procedure Debug (Str : in String);

   function Format_Int (Item  : in Integer;
                        Width : in Integer) return String
   is
      Temp_String : String (1 .. Width);
   begin
      Ada.Integer_Text_IO.Put (To => Temp_String,
                               Item => Item,
                               Base => 10);
      return Temp_String;
   end Format_Int;

   --  Read and return size of File, using Ada.Streams.Stream_IO
   --  On a 32-bit machine that allows files >2GB, I doubt this will
   --  work, but if anyone has a 2GB VCG file, then that's their
   --  problem!
   function Get_File_Size
     (File : in String) return Ada.Streams.Stream_IO.Count
   is
      package SIO renames Ada.Streams.Stream_IO;
      S : SIO.Count;
      F : SIO.File_Type;
   begin
      SIO.Open (F, SIO.In_File, File, "");
      S := SIO.Size (F);
      SIO.Close (F);
      return S;
   exception
      when others =>
         if SIO.Is_Open (F) then
            SIO.Close (F);
         end if;
         return 0;
   end Get_File_Size;

   procedure Put_Message_With_Duration (Message : in String;
                                        D       : in Duration)
   is
      package Duration_IO is new Ada.Text_IO.Fixed_IO (Duration);
      Hours   : Natural;
      Minutes : Natural;
      Seconds : Duration;
      Residue : Long_Float;
      LD      : Long_Float;
   begin
      LD := Long_Float (D);
      Hours := Natural (Long_Float'Floor (LD / 3600.0));
      Residue := LD - (Long_Float (Hours) * 3600.0);
      Minutes := Natural (Long_Float'Floor (Residue / 60.0));
      Seconds := Duration (Residue - (Long_Float (Minutes) * 60.0));
      Ada.Text_IO.Put (Message);
      Ada.Integer_Text_IO.Put (Item => Hours, Width => 4, Base => 10);
      Ada.Text_IO.Put (':');
      Ada.Integer_Text_IO.Put (Item => Minutes, Width => 2, Base => 10);
      Ada.Text_IO.Put (':');
      Duration_IO.Put (Item => Seconds, Fore => 2, Aft => 2, Exp => 0);
      Ada.Text_IO.New_Line;
   end Put_Message_With_Duration;


   ---------------------------------
   -- CMD - Command Line Handling --
   ---------------------------------
   package CMD
   --# own Valid, All_Files, Verbose, Version_Requested, Run_Simplifier,
   --#     Log_Output, Echo_Output, Sort_VCGs, Reverse_Order,
   --#     Processes, Exe_Switch;
   --# initializes Valid, All_Files, Version_Requested, Run_Simplifier,
   --#             Verbose, Log_Output, Echo_Output, Sort_VCGs, Reverse_Order,
   --#             Processes, Exe_Switch;
   is
      Valid             : Boolean := False;
      All_Files         : Boolean := False;
      Verbose           : Boolean := False;
      Version_Requested : Boolean := False;
      Run_Simplifier    : Boolean := True;
      Log_Output        : Boolean := False;
      Echo_Output       : Boolean := False;
      Sort_VCGs         : Boolean := False;
      Reverse_Order     : Boolean := False;
      Processes         : Positive := 1;
      Exe_Switch        : GNAT.OS_Lib.String_Access := null;

      --  Returns the arguments in the "sargs" section of the command
      --  line.  The returned array has exactly the number of elements
      --  required.  The returned array can be null (i.e. no elements)
      --  function Get_Sargs return GNAT.OS_Lib.Argument_List;
      SArgs : GNAT.OS_Lib.Argument_List_Access := null;

      ------------------------------------------------
      -- Reads and parses the command line and sets --
      -- program-wide flags as appropriate.         --
      ------------------------------------------------
      procedure Process_Command_Line (Switch_Char   : in Character);
      --# global in     GNAT.Command_Line.State;
      --#        in out Valid, All_Files, Verbose,
      --#               Run_Simplifier, Log_Output, Echo_Output,
      --#               Sort_VCGs, Reverse_Order, Processes, Exe_Switch, SArgs;

      --------------------------------------------
      -- Print command name, options and brief  --
      -- description of each to Standard_Output --
      --------------------------------------------
      procedure Usage;
      --# global out File_Sys;

   end CMD;

   ------------------------------------------------------------
   -- Work_Manager - provides coordinates the worker threads --
   -- and provides status output to the user                 --
   ------------------------------------------------------------

   package Work_Manager is

      MaxErrorStringIndex : constant Integer := 100;
      subtype ErrorStringIndex is Integer range 1 .. MaxErrorStringIndex;
      subtype ErrorString is String (ErrorStringIndex);

      NullErrorString : constant ErrorString := ErrorString'(others => ' ');

      subtype Job_Index is Natural;
      subtype Worker_ID is Natural;

      AnyFailed : Boolean := False;

      --  Keep track of the status of jobs in the list.
      --  At present this is not really used for anything, but it could be
      --  useful info if we add a monitor thread at a later date.
      type JobStatus is (Pending, InProgress, Finished, Failed);

      --  Record for details of each job.
      type Work_Package is record
         File_Name : GNAT.OS_Lib.String_Access;
         File_Size : Ada.Streams.Stream_IO.Count;
         Status : JobStatus := Pending;
         Worker : Worker_ID := 0;
         WhyFailed : ErrorString;
      end record;

      --  Work_List manages the list of jobs to be done.
      package Jobs is
         procedure Add_File (File : in String);
         procedure GetNextJob (Job_ID : out Job_Index);
         procedure JobFinished (Job : in Job_Index);
         procedure JobFailed (Job : in Job_Index; FailReason : in ErrorString);
         procedure Sort_Files_By_Size;
         procedure Display_Status_Banner;
         function Total_Number_Of_Files return Job_Index;
         function Number_Of_Files (Of_Status : in JobStatus) return Job_Index;
         procedure List_Jobs;
         function Get_File_Name (Job : in Job_Index) return String;
         function Get_HasFailed (Job : in Job_Index) return Boolean;
         function Get_WhyFailed (Job : in Job_Index) return ErrorString;
         procedure Clear;
      end Jobs;

   private

      package Work_Table is new GNAT.Table
            (Table_Component_Type => Work_Package,
             Table_Index_Type     => Job_Index,
             Table_Low_Bound      => 1,
             Table_Initial        => 1000,
             Table_Increment      => 100);

   end Work_Manager;

   package body Work_Manager
   --# own State;
   is

      package body Jobs is
         type Count_Per_Status is array (JobStatus) of Job_Index;

         Next_Job : Job_Index := 1;

         Job_Counts  : Count_Per_Status := Count_Per_Status'(others => 0);

         procedure Check_Invariant;

         procedure Check_Invariant
         is
            Total : Job_Index := 0;
         begin
            for I in JobStatus loop
               Total := Total + Job_Counts (I);
            end loop;
            if Total /= Work_Table.Last then
               Debug ("Invariant failure in Work_Manager");
               for I in JobStatus loop
                  Debug (Natural'Image (Job_Counts (I)));
               end loop;
               Debug (Natural'Image (Work_Table.Last));
            end if;
         end Check_Invariant;


         procedure Add_File (File : in String)
         is
         begin
            Work_Table.Increment_Last;

            Job_Counts (Pending) := Job_Counts (Pending) + 1;

            Work_Table.Table (Work_Table.Last) :=
               Work_Package'(File_Name => new String'(File),
                             File_Size => Get_File_Size (File),
                             Status => Pending,
                             Worker => 0,
                             WhyFailed => NullErrorString);

            Check_Invariant;
         end Add_File;

         procedure GetNextJob (Job_ID : out Job_Index)
         is
         begin
            --  Get the next job.
            Job_ID := Next_Job;

            --  Set the status of the job, and the worker.
            Work_Table.Table (Next_Job).Status := InProgress;

            Job_Counts (InProgress) := Job_Counts (InProgress) + 1;
            Job_Counts (Pending) := Job_Counts (Pending) - 1;

            Next_Job := Next_Job + 1;

            Check_Invariant;
         end GetNextJob;

         function Get_File_Name (Job : in Job_Index) return String
         is
         begin
            return Work_Table.Table (Job).File_Name.all;
         end Get_File_Name;

         function Get_HasFailed (Job : in Job_Index) return Boolean
         is
         begin
            return Work_Table.Table (Job).Status = Failed;
         end Get_HasFailed;

         function Get_WhyFailed (Job : in Job_Index) return ErrorString
         is
         begin
            return Work_Table.Table (Job).WhyFailed;
         end Get_WhyFailed;

         procedure JobFinished (Job : in Job_Index) is
         begin
            --  Mark the job as completed.
            Work_Table.Table (Job).Status := Finished;
            Job_Counts (Finished) := Job_Counts (Finished) + 1;
            Job_Counts (InProgress) := Job_Counts (InProgress) - 1;

            Check_Invariant;
         end JobFinished;

         --  Signal that a job has failed and record the reason.
         procedure JobFailed (Job : in Job_Index;
                              FailReason : in ErrorString) is
         begin
            --  Mark the job as failed.
            Work_Table.Table (Job).Status := Failed;
            Work_Table.Table (Job).WhyFailed := FailReason;
            Work_Manager.AnyFailed := True;

            Job_Counts (Failed) := Job_Counts (Failed) + 1;
            Job_Counts (InProgress) := Job_Counts (InProgress) - 1;

            Check_Invariant;
         end JobFailed;

         procedure Sort_Files_By_Size
         is
            Changed : Boolean;
            T       : Work_Package;
            use Ada.Streams.Stream_IO;
         begin

            if CMD.Sort_VCGs then
               --  Dumb but simple bubble sort algorithm.
               loop
                  Changed := False;

                  if CMD.Reverse_Order then
                     for X in Job_Index range 1 .. Work_Table.Last - 1 loop
                        if Work_Table.Table (X + 1).File_Size <
                           Work_Table.Table (X).File_Size then
                           T := Work_Table.Table (X);
                           Work_Table.Table (X) := Work_Table.Table (X + 1);
                           Work_Table.Table (X + 1) := T;
                           Changed := True;
                        end if;
                     end loop;
                  else
                     for X in Job_Index range 1 .. Work_Table.Last - 1 loop
                        if Work_Table.Table (X + 1).File_Size >
                           Work_Table.Table (X).File_Size then
                           T := Work_Table.Table (X);
                           Work_Table.Table (X) := Work_Table.Table (X + 1);
                           Work_Table.Table (X + 1) := T;
                           Changed := True;
                        end if;
                     end loop;
                  end if;

                  exit when not Changed;

               end loop;
            end if;
         end Sort_Files_By_Size;

         procedure Display_Status_Banner
         is
         begin
            Ada.Text_IO.Put_Line ("Job-ID Status     Filename");
            Ada.Text_IO.Put_Line ("====== ======     ========");
         end Display_Status_Banner;


         function Total_Number_Of_Files return Job_Index
         is
         begin
            return Work_Table.Last;
         end Total_Number_Of_Files;


         function Number_Of_Files (Of_Status : in JobStatus) return Job_Index
         is
         begin
            return Job_Counts (Of_Status);
         end Number_Of_Files;


         --  Display the list of jobs to do, to the screen.
         procedure List_Jobs
         is
         begin

            if Total_Number_Of_Files = 0 then
               Ada.Text_IO.Put_Line
                  ("No VCG files were found that require simplification");
            else
               Ada.Text_IO.Put_Line ("Files to be simplified are:");
               for I in Natural range 1 .. Total_Number_Of_Files loop
                  Ada.Text_IO.Put_Line (Work_Table.Table (I).File_Name.all &
                                        "," &
                                        Ada.Streams.Stream_IO.Count'Image
                                           (Work_Table.Table (I).File_Size) &
                                        " bytes");
               end loop;
               Ada.Text_IO.New_Line;
               Ada.Text_IO.Put_Line (Natural'Image (Total_Number_Of_Files) &
                                     " VCG files require simplification");
            end if;
            Ada.Text_IO.New_Line;
         end List_Jobs;

         procedure Clear is
         begin
            Work_Table.Set_Last (0);
         end Clear;
      end Jobs;

   end Work_Manager;
   ---------------------------------------------------------------------------


   --  Print tool banner to Standard_Output
   procedure Banner;
   --# global in     Spawner.Spadesimp_Exe;
   --#           out File_Sys;

   --  returns True if Left is older than (i.e. preceeds) Right
   function Is_Older (Left, Right : in GNAT.OS_Lib.OS_Time) return Boolean;

   --  Traverse all files and directories rooted at current-working
   --  directory, find VCG and PFS files that need simplifying, and put them
   --  into VCG_List.
   --
   --  a VCG file need simplifying if either:
   --   1) The CMD.All_Files flag is True, or
   --   2) The VCG (PFS) file has no corresponding SIV (SIP) file, or
   --   3) The VCG (PFS) file has a corresponding SIV (SIP) file, but the
   --      SIV (SIP) file's time stamp os older than that of the
   --      VCG (PFS) file.
   procedure Find_Files_To_Simp;
   --# global in     File_Sys;
   --#        in     CMD.All_Files;
   --#           out VCG_List.State;

   --  Run the Simplifier on all files in VCG_List.State
   procedure Simplify_Files;
   --# global in     VCG_List.State;
   --#        in     CMD.Verbose;
   --#        in     CMD.Reverse_Order;
   --#        in out File_Sys;

   --  put any errors found onto the screen
   procedure Report_Errors;

   ---------------------------------------------------------------------------
   --  CMD Package body                                                     --
   ---------------------------------------------------------------------------
   package body CMD
   is
      function Number_Of_Sargs return Natural;

      function Number_Of_Sargs return Natural
      is
         Count : Natural := 0;
      begin
         GNAT.Command_Line.Goto_Section ("sargs");

         loop
            case GNAT.Command_Line.Getopt ("*") is
               when Ada.Characters.Latin_1.NUL =>
                  exit;
               when '*' =>
                  Count := Count + 1;
               when others =>
                  null;
            end case;
         end loop;
         return Count;
      end Number_Of_Sargs;

      procedure Process_Command_Line (Switch_Char   : in Character)
      is
      begin
         Valid := True;

         --  See documentation of GNAT.Command_Line in
         --  GNAT library source in g-comlin.ads
         GNAT.Command_Line.Initialize_Option_Scan
           (Switch_Char              => Switch_Char,
            Stop_At_First_Non_Switch => False,
            Section_Delimiters       => "sargs");

         GNAT.Command_Line.Goto_Section ("");

         loop
            begin
               case GNAT.Command_Line.Getopt ("a v V l e n t r p= x=") is
                  when Ada.Characters.Latin_1.NUL =>
                     --  Must be start of next section...
                     exit;
                  when 'a' =>
                     All_Files := True;
                  when 'v' =>
                     Version_Requested := True;
                  when 'V' =>
                     Verbose := True;
                  when 't' =>
                     Sort_VCGs := True;
                  when 'r' =>
                     Reverse_Order := True;
                  when 'l' =>
                     Log_Output := True;
                  when 'e' =>
                     Echo_Output := True;
                  when 'n' =>
                     Run_Simplifier := False;
                  when 'p' =>
                     Processes := Positive'Value (GNAT.Command_Line.Parameter);
                  when 'x' =>
                     Exe_Switch := new String'(GNAT.Command_Line.Parameter);
                  when others =>
                     null;
               end case;
            exception
               when GNAT.Command_Line.Invalid_Switch =>
                  if GNAT.Command_Line.Full_Switch = "s" then
                     --  Must be beginning of /sargs section, so...
                     exit;
                  else
                     Ada.Text_IO.Put_Line ("Invalid Switch " & GNAT.Command_Line.Full_Switch);
                     Valid := False;
                  end if;
            end;
         end loop;

         --  If we have more than one thread running it makes no sense to send
         --  simplifier output to the screen.
         if Processes > 1 and Echo_Output then
            Ada.Text_IO.Put_Line ("Simplifier output cannot be echoed to the screen");
            Ada.Text_IO.Put_Line ("when more than 1 concurrent process is used");

            Valid := False;
         end if;


         loop
            declare
               S : constant String := GNAT.Command_Line.Get_Argument;
            begin
               exit when S'Length = 0;
               Ada.Text_IO.Put_Line ("Unexpected argument:" & S);
               Valid := False;
            end;
         end loop;


         declare
            NA : constant Natural := Number_Of_Sargs;
            subtype Arg_Index is Natural range 1 .. NA;
            subtype Args is GNAT.OS_Lib.Argument_List (Arg_Index);
            I : Positive := 1;
         begin
            SArgs := new Args'(others => null);
            GNAT.Command_Line.Goto_Section ("sargs");

            loop
               case GNAT.Command_Line.Getopt ("*") is
                  when Ada.Characters.Latin_1.NUL =>
                     exit;
                  when '*' =>
                     SArgs.all (I) := new String'
                       (GNAT.Command_Line.Full_Switch);
                     I := I + 1;
                  when others =>
                     null;
               end case;
            end loop;
            GNAT.OS_Lib.Normalize_Arguments (SArgs.all);
         end;

      exception
         when GNAT.Command_Line.Invalid_Switch    =>
            Ada.Text_IO.Put_Line ("Invalid Switch " & GNAT.Command_Line.Full_Switch);
            Valid := False;
         when GNAT.Command_Line.Invalid_Parameter =>
            Ada.Text_IO.Put_Line ("No parameter for " & GNAT.Command_Line.Full_Switch);
            Valid := False;
         when others =>
            Valid := False;
      end Process_Command_Line;

      procedure Usage
      is
      begin
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put_Line
           ("Usage: sparksimp" &
            " [-a] [-v] [-V] [-n] [-t] [-r] [-l] [-e] [-p=N] [-x=Simpexec]" &
            " [-sargs {simplifier_options}]");
         Ada.Text_IO.Put_Line
           ("Options: -a simplify all VCG files, ignoring time stamps");
         Ada.Text_IO.Put_Line
           ("         -v report version and terminate");
         Ada.Text_IO.Put_Line
           ("         -V verbose output");
         Ada.Text_IO.Put_Line
           ("         -n print commands but don't run Simplifier");
         Ada.Text_IO.Put_Line
           ("         -t Sort VCG files, largest first");
         Ada.Text_IO.Put_Line
           ("         -r reverse simplification order");
         Ada.Text_IO.Put_Line
           ("         -l log Simplifier output for XXX.vcg to XXX.log");
         Ada.Text_IO.Put_Line
           ("         -e echo Simplifier output to screen");
         Ada.Text_IO.Put_Line
           ("         -p=N use N concurrent processes for " &
            "running the Simplifier");
         Ada.Text_IO.Put_Line
           ("         -x=Simpexec, Simpexec specifies an alternative " &
            "Simplifier executable");
         Ada.Text_IO.Put_Line
           ("         -sargs pass all remaining options to Simplifier");
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put_Line (Version.Toolset_Support_Line1);
         Ada.Text_IO.Put_Line (Version.Toolset_Support_Line2);
         Ada.Text_IO.Put_Line (Version.Toolset_Support_Line3);
         Ada.Text_IO.Put_Line (Version.Toolset_Support_Line4);

      end Usage;

   end CMD;


   ---------------------------------------------------------------------------
   --  procedure Debug                                                      --
   ---------------------------------------------------------------------------
   procedure Debug (Str : in String)
   is
   begin
      if CMD.Verbose then
         Ada.Text_IO.Put_Line (Str);
      end if;
   end Debug;


   ---------------------------------------------------------------------------
   --  Log_Files package                                                    --
   ---------------------------------------------------------------------------

   --  This package is needed because we want to store a reference to the log
   --  file for each Simplifier. If the File_Type was stored with the other
   --  data, that record would become limited private and we could not copy the
   --  data (e.g. when compacting the Worker_Set).
   package Log_Files is

      type Log_File is limited private;

      type Log_File_Set is array (Positive range <>) of Log_File;

      procedure Initialize (File_Set :    out Log_File_Set);

      procedure Open (File_Name : in     String;
                      Index     :    out Positive;
                      File_Set  : in out Log_File_Set);

      procedure Close (Index    : in     Positive;
                       File_Set : in out Log_File_Set);

      function File_Type (Index    : Positive;
                          File_Set : Log_File_Set)
        return Ada.Text_IO.File_Type;

   private

      type Log_File is
         record
            In_Use : Boolean;
            FT     : Ada.Text_IO.File_Type;
         end record;

   end Log_Files;


   package body Log_Files is

      procedure Initialize (File_Set :    out Log_File_Set)
      is
      begin
         for I in File_Set'Range loop
            File_Set (I).In_Use := False;
            --  No initial value available for the File_Type component.
         end loop;
      end Initialize;

      procedure Open (File_Name : in     String;
                      Index     :    out Positive;
                      File_Set  : in out Log_File_Set)
      is
         procedure Set_Next_Free;
         procedure Set_Next_Free
         is
         begin
            Index := 1;
            while File_Set (Index).In_Use loop
               Index := Index + 1;
            end loop;
         end Set_Next_Free;
      begin
         Set_Next_Free;
         Ada.Text_IO.Create (File_Set (Index).FT,
                             Ada.Text_IO.Out_File,
                             File_Name);
         File_Set (Index).In_Use := True;
      end Open;

      procedure Close (Index    : in     Positive;
                       File_Set : in out Log_File_Set)
      is
      begin
         Ada.Text_IO.Close (File_Set (Index).FT);
         File_Set (Index).In_Use := False;
      end Close;

      function File_Type (Index    : Positive;
                          File_Set : Log_File_Set) return Ada.Text_IO.File_Type
      is
      begin
         return File_Set (Index).FT;
      end File_Type;

   end Log_Files;


   ---------------------------------------------------------------------------
   --  Workers package specification                                        --
   --                                                                       --
   --  This package provides the Start_Simp and Run_Simp procedures to      --
   --  run multiple simplifications in parallel.  The number of Workers is  --
   --  set by the /p option in the command line.                            --
   ---------------------------------------------------------------------------
   package Workers
   is

      Path          : GNAT.OS_Lib.String_Access;
      Spadesimp_Exe : GNAT.OS_Lib.String_Access;

      Spadesimp_Command : constant String := "spadesimp";

      procedure Locate_Spadesimp;

      type Worker_Set (Size : Positive) is limited private;

      procedure Initialize
                  (Work_Set : in out Worker_Set;
                   Options  : in     GNAT.OS_Lib.Argument_List_Access);

      procedure Start_Simp (The_Job  : in     Work_Manager.Job_Index;
                            Work_Set : in out Worker_Set);

      function Workers_Available (Work_Set : Worker_Set) return Natural;

      procedure Run_Simp (Work_Set : in out Worker_Set);

   private

      type Work_Data is
         record
            Job_ID       : Work_Manager.Job_Index;
            Start_Time   : Ada.Real_Time.Time;
            End_Time     : Ada.Real_Time.Time;
            Elapsed_Time : Duration;
            OK           : Boolean;
            WhyFailed    : Work_Manager.ErrorString;
            OP           : Positive;
         end record;

      type Work_Set_Data is array (Positive range <>) of Work_Data;

      type Worker_Set (Size : Positive) is
         record
            Worker_Count  : Positive;
            Working_Count : Natural;
            Files         : Work_Set_Data (1 .. Size);
            Procs         : SPARK.Expect.Multiprocess_Regexp_Array (1 .. Size);
            Logs          : Log_Files.Log_File_Set (1 .. Size);
         end record;

   end Workers;

   ---------------------------------------------------------------------------
   --  Wrap package                                                         --
   ---------------------------------------------------------------------------
   package Wrap is
      --  Copies SimpLine to the given File, wrapping the output
      --  at or near 80 columns, splitting onto multiple lines
      --  as needed.  A line may be wrapped at any of the following
      --  characters: ' ', '(', ')', '[', or ']'
      procedure CopyAndMaybeWrapLine (File     : in Ada.Text_IO.File_Type;
                                      SimpLine : in String);

   end Wrap;

   package body Wrap is

      ---------------------------------------------------------
      --  Most of this code has been borrowed and adapted from
      --  spade/simplifier/utils/wraps.  Functionally, it's
      --  the same as wrap_utility, but this code has been
      --  brought up to date for Ada95, GNAT, and -gnaty style
      ---------------------------------------------------------

      --  Maximum no. of output columns normally permitted
      MaxCols         : constant := 80;
      --  Columns to indent for 2nd. & subsequent wraps
      Indentation     : constant := 10;
      SubsequentLineWidth : constant := MaxCols - Indentation;

      --  Copies a line, wrapping if it appears to be necessary.
      procedure CopyAndMaybeWrapLine (File     : in Ada.Text_IO.File_Type;
                                      SimpLine : in String)
      is
         function IsALongLine (L : in String) return Boolean;

         --  Writes out the given slice of L
         procedure OutPartOfLine (L              : in String;
                                  FromCol, ToCol : in Positive);

         --  Writes out line-buffer L unchanged, without any line-wrapping.
         procedure OutUnchangedLine (L : in String);

         --  Write out line-buffer L, wrapping where necessary.
         procedure OutWrappedLine (L : in String);

         function IsALongLine (L : in String) return Boolean
         --  True if we need to wrap the line around - i.e. more than MaxCols.
         is
         begin
            return L'Length > MaxCols;
         end IsALongLine;

         procedure OutPartOfLine (L              : in String;
                                  FromCol, ToCol : in Positive)
         is
         begin
            Ada.Text_IO.Put (File, L (FromCol .. ToCol));
         end OutPartOfLine;

         procedure OutUnchangedLine (L : in String)
         is
         begin
            Ada.Text_IO.Put_Line (File, L);
         end OutUnchangedLine;

         procedure OutWrappedLine (L : in String)
         is
            procedure OutNextPartOfLine (L       : in     String;
                                         FromCol : in out Positive;
                                         InWidth : in     Positive);

            procedure Indent;

            OnCol, Width : Positive;

            procedure OutNextPartOfLine (L       : in     String;
                                         FromCol : in out Positive;
                                         InWidth : in     Positive)
            is
               function OKSplitChar (C : in Character) return Boolean;
               ToCol : Natural;

               function OKSplitChar (C : in Character) return Boolean
               --  Returns true if C is a space, parenthesis or bracket.
               is
                  use Ada.Characters.Latin_1;
               begin
                  return (C = Space) or else
                         (C = Left_Parenthesis) or else
                         (C = Right_Parenthesis) or else
                         (C = Left_Square_Bracket) or else
                         (C = Right_Square_Bracket);
               end OKSplitChar;

            begin --  OutNextPartOfLine
               ToCol := FromCol + InWidth - 1;
               --  if line can't be split at the exact length you want it,
               --  search left along the text for the first feasible place
               if not OKSplitChar (L (ToCol)) then      --  drat!
                  loop
                     ToCol := ToCol - 1;
                     exit when OKSplitChar (L (ToCol)) or else
                               (ToCol = FromCol);
                  end loop;
                  --  if the line can't be split at any point then search right
                  --  along the text i.e. the line will be longer than you want
                  if ToCol = FromCol then                      --  double drat!
                     ToCol := FromCol + InWidth - 1;
                     loop
                        exit when (ToCol >= L'Length) or else
                          OKSplitChar (L (ToCol));
                        ToCol := ToCol + 1;
                     end loop;
                  end if;
               end if;
               OutPartOfLine (L, FromCol, ToCol);
               FromCol := ToCol + 1;
            end OutNextPartOfLine;

            procedure Indent
            is
               subtype IndentIndex is Positive range 1 .. Indentation;
               subtype IndentString is String (IndentIndex);
               IndentC : constant IndentString := IndentString'(others => ' ');
            begin
               Ada.Text_IO.Put (File, IndentC);
            end Indent;

         begin --  OutWrappedLine
            OnCol := 1;       --  Start at column 1
            Width := MaxCols; --  To start with
            while OnCol + Width <= L'Length loop
               OutNextPartOfLine (L, OnCol, Width);
               Width := SubsequentLineWidth;
               Ada.Text_IO.New_Line (File, 1);
               Indent;
            end loop;
            if OnCol <= L'Length then
               OutPartOfLine (L, OnCol, L'Length);
               Ada.Text_IO.New_Line (File, 1);
            end if;
         end OutWrappedLine;

      begin --  CopyAndMaybeWrapLine
         --  if line is too long, wrap it, otherwise output it unchanged
         if IsALongLine (SimpLine) then
            OutWrappedLine (SimpLine);
         else
            OutUnchangedLine (SimpLine);
         end if;
      end CopyAndMaybeWrapLine;

   end Wrap;


   ---------------------------------------------------------------------------
   --  procedure Banner                                                     --
   ---------------------------------------------------------------------------
   procedure Banner
   is
   begin
      Ada.Text_IO.Put_Line ("SPARKSimp " &
        Version.Toolset_Distribution & " Edition, " &
        "Version " & Version.Toolset_Version &
        ", Date " & Version.Toolset_Build_Date &
        ", Build " & Version.Toolset_Build_Stamp);
      Ada.Text_IO.Put_Line (Version.Toolset_Copyright);

      --  Report location of simplifier binary - this should prevent
      --  "running the wrong simplifier by accident" problems
      if Workers.Spadesimp_Exe /= null then
         Ada.Text_IO.Put_Line ("Simplifier binary located at: " &
                               Workers.Spadesimp_Exe.all);
         Ada.Text_IO.New_Line;
      end if;
   end Banner;

   ---------------------------------------------------------------------------
   --  Workers package body                                                 --
   ---------------------------------------------------------------------------
   package body Workers
   is

      Simplifier_Options : GNAT.OS_Lib.Argument_List_Access;

      Pat : constant GNAT.Regpat.Pattern_Matcher :=
        GNAT.Regpat.Compile ("^.*\n", GNAT.Regpat.Multiple_Lines);

      Pat_Access : constant SPARK.Expect.Pattern_Matcher_Access :=
        new GNAT.Regpat.Pattern_Matcher'(Pat);

      procedure Locate_Spadesimp
      is
      begin
         if CMD.Exe_Switch /= null then
            --  simplifer executable specified by /x= switch
            Spadesimp_Exe := GNAT.OS_Lib.Locate_Exec_On_Path
               (CMD.Exe_Switch.all);
         else
            Path := GNAT.OS_Lib.Getenv ("PATH");

            if Path = null then
               Path := GNAT.OS_Lib.Getenv ("path");
            end if;

            if Path = null then
               Ada.Text_IO.Put_Line
                 ("Error: can't find PATH environment variable");
               Spadesimp_Exe := null;
            else
               Spadesimp_Exe := GNAT.OS_Lib.Locate_Exec_On_Path
                  (Spadesimp_Command);
            end if;
         end if;
      end Locate_Spadesimp;

      procedure Initialize (Work_Set : in out Worker_Set;
                            Options  : in     GNAT.OS_Lib.Argument_List_Access)
      is

      begin
         Locate_Spadesimp;
         Simplifier_Options     := Options;
         Work_Set.Worker_Count  := Work_Set.Procs'Length;
         Work_Set.Working_Count := 0;
         for I in Work_Set.Procs'Range loop
            Work_Set.Procs (I) := SPARK.Expect.Multiprocess_Regexp'
                                    (Descriptor => null,
                                     Regexp     => Pat_Access);
         end loop;
         --  No need to initialize Work_Set.Files (access to it is controlled
         --  by Working_Count).
         Log_Files.Initialize (Work_Set.Logs);
      end Initialize;

      procedure Start_Simp (The_Job  : in     Work_Manager.Job_Index;
                            Work_Set : in out Worker_Set)
      is

         procedure Create_Log_File (For_Worker : Natural);

         Worker : Natural;

         On_File : constant String :=
           Work_Manager.Jobs.Get_File_Name (The_Job);

         --  Find the first directory separator from the right hand
         --  end of File_Name, so we can split into the directory,
         --  and the plain file name
         Dir_Index : constant Natural := Ada.Strings.Fixed.Index
           (On_File,
            String1'(1 => GNAT.OS_Lib.Directory_Separator),
            Ada.Strings.Backward);

         --  Directory in which to run simplifier
         Dir : constant String := On_File (1 .. Dir_Index);

         --  Simple file name of VCG file to be simplified
         SF  : constant GNAT.OS_Lib.String_Access :=
           new String'(On_File (Dir_Index + 1 .. On_File'Last));

         Expect_Args : constant GNAT.OS_Lib.Argument_List :=
           (1 => SF) & Simplifier_Options.all;

         procedure Create_Log_File (For_Worker : Natural)
         is
            OK       : Boolean := False;
            Log_File : constant String :=
                         On_File (1 .. On_File'Last - 3) & "log";
         begin
            loop
               begin
                  Log_Files.Open (Log_File,
                                  Work_Set.Files (For_Worker).OP,
                                  Work_Set.Logs);
                  OK := True;
               exception
                  when Ada.IO_Exceptions.Use_Error |
                       Ada.IO_Exceptions.Name_Error =>
                     Ada.Text_IO.Put_Line ("Create failed - trying again...");
                     OK := False;
               end;
               exit when OK;
            end loop;
         end Create_Log_File;

         FD_Access : constant SPARK.Expect.Process_Descriptor_Access :=
           new SPARK.Expect.Process_Descriptor;

      begin
         Work_Set.Working_Count := Work_Set.Working_Count + 1;
         Worker := Work_Set.Working_Count;
         if CMD.Log_Output then
            Create_Log_File (Worker);
         end if;

         declare
            FN      : constant String :=
                        Work_Manager.Jobs.Get_File_Name (The_Job);
            Job_Str : constant String := Format_Int (Item  => The_Job,
                                                     Width => 6);
         begin
            --  Print a message to indicate the job has started.
            Ada.Text_IO.Put_Line
              (Job_Str & " Started  - " & FN);
         end;

         --  Create a Process_Descriptor that can be accessed via a
         --  Process_Descriptor_Access.  If one already exists then it was
         --  created for a previous simplification and recycled in Compact.
         --  This ensures that only the minimum number of objects are created
         --  on the heap and we don't need to worry about Free'ing them.
         if Work_Set.Procs (Worker).Descriptor = null then
            Work_Set.Procs (Worker).Descriptor :=
              new SPARK.Expect.Process_Descriptor;
         end if;

         Work_Set.Files (Worker).Start_Time := Ada.Real_Time.Clock;
         GNAT.Directory_Operations.Change_Dir (Dir);
         SPARK.Expect.Non_Blocking_Spawn (FD_Access.all,
                                         Spadesimp_Exe.all,
                                         Expect_Args,
                                         0,
                                         False);
         Work_Set.Procs (Worker).Descriptor := FD_Access;
         Work_Set.Files (Worker).Job_ID     := The_Job;
         Work_Set.Files (Worker).OK         := True;
         Work_Set.Files (Worker).WhyFailed  := Work_Manager.NullErrorString;
      end Start_Simp;

      function Workers_Available (Work_Set : Worker_Set) return Natural
      is
      begin
         return Work_Set.Worker_Count - Work_Set.Working_Count;
      end Workers_Available;

      procedure Run_Simp (Work_Set : in out Worker_Set)
      is

         procedure Compact (Removing : in     Positive);
         procedure Close_Log_File;

         Result        : SPARK.Expect.Expect_Match;
         Never_Timeout : constant Integer := -1;
         Worker        : Natural;
         Job_ID        : Work_Manager.Job_Index;

         procedure Close_Log_File
         is
            OK : Boolean := False;
         begin
            loop
               begin
                  Log_Files.Close (Work_Set.Files (Worker).OP,
                                   Work_Set.Logs);
                  OK := True;
               exception
                  when Ada.IO_Exceptions.Device_Error =>
                     --  if OP is still open, then try again!
                     if Ada.Text_IO.Is_Open
                         (Log_Files.File_Type (Work_Set.Files (Worker).OP,
                                               Work_Set.Logs))
                     then
                        Ada.Text_IO.Put_Line
                          ("Close failed with Device_Error - try again...");
                        OK := False;
                     else
                        Ada.Text_IO.Put_Line
                          ("Close failed with Device_Error - aborting...");
                        OK := True;
                     end if;
                  when Storage_Error =>
                     Ada.Text_IO.Put_Line
                       ("Close failed with Storage_Error - aborting...");
                     OK := True;
               end;
               exit when OK;
            end loop;
         end Close_Log_File;

         procedure Compact (Removing : in     Positive)
         is
            --  We need to preserve the pointer to the process descriptor
            --  so that it can be re-used in a later simplification.
            PD_Acc : constant SPARK.Expect.Process_Descriptor_Access :=
                       Work_Set.Procs (Removing).Descriptor;
         begin
            for I in Removing .. Work_Set.Working_Count - 1 loop
               Work_Set.Files (I) := Work_Set.Files (I + 1);
               Work_Set.Procs (I) := Work_Set.Procs (I + 1);
            end loop;
            Work_Set.Procs (Work_Set.Working_Count).Descriptor := PD_Acc;
            Work_Set.Working_Count := Work_Set.Working_Count - 1;
         end Compact;

         ---------------------------------------------------------
         --  This string is prodced by the Simplifier at the start
         --  of a line to signal a critical error.  This string
         --  must match that in spade/simplifier/src/utilities.pl
         --  in the clause write_error_preamble/0
         ---------------------------------------------------------
         Error_Preamble : constant String := "*** ERROR - ";

      begin  --  Code of Run_Simp
         loop
            SPARK.Expect.Expect (Result,
                               Worker,
                               Work_Set.Procs (1 .. Work_Set.Working_Count),
                               Never_Timeout);
            if Result in
                 1 .. SPARK.Expect.Expect_Match (Work_Set.Working_Count) then
               Worker := Integer (Result);
               declare
                  S : constant String :=
                    SPARK.Expect.Expect_Out_Match
                      (Work_Set.Procs (Worker).Descriptor.all);
                  Final_Char : Natural;
               begin
                  --  On NT, we want to turn the CR/LF sequence
                  --  coming from the Simplifier back into a
                  --  standard line-ending sequence, so...
                  if S'Length >= 2 and then
                    (S (S'Last) = Ada.Characters.Latin_1.LF and
                       S (S'Last - 1) = Ada.Characters.Latin_1.CR) then

                     Final_Char := S'Last - 2;

                     --  On Other platforms, the line might end in just
                     --  a single LF, so strip that as well if the case
                     --  above didn't apply.
                  elsif S'Length >= 1 and then
                  S (S'Last) = Ada.Characters.Latin_1.LF then

                     Final_Char := S'Last - 1;

                  else
                     Final_Char := S'Last;
                  end if;

                  if CMD.Log_Output then
                     --  wrap each line to the log file
                     Wrap.CopyAndMaybeWrapLine
                       (Log_Files.File_Type (Work_Set.Files (Worker).OP,
                                            Work_Set.Logs),
                        S (S'First .. Final_Char));
                  end if;

                  if CMD.Echo_Output then
                     Wrap.CopyAndMaybeWrapLine
                       (Ada.Text_IO.Standard_Output,
                        S (S'First .. Final_Char));
                  end if;

                  --  if an error is found pass out why
                  if S'Length >= Error_Preamble'Length and then
                    S (1 .. Error_Preamble'Length) = Error_Preamble then
                     Work_Set.Files (Worker).OK := False;
                     if S'Length <= Work_Manager.MaxErrorStringIndex then
                        Work_Set.Files (Worker).WhyFailed (1 .. S'Length) := S;
                     else
                        Work_Set.Files (Worker).WhyFailed :=
                          S (1 .. Work_Manager.MaxErrorStringIndex);
                     end if;
                  end if;

               end;

            elsif Result = SPARK.Expect.Expect_Timeout then
               --  Timeout is OK - go round again...
               Debug ("Expect timeout");

            elsif Result = SPARK.Expect.Expect_Full_Buffer then
               Debug ("Expect Full Buffer");
               exit;

            elsif Result = SPARK.Expect.Expect_Process_Died then
               Debug ("Expect Process Died with Worker = " &
                        Integer'Image (Worker));
               exit;

            else
               Debug ("Got an unexpected exception from Expect");
               exit;
            end if;
         end loop;

         --  Tidy up when a simplification has finished.
         if Worker = 0 then
            --  Exit from Run_Simp with error message;
            Debug ("Can't find completed Simplifier process.");
            return;
         end if;
         Job_ID := Work_Set.Files (Worker).Job_ID;

         Work_Set.Files (Worker).End_Time := Ada.Real_Time.Clock;
         Work_Set.Files (Worker).Elapsed_Time :=
           Ada.Real_Time.To_Duration (Work_Set.Files (Worker).End_Time -
                                       Work_Set.Files (Worker).Start_Time);

         SPARK.Expect.Close (Work_Set.Procs (Worker).Descriptor.all);

         if CMD.Log_Output then
            Debug ("Closing Log File");
            Close_Log_File;
         end if;

         Debug ("Worker is " & Integer'Image (Worker));
         Debug ("Job_ID is " & Integer'Image (Job_ID));

         if Work_Set.Files (Worker).OK then
            Work_Manager.Jobs.JobFinished (Job_ID);
         else
            Work_Manager.Jobs.JobFailed (Job_ID,
                                         Work_Set.Files (Worker).WhyFailed);
         end if;

         --  Display a message that the job is finished.
         declare
            Job_Str : constant String := Format_Int (Item  => Job_ID,
                                                     Width => 6);
         begin
            Put_Message_With_Duration
              (Job_Str & " Finished ", Work_Set.Files (Worker).Elapsed_Time);
         end;

         Compact (Removing => Worker);

      end Run_Simp;

   end Workers;

   ---------------------------------------------------------------------------
   --  function Is_Older                                                    --
   ---------------------------------------------------------------------------
   function Is_Older (Left, Right : in GNAT.OS_Lib.OS_Time) return Boolean
   is
      Result : Boolean;
      use GNAT.OS_Lib;
   begin
      if GM_Year (Left) = GM_Year (Right) then
         if GM_Month (Left) = GM_Month (Right) then
            if GM_Day (Left) = GM_Day (Right) then
               if GM_Hour (Left) = GM_Hour (Right) then
                  if GM_Minute (Left) = GM_Minute (Right) then
                     Result := GM_Second (Left) <
                       GM_Second (Right);
                  else
                     Result := GM_Minute (Left) <
                       GM_Minute (Right);
                  end if;
               else
                  Result := GM_Hour (Left) <
                    GM_Hour (Right);
               end if;
            else
               Result := GM_Day (Left) <
                 GM_Day (Right);
            end if;
         else
            Result := GM_Month (Left) <
              GM_Month (Right);
         end if;
      else
         Result := GM_Year (Left) <
           GM_Year (Right);
      end if;
      return Result;
   end Is_Older;


   ---------------------------------------------------------------------------
   --  procedure Find_Files_To_Simp                                         --
   ---------------------------------------------------------------------------
   procedure Find_Files_To_Simp
   is
      use GNAT.Directory_Operations;

      function Is_A_VCG_File (File : in String) return Boolean;

      function Is_A_PFS_File (File : in String) return Boolean;

      function File_Needs_Simplifying (File : in String) return Boolean;
      --# global in File_Sys,
      --#           CMD.All_Files;
      --# pre Is_A_VCG_File (File) or Is_A_PFS_File (File);

      procedure Scan_Directory (Dir : in Dir_Name_Str);

      function Is_A_VCG_File (File : in String) return Boolean
      is
         T : constant String := Ada.Characters.Handling.To_Lower (File);
      begin
         return (T'Length >= 5) and then
           (T (T'Last - 3) = '.' and
            T (T'Last - 2) = 'v' and
            T (T'Last - 1) = 'c' and
            T (T'Last)     = 'g');
      end Is_A_VCG_File;

      function Is_A_PFS_File (File : in String) return Boolean
      is
         T : constant String := Ada.Characters.Handling.To_Lower (File);
      begin
         return (T'Length >= 5) and then
           (T (T'Last - 3) = '.' and
            T (T'Last - 2) = 'p' and
            T (T'Last - 1) = 'f' and
            T (T'Last)     = 's');
      end Is_A_PFS_File;

      function File_Needs_Simplifying (File : in String) return Boolean
      is
         Result : Boolean;
      begin
         if Is_A_VCG_File (File) or Is_A_PFS_File (File) then

            if CMD.All_Files then
               Result := True;
            else
               declare
                  Simplified_File      : String (1 .. File'Length);
                  File_Time        : GNAT.OS_Lib.OS_Time;
                  Simplified_File_Time : GNAT.OS_Lib.OS_Time;
               begin
                  Simplified_File := File;
                  Simplified_File (Simplified_File'Last - 2) := 's';
                  Simplified_File (Simplified_File'Last - 1) := 'i';

                  --  Simplified VCG files end in ".siv", while
                  --  simplified PFS files end in ".sip", so...
                  if Is_A_VCG_File (File) then
                     Simplified_File (Simplified_File'Last) := 'v';
                  else
                     Simplified_File (Simplified_File'Last) := 'p';
                  end if;

                  if GNAT.IO_Aux.File_Exists (Simplified_File) then
                     File_Time := GNAT.OS_Lib.File_Time_Stamp (File);
                     Simplified_File_Time := GNAT.OS_Lib.File_Time_Stamp
                       (Simplified_File);

                     Result := Is_Older (Simplified_File_Time, File_Time);
                  else
                     --  Simplified file does not exist, so we definitely need
                     --  to simplify the VCG or PFS file
                     Result := True;
                  end if;
               end;
            end if;
         else
            Result := False;
         end if;
         return Result;
      end File_Needs_Simplifying;

      procedure Scan_Directory (Dir : in Dir_Name_Str)
      is
         D    : Dir_Type;
         Str  : String (1 .. 1024);
         Last : Natural;
      begin
         Open (D, Dir);
         loop
            Read (D, Str, Last);
            exit when Last = 0;

            declare
               F : constant String := Dir & Str (1 .. Last);
            begin
               if GNAT.OS_Lib.Is_Directory (F) then
                  Debug ("Found a directory : " & F);
                  --  Ignore "." and ".."
                  if ((Last = 1) and then (Str (1) = '.')) or
                    ((Last = 2) and then (Str (1) = '.' and
                                          Str (2) = '.')) then
                     null;
                  else
                     --  Recurse here
                     Scan_Directory (F & GNAT.OS_Lib.Directory_Separator);
                  end if;
               else
                  if Is_A_VCG_File (F) or Is_A_PFS_File (F) then
                     Debug ("Found a VCG or PFS file : " & F);
                     if File_Needs_Simplifying (F) then
                        Work_Manager.Jobs.Add_File (F);
                     end if;
                  end if;
               end if;
            end;

         end loop;
         Close (D);
      exception
         when others =>
            Close (D);
            raise;
      end Scan_Directory;

      CWD  : constant Dir_Name_Str := Get_Current_Dir;
   begin
      Scan_Directory (CWD);
   exception
      when others =>
         Ada.Text_IO.Put_Line ("Error scanning directories.");
         Work_Manager.Jobs.Clear;
   end Find_Files_To_Simp;

   ---------------------------------------------------------------------------
   --  procedure Simplify_Files                                             --
   ---------------------------------------------------------------------------
   procedure Simplify_Files
   is

      Start_Time   : Ada.Real_Time.Time;
      End_Time     : Ada.Real_Time.Time;
      Elapsed_Time : Duration;

      Working_Set : Workers.Worker_Set (CMD.Processes);

      Job_ID : Work_Manager.Job_Index;
      TF, PF, IPF, CF, FF : Work_Manager.Job_Index;
      WA : Natural;
   begin
      --  Do we have any work?
      if Work_Manager.Jobs.Total_Number_Of_Files /= 0 then

         Start_Time := Ada.Real_Time.Clock;

         Workers.Initialize (Working_Set,
                             CMD.SArgs);

         loop
            while Workers.Workers_Available (Working_Set) > 0 and
              Work_Manager.Jobs.Number_Of_Files
                 (Of_Status => Work_Manager.Pending) /= 0
            loop
               Work_Manager.Jobs.GetNextJob (Job_ID);
               Debug ("Starting Job with JobID =" & Integer'Image (Job_ID));
               Workers.Start_Simp (Job_ID,
                                   Working_Set);
            end loop;

            Workers.Run_Simp (Working_Set);

            TF   := Work_Manager.Jobs.Total_Number_Of_Files;
            PF   := Work_Manager.Jobs.Number_Of_Files
              (Work_Manager.Pending);
            IPF  := Work_Manager.Jobs.Number_Of_Files
              (Work_Manager.InProgress);
            CF   := Work_Manager.Jobs.Number_Of_Files
              (Work_Manager.Finished);
            FF   := Work_Manager.Jobs.Number_Of_Files
              (Work_Manager.Failed);
            WA  := Workers.Workers_Available (Working_Set);

            Debug ("Job finished...");
            Debug ("Total      " & Integer'Image (TF));
            Debug ("Pending    " & Integer'Image (PF));
            Debug ("InProgress " & Integer'Image (IPF));
            Debug ("Finished   " & Integer'Image (CF));
            Debug ("Failed     " & Integer'Image (FF));
            Debug ("WA      is " & Integer'Image (WA));

            exit when (CF + FF = TF) and (PF = 0) and (IPF = 0);

         end loop;

         End_Time := Ada.Real_Time.Clock;
         Elapsed_Time := Ada.Real_Time.To_Duration (End_Time - Start_Time);
         Put_Message_With_Duration ("Total elapsed time: ",
                                 Elapsed_Time);

      end if;

   end Simplify_Files;

   ---------------------------------------------------------------------------
   --  procedure Report_Errors                                              --
   ---------------------------------------------------------------------------
   procedure Report_Errors
   is
      Last_Job : Work_Manager.Job_Index;
   begin
      Last_Job := Work_Manager.Jobs.Total_Number_Of_Files;
      if Work_Manager.AnyFailed then
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put_Line
         ("The following files reported an error during simplification:");
         Ada.Text_IO.New_Line;
         --  number of files is always > 0 when report_errors is called
         for Job_Id in Natural range 1 .. Last_Job loop
            if Work_Manager.Jobs.Get_HasFailed (Job_Id) then
               Ada.Text_IO.Put_Line (Work_Manager.Jobs.Get_File_Name (Job_Id));
               Ada.Text_IO.Put_Line
               ("   " & Work_Manager.Jobs.Get_WhyFailed (Job_Id));
            end if;
         end loop;
      end if;
   end Report_Errors;

begin  --  Code of SPARKSimp

   if GNAT.OS_Lib.Directory_Separator = '/' then
      -- UNIX-like platforms - switches always begin with '-'
      CMD.Process_Command_Line ('-');
   else
      -- Windows platforms.  We might have been passed arguments with
      -- "-" or "/", so we peek at the first character of the
      -- first argument to see.

      if Ada.Command_Line.Argument_Count = 0 then
         -- No arguments - always valid - but call
         -- Process_Command_Line anyway to set up
         -- flags
         CMD.Process_Command_Line ('-');
      else
         declare
            First_Arg : constant String := Ada.Command_Line.Argument (1);
         begin
            if First_Arg (1) = '/' then
               CMD.Process_Command_Line ('/');
            elsif First_Arg (1) = '-' then
               CMD.Process_Command_Line ('-');
            else
               -- Must be illegal, but process anyway to set
               -- flags and raise errors
               CMD.Process_Command_Line ('-');
            end if;
         end;
      end if;
   end if;

   if CMD.Version_Requested then
      Banner;
   elsif CMD.Valid then
      Workers.Locate_Spadesimp;
      Banner;
      if Workers.Spadesimp_Exe = null then
         Ada.Text_IO.Put ("Error: Can't locate ");
         if CMD.Exe_Switch = null then
            Ada.Text_IO.Put_Line
               (Workers.Spadesimp_Command & " binary on PATH");
         else
            Ada.Text_IO.Put ("simplifier executable specified by ");
            Ada.Text_IO.Put_Line ("-x=" & CMD.Exe_Switch.all);
         end if;
      else
         Find_Files_To_Simp;
         Work_Manager.Jobs.List_Jobs;
         if Work_Manager.Jobs.Total_Number_Of_Files > 0 then
            if CMD.Run_Simplifier then
               Work_Manager.Jobs.Sort_Files_By_Size;
               Work_Manager.Jobs.Display_Status_Banner;
               Simplify_Files;
               Report_Errors;
            else
               Ada.Text_IO.Put_Line
                 ("Dry run mode - no simplifications performed");
            end if;
         end if;
      end if;

   else
      Banner;
      CMD.Usage;
   end if;

exception
   when E : others =>
      Ada.Text_IO.New_Line;
      Ada.Text_IO.Put_Line ("Unhandled Exception in SPARKSimp.");
      Ada.Text_IO.New_Line;
      Ada.Text_IO.Put_Line (Version.Toolset_Support_Line1);
      Ada.Text_IO.Put_Line (Version.Toolset_Support_Line2);
      Ada.Text_IO.Put_Line (Version.Toolset_Support_Line3);
      Ada.Text_IO.Put_Line (Version.Toolset_Support_Line4);
      Ada.Text_IO.New_Line;
      Ada.Text_IO.Put_Line ("Exception information:");
      Ada.Text_IO.Put_Line (Ada.Exceptions.Exception_Information (E));
      Ada.Text_IO.Put_Line ("Traceback:");
      Ada.Text_IO.Put_Line (GNAT.Traceback.Symbolic.Symbolic_Traceback (E));

end SPARKSimp;
------------------------------------------------------------------------------
