-- $Id: unitmanager.adb 13045 2009-04-20 08:41:19Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with SparkLex;
with RegularExpression;
with SPARK_IO;
with Directory_Operations;
with UnitManager.UnitStore;
with SparkMakeErrors;

package body UnitManager
--# own State is UnitManager.UnitStore.State;
is

   procedure Initialise (TheDirectories : in     StringList.Object;
                         Include        : in     StringList.Object;
                         Exclude        : in     StringList.Object;
                         RootFile       : in     EStrings.T;
                         Duplicates     : in     Boolean;
                         Success        :    out Boolean)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#        in out UnitStore.State;
   --#           out SparkLex.CurrLine;
   --# derives ErrorHandler.ErrorContext,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys,
   --#         Success,
   --#         UnitStore.State             from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          Duplicates,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          Exclude,
   --#                                          Include,
   --#                                          LexTokenManager.StringTable,
   --#                                          RootFile,
   --#                                          SPARK_IO.File_Sys,
   --#                                          TheDirectories,
   --#                                          UnitStore.State;
   is
      DirectoryIt : StringList.Iterator;
      FileIt      : StringList.Iterator;
      IncludeIt   : StringList.Iterator;
      ExcludeIt   : StringList.Iterator;

      CurrentDir     : EStrings.T;
      CurrentFile    : EStrings.T;

      AddThisFile    : Boolean;

      TheRegExp      : RegularExpression.Object;

      --------------------------------------------------------------------------
      procedure AddFile (CurrentFile : in     EStrings.T;
                         Duplicates  : in     Boolean;
                         Success     : in out Boolean)
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out UnitStore.State;
      --#           out SparkLex.CurrLine;
      --# derives ErrorHandler.ErrorContext,
      --#         LexTokenManager.StringTable,
      --#         UnitStore.State             from *,
      --#                                          CommandLineData.Content,
      --#                                          CurrentFile,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys &
      --#         SparkLex.CurrLine           from CommandLineData.Content,
      --#                                          CurrentFile,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys &
      --#         SPARK_IO.File_Sys,
      --#         Success                     from *,
      --#                                          CommandLineData.Content,
      --#                                          CurrentFile,
      --#                                          Dictionary.Dict,
      --#                                          Duplicates,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          UnitStore.State;
      is
         CurrentUnit    : Unit.Object;
      begin
         -- Extract the unit
         Unit.GetUnit (InFile  => CurrentFile,
                       TheUnit => CurrentUnit);

         if CurrentUnit = Unit.NullObject then
         -- This will be reported as warning and the unit ignored.
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidUnit,
                                    EStr1 => CurrentFile,
                                    EStr2 => EStrings.EmptyString,
                                    EStr3 => EStrings.EmptyString);
         else
            UnitStore.Add (TheUnit => CurrentUnit,
                           Added   => Success);
            if not Success then
               -- check to see if the filenames are different.  Okay if unit appears
               -- twice in same file, but fail if filenames are different, i.e.
               -- the same unit appears in more than one file.
               if EStrings.EqString
                   (CurrentFile,
                      UnitStore.Get (TheUnit => CurrentUnit.TheId).TheFile) then
                  Success := True;
               else
                  -- check to see whether duplicates are errors switch is set
                  -- report errors/warnings as appropriate
                  if Duplicates then
                     SparkMakeErrors.Report
                       (TheFault => SparkMakeErrors.DuplicateErrors,
                        EStr1 => CurrentUnit.TheId.TheName,
                        EStr2 => CurrentFile,
                        EStr3 => UnitStore.Get (TheUnit => CurrentUnit.TheId).TheFile);
                     Success := False;
                  else
                     SparkMakeErrors.Report
                       (TheFault => SparkMakeErrors.DuplicateOkay,
                        EStr1 => CurrentUnit.TheId.TheName,
                        EStr2 => CurrentFile,
                        EStr3 => UnitStore.Get (TheUnit => CurrentUnit.TheId).TheFile);
                     Success := True;
                  end if;
               end if;
            end if;
         end if;
      end AddFile;

      --------------------------------------------------------------------------
   begin
      Success := True;
      SparkLex.ClearLineContext;

      --ensure root file is added, even if not in current directory
      if not EStrings.IsEmpty (RootFile) then
         AddFile (RootFile, Duplicates, Success);
      end if;

      -- For all the directories
      --
      DirectoryIt := StringList.GetFirst (InList => TheDirectories);
      while Success and
        not StringList.IsNull (DirectoryIt) loop

         CurrentDir := StringList.Value (DirectoryIt);
         SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              "Processing directory ",
                              0);
         EStrings.PutLine (SPARK_IO.Standard_Output, CurrentDir);

         -- For all the include file regular expressions
         --
         IncludeIt := StringList.GetFirst (InList => Include);
         while Success and
           not StringList.IsNull (IncludeIt) loop

            -- For all the files matching this regular expression
            --
            TheRegExp := RegularExpression.Create (StringList.Value (IncludeIt));

            FileIt := StringList.GetFirst
              (InList => Directory_Operations.FindFiles
                 (Matching    => TheRegExp,
                  InDirectory => CurrentDir,
                  Recursively => True));

            while Success and
              not StringList.IsNull (FileIt) loop

               AddThisFile := True;
               CurrentFile := StringList.Value (FileIt);

               -- don't exclude the root file
               if not EStrings.EqString (CurrentFile, RootFile) then
                  -- For all the exclude file regular expressions
                  --
                  ExcludeIt := StringList.GetFirst (InList => Exclude);
                  while not StringList.IsNull (ExcludeIt) loop

                     if RegularExpression.Matches
                       (EStr      => CurrentFile,
                        TheRegExp => RegularExpression.Create
                          (StringList.Value (ExcludeIt))) then

                        AddThisFile := False;
                        exit;
                     end if;
                     ExcludeIt := StringList.Next (ExcludeIt);
                  end loop;
               end if;

               if AddThisFile then
                  AddFile (CurrentFile, Duplicates, Success);
               end if;

               FileIt := StringList.Next (FileIt);
            end loop;

            IncludeIt := StringList.Next (IncludeIt);
         end loop;

         DirectoryIt := StringList.Next (DirectoryIt);
      end loop;

   end Initialise;

   -------------------------------------------------------------

   function GetAllUnits return Units.Stack
   --# global in UnitStore.State;
   is
   begin
      return UnitStore.GetAllUnits;
   end GetAllUnits;

   -------------------------------------------------------------

   procedure GetFile (ForUnit : in     Unit.Id;
                      TheFile :    out EStrings.T;
                      Found   :    out Boolean)
   --# global in UnitStore.State;
   --# derives Found,
   --#         TheFile from ForUnit,
   --#                      UnitStore.State;
   is
   begin
      TheFile := UnitStore.Get (TheUnit => ForUnit).TheFile;
      Found := not EStrings.IsEmpty (TheFile);
   end GetFile;

   -------------------------------------------------------------

   procedure GetUnit (InFile  : in     EStrings.T;
                      TheUnit :    out Unit.Id;
                      Found   :    out Boolean)
   --# global in UnitStore.State;
   --# derives Found,
   --#         TheUnit from InFile,
   --#                      UnitStore.State;
   is
      CurrentUnit : Unit.Object;
      Id          : Unit.Id;
      AllUnits    : Units.Stack;
   begin
      Found := False;
      TheUnit := Unit.NullId;
      AllUnits := UnitStore.GetAllUnits;
      while not Units.IsEmpty (AllUnits) loop
         Units.Pop (TheStack => AllUnits,
                    TheUnit  => Id);
         CurrentUnit := UnitStore.Get (TheUnit => Id);
         if EStrings.EqString (InFile, CurrentUnit.TheFile) then
            Found := True;
            TheUnit := CurrentUnit.TheId;
            exit;
         end if;
      end loop;
   end GetUnit;

   -------------------------------------------------------------

   function Get (TheUnit : Unit.Id) return Unit.Object
   --# global in UnitStore.State;
   is
   begin
      return UnitStore.Get (TheUnit => TheUnit);
   end Get;

   -------------------------------------------------------------

   function Parent (OfUnit : Unit.Id) return Unit.Id
   --# global in UnitStore.State;
   is
      ParentUnit : Unit.Id;
   begin
      case OfUnit.TheKind is

         when Unit.PackageSpecificationUnit |
           Unit.MainProgramUnit |
           Unit.PackageBodyUnit =>
            -- These units do not have parents.
            ParentUnit := Unit.NullId;

         when Unit.SeparateBodyUnit =>

            ParentUnit :=
              UnitStore.GetBodyUnit (WithName => Unit.Prefix (OfUnit.TheName)).TheId;

         when Unit.ChildSpecificationUnit =>

            ParentUnit :=
              UnitStore.GetSpecificationUnit (WithName => Unit.Prefix (OfUnit.TheName)).TheId;

      end case;
      return ParentUnit;
   end Parent;

   -------------------------------------------------------------

   function PackageBody (ForUnit : Unit.Id) return Unit.Id
   --# global in UnitStore.State;
   is
      Result : Unit.Id;
   begin
      case ForUnit.TheKind is

         when Unit.SpecificationUnit =>

            Result := Unit.Id'(TheName => ForUnit.TheName,
                               TheKind => Unit.PackageBodyUnit);

         when Unit.PackageBodyUnit =>

            Result := ForUnit;

         when Unit.SeparateBodyUnit =>

            Result := ForUnit;
            while Result.TheKind /= Unit.PackageBodyUnit loop
               Result := Parent (Result);
            end loop;

         when Unit.MainProgramUnit =>

            Result := Unit.NullId;

      end case;
      return Result;
   end PackageBody;

   -------------------------------------------------------------

   function IsAComponent (ThisUnit   : Unit.Id;
                          OfThisUnit : Unit.Id) return Boolean
   --# global in UnitStore.State;
   --
   -- Returns True is ThisUnit is a component OfThisUnit according to
   -- the rules given below
   is
      Result     : Boolean := False;
      ParentUnit : Unit.Id;
   begin
      case ThisUnit.TheKind is

         when Unit.PrivateChildPackageSpecificationUnit =>

            -- If ThisUnit is a private child then it is a component
            -- OfThisUnit if OfThisUnit is its immediate parent.
            Result := EStrings.EqString
              (OfThisUnit.TheName, Parent (ThisUnit).TheName);

         when Unit.PublicChildPackageSpecificationUnit =>

            -- If ThisUnit is a public child then it is a component
            -- OfThisUnit if there is exactly one private parent
            -- between it and OfThisUnit and this private parent is an
            -- immediate child OfThisUnit.
            ParentUnit := Parent (OfUnit => ThisUnit);
            while ParentUnit /= Unit.NullId loop
               if ParentUnit.TheKind = Unit.PrivateChildPackageSpecificationUnit then
                  Result := EStrings.EqString
                    (OfThisUnit.TheName, Parent (ParentUnit).TheName);
                  exit;
               end if;
               ParentUnit := Parent (OfUnit => ParentUnit);
            end loop;

         when others =>

            null;

      end case;
      return Result;
   end IsAComponent;

   -------------------------------------------------------------

   function InheritedUnits (ForUnit : Unit.Id) return Units.Stack
   --# global in UnitStore.State;
   is
      Result : Units.Stack;
      TheUnit : Unit.Object;
      InheritedUnit : Unit.Object;
      It : StringList.Iterator;
   begin
      Result := Units.NullStack;
      TheUnit := UnitStore.Get (TheUnit => ForUnit);
      if TheUnit /= Unit.NullObject then
         It := StringList.GetFirst (InList => TheUnit.TheInheritedUnits);
         while not StringList.IsNull (It) loop
            InheritedUnit := UnitStore.GetSpecificationUnit (WithName => StringList.Value (It));
            if InheritedUnit /= Unit.NullObject then
               Units.Push (TheStack => Result,
                           TheUnit => InheritedUnit.TheId);
            end if;
            It := StringList.Next (It);
         end loop;
      end if;
      return Result;
   end InheritedUnits;

   -------------------------------------------------------------

   function WithedUnits (ForUnit : Unit.Id) return Units.Stack
   --# global in UnitStore.State;
   is
      Result : Units.Stack;
      TheUnit : Unit.Object;
      WithedUnit : Unit.Object;
      It : StringList.Iterator;
   begin
      Result := Units.NullStack;
      TheUnit := UnitStore.Get (TheUnit => ForUnit);
      if TheUnit /= Unit.NullObject then
         It := StringList.GetFirst (InList => TheUnit.TheWithedUnits);
         while not StringList.IsNull (It) loop
            WithedUnit := UnitStore.GetSpecificationUnit (WithName => StringList.Value (It));
            if WithedUnit /= Unit.NullObject then
               Units.Push (TheStack => Result,
                           TheUnit => WithedUnit.TheId);
            end if;
            It := StringList.Next (It);
         end loop;
      end if;
      return Result;
   end WithedUnits;

   -------------------------------------------------------------

   function WithedComponents (ForUnit : Unit.Id) return Units.Stack
   --# global in UnitStore.State;
   is
      TheWithedUnits : Units.Stack;
      Result         : Units.Stack;
      Id             : Unit.Id;
      ParentUnit     : Unit.Id;
   begin
      Result := Units.NullStack;
      TheWithedUnits := WithedUnits (ForUnit => ForUnit);
      while not Units.IsEmpty (TheWithedUnits) loop
         Units.Pop (TheStack => TheWithedUnits,
                    TheUnit  => Id);

         ParentUnit := ForUnit;
         while ParentUnit /= Unit.NullId loop
            if IsAComponent (ThisUnit => Id,
                             OfThisUnit => ParentUnit) then
               Units.Push (TheStack => Result,
                           TheUnit => Id);
            end if;
            ParentUnit := Parent (ParentUnit);
         end loop;
      end loop;
      return Result;
   end WithedComponents;

   -------------------------------------------------------------

   function SeparateUnits (ForUnit : Unit.Id) return Units.Stack
   --# global in UnitStore.State;
   is
      AllUnits    : Units.Stack;
      Id          : Unit.Id;
      Result      : Units.Stack;
      CurrentUnit : Unit.Object;
   begin
      Result := Units.NullStack;
      AllUnits := UnitStore.GetAllUnits;
      while not Units.IsEmpty (AllUnits) loop
         Units.Pop (TheStack => AllUnits,
                    TheUnit  => Id);
         CurrentUnit := UnitStore.Get (TheUnit => Id);
         if CurrentUnit.TheId.TheKind = Unit.SeparateBodyUnit then
            if Unit.AreEqual (Parent (CurrentUnit.TheId), ForUnit) then
               Units.Push (TheStack => Result,
                           TheUnit => CurrentUnit.TheId);
            end if;
         end if;
      end loop;
      return Result;
   end SeparateUnits;

   -------------------------------------------------------------

   function RequiredUnits (ForUnit : Unit.Id) return Units.Stack
   --# global in UnitStore.State;
   is
      Result              : Units.Stack;
      TheWithedComponents : Units.Stack;
      Id                  : Unit.Id;
   begin
      -- We'll need all the inherited units
      Result := InheritedUnits (ForUnit => ForUnit);

      if ForUnit.TheKind = Unit.SeparateBodyUnit then

         -- .. we need any withed components as no inherit is required
         TheWithedComponents := WithedComponents (ForUnit => ForUnit);

         while not Units.IsEmpty (TheWithedComponents) loop
            Units.Pop (TheStack => TheWithedComponents,
                       TheUnit  => Id);
            Units.Push (TheStack => Result,
                        TheUnit  => Id);
         end loop;

         -- We'll also need the body if it's a separate ...
         Units.Push (TheStack => Result,
                     TheUnit  => Parent (OfUnit => ForUnit));

      elsif ForUnit.TheKind = Unit.PackageBodyUnit then

         -- .. we need any withed components as no inherit is required
         TheWithedComponents := WithedComponents (ForUnit => ForUnit);

         while not Units.IsEmpty (TheWithedComponents) loop
            Units.Pop (TheStack => TheWithedComponents,
                       TheUnit  => Id);
            Units.Push (TheStack => Result,
                        TheUnit  => Id);
         end loop;

         -- ... and the spec ...
         Units.Push (TheStack => Result,
                     TheUnit => UnitStore.GetSpecificationUnit
                       (WithName => ForUnit.TheName).TheId);

      elsif ForUnit.TheKind in Unit.ChildSpecificationUnit then
         -- ... or the parent if it's a child package specification
         Units.Push (TheStack => Result,
                     TheUnit  => Parent (OfUnit => ForUnit));
      end if;
      return Result;
   end RequiredUnits;

   -------------------------------------------------------------

   function Components (ForUnit : Unit.Id) return Units.Stack
   --# global in UnitStore.State;
   is
      Result      : Units.Stack;
      Id          : Unit.Id;
      AllUnits    : Units.Stack;
   begin
      Result := Units.NullStack;
      AllUnits := UnitStore.GetAllUnits;
      while not Units.IsEmpty (AllUnits) loop
         Units.Pop (TheStack => AllUnits,
                    TheUnit  => Id);
         if IsAComponent (ThisUnit => Id,
                          OfThisUnit => ForUnit) then
            Units.Push (TheStack => Result,
                        TheUnit => Id);
         end if;
      end loop;
      return Result;
   end Components;


   -------------------------------------------------------------
   -- Find the root units. Slightly trickier than it may seem at
   -- first because we are not simply dealing with relationships
   -- between packages (eg package P requires package Q) but have
   -- to consider relationships between all compilation units.
   -- For example, q.adb requires q.ads. If q.adb isn't required
   -- by anything else (ie no separates) then it would seem to be
   -- a 'root' but we must not treat it as one because any
   -- package bodies are automatically added to the meta file
   -- after their corresponding specs. So the roots we want to
   -- find are:
   --  - any main programs;
   --  - any package specifications that are not required by
   --    other packages or main programs.
   function FindRoots return Units.Stack
   --# global in UnitStore.State;
   is
      AllUnits : Units.Stack;
      TheUnit  : Unit.Id;
      Result   : Units.Stack;

      -- Return value indicates whether there are any main programs
      -- or package specifications that require the given unit.
      function IsRequired (ThisUnit : Unit.Id) return Boolean
      --# global in UnitStore.State;
      is
         OtherUnits   : Units.Stack;
         ReqUnits     : Units.Stack;
         TheReqUnit   : Unit.Id;
         TheOtherUnit : Unit.Id;
         Found        : Boolean := False;
      begin

         OtherUnits := GetAllUnits;

         -- Check whether each other unit requires this unit.
         while not Found and not Units.IsEmpty (OtherUnits) loop

            Units.Pop (TheStack => OtherUnits,
                       TheUnit  => TheOtherUnit);

            -- Get the required units for the other unit
            ReqUnits := RequiredUnits (ForUnit => TheOtherUnit);

            -- Check each other unit that requires this unit. If
            -- any package specs or main programs require it then
            -- it can't be a root.
            while not Found and not Units.IsEmpty (ReqUnits) loop

               Units.Pop (TheStack => ReqUnits,
                          TheUnit  => TheReqUnit);

               if TheReqUnit = ThisUnit and
                  (TheOtherUnit.TheKind = Unit.MainProgramUnit or
                   TheOtherUnit.TheKind = Unit.PackageSpecificationUnit) then

                  Found := True;

               end if;

            end loop;

         end loop;

         return Found;

      end IsRequired;

   begin -- FindRoots

      AllUnits := GetAllUnits;
      Result := Units.NullStack;

      -- Check all the units we know about...
      while not Units.IsEmpty (AllUnits) loop

         Units.Pop (TheStack => AllUnits,
                    TheUnit  => TheUnit);

         -- If it's a main program, or a package spec that isn't
         -- required by any other units, then it must be a root.
         if TheUnit.TheKind = Unit.MainProgramUnit or
            (TheUnit.TheKind = Unit.PackageSpecificationUnit and
             not IsRequired (TheUnit)) then

            Units.Push (TheStack => Result,
                        TheUnit  => TheUnit);

         end if;

      end loop;

      return Result;

   end FindRoots;

end UnitManager;
