-- $Id: unitmanager-unitstore.adb 11375 2008-10-08 10:40:42Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with GNAT.HTable;
with EStrings;

package body UnitManager.UnitStore
is
   ---------------------------------------------------------------------------
   -- This package body is NOT SPARK, and should not be                     --
   -- presented to the Examiner                                             --
   ---------------------------------------------------------------------------


   MaxUnits : constant := 1000;
   type Index is range 1 .. MaxUnits;

   function Hash (Id : Unit.Id) return Index;

   package UnitTable is new GNAT.HTable.Simple_HTable
     (Header_Num => Index,
      Element    => Unit.Object,
      No_Element => Unit.NullObject,
      Key        => Unit.Id,
      Hash       => Hash,
      Equal      => Unit.AreEqual);

   ---------------------------------------------------------------------------

   function Key (FromId : Unit.Id) return EStrings.T
   --
   -- Returns a unique key (string) for the unit.
   is
      Result : EStrings.T := EStrings.EmptyString;
   begin
      EStrings.AppendExaminerString
        (EStr1 => Result,
         EStr2 => FromId.TheName);
      EStrings.AppendString
        (EStr => Result,
         Str  => Unit.Kind'Image (FromId.TheKind));
      return Result;
   end Key;

   --------------------------------------------------------------------------------

   function Hash (Id : Unit.Id) return Index
   is
      TheKey : EStrings.T;

      function RawHash (Key : String) return Index is

         type Uns is mod 2 ** 32;

         -- GNAT-Specific Import here
         function Rotate_Left (Value : Uns; Amount : Natural) return Uns;
         pragma Import (Intrinsic, Rotate_Left);

         Tmp : Uns := 0;
      begin
         for J in Key'Range loop
            Tmp := Rotate_Left (Tmp, 1) + Character'Pos (Key (J));
         end loop;

         return Index'First +
           Index'Base (Tmp mod Index'Range_Length); -- also GNAT-defined attrib
      end RawHash;

   begin
      TheKey := Key (FromId => Id);
      return RawHash (TheKey.Content (1 .. TheKey.Length));
   end Hash;

   --------------------------------------------------------------------------------

   procedure Add (TheUnit : in     Unit.Object;
                  Added   :    out Boolean)
   is
   begin
      if Get (TheUnit.TheId) = Unit.NullObject then
         Added := True;
         UnitTable.Set (K => TheUnit.TheId,
                        E => TheUnit);
      else
         Added := False;
      end if;
   end Add;

   --------------------------------------------------------------------------------

   function Get (TheUnit : Unit.Id) return Unit.Object
   is
   begin
      return UnitTable.Get (K => TheUnit);
   end Get;

   --------------------------------------------------------------------------------

   function GetBodyUnit (WithName : in EStrings.T) return Unit.Object
   is
      TheUnit : Unit.Object;
   begin
      for Kind in Unit.Kind range Unit.MainProgramUnit .. Unit.SeparateBodyUnit loop
         TheUnit := Get (TheUnit => Unit.Id'(TheName => WithName,
                                             TheKind => Kind));
         exit when TheUnit /= Unit.NullObject;
      end loop;
      return TheUnit;
   end GetBodyUnit;

   --------------------------------------------------------------------------------

   function GetSpecificationUnit (WithName : in EStrings.T) return Unit.Object
   is
      TheUnit : Unit.Object;
   begin
      for Kind in Unit.SpecificationUnit loop
         TheUnit := Get (TheUnit => Unit.Id'(TheName => WithName,
                                             TheKind => Kind));
         exit when TheUnit /= Unit.NullObject;
      end loop;
      return TheUnit;
   end GetSpecificationUnit;

   --------------------------------------------------------------------------------

   function GetAllUnits return Units.Stack
     is
      CurrentUnit : Unit.Object;
      TheUnits    : Units.Stack := Units.NullStack;
   begin
      CurrentUnit := UnitTable.Get_First;
      while CurrentUnit /= Unit.NullObject loop
         Units.Push (TheStack => TheUnits,
                     TheUnit  => CurrentUnit.TheId);
         CurrentUnit := UnitTable.Get_Next;
      end loop;
      return TheUnits;
   end GetAllUnits;

begin
   UnitTable.Reset;
end UnitManager.UnitStore;
