-- $Id: tokenmanager.ads 11375 2008-10-08 10:40:42Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with EStrings;
with SPARK_IO;
with SPSymbols;

--# inherit CommandLineData,
--#         Dictionary,
--#         ErrorHandler,
--#         EStrings,
--#         LexTokenManager,
--#         SparkLex,
--#         SparkMakeErrors,
--#         SPARK_IO,
--#         SPSymbols;

package TokenManager
is

   type Token is
      record
         Kind : SPSymbols.SPTerminal;
         Value : EStrings.T;
      end record;

   NullToken : constant Token :=
     Token'(Kind => SPSymbols.SPEND,
            Value => EStrings.EmptyString);

   type Iterator is private;

   NullIterator : constant Iterator;


   procedure GetFirstToken (FileId : in     SPARK_IO.File_Type;
                            It     :    out Iterator);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SPARK_IO.File_Sys;
   --#           out SparkLex.CurrLine;
   --# derives ErrorHandler.ErrorContext,
   --#         It,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          FileId,
   --#                                          LexTokenManager.StringTable,
   --#                                          SPARK_IO.File_Sys;

   procedure Next (It : in out Iterator);
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         It,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          It,
   --#                                          LexTokenManager.StringTable,
   --#                                          SparkLex.CurrLine,
   --#                                          SPARK_IO.File_Sys;

   procedure LookAhead (It : in out Iterator);
   --
   -- Allows the client to look at the next symbol after the last returned
   -- by Next or NextIdentifier. Only one lookahead is allowed. Multiple,
   -- consecutive calls to LookAhead will thus return the same symbol.
   -- Subsequent calls to Next and NextIdentifier function as normal (i.e. as if
   -- the lookahead had not been made).
   --
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         It,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          It,
   --#                                          LexTokenManager.StringTable,
   --#                                          SparkLex.CurrLine,
   --#                                          SPARK_IO.File_Sys;

   function IsNull (It : Iterator) return Boolean;

   function Current (It : Iterator) return Token;

   function ToString (Tok : Token) return EStrings.T;

private

   type Iterator is
      record
         File         : SPARK_IO.File_Type;
         -- The token we expect
         CurrentToken : Token;
         -- Sometimes it is necessary to read a token too far.
         -- If this is the case then NextToken stores what the token is so it can
         -- be returned on the next call to Next.
         NextToken    : Token;
         -- Allows the user to lookahead.
         IsLookAhead  : Boolean;
      end record;

   NullIterator : constant Iterator :=
     Iterator'(File         => SPARK_IO.Null_File,
               CurrentToken => NullToken,
               NextToken    => NullToken,
               IsLookAhead  => False);

end TokenManager;
