-- $Id: tokenmanager.adb 11375 2008-10-08 10:40:42Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with LexTokenManager;
with SparkLex;
with SPSymbols;
with SparkMakeErrors;

use type SPSymbols.SPTerminal;

package body TokenManager
is

   procedure GetIdentifier (It : in out Iterator)
   --# global in     CommandLineData.Content;
   --#        in     Dictionary.Dict;
   --#        in out ErrorHandler.ErrorContext;
   --#        in out LexTokenManager.StringTable;
   --#        in out SparkLex.CurrLine;
   --#        in out SPARK_IO.File_Sys;
   --# derives ErrorHandler.ErrorContext,
   --#         It,
   --#         LexTokenManager.StringTable,
   --#         SparkLex.CurrLine,
   --#         SPARK_IO.File_Sys           from CommandLineData.Content,
   --#                                          Dictionary.Dict,
   --#                                          ErrorHandler.ErrorContext,
   --#                                          It,
   --#                                          LexTokenManager.StringTable,
   --#                                          SparkLex.CurrLine,
   --#                                          SPARK_IO.File_Sys;
   is
      Unused      : Boolean;
      TokenKind   : SPSymbols.SPTerminal;
      LexValue    : LexTokenManager.LexValue;
      TokenValue  : EStrings.T;
      NextId      : EStrings.T;
   begin
      while It /= NullIterator loop
         -- is there more dotted notation?
         --# accept Flow, 10, Unused, "Ineffective assignment OK";
         SparkLex.ExaminerLex (ProgText   => It.File,
                               Token      => TokenKind,
                               LexVal     => LexValue,
                               PunctToken => Unused);

         -- No, this is the end of the identifier and we have read a token too far.
         if TokenKind /= SPSymbols.point then

            LexTokenManager.LexStringToString
              (LexStr => LexValue.TokenStr,
               Str    => TokenValue);

            It.NextToken := Token'(Kind => TokenKind,
                                   Value => TokenValue);

            exit;
         end if;

         -- Read the next part of the dotted identifier
         SparkLex.ExaminerLex (ProgText   => It.File,
                               Token      => TokenKind,
                               LexVal     => LexValue,
                               PunctToken => Unused);

         if TokenKind /= SPSymbols.identifier then
            -- Invlaid syntax return a null iterator
            It := NullIterator;
         else
            LexTokenManager.LexStringToString
              (LexStr => LexValue.TokenStr,
               Str    => NextId);

            EStrings.AppendString (It.CurrentToken.Value, ".");
            EStrings.AppendExaminerString (It.CurrentToken.Value, NextId);
         end if;
         --# end accept;
      end loop;
      --# accept Flow, 33, Unused, "Unused not references OK";
   end GetIdentifier;

   -----------------------------------------------------------------------------

   procedure Next (It : in out Iterator)
   is
      Unused      : Boolean;
      TokenKind   : SPSymbols.SPTerminal;
      TokenValue  : EStrings.T;
      LexValue    : LexTokenManager.LexValue;
   begin
      if It = NullIterator then
         SparkMakeErrors.Fatal ("calling TokenManager.Next with a null iterator.");

      elsif It.IsLookAhead then
         It.IsLookAhead := False;

      elsif It.NextToken /= NullToken then
         It.CurrentToken := It.NextToken;
         It.NextToken := NullToken;

      else
         --# accept Flow, 10, Unused, "Ineffective assignment OK";
         SparkLex.ExaminerLex (ProgText   => It.File,
                               Token      => TokenKind,
                               LexVal     => LexValue,
                               PunctToken => Unused);
         --# end accept;

         if TokenKind = SPSymbols.SPEND then
            It := NullIterator;
         else
            LexTokenManager.LexStringToString
              (LexStr => LexValue.TokenStr,
               Str    => TokenValue);

            It.CurrentToken := Token'(Kind => TokenKind,
                                      Value => TokenValue);

            if TokenKind = SPSymbols.identifier then
               GetIdentifier (It);
            end if;
         end if;
      end if;
      -- Ensure all values are lower case.
      It.CurrentToken.Value := EStrings.LowerCase (It.CurrentToken.Value);
      --# accept Flow, 33, Unused, "Unused not references OK";
   end Next;

   -----------------------------------------------------------------------------

   procedure LookAhead (It : in out Iterator)
   is
   begin
      if It = NullIterator then
         SparkMakeErrors.Fatal ("calling TokenManager.LookAhead with a null iterator.");
      elsif not It.IsLookAhead then
         Next (It);
         It.IsLookAhead := True;
      end if;
   end LookAhead;

   -----------------------------------------------------------------------------

   procedure GetFirstToken (FileId : in     SPARK_IO.File_Type;
                            It     :    out Iterator)
   is
      NewIt : Iterator;
   begin
      SparkLex.ClearLineContext;
      NewIt := Iterator'(File => FileId,
                         CurrentToken => NullToken,
                         NextToken    => NullToken,
                         IsLookAhead => False);
      Next (NewIt);
      It := NewIt;
   end GetFirstToken;

   -----------------------------------------------------------------------------

   function IsNull (It : Iterator) return Boolean
   is
   begin
      return It = NullIterator;
   end IsNull;

   -----------------------------------------------------------------------------

   function Current (It : Iterator) return Token
   is
      Result : Token;
   begin
      if It = NullIterator then
         Result := NullToken;
      else
         Result := It.CurrentToken;
      end if;
      return Result;
   end Current;

   -----------------------------------------------------------------------------

   function ToString (Tok : Token) return EStrings.T
   is
      --# hide ToString;
      Result : EStrings.T := EStrings.EmptyString;
   begin
      EStrings.AppendString
        (EStr => Result,
         Str => SPSymbols.SPTerminal'Image (Tok.Kind) & ": ");
      EStrings.AppendExaminerString
        (EStr1 => Result,
         EStr2 => Tok.Value);
      return Result;
   end ToString;

end TokenManager;
