-- $Id: stringlist.adb 13044 2009-04-20 08:40:02Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with SPARK_IO;
with SparkMakeErrors;

package body StringList
is
   ------------------------------------------------------------------------
   -- This package body is NOT SPARK
   ------------------------------------------------------------------------

   function "<=" (Left, Right : in EStrings.T) return Boolean
   is
   begin
      -- uses Standard."<=" for type String
      return Left.Content (1 .. Left.Length) <= Right.Content (1 .. Right.Length);
   end "<=";


   ------------------------------------------------------------------------
   -- Constructors
   ------------------------------------------------------------------------

   procedure AddToFront (ToList  : in out Object;
                         TheItem : in     EStrings.T)
   is
   begin
      ToList := new Node'(TheItem => TheItem,
                          Next    => ToList);
   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.AddToFront");
   end AddToFront;

   procedure AddToBack (ToList  : in out Object;
                        TheItem : in     EStrings.T)
   is
      CurrentNode : Object;
      NewNode     : Object;
   begin
      -- Create new node
      NewNode := new Node'(TheItem => TheItem,
                           Next    => NullObject);

      if ToList = NullObject then
         ToList := NewNode;
      else
         CurrentNode := ToList;
         -- Find the final node in the list
         while CurrentNode.Next /= NullObject loop
            CurrentNode := CurrentNode.Next;
         end loop;

         -- Link in the new node
         CurrentNode.Next := NewNode;
      end if;

   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.AddToBack");
   end AddToBack;


   procedure AddInLexOrder (ToList  : in out Object;
                            TheItem : in     EStrings.T)
   is
      CurrentNode : Object;
      PrevNode    : Object;
      NewNode     : Object;
   begin
      -- Create new node
      NewNode := new Node'(TheItem => TheItem,
                           Next    => NullObject);

      if ToList = NullObject then
         -- ToList is empty, so
         ToList := NewNode;
      else
         CurrentNode := ToList;
         PrevNode := NullObject;

         -- Find spot between PrevNode and CurrentNode where NewNode
         -- needs to be inserted.  Note uses "<=" operator for
         -- EString.T defined above.
         while CurrentNode.TheItem <= TheItem loop
            PrevNode := CurrentNode;
            CurrentNode := CurrentNode.Next;
            exit when CurrentNode = NullObject;
         end loop;

         -- Link NewNode in before CurrentNode
         NewNode.Next := CurrentNode;
         if PrevNode = NullObject then
            -- No previous node - NewNode must be the new head of ToList
            ToList := NewNode;
         else
            -- Link previous node to NewNode
            PrevNode.Next := NewNode;
         end if;
      end if;

   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.AddInLexOrder");
   end AddInLexOrder;


   function GetFirst (InList : Object) return Iterator
   is
   begin
      return Iterator (InList);
   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.GetFirst");
         return NullIterator;
   end GetFirst;


   function Next (It : Iterator) return Iterator
   is
   begin
      return Iterator (It.all.Next);
   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.Next");
         return NullIterator;
   end Next;

   ------------------------------------------------------------------------
   -- Accessors
   ------------------------------------------------------------------------

   function IsNull (It : Iterator) return Boolean
   is
   begin
      return It = NullIterator;
   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.IsNull");
         return False;
   end IsNull;

   function Value (It : Iterator) return EStrings.T
   is
   begin
      return It.all.TheItem;
   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.Value");
         return EStrings.EmptyString;
   end Value;


   ------------------------------------------------------------------------
   -- Debug support
   ------------------------------------------------------------------------

   procedure Output (TheList : in Object;
                     How     : in Orientation)
   is
      It : Iterator;
   begin
      It := GetFirst (InList => TheList);
      if IsNull (It) then
         SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              "<<Empty List>>",
                              0);
      else
         loop
            if How = Vertical then   -- expect stable expression
               EStrings.PutLine (File => SPARK_IO.Standard_Output,
                                        EStr => Value (It));
            else
               EStrings.PutString (File => SPARK_IO.Standard_Output,
                                          EStr => Value (It));
            end if;
            It := Next (It);
            exit when IsNull (It);
            if How = Horizontal then
               SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                                    ", ",
                                    0);
            end if;
         end loop;
      end if;
      if How = Horizontal then
         SPARK_IO.New_Line (File    => SPARK_IO.Standard_Output,
                            Spacing => 1);
      end if;
   exception
      when others =>
         SparkMakeErrors.Fatal ("Exception raised in StringList.Output");
   end Output;

end StringList;
