-- $Id: sparkmakecommandline.adb 13045 2009-04-20 08:41:19Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with CommandLine;
with SPARK_IO;
with Directory_Operations;
with RegularExpression;
with SparkMakeDebug;
with SparkMakeErrors;
with SystemErrors;
with CommandLineData;
with FileSystem;
with Version;

package body SparkMakeCommandLine
--# own State is RootFile,
--#              Duplicates,
--#              Debug,
--#              TheIncRegExps,
--#              TheExcRegExps,
--#              TheDirectories,
--#              MetaFile,
--#              IndexFile,
--#              NoMeta,
--#              NoIndex,
--#              AnnoCharSpecified,
--#              Path;
is

   Path       : PathType;
   RootFile   : EStrings.T;
   MetaFile   : EStrings.T;
   IndexFile  : EStrings.T;
   Debug      : Boolean;
   Duplicates : Boolean;

   NoMeta     : Boolean;
   NoIndex    : Boolean;

   AnnoCharSpecified : Boolean;

   TheDirectories : StringList.Object;
   TheIncRegExps  : StringList.Object;
   TheExcRegExps  : StringList.Object;

   ------------------------------------------------------------------------------

   function NormaliseRootFileName (N : EStrings.T)
                                  return EStrings.T
   --
   -- Ensures any relative path is fully expanded.
   is
      Result  : EStrings.T;
   begin
      Result := Directory_Operations.NormalizePathName
        (Name      => N,
         Directory => Directory_Operations.CurrentDirectory);
      return Result;
   end NormaliseRootFileName;

   ------------------------------------------------------------------------------

   function NormaliseDirName (N : EStrings.T)
                             return EStrings.T
   --
   -- Ensures any relative path is fully expanded and that
   -- there is a directory separator at the end.
   is
      Result  : EStrings.T;
      Success : Boolean;
   begin
      Result := NormaliseRootFileName (N);
      if Result.Length > 0 and then
        Result.Content (Result.Length) /= Directory_Operations.DirectorySeparator then
         EStrings.AppendChar (EStr    => Result,
                                     Ch      => Directory_Operations.DirectorySeparator,
                                     Success => Success);
         if not Success then
            Result := EStrings.EmptyString;
         end if;
      else
         Result := EStrings.EmptyString;
      end if;
      return Result;
   end NormaliseDirName;

   ------------------------------------------------------------------------------

   function RegExp (ForFile : in EStrings.T) return
     EStrings.T
   --
   -- Returns a regulat expression that will exactly match the given file.
   is
      Result  : EStrings.T;
      Char    : Character;
      Success : Boolean := True;
      Escape : constant Character := '\';
   begin
      Result := EStrings.EmptyString;
      for I in EStrings.Positions range 1 .. ForFile.Length loop
         Char := ForFile.Content (I);
         if Char = '\' or
           Char = '(' or
           Char = ')' or
           Char = '[' or
           Char = ']' or
           Char = '.' or
           Char = '*' or
           Char = '+' or
           Char = '?' or
           Char = '^' then
            -- We must escape these characters
            EStrings.AppendChar (EStr    => Result,
                                        Ch      => Escape,
                                        Success => Success);
         end if;
         exit when not Success;
         EStrings.AppendChar (EStr    => Result,
                                     Ch      => Char,
                                     Success => Success);
         exit when not Success;
      end loop;
      if not Success then
         Result := EStrings.EmptyString;
      end if;
      return Result;
   end RegExp;

   ------------------------------------------------------------------------------

   procedure ReportUsage
   --# global in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *;
   --
   -- Outputs the usage to the user.
   is
      --# hide ReportUsage;
   begin
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "Usage: sparkmake [option] [rootfile]",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "Options - all may be abbreviated to the shortest unique prefix",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "Options - all input options may be repeated as necessary",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "option         = input_option | output_option | behaviour_option | help_option",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "input_option   = dir_option | include_option | exclude_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "dir_option     = " &
                           "-" &
                           "directory=dirname - Look in and under dirname as well as cwd.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                      Default: none",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "include_option = " &
                           "-" &
                           "include=reg_exp   - Only include files if full path matches.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                      Default: *\.ad[bs]",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "exclude_option = " &
                           "-" &
                           "exclude=reg_exp   - Exclude files if full path matches.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                      Default: none",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "reg_exp        = term",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = char",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = elmt elmt ...     -- concatenation",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = *                 -- any string of 0 or more characters",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = ?                 -- matches any character",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = [char char ...]   -- matches any character listed",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "term           = [char - char]     -- matches any character in given range",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = nchr              -- matches given character",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = [nchr nchr ...]   -- matches any character listed",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = [^ nchr nchr ...] -- matches any character not listed",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = [char - char]     -- matches chars in given range",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = .                 -- matches any single character",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "elmt           = ( regexp )        -- parens used for grouping",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "char           = any character, including special characters",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "nchr           = any character except \()[].*+?^ or \char to match char",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "The characters '{' and '}' are NOT allowed to appear in any regular expression",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "behaviour_option = duplicates_option | annotation_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "duplicates_option = " &
                           "-" &
                           "duplicates_are_errors - Fail if duplicate units are found.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: false",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "annotation_option = " &
                           "-" &
                           "annotation_character  - Specify annotation character.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: #",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "output_option  = index_option | meta_option | noindex_option | nometa_option |",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                 path_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "index_option   = " &
                           "-" &
                           "index=file-spec          - The index file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default rootfile.idx",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "meta_option    = " &
                           "-" &
                           "meta=file-spec           - The meta file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default rootfile.smf",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "noindex_option = " &
                           "-" &
                           "noindexfile              - Suppress generation of index file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: false",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "nometa_option  = " &
                           "-" &
                           "nometafile               - Suppress generation of meta file.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: false",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "path_option    = " &
                           "-" &
                           "path=full | relative     - Produce relative or full pathnames.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: full",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "help_option    = helper_option | version_option",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "helper_option  = " &
                           "-" &
                           "help    - print off help information.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "version_option = " &
                           "-" &
                           "version - print off version information.",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "rootfile       = file-spec                 - The file to make.",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             Default: Produce index and metafile",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             for analysis of all files in and",
                         0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                         "                                             under current directory.",
                         0);
      SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line1, 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line2, 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line3, 0);
      SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Version.Toolset_Support_Line4, 0);

   end ReportUsage;

   ------------------------------------------------------------------------------

   procedure ReportVersion
   --# global in out SPARK_IO.FILE_SYS;
   --# derives SPARK_IO.FILE_SYS from *;
   --
   -- Outputs the usage to the user.
   is
      --# hide ReportVersion;
   begin
      SPARK_IO.Put_String
        (SPARK_IO.Standard_Output,
         "SPARKMake " & Version.Toolset_Distribution & " Edition, ", 0);
      SPARK_IO.Put_String
        (SPARK_IO.Standard_Output,
         "Version " & Version.Toolset_Version & ", Date " & Version.Toolset_Build_Date, 0);
      SPARK_IO.Put_Line
        (SPARK_IO.Standard_Output,
         ", Build " & Version.Toolset_Build_Stamp, 0);
      SPARK_IO.Put_Line
        (SPARK_IO.Standard_Output,
         Version.Toolset_Copyright, 0);
   end ReportVersion;

   ------------------------------------------------------------------------------

   procedure ProcessRootFileArgument (Arg     : in     EStrings.T;
                                      Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#           out RootFile;
   --# derives RootFile,
   --#         Success           from Arg &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Arg &
   --#         null              from Debug;
   is
   begin
      -- Expand the root file to its full path
      RootFile := NormaliseRootFileName (Arg);

      if Debug then
         SparkMakeDebug.ReportTextEText ("Found root file argument: ", Arg);
         SparkMakeDebug.ReportTextEText ("Expanded to: ", RootFile);
      end if;

      -- Does the file exist?
      if Directory_Operations.IsFile (RootFile) then
         Success := True;
      else
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.CannotFindFile,
                                 EStr1 => RootFile,
                                 EStr2 => EStrings.EmptyString,
                                 EStr3 => EStrings.EmptyString);
      end if;
   end ProcessRootFileArgument;

   ------------------------------------------------------------------------------

   procedure ProcessIncludeSwitch (Switch  : in     EStrings.T;
                                   Arg     : in     EStrings.T;
                                   Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out TheIncRegExps;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Switch &
   --#         Success           from Arg &
   --#         TheIncRegExps     from *,
   --#                                Arg &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found include switch: ", Arg);
      end if;

      if EStrings.IsEmpty (Arg) or else
        RegularExpression.IsNull
        (RegularExpression.Create (Arg)) then
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                 EStr1 => Arg,
                                 EStr2 => Switch,
                                 EStr3 => EStrings.EmptyString);
      else
         Success := True;
         StringList.AddToFront (ToList  => TheIncRegExps,
                                TheItem => Arg);
      end if;
   end ProcessIncludeSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessExcludeSwitch (Switch  : in     EStrings.T;
                                   Arg     : in     EStrings.T;
                                   Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out TheExcRegExps;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Switch &
   --#         Success           from Arg &
   --#         TheExcRegExps     from *,
   --#                                Arg &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found exclude switch: ", Arg);
      end if;

      if EStrings.IsEmpty (Arg) or else
        RegularExpression.IsNull
        (RegularExpression.Create (Arg)) then
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                 EStr1 => Arg,
                                 EStr2 => Switch,
                                 EStr3 => EStrings.EmptyString);
      else
         Success := True;
         StringList.AddToFront (ToList  => TheExcRegExps,
                                TheItem => Arg);
      end if;
   end ProcessExcludeSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessDirectorySwitch (Switch  : in     EStrings.T;
                                     Arg     : in     EStrings.T;
                                     Success :    out Boolean)
   --# global in     Debug;
   --#        in out SPARK_IO.FILE_SYS;
   --#        in out TheDirectories;
   --# derives SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Switch &
   --#         Success           from Arg &
   --#         TheDirectories    from *,
   --#                                Arg &
   --#         null              from Debug;
   is
      TheDir : EStrings.T;
   begin
      TheDir := NormaliseDirName (Arg);

      if Debug then
         SparkMakeDebug.ReportTextEText ("Found directory switch: ", Arg);
         SparkMakeDebug.ReportTextEText ("Normalised to: ", TheDir);
      end if;

      if Directory_Operations.IsDirectory (TheDir) then
         Success := True;
         -- the current directory is used by default so don't add it again
         if not EStrings.EqString (TheDir, Directory_Operations.CurrentDirectory) then
            StringList.AddToBack (ToList  => TheDirectories,
                                  TheItem => TheDir);
         end if;
      else
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                 EStr1 => TheDir,
                                 EStr2 => Switch,
                                 EStr3 => EStrings.EmptyString);
      end if;
   end ProcessDirectorySwitch;

   ------------------------------------------------------------------------------

   procedure ProcessIndexSwitch (Switch  : in     EStrings.T;
                                 Arg     : in     EStrings.T;
                                 Success :    out Boolean)
   --# global in     Debug;
   --#        in out IndexFile;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives IndexFile         from *,
   --#                                Arg &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                IndexFile,
   --#                                Switch &
   --#         Success           from IndexFile &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found index switch: ", Arg);
      end if;

      if EStrings.IsEmpty (IndexFile) then
         Success := True;
         IndexFile := Arg;
         FileSystem.CheckExtension
           (IndexFile,
            FileSystem.CaseOfFileExtensions
              (CommandLineData.DEFAULT_INDEX_EXTENSION));

      else
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1 => Switch,
                                 EStr2 => EStrings.EmptyString,
                                 EStr3 => EStrings.EmptyString);
      end if;
   end ProcessIndexSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessMetaSwitch (Switch  : in     EStrings.T;
                                Arg     : in     EStrings.T;
                                Success :    out Boolean)
   --# global in     Debug;
   --#        in out MetaFile;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives MetaFile          from *,
   --#                                Arg &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                MetaFile,
   --#                                Switch &
   --#         Success           from MetaFile &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found meta switch: ", Arg);
      end if;

      if EStrings.IsEmpty (MetaFile) then
         Success := True;
         MetaFile := Arg;
         FileSystem.CheckExtension
           (MetaFile,
            FileSystem.CaseOfFileExtensions
              (CommandLineData.MetaFileExtension));
      else
         -- duplicate meta file switch
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1 => Switch,
                                 EStr2 => EStrings.EmptyString,
                                 EStr3 => EStrings.EmptyString);
      end if;
   end ProcessMetaSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessPathSwitch (Switch  : in     EStrings.T;
                                Arg     : in     EStrings.T;
                                Success :    out Boolean)
   --# global in     Debug;
   --#        in out Path;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives Path,
   --#         Success           from Arg,
   --#                                Path &
   --#         SPARK_IO.FILE_SYS from *,
   --#                                Arg,
   --#                                Path,
   --#                                Switch &
   --#         null              from Debug;
   is
   begin
      if Debug then
         SparkMakeDebug.ReportTextEText ("Found path switch: ", Arg);
      end if;

      if Path = Undefined then

         if EStrings.Eq1String (Arg, "full") or
           EStrings.Eq1String (Arg, "ful") or
           EStrings.Eq1String (Arg, "fu") or
           EStrings.Eq1String (Arg, "f") then

            Success := True;
            Path := Full;

         elsif EStrings.Eq1String (Arg, "relative") or
           EStrings.Eq1String (Arg, "relativ") or
           EStrings.Eq1String (Arg, "relati") or
           EStrings.Eq1String (Arg, "relat") or
           EStrings.Eq1String (Arg, "rela") or
           EStrings.Eq1String (Arg, "rel") or
           EStrings.Eq1String (Arg, "re") or
           EStrings.Eq1String (Arg, "r")  then

            Success := True;
            Path := Relative;

         else

            Success := False;
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                    EStr1 => Arg,
                                    EStr2 => Switch,
                                    EStr3 => EStrings.EmptyString);
         end if;
      else
         -- duplicate path switch
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1 => Switch,
                                 EStr2 => EStrings.EmptyString,
                                 EStr3 => EStrings.EmptyString);
      end if;
   end ProcessPathSwitch;

   ------------------------------------------------------------------------------

   procedure ProcessAnnoChar (Switch  : in     EStrings.T;
                              Arg     : in     EStrings.T;
                              Success :    out Boolean)
   --# global in     Debug;
   --#        in out AnnoCharSpecified;
   --#        in out CommandLineData.Content;
   --#        in out SPARK_IO.FILE_SYS;
   --# derives AnnoCharSpecified,
   --#         Success                 from AnnoCharSpecified,
   --#                                      Arg &
   --#         CommandLineData.Content from *,
   --#                                      AnnoCharSpecified,
   --#                                      Arg &
   --#         SPARK_IO.FILE_SYS       from *,
   --#                                      AnnoCharSpecified,
   --#                                      Arg,
   --#                                      Switch &
   --#         null                    from Debug;
   is
   begin

      if Debug then
         SparkMakeDebug.ReportTextEText ("Found annotation character argument: ", Arg);
      end if;

      if not AnnoCharSpecified  then

         if Arg.Length = 1 then

            Success := True;
            AnnoCharSpecified := True;
            -- set AnnoChar in the Examiner
            CommandLineData.Content.AnnoChar := Arg.Content (1);  -- expect warning here

         else

            Success := False;
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidArgument,
                                    EStr1 => Arg,
                                    EStr2 => Switch,
                                    EStr3 => EStrings.EmptyString);
         end if;
      else
         -- duplicate annochar switch
         Success := False;
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.DuplicateSwitch,
                                 EStr1 => Switch,
                                 EStr2 => EStrings.EmptyString,
                                 EStr3 => EStrings.EmptyString);
      end if;
   end ProcessAnnoChar;

   ------------------------------------------------------------------------------

   procedure Process (Success        : out Boolean;
                      HelpOrVerFound : out Boolean)
   --# global in     CommandLine.State;
   --#        in out CommandLineData.Content;
   --#        in out SPARK_IO.FILE_SYS;
   --#           out AnnoCharSpecified;
   --#           out Debug;
   --#           out Duplicates;
   --#           out IndexFile;
   --#           out MetaFile;
   --#           out NoIndex;
   --#           out NoMeta;
   --#           out Path;
   --#           out RootFile;
   --#           out TheDirectories;
   --#           out TheExcRegExps;
   --#           out TheIncRegExps;
   --# derives AnnoCharSpecified,
   --#         Debug,
   --#         Duplicates,
   --#         HelpOrVerFound,
   --#         IndexFile,
   --#         MetaFile,
   --#         NoIndex,
   --#         NoMeta,
   --#         Path,
   --#         RootFile,
   --#         Success,
   --#         TheDirectories,
   --#         TheExcRegExps,
   --#         TheIncRegExps           from CommandLine.State &
   --#         CommandLineData.Content,
   --#         SPARK_IO.FILE_SYS       from *,
   --#                                      CommandLine.State;
   is
      Switch            : EStrings.T;
      Argument          : EStrings.T;
      TheRegExp         : EStrings.T;
      SwitchOrArgFound  : Boolean;
      Done              : Boolean;
      OptionOk          : Boolean;
   begin
      Path              := Undefined;
      RootFile          := EStrings.EmptyString;
      MetaFile          := EStrings.EmptyString;
      IndexFile         := EStrings.EmptyString;
      AnnoCharSpecified := False;
      Debug             := False;
      Duplicates        := False;
      TheIncRegExps     := StringList.NullObject;
      TheExcRegExps     := StringList.NullObject;
      Success           := True;
      Done              := False;
      HelpOrVerFound    := False;
      NoIndex           := False;
      NoMeta            := False;

      -- The current directory is always assumed
      TheDirectories := StringList.NullObject;
      StringList.AddToFront (ToList  => TheDirectories,
                             TheItem => Directory_Operations.CurrentDirectory);

      -- Setup the command line
      CommandLine.Setup;

      -- Read options
      while not Done and not HelpOrVerFound loop

         CommandLine.Read (Switch => Switch,
                           Argument => Argument,
                           Success => SwitchOrArgFound);

         if SwitchOrArgFound then

            if EStrings.IsEmpty (Switch) then

               -- ARGUMENT: root file
               ProcessRootFileArgument (Arg => Argument,
                                        Success => OptionOk);

            elsif EStrings.Eq1String (Switch, "help") or
              EStrings.Eq1String (Switch, "hel") or
              EStrings.Eq1String (Switch, "he") or
              EStrings.Eq1String (Switch, "h") then

               -- SWITCH: help
               ReportUsage;
               HelpOrVerFound := True;
               OptionOk := True;

            elsif EStrings.Eq1String (Switch, "version") or
              EStrings.Eq1String (Switch, "versio") or
              EStrings.Eq1String (Switch, "versi") or
              EStrings.Eq1String (Switch, "vers") or
              EStrings.Eq1String (Switch, "ver") or
              EStrings.Eq1String (Switch, "ve") or
              EStrings.Eq1String (Switch, "v") then

               -- SWITCH: version
               ReportVersion;
               HelpOrVerFound := True;
               OptionOk := True;

            elsif EStrings.Eq1String (Switch, "path") or
              EStrings.Eq1String (Switch, "pat") or
              EStrings.Eq1String (Switch, "pa") or
              EStrings.Eq1String (Switch, "p") then

               -- SWITCH: path
               ProcessPathSwitch (Switch => Switch,
                                  Arg => Argument,
                                  Success => OptionOk);

            elsif EStrings.Eq1String (Switch, "directory") or
              EStrings.Eq1String (Switch, "director") or
              EStrings.Eq1String (Switch, "directo") or
              EStrings.Eq1String (Switch, "direct") or
              EStrings.Eq1String (Switch, "direc") or
              EStrings.Eq1String (Switch, "dire") or
              EStrings.Eq1String (Switch, "dir") or
              EStrings.Eq1String (Switch, "di") then

               -- SWITCH: directory
               ProcessDirectorySwitch (Switch => Switch,
                                       Arg => Argument,
                                       Success => OptionOk);

            elsif EStrings.Eq1String (Switch, "include") or
              EStrings.Eq1String (Switch, "includ") or
              EStrings.Eq1String (Switch, "inclu") or
              EStrings.Eq1String (Switch, "incl") or
              EStrings.Eq1String (Switch, "inc") then

               -- SWITCH: include
               ProcessIncludeSwitch (Switch => Switch,
                                     Arg => Argument,
                                     Success => OptionOk);

            elsif EStrings.Eq1String (Switch, "exclude") or
              EStrings.Eq1String (Switch, "exclud") or
              EStrings.Eq1String (Switch, "exclu") or
              EStrings.Eq1String (Switch, "excl") or
              EStrings.Eq1String (Switch, "exc") or
              EStrings.Eq1String (Switch, "ex") or
              EStrings.Eq1String (Switch, "e") then

               -- SWITCH: exclude
               ProcessExcludeSwitch (Switch => Switch,
                                     Arg => Argument,
                                     Success => OptionOk);

            elsif EStrings.Eq1String (Switch, "meta") or
              EStrings.Eq1String (Switch, "met") or
              EStrings.Eq1String (Switch, "me") or
              EStrings.Eq1String (Switch, "m") then

               -- SWITCH: meta
               ProcessMetaSwitch (Switch => Switch,
                                  Arg => Argument,
                                  Success => OptionOk);

            elsif EStrings.Eq1String (Switch, "index") or
              EStrings.Eq1String (Switch, "inde") or
              EStrings.Eq1String (Switch, "ind") then

               -- SWITCH: index
               ProcessIndexSwitch (Switch => Switch,
                                   Arg => Argument,
                                   Success => OptionOk);

            elsif EStrings.Eq1String (Switch, "noindexfile") or
                  EStrings.Eq1String (Switch, "noindexfil") or
                  EStrings.Eq1String (Switch, "noindexfi") or
                  EStrings.Eq1String (Switch, "noindexf") or
                  EStrings.Eq1String (Switch, "noindex") or
                  EStrings.Eq1String (Switch, "noinde") or
                  EStrings.Eq1String (Switch, "noind") or
                  EStrings.Eq1String (Switch, "noin") or
                  EStrings.Eq1String (Switch, "noi") then

               -- SWITCH: noindexfile
               NoIndex := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found noindexfile switch");

            elsif EStrings.Eq1String (Switch, "nometafile") or
                  EStrings.Eq1String (Switch, "nometafil") or
                  EStrings.Eq1String (Switch, "nometafi") or
                  EStrings.Eq1String (Switch, "nometaf") or
                  EStrings.Eq1String (Switch, "nometa") or
                  EStrings.Eq1String (Switch, "nomet") or
                  EStrings.Eq1String (Switch, "nome") or
                  EStrings.Eq1String (Switch, "nom") then

               -- SWITCH: nometafile
               NoMeta := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found nometafile switch");

            elsif EStrings.Eq1String (Switch, "debug") then
               -- SWITCH: debug
               Debug := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found debug switch");

            elsif EStrings.Eq1String (Switch, "duplicates_are_errors") or
                  EStrings.Eq1String (Switch, "duplicates_are_error") or
                  EStrings.Eq1String (Switch, "duplicates_are_erro") or
                  EStrings.Eq1String (Switch, "duplicates_are_err") or
                  EStrings.Eq1String (Switch, "duplicates_are_er") or
                  EStrings.Eq1String (Switch, "duplicates_are_e") or
                  EStrings.Eq1String (Switch, "duplicates_are_") or
                  EStrings.Eq1String (Switch, "duplicates_are") or
                  EStrings.Eq1String (Switch, "duplicates_ar") or
                  EStrings.Eq1String (Switch, "duplicates_a") or
                  EStrings.Eq1String (Switch, "duplicates_") or
                  EStrings.Eq1String (Switch, "duplicates") or
                  EStrings.Eq1String (Switch, "duplicate") or
                  EStrings.Eq1String (Switch, "duplicat") or
                  EStrings.Eq1String (Switch, "duplica") or
                  EStrings.Eq1String (Switch, "duplic") or
                  EStrings.Eq1String (Switch, "dupli") or
                  EStrings.Eq1String (Switch, "dupl") or
                  EStrings.Eq1String (Switch, "dup") or
                  EStrings.Eq1String (Switch, "du") then

               -- SWITCH: duplicates
               Duplicates := True;
               OptionOk := True;
               SparkMakeDebug.ReportText ("Found duplicates switch");

            elsif EStrings.Eq1String (Switch, "annotation_character") or
                  EStrings.Eq1String (Switch, "annotation_characte") or
                  EStrings.Eq1String (Switch, "annotation_charact") or
                  EStrings.Eq1String (Switch, "annotation_charac") or
                  EStrings.Eq1String (Switch, "annotation_chara") or
                  EStrings.Eq1String (Switch, "annotation_char") or
                  EStrings.Eq1String (Switch, "annotation_cha") or
                  EStrings.Eq1String (Switch, "annotation_ch") or
                  EStrings.Eq1String (Switch, "annotation_c") or
                  EStrings.Eq1String (Switch, "annotation_") or
                  EStrings.Eq1String (Switch, "annotation") or
                  EStrings.Eq1String (Switch, "annotatio") or
                  EStrings.Eq1String (Switch, "annotati") or
                  EStrings.Eq1String (Switch, "annotat") or
                  EStrings.Eq1String (Switch, "annota") or
                  EStrings.Eq1String (Switch, "annot") or
                  EStrings.Eq1String (Switch, "anno") or
                  EStrings.Eq1String (Switch, "ann") or
                  EStrings.Eq1String (Switch, "an") or
                  EStrings.Eq1String (Switch, "a") then

               ProcessAnnoChar (Switch => Switch,
                                Arg => Argument,
                                Success => OptionOk);


            else
               -- unrecognised switch
               OptionOk := False;
               SparkMakeErrors.Report (TheFault => SparkMakeErrors.InvalidSwitch,
                                       EStr1 => Switch,
                                       EStr2 => EStrings.EmptyString,
                                       EStr3 => EStrings.EmptyString);
            end if;

            Success := Success and OptionOk;

         else
            -- nothing more on the command line.
            Done := True;
         end if;
      end loop;

      -- If usage or version info requested then don't continue with any other processing
      if Success and not HelpOrVerFound then

         -- The command line has parsed OK

         -- Set non specified switches to default values.

         if Path = Undefined then
            Path := Full;
         end if;

         if EStrings.IsEmpty (IndexFile) then
            -- Index file not specified so index file is <rootfile>.idx
            if EStrings.IsEmpty (RootFile) then
               EStrings.CopyString (IndexFile, "spark");
            else
               IndexFile := RootFile;
            end if;
            Directory_Operations.SetExtension
              (Path => IndexFile,
               Ext => FileSystem.CaseOfFileExtensions
                 (CommandLineData.DEFAULT_INDEX_EXTENSION));
            if Debug then
               SparkMakeDebug.ReportTextEText ("Using default index file: ", IndexFile);
            end if;
         end if;

         if EStrings.IsEmpty (MetaFile) then
            -- Meta file not specified so meta file is <rootfile>.smf
            if EStrings.IsEmpty (RootFile) then
               EStrings.CopyString (MetaFile, "spark");
            else
               MetaFile := RootFile;
            end if;
            Directory_Operations.SetExtension
              (Path => MetaFile,
               Ext => FileSystem.CaseOfFileExtensions
                 (CommandLineData.MetaFileExtension));
            if Debug then
               SparkMakeDebug.ReportTextEText ("Using default meta file: ", MetaFile);
            end if;
         end if;

         if StringList.IsNull (StringList.GetFirst (TheIncRegExps)) then
            -- No include was specified so use the default (GNAT file naming convention).
            EStrings.CopyString (Argument, "*\.ad[bs]");
            StringList.AddToFront (ToList  => TheIncRegExps,
                                   TheItem => Argument);
            if Debug then
               SparkMakeDebug.ReportText ("No include switch. Will use the GNAT naming convention");
            end if;
         end if;

         if not EStrings.IsEmpty (RootFile) then
            -- Make sure the root file will be included.
            TheRegExp := RegExp (ForFile => RootFile);
            if EStrings.IsEmpty (TheRegExp) then
               SystemErrors.FatalError
                 (SysErr => SystemErrors.OtherInternalError,
                  Msg => "Cannot generate regular expression for root file.");
            else
               SparkMakeDebug.ReportTextEText ("Root file will be included by ", TheRegExp);
               StringList.AddToFront (ToList  => TheIncRegExps,
                                      TheItem => TheRegExp);
            end if;
         end if;

      end if;
   end Process;


   function PathRequired return PathType
   --# global in Path;
   is
   begin
      return Path;
   end PathRequired;

   function MetaFileName return EStrings.T
   --# global in MetaFile;
   is
   begin
      return MetaFile;
   end MetaFileName;

   function IndexFileName return EStrings.T
   --# global in IndexFile;
   is
   begin
      return IndexFile;
   end IndexFileName;

   function RootFileName return EStrings.T
   --# global in RootFile;
   is
   begin
      return RootFile;
   end RootFileName;

   function DuplicatesError return Boolean
   --# global in Duplicates;
   is
   begin
      return Duplicates;
   end DuplicatesError;

   function DebugOn return Boolean
   --# global in Debug;
   is
   begin
      return Debug;
   end DebugOn;

   function NoIndexFile return Boolean
   --# global in NoIndex;
   is
   begin
      return NoIndex;
   end NoIndexFile;

   function NoMetaFile return Boolean
   --# global in NoMeta;
   is
   begin
      return NoMeta;
   end NoMetaFile;

   function TheDirectoryNames return StringList.Object
   --# global in TheDirectories;
   is
   begin
      return TheDirectories;
   end TheDirectoryNames;

   function TheIncFileRegExps return StringList.Object
   --# global in TheIncRegExps;
   is
   begin
      return TheIncRegExps;
   end TheIncFileRegExps;

   function TheExcFileRegExps return StringList.Object
   --# global in TheExcRegExps;
   is
   begin
      return TheExcRegExps;
   end TheExcFileRegExps;

end SparkMakeCommandLine;
