-- $Id: sparkmake.adb 13045 2009-04-20 08:41:19Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--                                                                            --
--  Specification                                                             --
--                                                                            --
--  INPUT OPTIONS                                                             --
--  -------------                                                             --
--                                                                            --
--  Command Line                                                              --
--  1. Invoking SPARKMake with no arguments will output an error message      --
--  2. All switches can be cut down to their smallest unique value            --
--                                                                            --
--  /directory option                                                         --
--  1. The default is the current directory                                   --
--  2. Any other directories, specified by this option are also searched      --
--  3. This switch may be repeated and the union of the default and all       --
--     explicitly stated directories are searched                             --
--  4. An error is reported if the specified directory does not exist         --
--  5. An error will be reported if there is no argument                      --
--  6. A relative path may be used                                            --
--                                                                            --
--  /include option                                                           --
--  1. The default will find all *.ads and *.adb files (GNAT naming conv.)    --
--  2. A specified include option overrides the default                       --
--  3. This switch may be duplicated and the union will be taken              --
--  4. An error will be reported if there is no argument                      --
--  5. An error will be reported if the argument is not a regular expression  --
--  6. An error is output if the /inc option includes a file with an invalid  --
--     compilation unit.                                                      --
--                                                                            --
--  /exclude                                                                  --
--  1. The default excludes no files                                          --
--  2. When specified, files matching this expression are excluded            --
--  3. This switch may be duplicated and the union will be taken              --
--  4. An error will be reported if there is no argument                      --
--  5. An error will be reported if the argument is not a regular expression  --
--                                                                            --
--  Argument                                                                  --
--  1. The argument consitutes the root file                                  --
--  2. It may appear anywhere in the command line                             --
--  3. An error is reported if the root file cannot be found                  --
--  4. The root file is always included regardless of the /inc, /exc switches --
--  5. If no argument is provided then sparkmake will produce an index and    --
--     meta file for the analysis of all files in the current directory (and  --
--     any subdirectories).                                                   --
--                                                                            --
--  OUTPUT OPTIONS                                                            --
--  --------------                                                            --
--                                                                            --
--  /index                                                                    --
--  1. The default is the root file name (or spark.idx if no root file given) --
--  2. If the root file name has an extension it is replaced with idx         --
--  3. If the root file has no extension then idx is used                     --
--  4. When specified the argument constitues the index file                  --
--  5. If the file does not exist it is created                               --
--  6. If the file exists then it will be overwritten                         --
--  7. The index file contains entries for:                                   --
--    a. All files specified by the /include switch                           --
--    b. That are not excluded by the /exclude switch                         --
--    c. The root file                                                        --
--  8. The index file always specifies full pathnames                         --
--  9. The format of the index files matches that described in the Examiner   --
--       user manual.                                                         --
--  10. An error is output if the same unit is duplicated in more than one file--
--                                                                            --
--  /meta                                                                     --
--  1. The default is the root file name (or spark.smf if no root file given) --
--  2. If the root file name has an extension it is replaced with smf         --
--  3. If the root file has no extension then smf is used                     --
--  4. When specified the argument constitues the meta file                   --
--  5. If the file does not exist it is created                               --
--  6. If the file exists then it will be overwritten                         --
--  7. The meta file always specifies full path names                         --
--  8. The root of the make is the root file specified as an argument on the  --
--        command line                                                        --
--  9. The make process will attempt to make an entire closure for the root   --
--        file. i.e. it will make all the dependent units AND their bodies    --
--                   and separates.                                           --
--  10. All files involved in the make will have an entry in the index file   --
--  11. If a unit cannot be found it is ignored.                              --
--                                                                            --
--  /noindexfile                                                              --
--  1. Default is False.                                                      --
--  2. If True then the index file will not be generated.                     --
--                                                                            --
--  /nometafile                                                               --
--  1. Default is False.                                                      --
--  2. If True then the meta file will not be generated.                      --
--                                                                            --
--  /path                                                                     --
--  1. Default is "full"                                                      --
--  2. User can select "full" or "relative"                                   --
--                                                                            --
--  BEHAVIOUR OPTIONS                                                         --
--  -----------------                                                         --
--                                                                            --
--  /duplicates_are_errors                                                    --
--  1. Duplicate units are treated as an error                                --
--                                                                            --
--  /annotation_character                                                     --
--  1. The annotation character can be specified                              --
--  2. If it is not a character it will not be accepted                       --
--  3. If not specified '#' will be used                                      --
--------------------------------------------------------------------------------

with Ada.Exceptions;
with GNAT.Traceback.Symbolic;
with CommandLineData;
with EStrings;
with ELStrings;
with ScreenEcho;
with SPARK_IO;
with ErrorHandler;
with LexTokenManager;
with Dictionary;
with Unit;
with Units;
with UnitManager;
with SparkLex;
with SparkMakeCommandLine;
with SparkMakeErrors;
with SparkMakeDebug;
with Directory_Operations;

use type Unit.Kind;
use type Unit.Object;
use type Unit.Id;
use type SPARK_IO.File_Status;
use type SPARK_IO.File_Type;
use type SparkMakeCommandLine.PathType;

--# inherit CommandLine,
--#         CommandLineData,
--#         Dictionary,
--#         ELStrings,
--#         ErrorHandler,
--#         EStrings,
--#         LexTokenManager,
--#         Directory_Operations,
--#         SparkLex,
--#         SparkMakeCommandLine,
--#         SparkMakeDebug,
--#         SparkMakeErrors,
--#         SPARK_IO,
--#         Unit,
--#         UnitManager,
--#         Units;

--# main_program

procedure Sparkmake
--# global in     CommandLine.State;
--#        in out ErrorHandler.ErrorContext;
--#        in out LexTokenManager.StringTable;
--#        in out SPARK_IO.File_Sys;
--#        in out UnitManager.State;
--#           out CommandLineData.Content;
--#           out Dictionary.Dict;
--#           out SparkLex.CurrLine;
--#           out SparkMakeCommandLine.State;
--# derives CommandLineData.Content,
--#         SparkMakeCommandLine.State  from CommandLine.State &
--#         Dictionary.Dict             from LexTokenManager.StringTable,
--#                                          SPARK_IO.File_Sys &
--#         ErrorHandler.ErrorContext,
--#         LexTokenManager.StringTable,
--#         SparkLex.CurrLine,
--#         SPARK_IO.File_Sys,
--#         UnitManager.State           from CommandLine.State,
--#                                          ErrorHandler.ErrorContext,
--#                                          LexTokenManager.StringTable,
--#                                          SPARK_IO.File_Sys,
--#                                          UnitManager.State;
is

   -- The entry type signifiers must all be the same length
   Main_Program_Sig  : constant String := "main_program  is in ";
   Specification_Sig : constant String := "specification is in ";
   Body_Sig          : constant String := "body          is in ";
   Subunit_Sig       : constant String := "subunit       is in ";
   Component_Sig     : constant String := "components    are   ";
   -- This must equal the length of the above signifiers
   Sig_Length        : constant EStrings.Lengths := 20;

   -- The files output by this program
   TheMetaFile   : SPARK_IO.File_Type;

   Success        : Boolean;
   HelpOrVerFound : Boolean;

   --------------------------------------------------------------------------

   procedure OpenOrCreateFile (File    : in     EStrings.T;
                               Mode    : in     SPARK_IO.File_Mode;
                               Id      :    out SPARK_IO.File_Type;
                               Success :    out Boolean)
   --# global in out SPARK_IO.File_Sys;
   --# derives Id,
   --#         SPARK_IO.File_Sys,
   --#         Success           from File,
   --#                                Mode,
   --#                                SPARK_IO.File_Sys;
   is
      --# hide OpenOrCreateFile;

      Status : SPARK_IO.File_Status;

      function Str (EStr : EStrings.T) return String
      is
      begin
         return EStr.Content (1 .. EStr.Length);
      end Str;

   begin
      SPARK_IO.Open (File         => Id,
                     Mode_Of_File => Mode,
                     Name_Length  => File.Length,
                     Name_Of_File => Str (File),
                     Form_Of_File => "",
                     Status       => Status);

      if Status = SPARK_IO.Name_Error then
         SPARK_IO.Create (File         => Id,
                          Name_Length  => File.Length,
                          Name_Of_File => Str (File),
                          Form_Of_File => "",
                          Status       => Status);
      end if;
      Success := Status = SPARK_IO.Ok;
      if not Success then
         SparkMakeErrors.Report (TheFault => SparkMakeErrors.CannotOpenFile,
                                 EStr1 => File,
                                 EStr2 => EStrings.EmptyString,
                                 EStr3 => EStrings.EmptyString);
      end if;
   end OpenOrCreateFile;

   --------------------------------------------------------------------------

   procedure CloseFile (File   : in     EStrings.T;
                        FileId : in out SPARK_IO.File_Type)
   --# global in out SPARK_IO.File_Sys;
   --# derives FileId            from * &
   --#         SPARK_IO.File_Sys from *,
   --#                                File,
   --#                                FileId;
   is
      Status : SPARK_IO.File_Status;
   begin
      if FileId /= SPARK_IO.Null_File then
         SPARK_IO.Close (File   => FileId,
                         Status => Status);
         if Status /= SPARK_IO.Ok then
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.CannotCloseFile,
                                    EStr1 => File,
                                    EStr2 => EStrings.EmptyString,
                                    EStr3 => EStrings.EmptyString);
         end if;
      end if;
   end CloseFile;

   --------------------------------------------------------------------------

   procedure OutputDebug (TheFile : in EStrings.T;
                          Indent  : in Integer)
   --# derives null from Indent,
   --#                   TheFile;
   is
      EStr : EStrings.T;
   begin
      EStr := EStrings.EmptyString;
      for I in Integer range 1 .. Indent loop
         EStrings.AppendString (EStr, " ");
      end loop;
      EStrings.AppendExaminerString (EStr, TheFile);
      SparkMakeDebug.ReportTextEText
        (Text => "Make: ",
         EText => EStr);
   end OutputDebug;

   --------------------------------------------------------------------------

   function QuoteIfNeeded (F : in EStrings.T) return
     EStrings.T
   is
      Result : EStrings.T;

      function ContainsSpace return Boolean
      --# global in F;
      is
         Result : Boolean := False;
      begin
         for I in EStrings.Positions range 1 .. F.Length loop
            if F.Content (I) = ' ' then
               Result := True;
               exit;
            end if;
         end loop;
         return Result;
      end ContainsSpace;

   begin
      if ContainsSpace then
         Result := EStrings.EmptyString;
         EStrings.AppendString (Result, """");
         EStrings.AppendExaminerString (Result, F);
         EStrings.AppendString (Result, """");
      else
         Result := F;
      end if;
      return Result;
   end QuoteIfNeeded;

   --------------------------------------------------------------------------

   procedure Make (TheFile : in EStrings.T)
   --# global in     SparkMakeCommandLine.State;
   --#        in     TheMetaFile;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                SparkMakeCommandLine.State,
   --#                                TheFile,
   --#                                TheMetaFile;
   --#
   is

      --# hide Make;

      TheUnit : Unit.Id;
      Found   : Boolean;

      MainProgramFound : Boolean := False;
      MainProgramName  : EStrings.T := EStrings.EmptyString;

      -- Stores which units have been placed in the meta file.
      TheMadeStack  : Units.Stack;

      -- Root units for the make
      RootStack   : Units.Stack;
      TheRootUnit : Unit.Id;

      -- For debug purposes
      Indent        : Integer := 0;

      --------------------------------------------------------------------------

      -- Forward declaration.
      -- MakeUnit and MakeUnits are mutually recursive.

      procedure MakeUnits (TheUnits : in Units.Stack);

      --------------------------------------------------------------------------

      procedure MakeUnit (TheUnit : in Unit.Id)
      is
         Found    : Boolean;
         TheUnits : Units.Stack;
         TheFile  : EStrings.T;
      begin

         -- don't make something twice or we'll recurse forever!
         --
         if not Units.InStack (TheUnit  => TheUnit,
                               TheStack => TheMadeStack) then

            -- Check we know about this unit
            UnitManager.GetFile (ForUnit => TheUnit,
                                 TheFile => TheFile,
                                 Found   => Found);

            if Found then
               -- Record the fact we've made this unit.
               --
               Units.Push (TheMadeStack, TheUnit);

               if SparkMakeCommandLine.DebugOn then
                  OutputDebug (TheFile => TheFile,
                               Indent => Indent);
               end if;

               -- Get the required units
               --
               TheUnits := UnitManager.RequiredUnits (ForUnit => TheUnit);

               if not Units.IsEmpty (TheUnits) then
                  Indent := Indent + 3;

                  -- Make the required units first
                  --
                  MakeUnits (TheUnits => TheUnits);

                  Indent := Indent - 3;
               end if;

               -- Write this filename for this unit to the meta file.
               --
               if SparkMakeCommandLine.PathRequired = SparkMakeCommandLine.Full then
                  EStrings.PutLine (TheMetaFile, QuoteIfNeeded (TheFile));
               else
                  EStrings.PutLine
                    (TheMetaFile,
                     QuoteIfNeeded
                       (Directory_Operations.RelativeName
                          (OfThisFileOrDir => TheFile,
                           ToThisDir => Directory_Operations.CurrentDirectory)));
               end if;

               -- If we find multiple main programs in the make then warn the user
               -- because the resulting meta file will not Examiner cleanly.
               if TheUnit.TheKind = Unit.MainProgramUnit then
                  if MainProgramFound then -- Don't warn for the first one we find!
                     SparkMakeErrors.Report (TheFault => SparkMakeErrors.MultipleMainPrograms,
                                             EStr1 => MainProgramName,
                                             EStr2 => TheFile,
                                             EStr3 => EStrings.EmptyString);
                  else
                     MainProgramName := TheFile; -- only update this the first time round
                  end if;
                  MainProgramFound := True;
               end if;

               if TheUnit.TheKind in Unit.SpecificationUnit then
                  -- This was a spec so make the body
                  MakeUnit (TheUnit => UnitManager.PackageBody (ForUnit => TheUnit));
               else
                  -- This was a body so make any separates
                  MakeUnits (TheUnits => UnitManager.SeparateUnits (ForUnit => TheUnit));
               end if;
            end if;
         end if;
      end MakeUnit;

      --------------------------------------------------------------------------

      procedure MakeUnits (TheUnits : in Units.Stack)
      is
         TheUnit    : Unit.Id;
         LocalUnits : Units.Stack;
      begin
         LocalUnits := TheUnits;
         while not Units.IsEmpty (LocalUnits) loop

            Units.Pop (TheStack => LocalUnits,
                       TheUnit => TheUnit);

            MakeUnit (TheUnit => TheUnit);

         end loop;
      end MakeUnits;

      --------------------------------------------------------------------------

      procedure WarnOfUnitsNotInMake
      --# global TheMadeStack;
      is
         AllUnits : Units.Stack;
         TheUnit  : Unit.Id;
         Found    : Boolean;
         TheFile  : EStrings.T;

         PrintedWarningHeader : Boolean := False;

      begin

         AllUnits := UnitManager.GetAllUnits;

         -- If any unit is not in the made stack then it wasn't written to the
         -- meta file. Warn the user that this is the case.

         while not Units.IsEmpty (AllUnits) loop

            Units.Pop (TheStack => AllUnits,
                       TheUnit  => TheUnit);

            if not Units.InStack (TheUnit  => TheUnit,
                                  TheStack => TheMadeStack) then

               -- Check we know about this unit
               UnitManager.GetFile (ForUnit => TheUnit,
                                    TheFile => TheFile,
                                    Found   => Found);

               if Found then

                  if not PrintedWarningHeader then
                     SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                                          "The following units were not included in the meta file:",
                                          0);
                     PrintedWarningHeader := True;
                  end if;

                  EStrings.PutLine (SPARK_IO.Standard_Output,
                                      TheFile);
               end if;

            end if;

         end loop;

      end WarnOfUnitsNotInMake;

      --------------------------------------------------------------------------

   begin

      -- If no root filename specified then do it for the whole directory
      if EStrings.IsEmpty (TheFile) then

         RootStack := UnitManager.FindRoots; -- Get all the root units for this make

         -- Nothing made so far.
         TheMadeStack := Units.NullStack;

         -- Iterate through all the roots, making the metafile entries
         -- for each one in turn.
         while not Units.IsEmpty (RootStack) loop
            Units.Pop (TheStack => RootStack,
                       TheUnit  => TheRootUnit);

            -- kick off the recursive make for this root.
            MakeUnit (TheUnit => TheRootUnit);

         end loop;

      else
         -- A filename was supplied so use that as the root for the make

         -- Nothing made so far.
         TheMadeStack := Units.NullStack;

         -- Get the unit in this file
         UnitManager.GetUnit (InFile  => TheFile,
                              TheUnit => TheUnit,
                              Found   => Found);

         if Found then
            -- kick off the recursive make
            MakeUnit (TheUnit => TheUnit);
         else
            SparkMakeErrors.Report (TheFault => SparkMakeErrors.CannotFindFile,
                                    EStr1 => TheFile,
                                    EStr2 => EStrings.EmptyString,
                                    EStr3 => EStrings.EmptyString);
         end if;
      end if;

      -- The user may be expecting *all* files to be included in the make
      -- so warn about any that were not put in the meta file (ie those
      -- that are present in AllUnits but not in TheMadeStack).
      WarnOfUnitsNotInMake;

   end Make;

   --------------------------------------------------------------------------

   function MaxUnitNameLength (TheStack : in Units.Stack)
                               return EStrings.Lengths
   is
      Iterator : Units.Iterator;
      NextUnit : Unit.Id;
      MaxId    : EStrings.Lengths;
   begin
      Units.Init_Iterator (TheStack, Iterator);
      MaxId := 0;

      while not Units.Iterated (Iterator) loop
         Units.Iterate (Iterator, NextUnit);
         if NextUnit.TheName.Length > MaxId then
            MaxId := NextUnit.TheName.Length;
         end if;
      end loop;

      return MaxId;
   end MaxUnitNameLength;

   --------------------------------------------------------------------------

   procedure TabToPosition (ELStr       : in out ELStrings.T;
                            TabPosition : in     ELStrings.Lengths)
   --# derives ELStr from *,
   --#                    TabPosition;
   is
   begin
      for I in ELStrings.Lengths range ELStr.Length .. TabPosition
      loop
         ELStrings.AppendString (ELStr, " ");
      end loop;
   end TabToPosition;

   --------------------------------------------------------------------------

   function Line (ForUnit : in Unit.Object;
                  TabPosition : EStrings.Lengths)
                  return ELStrings.T
   --# global in SparkMakeCommandLine.State;
   --
   -- Returns <unit name> <specification|body> is in <file name>
   is
      Temp         : ELStrings.T;
   begin
      Temp := ELStrings.EmptyString;
      ELStrings.AppendExaminerString
        (ELStr => Temp,
         EStr  => ForUnit.TheId.TheName);

      TabToPosition (ELStr => Temp, TabPosition => TabPosition);

      case ForUnit.TheId.TheKind is

         when Unit.MainProgramUnit   =>
            ELStrings.AppendString
              (EStr => Temp,
               Str  => Main_Program_Sig);

         when Unit.SpecificationUnit =>
            ELStrings.AppendString
              (EStr => Temp,
               Str  => Specification_Sig);

         when Unit.PackageBodyUnit   =>
            ELStrings.AppendString
              (EStr => Temp,
               Str  => Body_Sig);

         when Unit.SeparateBodyUnit   =>
            ELStrings.AppendString
              (EStr => Temp,
               Str  => Subunit_Sig);

      end case;

      if SparkMakeCommandLine.PathRequired = SparkMakeCommandLine.Full then
         ELStrings.AppendExaminerString
           (ELStr => Temp,
            EStr  => QuoteIfNeeded (ForUnit.TheFile));
      else
         ELStrings.AppendExaminerString
           (ELStr => Temp,
            EStr  => QuoteIfNeeded
              (Directory_Operations.RelativeName (OfThisFileOrDir => ForUnit.TheFile,
                                                  ToThisDir => Directory_Operations.CurrentDirectory)));
      end if;

      return Temp;
   end Line;

   --------------------------------------------------------------------------

   procedure BuildIndexFile (TheName : in     EStrings.T;
                             Success :    out Boolean)
   --# global in     SparkMakeCommandLine.State;
   --#        in     UnitManager.State;
   --#        in out SPARK_IO.File_Sys;
   --# derives SPARK_IO.File_Sys from *,
   --#                                SparkMakeCommandLine.State,
   --#                                TheName,
   --#                                UnitManager.State &
   --#         Success           from SPARK_IO.File_Sys,
   --#                                TheName;

   is
      CurrentComponent : Unit.Id;
      TheComponents    : Units.Stack;
      AllUnits         : Units.Stack;
      CurrentUnit      : Unit.Object;
      Id               : Unit.Id;
      FirstComponent   : Boolean;
      TheIndexFile     : SPARK_IO.File_Type;
      TabPosition      : ELStrings.Lengths;
      ComponentLine    : ELStrings.T;
   begin
      SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                           "Building index file ",
                           0);

      EStrings.PutLine (SPARK_IO.Standard_Output,
                               TheName);

      OpenOrCreateFile (File    => TheName,
                        Mode    => SPARK_IO.Out_File,
                        Id      => TheIndexFile,
                        Success => Success);

      if Success then

         AllUnits := UnitManager.GetAllUnits;
         Units.Sort (AllUnits);

         TabPosition := MaxUnitNameLength (AllUnits);

         while not Units.IsEmpty (AllUnits) loop

            Units.Pop (TheStack => AllUnits,
                       TheUnit  => Id);

            CurrentUnit := UnitManager.Get (TheUnit => Id);

            Unit.OutputObject (CurrentUnit);

            -- Write the file -> unit mapping to the index file
            ELStrings.PutLine
              (File => TheIndexFile,
               EStr => Line (ForUnit => CurrentUnit,
                             TabPosition => TabPosition));

            -- if this is a specification
            -- Write any component information to the index file
            if CurrentUnit.TheId.TheKind in Unit.SpecificationUnit then
               TheComponents := UnitManager.Components (ForUnit => CurrentUnit.TheId);

               FirstComponent := True;
               ComponentLine := ELStrings.ToExaminerLongString
                 (CurrentUnit.TheId.TheName);

               while not Units.IsEmpty (TheComponents) loop

                  Units.Pop (TheStack => TheComponents,
                             TheUnit  => CurrentComponent);

                  if FirstComponent then
                     -- Create the <Unit_Name> components are
                     FirstComponent := False;
                     TabToPosition (ELStr       => ComponentLine,
                                    TabPosition => TabPosition);
                     ELStrings.AppendString
                       (EStr => ComponentLine,
                        Str  => Component_Sig);
                  else
                     -- Add comma to continue list on new line
                     ELStrings.AppendString
                       (EStr => ComponentLine,
                        Str  => ",");
                     ELStrings.PutLine
                       (File => TheIndexFile,
                        EStr => ComponentLine);

                     -- Align component names
                     ComponentLine := ELStrings.EmptyString;
                     TabToPosition (ELStr       => ComponentLine,
                                    TabPosition => TabPosition + Sig_Length);
                  end if;

                  -- Append a component name
                  ELStrings.AppendExaminerString
                    (ELStr => ComponentLine,
                     EStr  => CurrentComponent.TheName);

               end loop;
               if not FirstComponent then
                  ELStrings.PutLine
                    (File => TheIndexFile,
                     EStr => ComponentLine);
               end if;
            end if;
         end loop;
      end if;
      --# accept Flow, 10, TheIndexFile, "Ineffective assignment here OK";
      CloseFile (TheName,
                 TheIndexFile);
   end BuildIndexFile;

   ----------------------------------------------------------------------------------

   procedure BuildMetaFile (TheName : in EStrings.T;
                            FromRoot : in EStrings.T)
   --# global in     SparkMakeCommandLine.State;
   --#        in out SPARK_IO.File_Sys;
   --#           out TheMetaFile;
   --# derives SPARK_IO.File_Sys from *,
   --#                                FromRoot,
   --#                                SparkMakeCommandLine.State,
   --#                                TheName &
   --#         TheMetaFile       from SPARK_IO.File_Sys,
   --#                                TheName;
   is
      Success : Boolean;
   begin
      SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                           "Building meta file ",
                           0);

      EStrings.PutLine (SPARK_IO.Standard_Output,
                               TheName);

      OpenOrCreateFile (File    => TheName,
                        Mode    => SPARK_IO.Out_File,
                        Id      => TheMetaFile,
                        Success => Success);

      if Success then
         Make (FromRoot);
         CloseFile (TheName, TheMetaFile);
      end if;
   end BuildMetaFile;

begin -- Set Up
   CommandLineData.Initialize;

   LexTokenManager.InitialiseStringTable;
   Dictionary.Initialize (False);
   ErrorHandler.SparkMakeInit;
   SparkLex.ClearLineContext;

   -- Read the command line
   SparkMakeCommandLine.Process (Success, HelpOrVerFound);

   if Success and not HelpOrVerFound then
      UnitManager.Initialise
        (TheDirectories => SparkMakeCommandLine.TheDirectoryNames,
         Include => SparkMakeCommandLine.TheIncFileRegExps,
         Exclude => SparkMakeCommandLine.TheExcFileRegExps,
         RootFile => SparkMakeCommandLine.RootFileName,
         Duplicates => SparkMakeCommandLine.DuplicatesError,
         Success => Success);

      if Success then
         if not SparkMakeCommandLine.NoIndexFile then
            BuildIndexFile (TheName => SparkMakeCommandLine.IndexFileName,
                            Success => Success);
         end if;

         if Success then
            if not SparkMakeCommandLine.NoMetaFile then
               --# accept Flow, 10, TheMetaFile, "Ineffective assignment here OK";
               BuildMetaFile (TheName => SparkMakeCommandLine.MetaFileName,
                              FromRoot => SparkMakeCommandLine.RootFileName);
               --# end accept;
            end if;
         end if;
      end if;
   end if;
   --# accept Flow, 33, TheMetaFile, "Metafile not referenced here";
exception
   --# hide Sparkmake;
   when E : others =>
      ScreenEcho.New_Line (1);
      ScreenEcho.Put_Line ("Unhandled Exception in SPARKMake");
      ScreenEcho.Put_Line ("Exception information:");
      ScreenEcho.Put_Line (Ada.Exceptions.Exception_Information (E));
      ScreenEcho.Put_Line ("Traceback:");
      ScreenEcho.Put_Line (GNAT.Traceback.Symbolic.Symbolic_Traceback (E));
end Sparkmake;
