-- $Id: directory_operations.adb 13046 2009-04-20 08:52:00Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with GNAT.Directory_Operations;
with GNAT.OS_Lib;
with StringUtilities;
with SystemErrors;

package body Directory_Operations
is

   function Str (E : EStrings.T) return String
   is
   begin
      return E.Content (1 .. E.Length);
   end Str;

   --------------------------------------------------------------------------------

   function Prefix (EStr             : EStrings.T;
                    Separator        : Character;
                    IncludeSeparator : Boolean) return
     EStrings.T
   -- Returns the text up to the last occurrance of the separator.
   -- If IncludeSeparator is true then the string returned ends in the separator
   -- otherwise it does not.
   is
      Result : EStrings.T := EStrings.EmptyString;
      Last   : Integer := 0;
   begin
      for I in reverse Integer range 1 .. EStr.Length loop
         if I /= EStr.Length and then
           EStr.Content (I) = Separator then
            Last := I;
            exit;
         end if;
      end loop;

      if Last /= 0 then
         if IncludeSeparator then
            Result := StringUtilities.Section (EStr     => EStr,
                                               StartPos => 1,
                                               Length   => Last);
         else
            Result := StringUtilities.Section (EStr     => EStr,
                                               StartPos => 1,
                                               Length   => Last - 1);
         end if;
      end if;
      return Result;
   end Prefix;

   --------------------------------------------------------------------------------

   function DirectorySeparator return Character
   is
   begin
      return GNAT.OS_Lib.Directory_Separator;
   end DirectorySeparator;

   --------------------------------------------------------------------------------

   function CurrentDirectory return EStrings.T
   is
      Result : EStrings.T;
   begin
      EStrings.CopyString (EStr => Result,
                                  Str => GNAT.Directory_Operations.Get_Current_Dir);
      return Result;
   end CurrentDirectory;

   --------------------------------------------------------------------------------

   function FindFiles (Matching    : RegularExpression.Object;
                       InDirectory : EStrings.T;
                       Recursively : Boolean) return StringList.Object
   is

      TheResult : StringList.Object;

      procedure Scan_Directory (Dir : in GNAT.Directory_Operations.Dir_Name_Str)
      is
         D    : GNAT.Directory_Operations.Dir_Type;
         Str  : String (1 .. 1024);
         Last : Natural;
      begin
         GNAT.Directory_Operations.Open (D, Dir);
         loop
            GNAT.Directory_Operations.Read (D, Str, Last);
            exit when Last = 0;

            declare
               F : constant String := Dir & Str (1 .. Last);
               EStr : EStrings.T;
            begin
               if GNAT.OS_Lib.Is_Directory (F) then

                  --  Ignore "." and ".."
                  if ((Last = 1) and then (Str (1) = '.')) or
                    ((Last = 2) and then (Str (1) = '.' and
                                            Str (2) = '.')) then
                     null;
                  elsif Recursively then
                     --  Recurse here
                     Scan_Directory (F & GNAT.OS_Lib.Directory_Separator);
                  end if;
               else
                  -- Does this file match the regular expression?
                  EStrings.CopyString (EStr => EStr,
                                       Str  => F);
                  if RegularExpression.Matches (EStr      => EStr,
                                                TheRegExp => Matching) then
                     StringList.AddInLexOrder (ToList => TheResult,
                                               TheItem => EStr);
                  end if;
               end if;
            end;

         end loop;
         GNAT.Directory_Operations.Close (D);
      exception
         when others =>
            GNAT.Directory_Operations.Close (D);
            raise;
      end Scan_Directory;

   begin
      TheResult := StringList.NullObject;
      Scan_Directory (Str (InDirectory));
      return TheResult;
   end FindFiles;

   --------------------------------------------------------------------------------

   function FileExtension (Path : EStrings.T) return
     EStrings.T
   is
      Result : EStrings.T;
   begin
      EStrings.CopyString
        (EStr => Result,
         Str => GNAT.Directory_Operations.File_Extension
           (Path => Str (Path)));
      return Result;
   end FileExtension;

   --------------------------------------------------------------------------------

   procedure SetExtension (Path : in out EStrings.T;
                           Ext  : in     EStrings.T)
   is
   begin
      if not EStrings.IsEmpty (FileExtension (Path)) then
         -- Has an extension so remove it
         Path := Prefix (Path, '.', False);
      end if;
      -- Add the given extension.
      EStrings.AppendString (EStr => Path,
                                    Str => ".");
      EStrings.AppendExaminerString (EStr1 => Path,
                                            EStr2 => Ext);
   end SetExtension;

   --------------------------------------------------------------------------------

   function FileName (Path : EStrings.T) return EStrings.T
   is
      Result : EStrings.T;
   begin
      EStrings.CopyString
        (EStr => Result,
         Str => GNAT.Directory_Operations.File_Name
           (Path => Str (Path)));
      return Result;
   end FileName;

   --------------------------------------------------------------------------------

   function IsDirectory (Path : EStrings.T) return Boolean
   is
   begin
      return GNAT.OS_Lib.Is_Directory (Name => Str (Path));
   end IsDirectory;

   --------------------------------------------------------------------------------

   function IsFile (Path : EStrings.T) return Boolean
   is
   begin
      return GNAT.OS_Lib.Is_Regular_File (Name => Str (Path));
   end IsFile;

   --------------------------------------------------------------------------------

   function NormalizePathName
     (Name      : EStrings.T;
      Directory : EStrings.T)
     return EStrings.T
   is
      Result : EStrings.T;
   begin
      EStrings.CopyString
        (EStr => Result,
         Str => GNAT.OS_Lib.Normalize_Pathname
           (Name          => Str (Name),
            Directory     => Str (Directory),
            Resolve_Links => False));
      return Result;
   end NormalizePathName;

   --------------------------------------------------------------------------------

   procedure NormaliseDir (D : in out EStrings.T)
   is
      Unused : Boolean;
   begin
      if not IsDirectory (D) then
         SystemErrors.FatalError
           (SysErr => SystemErrors.OtherInternalError,
            Msg => "Internal error.");
      elsif D.Content (D.Length) /= DirectorySeparator then
         EStrings.AppendChar (EStr => D,
                                     Ch  => DirectorySeparator,
                                     Success => Unused);
      end if;
   end NormaliseDir;

   --------------------------------------------------------------------------------

   function UpDir (D : EStrings.T) return EStrings.T
   is
      Result : EStrings.T := EStrings.EmptyString;
   begin
      if not IsDirectory (D) then
         SystemErrors.FatalError
           (SysErr => SystemErrors.OtherInternalError,
            Msg => "Internal error.");
      else
         Result := Prefix (EStr             => D,
                           Separator        => DirectorySeparator,
                           IncludeSeparator => True);
      end if;
      return Result;
   end UpDir;

   --------------------------------------------------------------------------------

   function RelativeName
     (OfThisFileOrDir : EStrings.T;
      ToThisDir       : EStrings.T) return EStrings.T
   is

      TheCommonBit : EStrings.T;
      OfThis       : EStrings.T;
      ToThis       : EStrings.T;
      Result       : EStrings.T;
      Unused       : Boolean;

      function CommonPath (P1, P2 : EStrings.T) return
        EStrings.T is
      --
      -- Returns the common directory.
      -- The last char is always the directory separator
         Result : EStrings.T := EStrings.EmptyString;
         Unused : Boolean;
      begin
         for I in Integer range 1 .. P1.Length loop
            exit when I > P2.Length;
            exit when P1.Content (I) /= P2.Content (I);
            EStrings.AppendChar (EStr => Result,
                                        Ch  => P1.Content (I),
                                        Success => Unused);
         end loop;
         if Result.Content (Result.Length) /= DirectorySeparator then
            Result := Prefix (Result, DirectorySeparator, True);
         end if;
         return Result;
      end CommonPath;

   begin
      Result := EStrings.EmptyString;
      ToThis := ToThisDir;
      OfThis := OfThisFileOrDir;

      -- Check the input parameters make sense.
      if IsFile (OfThis) and not IsDirectory (OfThis) then
         OfThis := Prefix (OfThis, DirectorySeparator, True);
      end if;

      if IsDirectory (ToThis) then
         NormaliseDir (ToThis);
      end if;

      if not IsDirectory (OfThis) or not IsDirectory (ToThis) then
         SystemErrors.FatalError
           (SysErr => SystemErrors.OtherInternalError,
            Msg => "Internal error.");
      end if;

      TheCommonBit := CommonPath (OfThis, ToThis);

      if EStrings.IsEmpty (TheCommonBit) then
         Result := OfThisFileOrDir;

      elsif EStrings.EqString (TheCommonBit, ToThis) then
         Result := StringUtilities.Section
           (EStr     => OfThisFileOrDir,
            StartPos => TheCommonBit.Length + 1,
            Length   => OfThisFileOrDir.Length - TheCommonBit.Length);

      else
         loop
            ToThis := UpDir (ToThis);
            EStrings.AppendString (EStr => Result,
                                          Str  => "..");
            EStrings.AppendChar (EStr => Result,
                                        Ch  => DirectorySeparator,
                                        Success => Unused);
            exit when EStrings.EqString (ToThis, TheCommonBit);
         end loop;
         EStrings.AppendExaminerString
           (EStr1 => Result,
            EStr2 => StringUtilities.Section
              (EStr     => OfThisFileOrDir,
               StartPos => TheCommonBit.Length + 1,
               Length   => OfThisFileOrDir.Length - TheCommonBit.Length));
      end if;
      return Result;
   end RelativeName;


end Directory_Operations;
