-- $Id: sparklalr_parser.adb 11998 2009-01-02 14:42:09Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Command_Line_Options;
with Sparklalr_Goto;
use type Sparklalr_Goto.Next_T;
with Symbols_Dump;
with Sparklalr_Memory;

package body Sparklalr_Parser
   --# own State is
   --#   Pa_List,
   --#   Reduce_State,
   --#   Unique_Reduce_State,
   --#   Pa_Count,
   --#   Pat_Count,
   --#   Pa_Array;
is

   subtype Pa_Array_Range is Positive range 1 .. Pa_Table_Size;
   type Pa_Rec is record
      Index                  : Sparklalr_Common.Term_Range;
      The_Entry : Sparklalr_Common.Short_Int;
      Pa_Next                : Pt_Pa_Rec;
   end record;
   type Pa_Array_Array_T is array (Pa_Array_Range) of Pa_Rec;
   type Pa_Array_T is record
      The_Array : Pa_Array_Array_T;
      Top       : Pt_Pa_Rec;
   end record;

   type State_Set is array (Sparklalr_Common.State_Range) of Boolean;
   type Pa_List_T is array (Sparklalr_Common.State_Range) of Pt_Pa_Rec; -- PARSING ACTION TABLE

   Pa_List             : Pa_List_T;
   Reduce_State        : State_Set;
   Unique_Reduce_State : State_Set;
   Pa_Count            : Integer;
   Pat_Count           : Integer;
   Pa_Array            : Pa_Array_T;

   -- Local procedures/functions
   procedure Sort_Pa (Head : in out Pt_Pa_Rec)
      --# global in out Pa_Array;
      --# derives Head,
      --#         Pa_Array from Head,
      --#                       Pa_Array;
   is
      P, Pnext, Q : Pt_Pa_Rec;
      Found       : Boolean;
   begin
      if Head /= 0 then
         P                                 := Pa_Array.The_Array (Head).Pa_Next;
         Pa_Array.The_Array (Head).Pa_Next := 0;
      else
         P := 0;
      end if;
      while P /= 0 loop
         Pnext := Pa_Array.The_Array (P).Pa_Next;
         if Pa_Array.The_Array (P).The_Entry >= Pa_Array.The_Array (Head).The_Entry then
            Pa_Array.The_Array (P).Pa_Next := Head;
            Head                           := P;
         else
            Q     := Head;
            Found := False;
            loop
               if Pa_Array.The_Array (Q).Pa_Next /= 0 then
                  if Pa_Array.The_Array (P).The_Entry >=
                     Pa_Array.The_Array (Pa_Array.The_Array (Q).Pa_Next).The_Entry then
                     Found := True;
                  else
                     Q := Pa_Array.The_Array (Q).Pa_Next;
                  end if;
               else
                  Found := True;
               end if;
               exit when Found;
            end loop;
            Pa_Array.The_Array (P).Pa_Next := Pa_Array.The_Array (Q).Pa_Next;
            Pa_Array.The_Array (Q).Pa_Next := P;
         end if;
         P := Pnext;
      end loop;
   end Sort_Pa;
   -- End local procedures/functions

   procedure Init_Pa_List
      --# global out Pa_List;
      --# derives Pa_List from ;
   is
   begin
      Pa_List := Pa_List_T'(others => 0);
   end Init_Pa_List;

   procedure Init_Pat_Count
      --# global out Pat_Count;
      --# derives Pat_Count from ;
   is
   begin
      Pat_Count := 0;
   end Init_Pat_Count;

   procedure Initialise
      --# global out Pat_Count;
      --#        out Pa_Array;
      --#        out Pa_Count;
      --#        out Pa_List;
      --#        out Reduce_State;
      --#        out Unique_Reduce_State;
      --# derives Pat_Count,
      --#         Pa_Array,
      --#         Pa_Count,
      --#         Pa_List,
      --#         Reduce_State,
      --#         Unique_Reduce_State from ;
   is
   begin
      Init_Pa_List;
      Init_Pat_Count;
      Reduce_State        := State_Set'(others => False);
      Unique_Reduce_State := State_Set'(others => False);
      Pa_Count            := 0;
      Pa_Array            :=
        Pa_Array_T'
        (The_Array => Pa_Array_Array_T'(others => Pa_Rec'(Index => 0, The_Entry => 0, Pa_Next => 0)),
         Top       => 0);
   end Initialise;

   procedure Gen_State_Info
      --# global in     Pa_Array;
      --#        in     Pa_List;
      --#        in     Sparklalr_Memory.Stat_No;
      --#        in out Reduce_State;
      --#        in out Unique_Reduce_State;
      --# derives Reduce_State,
      --#         Unique_Reduce_State from *,
      --#                                  Pa_Array,
      --#                                  Pa_List,
      --#                                  Sparklalr_Memory.Stat_No;
   is
      Ptr, List_Start  : Pt_Pa_Rec;
      Is_Unique_Reduce : Boolean;
      St               : Integer;
   begin
      St := 1;
      while St <= Sparklalr_Memory.Get_Stat_No loop
         Ptr              := Pa_List (St);
         List_Start       := Ptr;
         Is_Unique_Reduce := True;
         while Ptr /= 0 loop
            if (Pa_Array.The_Array (Ptr).The_Entry > Sparklalr_Common.Prod_Lim) or
               (Pa_Array.The_Array (Ptr).The_Entry = 1) then
               Is_Unique_Reduce := False;
            else
               Reduce_State (St) := True;
               Is_Unique_Reduce  := Is_Unique_Reduce and
                 (Pa_Array.The_Array (Ptr).The_Entry =
                    Pa_Array.The_Array (List_Start).The_Entry);
            end if;
            Ptr := Pa_Array.The_Array (Ptr).Pa_Next;
         end loop;
         if Is_Unique_Reduce then
            Unique_Reduce_State (St) := True;
         end if;
         St := St + 1;
      end loop;
   end Gen_State_Info;

   procedure Pa_Search (State_Index, Term_Index : in Integer; Result : out Integer; Pl : out Pt_Pa_Rec)
      --# global in Pa_Array;
      --#        in Pa_List;
      --# derives Pl,
      --#         Result from Pa_Array,
      --#                     Pa_List,
      --#                     State_Index,
      --#                     Term_Index;
   is
      Found : Boolean;
      Plist : Pt_Pa_Rec;
   begin
      Result := 0;
      Pl     := 0;
      Found  := False;
      Plist  := Pa_List (State_Index);
      while (Plist /= 0) and not Found loop
         if Pa_Array.The_Array (Plist).Index = Term_Index then
            Pl     := Plist;
            Result := Pa_Array.The_Array (Plist).The_Entry;
            Found  := True;
         else
            Plist := Pa_Array.The_Array (Plist).Pa_Next;
         end if;
      end loop;
   end Pa_Search;

   procedure Pa_Insert (State_Index, Term_Index, Insertion : in Integer)
      -- INSERTS A NEW ENTRY INTO THE PARSING ACTION TABLE
      --# global in     Command_Line_Options.State;
      --#        in     Symbols_Dump.State;
      --#        in out Pa_Array;
      --#        in out Pa_Count;
      --#        in out Pa_List;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Pa_Array             from *,
      --#                                   Insertion,
      --#                                   Pa_List,
      --#                                   State_Index,
      --#                                   Term_Index &
      --#         Pa_Count             from * &
      --#         Pa_List              from *,
      --#                                   Pa_Array,
      --#                                   State_Index,
      --#                                   Term_Index &
      --#         Sparklalr_IO.Outputs from *,
      --#                                   Command_Line_Options.State,
      --#                                   Insertion,
      --#                                   State_Index,
      --#                                   Symbols_Dump.State,
      --#                                   Term_Index;
   is
      Posn             : Integer;
      Result_Pa_Search : Integer;
      Pl               : Pt_Pa_Rec;
   begin
      if Command_Line_Options.Get_Debug_Level (6) then
         Sparklalr_IO.Put (Sparklalr_IO.Standard_Output, " PA(STATE=");
         Sparklalr_IO.Put_Int (Sparklalr_IO.Standard_Output, State_Index, 3);
         Sparklalr_IO.Put (Sparklalr_IO.Standard_Output, ",TERMINAL=");
         Posn := 23;
         --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
         Sparklalr_Common.Print (Sparklalr_IO.Standard_Output, Symbols_Dump.Get_Term_Set (Term_Index), Posn, 23, False);
         --# end accept;
         Sparklalr_IO.Put (Sparklalr_IO.Standard_Output, ") = ");
         Sparklalr_IO.Put_Int (Sparklalr_IO.Standard_Output, Insertion, 3);
         Sparklalr_IO.New_Line (Sparklalr_IO.Standard_Output);
      end if;
      Pa_Count := Pa_Count + 1;
      Pa_Search (State_Index, Term_Index, Result_Pa_Search, Pl);
      if Result_Pa_Search /= 0 then
         Pa_Array.The_Array (Pl).The_Entry := Insertion;
      else
         Pa_Array.Top                      := Pa_Array.Top + 1;
         Pa_Array.The_Array (Pa_Array.Top) :=
           Pa_Rec'(Index => Term_Index, The_Entry => Insertion, Pa_Next => Pa_List (State_Index));
         Pa_List (State_Index)             := Pa_Array.Top;
      end if;
   end Pa_Insert;

   function Action_Equal (Act1, Act2 : in Pt_Pa_Rec) return Boolean
      --# global in Pa_Array;
        is
      Found    : Boolean;
      A2       : Pt_Pa_Rec;
      C1, C2   : Integer;
      Act1_Tmp : Pt_Pa_Rec;

      function Eq_Pa_Size (A1, A2 : in Pt_Pa_Rec) return Boolean
         --# global in Pa_Array;
           is
         A1_Tmp : Pt_Pa_Rec;
         A2_Tmp : Pt_Pa_Rec;
      begin
         A1_Tmp := A1;
         A2_Tmp := A2;
         while (A1_Tmp /= 0) and (A2_Tmp /= 0) loop
            A1_Tmp := Pa_Array.The_Array (A1_Tmp).Pa_Next;
            A2_Tmp := Pa_Array.The_Array (A2_Tmp).Pa_Next;
         end loop;
         return A1_Tmp = A2_Tmp;
      end Eq_Pa_Size;

   begin
      Act1_Tmp := Act1;
      Found    := False;
      if Eq_Pa_Size (Act1_Tmp, Act2) then
         while Act1_Tmp /= 0 loop
            C1    := Pa_Array.The_Array (Act1_Tmp).Index;
            C2    := Pa_Array.The_Array (Act1_Tmp).The_Entry;
            A2    := Act2;
            Found := False;
            while (A2 /= 0) and not Found loop
               if Pa_Array.The_Array (A2).Index = C1 then
                  if Pa_Array.The_Array (A2).The_Entry = C2 then
                     Found := True;
                  else
                     A2 := 0;
                  end if;
               else
                  A2 := Pa_Array.The_Array (A2).Pa_Next;
               end if;
            end loop;
            if Found then
               Act1_Tmp := Pa_Array.The_Array (Act1_Tmp).Pa_Next;
            else
               Act1_Tmp := 0;
            end if;
         end loop;
      end if;
      return Found;
   end Action_Equal;

   procedure Pa_Stats (F : in Sparklalr_IO.File_Type)
      --# global in     Pa_Count;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Sparklalr_IO.Outputs from *,
      --#                                   F,
      --#                                   Pa_Count;
   is
   begin
      Sparklalr_IO.Put_Int (F, Pa_Count, 6);
      Sparklalr_IO.Put_Line (F, " PARSING ACTIONS GENERATED");
   end Pa_Stats;

   procedure Dump_Actions (F : in Sparklalr_IO.File_Type; Nstate : in Integer)
      --# global in     Pa_Array;
      --#        in     Pa_List;
      --#        in     Symbols_Dump.State;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Sparklalr_IO.Outputs from *,
      --#                                   F,
      --#                                   Nstate,
      --#                                   Pa_Array,
      --#                                   Pa_List,
      --#                                   Symbols_Dump.State;
   is
      Pl               : Pt_Pa_Rec;
      I, J             : Integer;
      Posn             : Integer;
      Result_Pa_Search : Integer;
   begin
      Posn := 1;
      I    := 0;
      while I <= Symbols_Dump.Get_Nterms loop
         --# accept F, 10, Pl, "Ineffective assigment to Pl here expected and OK";
         Pa_Search (Nstate, I, Result_Pa_Search, Pl);
         --# end accept;
         if Result_Pa_Search /= 0 then
            Sparklalr_Common.Put_N_Chars (F, ' ', 4);
            Posn := Posn + 4;
            --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
            Sparklalr_Common.Print (F, Symbols_Dump.Get_Term_Set (I), Posn, 4, False);
            --# end accept;
            --# accept F, 10, Pl, "Ineffective assigment to Pl here expected and OK";
            Pa_Search (Nstate, I, J, Pl);
            --# end accept;
            if J > Sparklalr_Common.Prod_Lim then
               Sparklalr_IO.Put (F, " SHIFT ");
               Sparklalr_IO.Put_Int (F, J - Sparklalr_Common.Prod_Lim, 4);
            else
               if (J = 1) or (J = -1) then
                  if J = 1 then
                     Sparklalr_IO.Put (F, " ACCEPT ");
                  else
                     Sparklalr_IO.Put (F, " ERROR ");
                  end if;
               else
                  Sparklalr_IO.Put (F, " REDUCE ");
                  Sparklalr_IO.Put_Int (F, J, 4);
               end if;
            end if;
            Sparklalr_IO.New_Line (F);
            Posn := 1;
         end if;
         I := I + 1;
      end loop;
      --# accept F, 33, Pl, "Pl is unused OK";
   end Dump_Actions;

   procedure Action_Gen (State_Var : in Integer)
      --# global in     Unique_Reduce_State;
      --#        in out Pat_Count;
      --#        in out Pa_Array;
      --#        in out Pa_List;
      --# derives Pat_Count from *,
      --#                        Pa_Array,
      --#                        Pa_List,
      --#                        State_Var,
      --#                        Unique_Reduce_State &
      --#         Pa_Array,
      --#         Pa_List   from Pa_Array,
      --#                        Pa_List,
      --#                        State_Var;
   is

      Curr_Entry            : Integer;
      Red_Count             : Sparklalr_Common.Production_Count;
      Exit_Flag : Boolean;
      P                     : Pt_Pa_Rec;
      Next                  : Sparklalr_Goto.Next_T;
      First_Var             : Sparklalr_Common.Term_Range;

      procedure Write_Cond (Next : in Sparklalr_Goto.Next_T; First_Var : in Sparklalr_Common.Term_Range)
         --# global in out Pat_Count;
         --# derives Pat_Count from *,
         --#                        First_Var,
         --#                        Next;
      is
         T        : Sparklalr_Common.Term_Range;
         Next_Tmp : Sparklalr_Goto.Next_T;
      begin
         Next_Tmp  := Next;
         Pat_Count := Pat_Count + 1;
         if Next_Tmp /= Sparklalr_Goto.Next_False_Const then
            T := First_Var;
            loop
               T := T + 1;
               if Sparklalr_Goto.Get_Next (Next_Tmp, T) then
                  Pat_Count := Pat_Count + 1;
                  Sparklalr_Goto.Set_Next (Next_Tmp, T, False);
               end if;
               exit when Next_Tmp = Sparklalr_Goto.Next_False_Const;
            end loop;
         end if;
      end Write_Cond;

   begin
      Sort_Pa (Pa_List (State_Var));
      P := Pa_List (State_Var);
      while P /= 0 loop
         Curr_Entry            := Pa_Array.The_Array (P).The_Entry;
         First_Var             := Pa_Array.The_Array (P).Index;
         Next                  := Sparklalr_Goto.Next_False_Const;
         Exit_Flag := False;
         Red_Count             := 0;
         loop
            P := Pa_Array.The_Array (P).Pa_Next;
            if P /= 0 then
               if Pa_Array.The_Array (P).The_Entry = Curr_Entry then
                  if Pa_Array.The_Array (P).Index < First_Var then
                     Sparklalr_Goto.Set_Next (Next, First_Var, True);
                     First_Var := Pa_Array.The_Array (P).Index;
                  else
                     Sparklalr_Goto.Set_Next (Next, Pa_Array.The_Array (P).Index, True);
                  end if;
               else
                  Exit_Flag := True;
               end if;
            else
               Exit_Flag := True;
            end if;
            exit when Exit_Flag;
         end loop;
         if Curr_Entry >= 0 then
            if Curr_Entry <= Sparklalr_Common.Prod_Lim then
               Red_Count := Red_Count + 1;
            end if;
            if (Red_Count = 1) and (P = 0) and Unique_Reduce_State (State_Var) then
               Pat_Count := Pat_Count + 1;
            else
               Write_Cond (Next, First_Var);
            end if;
         end if;
      end loop;
   end Action_Gen;

   procedure Action_Gen_Pa_Out (F : in Sparklalr_IO.File_Type; State_Var : in Integer; Curr_Pat_Index : in out Integer)
      --# global in     Sparklalr_Memory.Prod_Sum;
      --#        in     Symbols_Dump.State;
      --#        in     Unique_Reduce_State;
      --#        in out Pa_Array;
      --#        in out Pa_List;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Curr_Pat_Index       from *,
      --#                                   Pa_Array,
      --#                                   Pa_List,
      --#                                   State_Var,
      --#                                   Unique_Reduce_State &
      --#         Pa_Array,
      --#         Pa_List              from Pa_Array,
      --#                                   Pa_List,
      --#                                   State_Var &
      --#         Sparklalr_IO.Outputs from *,
      --#                                   Curr_Pat_Index,
      --#                                   F,
      --#                                   Pa_Array,
      --#                                   Pa_List,
      --#                                   Sparklalr_Memory.Prod_Sum,
      --#                                   State_Var,
      --#                                   Symbols_Dump.State,
      --#                                   Unique_Reduce_State;
   is

      Red_Count             : Sparklalr_Common.Production_Count;
      Exit_Flag : Boolean;
      P                     : Pt_Pa_Rec;
      Next                  : Sparklalr_Goto.Next_T;
      First_Var             : Sparklalr_Common.Term_Range;
      Curr_Entry            : Integer;

      procedure Write_Act (F : in Sparklalr_IO.File_Type; Curr_Entry : in Integer)
         --# global in     Sparklalr_Memory.Prod_Sum;
         --#        in     Symbols_Dump.State;
         --#        in out Sparklalr_IO.Outputs;
         --# derives Sparklalr_IO.Outputs from *,
         --#                                   Curr_Entry,
         --#                                   F,
         --#                                   Sparklalr_Memory.Prod_Sum,
         --#                                   Symbols_Dump.State;
      is
         Posn : Integer;
      begin
         Posn := 22;
         if Curr_Entry /= 1 then -- ACCEPT
            if Curr_Entry > Sparklalr_Common.Prod_Lim then
               Sparklalr_IO.Put (F, ", SPParseAct'(Shift, ");
               Sparklalr_IO.Put_Int (F, Curr_Entry - Sparklalr_Common.Prod_Lim, 1);
               Sparklalr_IO.Put (F, ", NoSym, NoRed, NoProd)");
            else
               Sparklalr_IO.Put_Line (F, ", SPParseAct'(Reduce, SPProductions.NoState, ");
               Sparklalr_Common.Put_N_Chars (F, ' ', 22);
               --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
               Symbols_Dump.Print_String_Sym
                 (F,
                  Sparklalr_Common.Sp_Symbol_Str,
                  Sparklalr_Memory.Get_Prod_Sum (Curr_Entry, 1) + Sparklalr_Common.Nt_Base,
                  Posn,
                  22,
                  False);
               --# end accept;
               Sparklalr_IO.New_Line (F);
               Sparklalr_Common.Put_N_Chars (F, ' ', 22);
               Sparklalr_IO.Put (F, ", ");
               Sparklalr_IO.Put_Int (F, Sparklalr_Memory.Get_Prod_Sum (Curr_Entry, 2), 1);
               Sparklalr_IO.Put (F, ", ");
               Sparklalr_IO.Put_Int (F, Curr_Entry, 1);
               Sparklalr_IO.Put_Char (F, ')');
            end if;
         else
            Sparklalr_IO.Put (F, ", AcceptAction");
         end if;
      end Write_Act;

      procedure Write_Cond
        (F              : in Sparklalr_IO.File_Type;
         Next           : in Sparklalr_Goto.Next_T;
         First_Var      : in Sparklalr_Common.Term_Range;
         Curr_Entry     : in Integer;
         Curr_Pat_Index : in out Integer)
         --# global in     Sparklalr_Memory.Prod_Sum;
         --#        in     Symbols_Dump.State;
         --#        in out Sparklalr_IO.Outputs;
         --# derives Curr_Pat_Index       from *,
         --#                                   First_Var,
         --#                                   Next &
         --#         Sparklalr_IO.Outputs from *,
         --#                                   Curr_Entry,
         --#                                   Curr_Pat_Index,
         --#                                   F,
         --#                                   First_Var,
         --#                                   Next,
         --#                                   Sparklalr_Memory.Prod_Sum,
         --#                                   Symbols_Dump.State;
      is
         T        : Sparklalr_Common.Term_Range;
         Posn     : Integer;
         Next_Tmp : Sparklalr_Goto.Next_T;
      begin
         Sparklalr_IO.Put_Line (F, "                         SymActionPair'(");
         Next_Tmp       := Next;
         Curr_Pat_Index := Curr_Pat_Index + 1;
         Posn           := 21;
         if Next_Tmp = Sparklalr_Goto.Next_False_Const then
            --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
            Sparklalr_Common.Print2 (F, Sparklalr_Common.Sp_Symbol_Str, Symbols_Dump.Get_Term_Set (First_Var), Posn, 21, False);
            --# end accept;
            Write_Act (F, Curr_Entry);
            Sparklalr_IO.Put_Char (F, ')');
         else
            --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
            Sparklalr_Common.Print2 (F, Sparklalr_Common.Sp_Symbol_Str, Symbols_Dump.Get_Term_Set (First_Var), Posn, 21, False);
            --# end accept;
            Write_Act (F, Curr_Entry);
            Sparklalr_IO.Put_Char (F, ')');
            T := First_Var;
            loop
               T := T + 1;
               if Sparklalr_Goto.Get_Next (Next_Tmp, T) then
                  Sparklalr_IO.Put_Line (F, ",");
                  Sparklalr_IO.Put_Int (F, Curr_Pat_Index, 5);
                  Sparklalr_IO.Put_Line (F, "                      => SymActionPair'(");
                  Curr_Pat_Index := Curr_Pat_Index + 1;
                  Posn           := 21;
                  --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
                  Sparklalr_Common.Print2 (F, Sparklalr_Common.Sp_Symbol_Str, Symbols_Dump.Get_Term_Set (T), Posn, 13, False);
                  --# end accept;
                  Write_Act (F, Curr_Entry);
                  Sparklalr_IO.Put_Char (F, ')');
                  Sparklalr_Goto.Set_Next (Next_Tmp, T, False);
               end if;
               exit when Next_Tmp = Sparklalr_Goto.Next_False_Const;
            end loop;
         end if;
      end Write_Cond;

   begin
      Sort_Pa (Pa_List (State_Var));
      P := Pa_List (State_Var);
      while P /= 0 loop
         Curr_Entry            := Pa_Array.The_Array (P).The_Entry;
         First_Var             := Pa_Array.The_Array (P).Index;
         Next                  := Sparklalr_Goto.Next_False_Const;
         Exit_Flag := False;
         Red_Count             := 0;
         loop
            P := Pa_Array.The_Array (P).Pa_Next;
            if P /= 0 then
               if Pa_Array.The_Array (P).The_Entry = Curr_Entry then
                  if Pa_Array.The_Array (P).Index < First_Var then
                     Sparklalr_Goto.Set_Next (Next, First_Var, True);
                     First_Var := Pa_Array.The_Array (P).Index;
                  else
                     Sparklalr_Goto.Set_Next (Next, Pa_Array.The_Array (P).Index, True);
                  end if;
               else
                  Exit_Flag := True;
               end if;
            else
               Exit_Flag := True;
            end if;
            exit when Exit_Flag;
         end loop;
         if Curr_Entry >= 0 then
            if Curr_Entry <= Sparklalr_Common.Prod_Lim then
               Red_Count := Red_Count + 1;
            end if;
            if (Red_Count = 1) and (P = 0) and Unique_Reduce_State (State_Var) then
               Sparklalr_IO.Put (F, "SymActionPair'(Default");
               Curr_Pat_Index := Curr_Pat_Index + 1;
               Write_Act (F, Curr_Entry);
               Sparklalr_IO.Put_Char (F, ')');
            else
               if P = 0 then
                  Write_Cond (F, Next, First_Var, Curr_Entry, Curr_Pat_Index);
               else
                  Write_Cond (F, Next, First_Var, Curr_Entry, Curr_Pat_Index);
                  Sparklalr_IO.Put_Line (F, ",");
                  Sparklalr_IO.Put_Int (F, Curr_Pat_Index, 5);
                  Sparklalr_IO.Put (F, " => ");
               end if;
            end if;
         end if;
      end loop;
   end Action_Gen_Pa_Out;

   procedure Action_Gen_Pa_Out_Sp (F : in Sparklalr_IO.File_Type; State_Var : in Integer; Curr_Pat_Index : in out Integer)
      --# global in     Sparklalr_Memory.Prod_Sum;
      --#        in     Symbols_Dump.State;
      --#        in     Unique_Reduce_State;
      --#        in out Pa_Array;
      --#        in out Pa_List;
      --#        in out Sparklalr_IO.Outputs;
      --# derives Curr_Pat_Index       from *,
      --#                                   Pa_Array,
      --#                                   Pa_List,
      --#                                   State_Var,
      --#                                   Unique_Reduce_State &
      --#         Pa_Array,
      --#         Pa_List              from Pa_Array,
      --#                                   Pa_List,
      --#                                   State_Var &
      --#         Sparklalr_IO.Outputs from *,
      --#                                   Curr_Pat_Index,
      --#                                   F,
      --#                                   Pa_Array,
      --#                                   Pa_List,
      --#                                   Sparklalr_Memory.Prod_Sum,
      --#                                   State_Var,
      --#                                   Symbols_Dump.State,
      --#                                   Unique_Reduce_State;
   is

      Red_Count             : Sparklalr_Common.Production_Count;
      Exit_Flag : Boolean;
      P                     : Pt_Pa_Rec;
      Posn                  : Integer;
      Next                  : Sparklalr_Goto.Next_T;
      First_Var             : Sparklalr_Common.Term_Range;
      Curr_Entry            : Integer;

      procedure Write_Act
        (F          : in Sparklalr_IO.File_Type;
         Posn       : in out Integer;
         Tab        : in Integer;
         Curr_Entry : in Integer)
         --# global in     Sparklalr_Memory.Prod_Sum;
         --#        in     Symbols_Dump.State;
         --#        in out Sparklalr_IO.Outputs;
         --# derives Posn                 from *,
         --#                                   Curr_Entry,
         --#                                   Sparklalr_Memory.Prod_Sum,
         --#                                   Symbols_Dump.State,
         --#                                   Tab &
         --#         Sparklalr_IO.Outputs from *,
         --#                                   Curr_Entry,
         --#                                   F,
         --#                                   Posn,
         --#                                   Sparklalr_Memory.Prod_Sum,
         --#                                   Symbols_Dump.State,
         --#                                   Tab;
      is
      begin
         if Curr_Entry /= 1 then -- ACCEPT
            if Curr_Entry > Sparklalr_Common.Prod_Lim then
               Sparklalr_IO.Put (F, " + (ShiftAct + State * ");
               Sparklalr_IO.Put_Int (F, Curr_Entry - Sparklalr_Common.Prod_Lim, 1);
               Sparklalr_IO.Put_Char (F, ')');
            else
               Sparklalr_IO.Put (F, " + (ReduceAct + ((Symbol * (SPSymbols.SPNonTerminal'Pos (");
               Posn := Posn + 72;
               Symbols_Dump.Print_String_Sym
                 (F,
                  Sparklalr_Common.Sp_Symbol_Str,
                  Sparklalr_Memory.Get_Prod_Sum (Curr_Entry, 1) + Sparklalr_Common.Nt_Base,
                  Posn,
                  Tab,
                  False);
               Sparklalr_IO.Put (F, ") - FirstNonTerminal) + RedBy * ");
               Sparklalr_IO.Put_Int (F, Sparklalr_Memory.Get_Prod_Sum (Curr_Entry, 2), 1);
               Sparklalr_IO.Put (F, ") + ProdNo * ");
               Sparklalr_IO.Put_Int (F, Curr_Entry, 1);
               Sparklalr_IO.Put (F, "))");
               Posn := Posn + 32;
            end if;
         else
            Sparklalr_IO.Put (F, " + AcceptAct");
            Posn := Posn + 12;
         end if;
      end Write_Act;

      procedure Write_Cond
        (F              : in Sparklalr_IO.File_Type;
         Next           : in Sparklalr_Goto.Next_T;
         First_Var      : in Sparklalr_Common.Term_Range;
         Curr_Entry     : in Integer;
         Curr_Pat_Index : in out Integer)
         --# global in     Sparklalr_Memory.Prod_Sum;
         --#        in     Symbols_Dump.State;
         --#        in out Sparklalr_IO.Outputs;
         --# derives Curr_Pat_Index       from *,
         --#                                   First_Var,
         --#                                   Next &
         --#         Sparklalr_IO.Outputs from *,
         --#                                   Curr_Entry,
         --#                                   Curr_Pat_Index,
         --#                                   F,
         --#                                   First_Var,
         --#                                   Next,
         --#                                   Sparklalr_Memory.Prod_Sum,
         --#                                   Symbols_Dump.State;
      is
         T        : Sparklalr_Common.Term_Range;
         Posn     : Integer;
         Next_Tmp : Sparklalr_Goto.Next_T;
      begin
         Next_Tmp       := Next;
         Curr_Pat_Index := Curr_Pat_Index + 1;
         if Next_Tmp = Sparklalr_Goto.Next_False_Const then
            Sparklalr_IO.Put (F, "SPSymbols.SPTerminal'Pos (");
            Posn := 36;
            Sparklalr_Common.Print2 (F, Sparklalr_Common.Sp_Symbol_Str, Symbols_Dump.Get_Term_Set (First_Var), Posn, 10, False);
            Sparklalr_IO.Put_Char (F, ')');
            Posn := Posn + 1;
            --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
            Write_Act (F, Posn, 10, Curr_Entry);
         --# end accept;
         else
            Sparklalr_IO.Put (F, "SPSymbols.SPTerminal'Pos (");
            Posn := 26;
            Sparklalr_Common.Print2 (F, Sparklalr_Common.Sp_Symbol_Str, Symbols_Dump.Get_Term_Set (First_Var), Posn, 10, False);
            Sparklalr_IO.Put_Char (F, ')');
            Posn := Posn + 1;
            --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
            Write_Act (F, Posn, 10, Curr_Entry);
            --# end accept;
            T := First_Var;
            loop
               T := T + 1;
               if Sparklalr_Goto.Get_Next (Next_Tmp, T) then
                  Sparklalr_IO.Put_Line (F, ",");
                  Sparklalr_IO.Put_Int (F, Curr_Pat_Index, 5);
                  Sparklalr_IO.Put (F, " => ");
                  Curr_Pat_Index := Curr_Pat_Index + 1;
                  Sparklalr_IO.Put (F, "SPSymbols.SPTerminal'Pos (");
                  Posn := 26;
                  Sparklalr_Common.Print2 (F, Sparklalr_Common.Sp_Symbol_Str, Symbols_Dump.Get_Term_Set (T), Posn, 10, False);
                  Sparklalr_IO.Put_Char (F, ')');
                  Posn := Posn + 1;
                  --# accept F, 10, Posn, "Ineffective assigment to Posn here expected and OK";
                  Write_Act (F, Posn, 10, Curr_Entry);
                  --# end accept;
                  Sparklalr_Goto.Set_Next (Next_Tmp, T, False);
               end if;
               exit when Next_Tmp = Sparklalr_Goto.Next_False_Const;
            end loop;
         end if;
      end Write_Cond;

   begin
      Sort_Pa (Pa_List (State_Var));
      Posn := 0;
      P    := Pa_List (State_Var);
      while P /= 0 loop
         Curr_Entry            := Pa_Array.The_Array (P).The_Entry;
         First_Var             := Pa_Array.The_Array (P).Index;
         Next                  := Sparklalr_Goto.Next_False_Const;
         Exit_Flag := False;
         Red_Count             := 0;
         loop
            P := Pa_Array.The_Array (P).Pa_Next;
            if P /= 0 then
               if Pa_Array.The_Array (P).The_Entry = Curr_Entry then
                  if Pa_Array.The_Array (P).Index < First_Var then
                     Sparklalr_Goto.Set_Next (Next, First_Var, True);
                     First_Var := Pa_Array.The_Array (P).Index;
                  else
                     Sparklalr_Goto.Set_Next (Next, Pa_Array.The_Array (P).Index, True);
                  end if;
               else
                  Exit_Flag := True;
               end if;
            else
               Exit_Flag := True;
            end if;
            exit when Exit_Flag;
         end loop;
         if Curr_Entry >= 0 then
            if Curr_Entry <= Sparklalr_Common.Prod_Lim then
               Red_Count := Red_Count + 1;
            end if;
            if (Red_Count = 1) and (P = 0) and Unique_Reduce_State (State_Var) then
               Sparklalr_IO.Put (F, "Default");
               Curr_Pat_Index := Curr_Pat_Index + 1;
               Write_Act (F, Posn, 10, Curr_Entry);
            else
               if P = 0 then
                  Write_Cond (F, Next, First_Var, Curr_Entry, Curr_Pat_Index);
               else
                  Write_Cond (F, Next, First_Var, Curr_Entry, Curr_Pat_Index);
                  Sparklalr_IO.Put_Line (F, ",");
                  Sparklalr_IO.Put_Int (F, Curr_Pat_Index, 5);
                  Sparklalr_IO.Put (F, " => ");
               end if;
            end if;
         end if;
      end loop;
   end Action_Gen_Pa_Out_Sp;

   function Get_Reduce_State (S : in Sparklalr_Common.State_Range) return Boolean
      --# global in Reduce_State;
        is
   begin
      return Reduce_State (S);
   end Get_Reduce_State;

   function Get_Pat_Count return Integer
      --# global in Pat_Count;
        is
   begin
      return Pat_Count;
   end Get_Pat_Count;

   function Get_Pa_List (I : in Sparklalr_Common.State_Range) return Pt_Pa_Rec
      --# global in Pa_List;
        is
   begin
      return Pa_List (I);
   end Get_Pa_List;

end Sparklalr_Parser;
