-- $Id: sparklalr_io.adb 12159 2009-01-15 11:13:48Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Ada.Integer_Text_IO;
with Ada.Command_Line;
with Ada.Strings.Fixed;
with Ada.Characters.Handling; -- This is for to_upper statement
with GNAT.OS_Lib; -- This is for exit statement
with Ada.Characters.Latin_1;

package body Sparklalr_IO is
   --# hide Sparklalr_IO;

   Len_Full_File_Name : constant := 9;

   subtype Full_File_Name is String (1 .. Len_Full_File_Name);

   procedure Put_Char (Fil : in File_Type; C : in Character) is
   begin
      Ada.Text_IO.Put (Ada.Text_IO.File_Type (Fil), C);
   end Put_Char;

   procedure Get_Char (Fil : in File_Type; C : out Character) is
   begin
      Ada.Text_IO.Get (Ada.Text_IO.File_Type (Fil), C);
   end Get_Char;

   procedure Put_Int (Fil : in File_Type; I : in Integer; Width : in Natural) is
   begin
      Ada.Integer_Text_IO.Put (Ada.Text_IO.File_Type (Fil), I, Width);
   end Put_Int;

   procedure Skip_Int (Fil : in File_Type; Width : in Natural) is
      I : Integer;
      pragma Unreferenced (I);
   begin
      Ada.Integer_Text_IO.Get (Ada.Text_IO.File_Type (Fil), I, Width);
   end Skip_Int;

   procedure Put (Fil : in File_Type; S : in String) is
   begin
      Ada.Text_IO.Put (Ada.Text_IO.File_Type (Fil), S);
   end Put;

   procedure Put_Line (Fil : in File_Type; S : in String) is
   begin
      Ada.Text_IO.Put_Line (Ada.Text_IO.File_Type (Fil), S);
   end Put_Line;

   procedure New_Line (Fil : in File_Type) is
   begin
      Ada.Text_IO.New_Line (Ada.Text_IO.File_Type (Fil));
   end New_Line;

   procedure Skip_Line (Fil : in File_Type; Spacing : in Positive) is
   begin
      Ada.Text_IO.Skip_Line (Ada.Text_IO.File_Type (Fil), Ada.Text_IO.Positive_Count (Spacing));
   end Skip_Line;

   function End_Of_Line (Fil : in File_Type) return Boolean is
      Peek   : Character;
      Result : Boolean;
   begin
      -- Windows line endings: Carriage return | Line feed
      -- Unix line endings:    Line feed
      --
      -- End_Of_Line is extended so that, on all platforms, if the next
      -- character is carriage return, end of line holds. This enables the
      -- parser generator to operates correctly on both Windows and Unix with
      -- both Windows and Unix line endings.
      Look_Ahead (Fil, Peek);
      if Peek = Ada.Characters.Latin_1.CR then
         Result := True;
      else
         Result := Ada.Text_IO.End_Of_Line (Ada.Text_IO.File_Type (Fil));
      end if;
      return Result;
   end End_Of_Line;

   function End_Of_File (Fil : in File_Type) return Boolean is
   begin
      return Ada.Text_IO.End_Of_File (Ada.Text_IO.File_Type (Fil));
   end End_Of_File;

   procedure Look_Ahead (Fil : in File_Type; C : out Character) is
      Eol : Boolean;
   begin
      Ada.Text_IO.Look_Ahead (Ada.Text_IO.File_Type (Fil), C, Eol);
   end Look_Ahead;

   function Argument_Count return Natural is
   begin
      return Ada.Command_Line.Argument_Count;
   end Argument_Count;

   function Argument (Number : Positive) return Arg_String is
      Result : Arg_String;
   begin
      Ada.Strings.Fixed.Move (Ada.Command_Line.Argument (Number), Result);
      return Result;
   end Argument;

   procedure To_Upper (S : in out String) is
   begin
      S := Ada.Characters.Handling.To_Upper (S);
   end To_Upper;

   function Trim_Length (Arg : in Arg_String) return Natural is
   begin
      return Ada.Strings.Fixed.Trim (Arg, Ada.Strings.Right)'Length;
   end Trim_Length;

   procedure Exit_St (S : in String; Status : in Typ_Exit) is
   begin
      Put_Line (Standard_Output, S);
      case Status is
         when Ok =>
            GNAT.OS_Lib.OS_Exit (0);
         when Error =>
            GNAT.OS_Lib.OS_Exit (1);
         when Warning =>
            GNAT.OS_Lib.OS_Exit (2);
      end case;
   end Exit_St;

   procedure Rewrite
     (Fil         : out File_Type;
      Name        : in File_Name;
      Extn        : in File_Extn;
      File_Status : out Boolean)
   is
      Ff : Full_File_Name;
   begin
      Ff := Name & Extn;
      Ada.Text_IO.Create (File => Ada.Text_IO.File_Type (Fil), Name => Ff);
      File_Status := True;
   end Rewrite;

   procedure Reset
     (Fil         : out File_Type;
      Name        : in File_Name;
      Extn        : in File_Extn;
      File_Status : out Boolean)
   is
      Ff : Full_File_Name;
   begin
      Ff := Name & Extn;
      Ada.Text_IO.Open (Ada.Text_IO.File_Type (Fil), Ada.Text_IO.In_File, Ff);
      File_Status := True;
   end Reset;

   function Standard_Output return File_Type is
      function File_Type_2_File_Type (F : in Ada.Text_IO.File_Type) return File_Type is
      begin
         return File_Type (F);
      end File_Type_2_File_Type;

   begin
      return File_Type_2_File_Type (Ada.Text_IO.Standard_Output);
   end Standard_Output;

end Sparklalr_IO;
