-- $Id: sparkprogram.adb 13082 2009-04-22 09:44:30Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Ada.Characters.Latin_1,
  Ada.Characters.Handling,
  CommandLineData,
  EStrings,
  Heap,
  LexTokenManager,
  RelationAlgebra,
  SeqAlgebra,
  SparkFormatCommandLineData,
  SparkLex,
  SPSymbols;
use type SparkFormatCommandLineData.ExpandOrCompress,
         SPSymbols.SPTerminal,
         SeqAlgebra.MemberOfSeq,
         LexTokenManager.StrCompResult,
         LexTokenManager.LexString;

package body SPARKProgram
is

   --# inherit Ada.Characters.Latin_1,
   --#         EStrings;
   package WhiteSpace
   is
      function IsWhiteSpace (Char : Character) return Boolean;

      procedure Skip (InputLine : in     EStrings.T;
                      Index     : in out EStrings.Lengths);
      --# derives Index from *,
      --#                    InputLine;
   end WhiteSpace;

   --# inherit CommandLineData,
   --#         Dictionary,
   --#         ErrorHandler,
   --#         EStrings,
   --#         LexTokenManager,
   --#         SparkLex,
   --#         SPARK_IO,
   --#         SPSymbols,
   --#         WhiteSpace;
   package File_IO
   is

      procedure Create (TemporaryFile : in out SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         TemporaryFile     from SPARK_IO.File_Sys,
      --#                                TemporaryFile;

      procedure Reset (TemporaryFile : in out SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         TemporaryFile     from *,
      --#                                TemporaryFile;

      procedure Delete (TemporaryFile : in out SPARK_IO.File_Type);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys,
      --#         TemporaryFile     from *,
      --#                                TemporaryFile;

      procedure Lex (TemporaryFile : in     SPARK_IO.File_Type;
                     Token         :    out SPSymbols.SPTerminal;
                     LexVal        :    out LexTokenManager.LexValue);
      --# global in     CommandLineData.Content;
      --#        in     Dictionary.Dict;
      --#        in out ErrorHandler.ErrorContext;
      --#        in out LexTokenManager.StringTable;
      --#        in out SparkLex.Currline;
      --#        in out SPARK_IO.File_Sys;
      --# derives ErrorHandler.ErrorContext,
      --#         LexTokenManager.StringTable,
      --#         LexVal,
      --#         SparkLex.Currline,
      --#         SPARK_IO.File_Sys,
      --#         Token                       from CommandLineData.Content,
      --#                                          Dictionary.Dict,
      --#                                          ErrorHandler.ErrorContext,
      --#                                          LexTokenManager.StringTable,
      --#                                          SparkLex.Currline,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TemporaryFile;

   end File_IO;

   --# inherit Heap,
   --#         LexTokenManager,
   --#         SeqAlgebra,
   --#         WhiteSpace;
   package Iteration is
      type Iterator is private;

      procedure Initialise (TheHeap            : in     Heap.HeapRecord;
                            TheSeq             : in     SeqAlgebra.Seq;
                            AlphabeticOrdering : in     Boolean;
                            TheIterator        :    out Iterator);
      --# global in LexTokenManager.StringTable;
      --# derives TheIterator from AlphabeticOrdering,
      --#                          LexTokenManager.StringTable,
      --#                          TheHeap,
      --#                          TheSeq;

      procedure Next (TheHeap     : in     Heap.HeapRecord;
                      TheIterator : in out Iterator);
      --# global in LexTokenManager.StringTable;
      --# derives TheIterator from *,
      --#                          LexTokenManager.StringTable,
      --#                          TheHeap;

      function Complete (TheIterator : Iterator) return Boolean;

      function CurrentString (TheIterator : Iterator)
                              return LexTokenManager.LexString;

      function CurrentMember (TheIterator : Iterator)
                              return SeqAlgebra.MemberOfSeq;

   private

      type Iterator is record
         FirstMember        : SeqAlgebra.MemberOfSeq;
         CurrentMember      : SeqAlgebra.MemberOfSeq;
         AlphabeticOrdering : Boolean;
         Placeholder        : LexTokenManager.LexString;
         Complete           : Boolean;
      end record;

   end Iteration;

   --# inherit Ada.Characters.Handling,
   --#         CommandLineData,
   --#         EStrings,
   --#         SparkFormatCommandLineData,
   --#         SPARK_IO,
   --#         WhiteSpace;
   package Annotations is
      type Anno_Type is private;

      procedure Initialize (This       :    out Anno_Type;
                            AnnoIntro  : in     String;
                            AnnoSucc   : in     String;
                            AnnoIndent : in     Natural);
      --# global in CommandLineData.Content;
      --# derives This from AnnoIndent,
      --#                   AnnoIntro,
      --#                   AnnoSucc,
      --#                   CommandLineData.Content;

      function Intro (This : Anno_Type) return EStrings.T;

      function Indent (This : Anno_Type) return Natural;

      procedure IsAnnoStart (This      : in     Anno_Type;
                             InputLine : in     EStrings.T;
                             Index     : in out EStrings.Lengths;
                             OK        :    out Boolean);
      --# derives Index,
      --#         OK    from Index,
      --#                    InputLine,
      --#                    This;

      function IsStartOf (This      : Anno_Type;
                          InputLine : EStrings.T)
                              return Boolean;

      function IsEndOf (This        : Anno_Type;
                        InputLine : EStrings.T) return Boolean;

      function ProperStartCol (StartCol : EStrings.Positions)
         return EStrings.Positions;

      procedure Write (This     : Anno_Type;
                       Output   : in SPARK_IO.File_Type;
                       StartCol : in EStrings.Positions);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Output,
      --#                                StartCol,
      --#                                This;

      procedure WriteIntro (This     : in Anno_Type;
                            Output   : in SPARK_IO.File_Type;
                            StartCol : in EStrings.Positions);
      --# global in out SPARK_IO.File_Sys;
      --# derives SPARK_IO.File_Sys from *,
      --#                                Output,
      --#                                StartCol,
      --#                                This;

      function Name1StartCol (This     : in Anno_Type;
                              StartCol : in EStrings.Positions)
                              return EStrings.Positions;

   private
      subtype AnnoStartBounds is Positive range 1 .. 3;
      subtype AnnoStart_Type is String (AnnoStartBounds);

      type Anno_Type is record
         AnnoStart   : AnnoStart_Type;
         AnnoIntro   : EStrings.T;
         AnnoSucc    : EStrings.T;
         AnnoIndent  : Natural;
      end record;
   end Annotations;

   --# inherit Ada.Characters.Handling,
   --#         Ada.Characters.Latin_1,
   --#         Annotations,
   --#         CommandLineData,
   --#         Dictionary,
   --#         ELStrings,
   --#         ErrorHandler,
   --#         EStrings,
   --#         File_IO,
   --#         Heap,
   --#         Iteration,
   --#         LexTokenManager,
   --#         RelationAlgebra,
   --#         SeqAlgebra,
   --#         SparkFormatCommandLineData,
   --#         SparkLex,
   --#         SPARKProgram,
   --#         SPARK_IO,
   --#         SPSymbols,
   --#         Statistics,
   --#         WhiteSpace;
   package Reformatter is
      Allow_Dotted_Names    : constant Boolean := True;
      Disallow_Dotted_Names : constant Boolean := False;

      type State is private;

      procedure Initialise (Anno        : in     Annotations.Anno_Type;
                            Alphabetic  : in     Boolean;
                            DottedNames : in     Boolean;
                            TheHeap     : in out Heap.HeapRecord;
                            This        :    out State);
      --# global in out Statistics.TableUsage;
      --# derives Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    TheHeap &
      --#         This                  from Alphabetic,
      --#                                    Anno,
      --#                                    DottedNames,
      --#                                    TheHeap;

      procedure Parse (This          : in out State;
                       TheHeap       : in out Heap.HeapRecord;
                       TemporaryFile : in out SPARK_IO.File_Type);
      --# global in     CommandLineData.Content;
      --#        in out LexTokenManager.StringTable;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --# derives LexTokenManager.StringTable,
      --#         SPARK_IO.File_Sys           from *,
      --#                                          CommandLineData.Content,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TemporaryFile,
      --#                                          This &
      --#         Statistics.TableUsage,
      --#         TheHeap,
      --#         This                        from *,
      --#                                          CommandLineData.Content,
      --#                                          LexTokenManager.StringTable,
      --#                                          SPARK_IO.File_Sys,
      --#                                          TemporaryFile,
      --#                                          TheHeap,
      --#                                          This &
      --#         TemporaryFile               from *;

      procedure Reformat (This          : in     State;
                          TheHeap       : in out Heap.HeapRecord;
                          TemporaryFile : in out SPARK_IO.File_Type;
                          Output        : in     SPARK_IO.File_Type;
                          Success       :    out Boolean);
      --# global in     CommandLineData.Content;
      --#        in     LexTokenManager.StringTable;
      --#        in     SparkFormatCommandLineData.Content;
      --#        in out SPARK_IO.File_Sys;
      --#        in out Statistics.TableUsage;
      --# derives SPARK_IO.File_Sys     from *,
      --#                                    CommandLineData.Content,
      --#                                    LexTokenManager.StringTable,
      --#                                    Output,
      --#                                    SparkFormatCommandLineData.Content,
      --#                                    TemporaryFile,
      --#                                    TheHeap,
      --#                                    This &
      --#         Statistics.TableUsage,
      --#         TheHeap               from *,
      --#                                    LexTokenManager.StringTable,
      --#                                    SparkFormatCommandLineData.Content,
      --#                                    TheHeap,
      --#                                    This &
      --#         Success               from This &
      --#         TemporaryFile         from *,
      --#                                    This;

      procedure Finalise (This    : in out State;
                          TheHeap : in out Heap.HeapRecord);
      --# derives TheHeap,
      --#         This    from *,
      --#                      This;

   private
      type Relation_Type is record
         ModifierRel : RelationAlgebra.Relation;
         TypeRel     : RelationAlgebra.Relation;
         PropertyRel : RelationAlgebra.Relation;
      end record;

      type VariableModifier is (Unmoded, InMode, OutMode, InOutMode,
                                 ProtectedUnmoded, ProtectedIn, ProtectedOut,
                                 TaskModifier);
      type Modifier_Use is array (VariableModifier) of Boolean;

      type Statistics_Type is record
         StartCol            : EStrings.Positions;
         MaxModifierLength   : EStrings.Lengths;
         MaxPrimaryIdLength  : EStrings.Lengths;
         MaxTypeMarkLength   : EStrings.Lengths;
         ModifiersPresent    : Modifier_Use;
         TypeMarkPresent     : Boolean;
         PropertyListPresent : Boolean;
      end record;

      type State is record
         Anno               : Annotations.Anno_Type;
         Relations          : Relation_Type;
         ParseStats         : Statistics_Type;
         AlphabeticOrdering : Boolean;
         AllowDottedNames   : Boolean;
         Success            : Boolean;
      end record;
   end Reformatter;

   package body WhiteSpace
   is

      function IsWhiteSpace (Char : Character) return Boolean
      is
      begin
         --Note that: Ada.Characters.Latin_1.HT is a tab.
         return Char = Ada.Characters.Latin_1.Space or Char = Ada.Characters.Latin_1.HT;
      end IsWhiteSpace;

      procedure Skip (InputLine : in     EStrings.T;
                      Index     : in out EStrings.Lengths)
      is
      begin
         loop
            exit when not IsWhiteSpace (InputLine.Content (Index)) or else Index = InputLine.Length;
            Index := Index + 1;
         end loop;
      end Skip;

   end WhiteSpace;

   package body File_IO
   is

      procedure Create (TemporaryFile : in out SPARK_IO.File_Type)
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SPARK_IO.Create (TemporaryFile, 0, "", "", Unused);
         -- Not sure what we can do if we can't create a temporary file
      end Create;

      procedure Reset (TemporaryFile : in out SPARK_IO.File_Type)
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SPARK_IO.Reset (TemporaryFile, SPARK_IO.In_File, Unused);
         -- not sure what we can do if we can't reset the file
      end Reset;

      procedure Delete (TemporaryFile : in out SPARK_IO.File_Type)
      is
         Unused : SPARK_IO.File_Status;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SPARK_IO.Delete (TemporaryFile, Unused);
         -- not sure what we can do if we can't delete the file
      end Delete;

      procedure Lex (TemporaryFile : in     SPARK_IO.File_Type;
                     Token         :    out SPSymbols.SPTerminal;
                     LexVal        :    out LexTokenManager.LexValue)
      is
         Unused : Boolean;
      begin
         --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
         --#        Flow_Message, 33, Unused, "The variable Unused is not required";
         SparkLex.SPARKFormatLex (TemporaryFile, Token, LexVal, Unused);
      end Lex;

   end File_IO;

   procedure Copy (Input, Output : in SPARK_IO.File_Type)
   is
      InputLine : EStrings.T;
   begin
      while not SPARK_IO.End_Of_File (Input) loop
         EStrings.GetLine (Input, InputLine);
         EStrings.PutLine (Output, InputLine);
      end loop;
   end Copy;

   package body Iteration is separate;

   package body Annotations is separate;

   package body Reformatter is separate;

   procedure ReformatAnnotations (Input, Output : in SPARK_IO.File_Type)
   is

      type StateType is (ParsingSPARKCode, ParsingGlobalsList,
                         ParsingDerivesList, ParsingInheritClause,
                         ParsingOwnVariableClause,
                         ParsingRefinementDefinition,
                         ParsingInitializationSpecification);


      --# inherit Ada.Characters.Handling,
      --#         CommandLineData,
      --#         EStrings,
      --#         SPARK_IO,
      --#         WhiteSpace;
      package Annotation
      --# own StartString;
      is

         procedure Initialize;
         --# global in     CommandLineData.Content;
         --#           out StartString;
         --# derives StartString from CommandLineData.Content;

         procedure IsStart (InputLine : in     EStrings.T;
                            Index     : in out EStrings.Lengths;
                            OK        :    out Boolean);
         --# global in StartString;
         --# derives Index,
         --#         OK    from Index,
         --#                    InputLine,
         --#                    StartString;

         procedure Write (Output   : in SPARK_IO.File_Type;
                          StartCol : in EStrings.Positions);
         --# global in     StartString;
         --#        in out SPARK_IO.File_Sys;
         --# derives SPARK_IO.File_Sys from *,
         --#                                Output,
         --#                                StartCol,
         --#                                StartString;

      end Annotation;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         EStrings,
      --#         WhiteSpace;
      package SPARKKeywordMatching
      is
         function Match (InputLine    : EStrings.T;
                         Index        : EStrings.Lengths;
                         MatchKeyword : EStrings.T) return Boolean;

         function MatchAtStart (InputLine    : EStrings.T;
                                MatchKeyword : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end SPARKKeywordMatching;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package AcceptList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end AcceptList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package EndList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end EndList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package ForList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end ForList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package FunctionList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end FunctionList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package TypeList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end TypeList;

      --# inherit Annotation,
      --#         EStrings,
      --#         SPARKKeywordMatching,
      --#         WhiteSpace;
      package AssertList
      is
         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;
      end AssertList;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         CommandLineData,
      --#         Dictionary,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         Heap,
      --#         LexTokenManager,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         WhiteSpace;
      package DerivesList
      --# own State;
      is

         procedure Initialize;
         --# global in out LexTokenManager.StringTable;
         --#           out State;
         --# derives LexTokenManager.StringTable,
         --#         State                       from LexTokenManager.StringTable;

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;

         procedure Parse (TemporaryFile : in out SPARK_IO.File_Type;
                          StartCol      :    out EStrings.Positions;
                          TheHeap       : in out Heap.HeapRecord;
                          Imports       :    out SeqAlgebra.Seq;
                          Exports       :    out SeqAlgebra.Seq;
                          Dependencies  :    out RelationAlgebra.Relation;
                          OK            :    out Boolean);
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in     State;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out LexTokenManager.StringTable;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#           out SparkLex.Currline;
         --# derives Dependencies,
         --#         Exports,
         --#         Imports                     from TheHeap &
         --#         ErrorHandler.ErrorContext,
         --#         LexTokenManager.StringTable,
         --#         OK,
         --#         SparkLex.Currline,
         --#         SPARK_IO.File_Sys,
         --#         TheHeap                     from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SparkFormatCommandLineData.Content,
         --#                                          SPARK_IO.File_Sys,
         --#                                          State,
         --#                                          TemporaryFile,
         --#                                          TheHeap &
         --#         StartCol                    from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TemporaryFile &
         --#         Statistics.TableUsage       from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SparkFormatCommandLineData.Content,
         --#                                          SPARK_IO.File_Sys,
         --#                                          State,
         --#                                          TemporaryFile,
         --#                                          TheHeap &
         --#         TemporaryFile               from *;

         procedure Reformat (OK            : in     Boolean;
                             TemporaryFile : in out SPARK_IO.File_Type;
                             Output        : in     SPARK_IO.File_Type;
                             StartCol      : in     EStrings.Positions;
                             TheHeap       : in out Heap.HeapRecord;
                             Exports       : in     SeqAlgebra.Seq;
                             Dependencies  : in     RelationAlgebra.Relation);
         --# global in     Annotation.StartString;
         --#        in     LexTokenManager.StringTable;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in     State;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --# derives SPARK_IO.File_Sys     from *,
         --#                                    Annotation.StartString,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.StringTable,
         --#                                    OK,
         --#                                    Output,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    StartCol,
         --#                                    State,
         --#                                    TemporaryFile,
         --#                                    TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap               from *,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.StringTable,
         --#                                    OK,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    State,
         --#                                    TheHeap &
         --#         TemporaryFile         from *,
         --#                                    OK;

      end DerivesList;

      package Subprogram
      is
         type SubprogramType is (IsFunction, IsProcedure, IsUnknown);
      end Subprogram;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         CommandLineData,
      --#         DerivesList,
      --#         Dictionary,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         Heap,
      --#         LexTokenManager,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         Subprogram,
      --#         WhiteSpace;
      package DeclareList
      is

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;
         pragma Unreferenced (IsEnd); -- for future use

      end DeclareList;

      --# inherit Ada.Characters.Handling,
      --#         Annotation,
      --#         CommandLineData,
      --#         DerivesList,
      --#         Dictionary,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         Heap,
      --#         LexTokenManager,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         Subprogram,
      --#         WhiteSpace;
      package ProofList
      is

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;
         pragma Unreferenced (IsEnd); -- for future use

      end ProofList;

      --# inherit AcceptList,
      --#         Ada.Characters.Handling,
      --#         Annotation,
      --#         AssertList,
      --#         CommandLineData,
      --#         DeclareList,
      --#         DerivesList,
      --#         Dictionary,
      --#         EndList,
      --#         ErrorHandler,
      --#         EStrings,
      --#         File_IO,
      --#         ForList,
      --#         FunctionList,
      --#         Heap,
      --#         LexTokenManager,
      --#         ProofList,
      --#         RelationAlgebra,
      --#         SeqAlgebra,
      --#         SparkFormatCommandLineData,
      --#         SPARKKeywordMatching,
      --#         SparkLex,
      --#         SPARKProgram,
      --#         SPARK_IO,
      --#         SPSymbols,
      --#         Statistics,
      --#         Subprogram,
      --#         TypeList,
      --#         WhiteSpace;
      package GlobalsList
      --# own ConsecutiveEmptyLines;
      --# initializes ConsecutiveEmptyLines;
      is

         type Modes is (Unmoded, InMode, InOutMode, OutMode);
         type GlobalVariablesType is array (Modes) of SeqAlgebra.Seq;

         function IsStart (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         function IsEnd (InputLine : EStrings.T) return Boolean;
         --# global in Annotation.StartString;

         procedure IncrementConsecutiveEmptyLines;
         --# global in out ConsecutiveEmptyLines;
         --# derives ConsecutiveEmptyLines from *;

         procedure ResetConsecutiveEmptyLines;
         --# global out ConsecutiveEmptyLines;
         --# derives ConsecutiveEmptyLines from ;

         function NoEmptyLines return Boolean;
         --# global in ConsecutiveEmptyLines;

         procedure Parse (TemporaryFile   : in out SPARK_IO.File_Type;
                          StartCol        :    out EStrings.Positions;
                          TheHeap         : in out Heap.HeapRecord;
                          GlobalVariables :    out GlobalVariablesType;
                          OK              :    out Boolean);
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out LexTokenManager.StringTable;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#           out SparkLex.Currline;
         --# derives ErrorHandler.ErrorContext,
         --#         LexTokenManager.StringTable,
         --#         OK,
         --#         SparkLex.Currline,
         --#         SPARK_IO.File_Sys,
         --#         StartCol                    from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TemporaryFile &
         --#         GlobalVariables             from TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap                     from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TemporaryFile,
         --#                                          TheHeap &
         --#         TemporaryFile               from *;

         procedure Reformat (OK                   : in     Boolean;
                             TemporaryFile        : in out SPARK_IO.File_Type;
                             Output               : in     SPARK_IO.File_Type;
                             StartCol             : in     EStrings.Positions;
                             TheHeap              : in out Heap.HeapRecord;
                             FunctionOrProcedure  : in     Subprogram.SubprogramType;
                             GlobalVariables      : in     GlobalVariablesType;
                             Imports              : in     SeqAlgebra.Seq;
                             Exports              : in     SeqAlgebra.Seq);
         --# global in     Annotation.StartString;
         --#        in     LexTokenManager.StringTable;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in out ConsecutiveEmptyLines;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --# derives ConsecutiveEmptyLines from  &
         --#         SPARK_IO.File_Sys     from *,
         --#                                    Annotation.StartString,
         --#                                    ConsecutiveEmptyLines,
         --#                                    Exports,
         --#                                    FunctionOrProcedure,
         --#                                    GlobalVariables,
         --#                                    Imports,
         --#                                    LexTokenManager.StringTable,
         --#                                    OK,
         --#                                    Output,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    StartCol,
         --#                                    TemporaryFile,
         --#                                    TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap               from *,
         --#                                    Exports,
         --#                                    FunctionOrProcedure,
         --#                                    GlobalVariables,
         --#                                    Imports,
         --#                                    OK,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    TheHeap &
         --#         TemporaryFile         from *,
         --#                                    OK;

      end GlobalsList;

      FunctionOrProcedure : Subprogram.SubprogramType;
      State               : StateType;
      InputLine           : EStrings.T;
      TheHeap             : Heap.HeapRecord;
      GlobalsListFile     : SPARK_IO.File_Type;
      DerivesListFile     : SPARK_IO.File_Type;
      InheritClauseFile   : SPARK_IO.File_Type;
      InitializeSpecFile  : SPARK_IO.File_Type;
      OwnVarClauseFile    : SPARK_IO.File_Type;
      GlobalVariables     : GlobalsList.GlobalVariablesType;
      Imports             : SeqAlgebra.Seq;
      Exports             : SeqAlgebra.Seq;
      Dependencies        : RelationAlgebra.Relation;
      StartCol            : EStrings.Positions;
      GlobalsListOK       : Boolean;
      DerivesListOK       : Boolean;
      InheritAnno         : Annotations.Anno_Type;
      InitializesAnno     : Annotations.Anno_Type;
      OwnVarAnno          : Annotations.Anno_Type;
      InheritClause       : Reformatter.State;
      InitializationSpec  : Reformatter.State;
      OwnVariableClause   : Reformatter.State;
      File_IO_Required    : Boolean;
      ReformatSuccessful  : Boolean;

      package body Annotation
      is

         StartString : EStrings.T;

         procedure Initialize
         is
            Unused : Boolean;
         begin
            EStrings.CopyString (StartString, "--");
            --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
            --#        Flow_Message, 33, Unused, "The variable Unused is not required";
            EStrings.AppendChar (StartString,
                                        CommandLineData.Content.AnnoChar,
                                        Unused);
            -- we know there's always room
         end Initialize;

         procedure IsStart (InputLine : in     EStrings.T;
                            Index     : in out EStrings.Lengths;
                            OK        :    out Boolean)
         is
         begin

            WhiteSpace.Skip (InputLine, Index);

            if InputLine.Length - Index >= 2 then
               if InputLine.Content (Index) = StartString.Content (1) and then
                  InputLine.Content (Index + 1) = StartString.Content (2) and then
                  InputLine.Content (Index + 2) = StartString.Content (3) then
                  Index := Index + 3;
                  OK := True;
               else
                  OK := False;
               end if;
            else
               OK := False;
            end if;

         end IsStart;

         procedure Write (Output   : in SPARK_IO.File_Type;
                          StartCol : in EStrings.Positions)
         is
         begin
            SPARK_IO.Set_Col (Output, StartCol);
            EStrings.PutString (Output, StartString);
         end Write;

      end Annotation;

      package body SPARKKeywordMatching
      is

         -- InputLine    - A line from the spark input file.
         -- Index        - An index position, expected to inside InputLine.
         -- MatchKeyword - A keyword to match.
         --
         -- The function returns true where InputLine, at position index, contains
         -- the keyword MatchKeyword. For a match, the keyword must be terminated
         -- by a newline or white space.
         function Match (InputLine    : EStrings.T;
                         Index        : EStrings.Lengths;
                         MatchKeyword : EStrings.T) return Boolean
         is
            IsMatch         : Boolean;
            CharFromLine    : Character;
            CharFromKeyword : Character;
         begin

            -- EStrings.Lengths'First is zero. This is used to describe empty strings.
            -- Can not find a match against an empty string.
            if Index = EStrings.Lengths'First then
               IsMatch := False;
            else
               IsMatch := True;
               for I in EStrings.Lengths range 1 .. MatchKeyword.Length loop
                  if (Index - 1) + I >= (InputLine.Length + 1) then
                     -- Reached or exceeded the end of line. Can not be a match.
                     IsMatch := False;
                     exit;
                  end if;

                  --# assert MatchKeyword=MatchKeyword% and
                  --#        Index > EStrings.Lengths'First and
                  --#        Index <= EStrings.Lengths'Last and
                  --#        I>=1 and I<=MatchKeyword.Length and
                  --#        (((Index - 1) + I) >= EStrings.Lengths'First) and
                  --#        (((Index - 1) + I) <= EStrings.Lengths'Last);

                  CharFromLine := Ada.Characters.Handling.To_Lower (InputLine.Content ((Index - 1) + I));
                  CharFromKeyword := Ada.Characters.Handling.To_Lower (MatchKeyword.Content (I));
                  if not (CharFromLine = CharFromKeyword) then
                     -- Are not the same at this index. Can not be a match.
                     IsMatch := False;
                     exit;
                  end if;
               end loop;

               -- If detected a keyword match, then investigate what appears after
               -- the keyword.
               if IsMatch then
                     -- If not terminated by line feed, is only a match if followed by
                     -- white space.
                  if ((Index + MatchKeyword.Length) <= InputLine.Length) then
                     IsMatch := WhiteSpace.IsWhiteSpace (InputLine.Content (Index + MatchKeyword.Length));
                  end if;
               end if;
            end if;

            return IsMatch;
         end Match;

         -- InputLine    - A line from the spark input file.
         -- MatchKeyword - A sparkkwyword to match.
         --
         -- The function returns true where InputLine is an annotation line, starting
         -- with the keyword MatchKeyword.
         function MatchAtStart (InputLine    : EStrings.T;
                                MatchKeyword : EStrings.T) return Boolean
         is
            Index : EStrings.Lengths;
            OK    : Boolean;
         begin
            if InputLine.Length > 0 then
               Index := 1;
               Annotation.IsStart (InputLine, Index, OK);
               if OK then
                  WhiteSpace.Skip (InputLine, Index);
                  OK := Match (InputLine, Index, MatchKeyword);
               end if;
            else
               OK := False;
            end if;

            return OK;
         end MatchAtStart;

      end SPARKKeywordMatching;

      package body AcceptList
      is
         -- "accept" has 6 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(6,
                                        EStrings.Contents'('a', 'c', 'c', 'e', 'p', 't', others => ' '));

         -- checks it's at the start of a line up to "--# accept"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end AcceptList;

      package body EndList
      is
         -- "end" has 3 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(3,
                                        EStrings.Contents'('e', 'n', 'd', others => ' '));

         -- checks it's at the start of a line up to "--# end"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end EndList;

      package body ForList
      is
         -- "for" has 3 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(3,
                                        EStrings.Contents'('f', 'o', 'r', others => ' '));

         -- checks it's at the start of a line up to "--# for"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end ForList;

      package body FunctionList
      is
         -- "function" has 8 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(8,
                                        EStrings.Contents'('f', 'u', 'n', 'c', 't', 'i', 'o', 'n', others => ' '));

         -- checks it's at the start of a line up to "--# function"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end FunctionList;

      package body TypeList
      is
         -- "type" has 4 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(4,
                                        EStrings.Contents'('t', 'y', 'p', 'e', others => ' '));

         -- checks it's at the start of a line up to "--# type"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end TypeList;

      package body AssertList
      is
         -- "assert" has 6 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(6,
                                        EStrings.Contents'('a', 's', 's', 'e', 'r', 't', others => ' '));

         -- checks it's at the start of a line up to "--# assert"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

      end AssertList;

      package body DerivesList
      --# own State is MultiplyToken, RWNullToken;
      is
         -- "derives" has 7 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(7,
                                        EStrings.Contents'('d', 'e', 'r', 'i', 'v', 'e', 's', others => ' '));

         MultiplyToken, RWNullToken : Natural;

         procedure Initialize
         --# global in out LexTokenManager.StringTable;
         --#           out MultiplyToken;
         --#           out RWNullToken;
         --# derives LexTokenManager.StringTable,
         --#         MultiplyToken,
         --#         RWNullToken                 from LexTokenManager.StringTable;
         is

            MultiplyString : EStrings.T;
            RWNullString   : EStrings.T;

            procedure InsertString (Str       : in     EStrings.T;
                                    LexStrRef :    out Natural)
            --# global in out LexTokenManager.StringTable;
            --# derives LexStrRef,
            --#         LexTokenManager.StringTable from LexTokenManager.StringTable,
            --#                                          Str;
            --# pre Str.Length >= 1;
            is
               LexStr : LexTokenManager.LexString;
            begin
               LexTokenManager.InsertExaminerString (Str, LexStr);
               LexStrRef := LexTokenManager.LexStringRef (LexStr);
            end InsertString;

         begin
            EStrings.CopyString (MultiplyString, "*");
            InsertString (MultiplyString, MultiplyToken);
            EStrings.CopyString (RWNullString, "null");
            InsertString (RWNullString, RWNullToken);
         end Initialize;

         -- checks it's at the start of a line up to "--# derives"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

         -- checks for ";" at end of line
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result : Boolean;
         begin

            Result := False;

            for I in EStrings.Lengths range EStrings.Positions'First .. InputLine.Length loop
               if InputLine.Content (I) = ';' then
                  Result := True;
                  exit;
               end if;
            end loop;

            return Result;

         end IsEnd;

         procedure Parse (TemporaryFile : in out SPARK_IO.File_Type;
                          StartCol      :    out EStrings.Positions;
                          TheHeap       : in out Heap.HeapRecord;
                          Imports       :    out SeqAlgebra.Seq;
                          Exports       :    out SeqAlgebra.Seq;
                          Dependencies  :    out RelationAlgebra.Relation;
                          OK            :    out Boolean)
         --# global in     CommandLineData.Content;
         --#        in     Dictionary.Dict;
         --#        in     MultiplyToken;
         --#        in     RWNullToken;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in out ErrorHandler.ErrorContext;
         --#        in out LexTokenManager.StringTable;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --#           out SparkLex.Currline;
         --# derives Dependencies,
         --#         Exports,
         --#         Imports                     from TheHeap &
         --#         ErrorHandler.ErrorContext,
         --#         LexTokenManager.StringTable,
         --#         OK,
         --#         SparkLex.Currline,
         --#         SPARK_IO.File_Sys,
         --#         TheHeap                     from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          MultiplyToken,
         --#                                          RWNullToken,
         --#                                          SparkFormatCommandLineData.Content,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TemporaryFile,
         --#                                          TheHeap &
         --#         StartCol                    from CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TemporaryFile &
         --#         Statistics.TableUsage       from *,
         --#                                          CommandLineData.Content,
         --#                                          Dictionary.Dict,
         --#                                          ErrorHandler.ErrorContext,
         --#                                          LexTokenManager.StringTable,
         --#                                          MultiplyToken,
         --#                                          RWNullToken,
         --#                                          SparkFormatCommandLineData.Content,
         --#                                          SPARK_IO.File_Sys,
         --#                                          TemporaryFile,
         --#                                          TheHeap &
         --#         TemporaryFile               from *;
         is

            Token       : SPSymbols.SPTerminal;
            LexVal      : LexTokenManager.LexValue;

            procedure ParseDependencyClauses (TemporaryFile : in     SPARK_IO.File_Type;
                                              Token         : in out SPSymbols.SPTerminal;
                                              LexVal        : in     LexTokenManager.LexValue;
                                              TheHeap       : in out Heap.HeapRecord;
                                              Exports       : in     SeqAlgebra.Seq;
                                              Imports       : in     SeqAlgebra.Seq;
                                              Dependencies  : in     RelationAlgebra.Relation;
                                              OK            :    out Boolean)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in     MultiplyToken;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out ErrorHandler.ErrorContext;
            --#        in out LexTokenManager.StringTable;
            --#        in out SparkLex.Currline;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives ErrorHandler.ErrorContext,
            --#         LexTokenManager.StringTable,
            --#         SparkLex.Currline,
            --#         SPARK_IO.File_Sys,
            --#         Statistics.TableUsage,
            --#         TheHeap,
            --#         Token                       from *,
            --#                                          CommandLineData.Content,
            --#                                          Dependencies,
            --#                                          Dictionary.Dict,
            --#                                          ErrorHandler.ErrorContext,
            --#                                          Exports,
            --#                                          Imports,
            --#                                          LexTokenManager.StringTable,
            --#                                          LexVal,
            --#                                          MultiplyToken,
            --#                                          RWNullToken,
            --#                                          SparkFormatCommandLineData.Content,
            --#                                          SparkLex.Currline,
            --#                                          SPARK_IO.File_Sys,
            --#                                          TemporaryFile,
            --#                                          TheHeap,
            --#                                          Token &
            --#         OK                          from CommandLineData.Content,
            --#                                          Dependencies,
            --#                                          Dictionary.Dict,
            --#                                          ErrorHandler.ErrorContext,
            --#                                          Exports,
            --#                                          Imports,
            --#                                          LexTokenManager.StringTable,
            --#                                          LexVal,
            --#                                          MultiplyToken,
            --#                                          RWNullToken,
            --#                                          SparkFormatCommandLineData.Content,
            --#                                          SparkLex.Currline,
            --#                                          SPARK_IO.File_Sys,
            --#                                          TemporaryFile,
            --#                                          TheHeap,
            --#                                          Token;
            is

               SimilarExports       : SeqAlgebra.Seq;
               ImportsForThisExport : SeqAlgebra.Seq;
               NextLexVal           : LexTokenManager.LexValue;
               SomeExports          : Boolean;

               procedure ParseExportList (TemporaryFile  : in     SPARK_IO.File_Type;
                                          Token          : in out SPSymbols.SPTerminal;
                                          LexVal         : in out LexTokenManager.LexValue;
                                          TheHeap        : in out Heap.HeapRecord;
                                          Exports        : in     SeqAlgebra.Seq;
                                          SimilarExports : in     SeqAlgebra.Seq;
                                          OK             :    out Boolean)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     RWNullToken;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out ErrorHandler.ErrorContext;
               --#        in out LexTokenManager.StringTable;
               --#        in out SparkLex.Currline;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives ErrorHandler.ErrorContext,
               --#         LexTokenManager.StringTable,
               --#         LexVal,
               --#         OK,
               --#         SparkLex.Currline,
               --#         SPARK_IO.File_Sys,
               --#         Token                       from CommandLineData.Content,
               --#                                          Dictionary.Dict,
               --#                                          ErrorHandler.ErrorContext,
               --#                                          LexTokenManager.StringTable,
               --#                                          LexVal,
               --#                                          SparkFormatCommandLineData.Content,
               --#                                          SparkLex.Currline,
               --#                                          SPARK_IO.File_Sys,
               --#                                          TemporaryFile,
               --#                                          Token &
               --#         Statistics.TableUsage,
               --#         TheHeap                     from *,
               --#                                          CommandLineData.Content,
               --#                                          Dictionary.Dict,
               --#                                          ErrorHandler.ErrorContext,
               --#                                          Exports,
               --#                                          LexTokenManager.StringTable,
               --#                                          LexVal,
               --#                                          RWNullToken,
               --#                                          SimilarExports,
               --#                                          SparkFormatCommandLineData.Content,
               --#                                          SparkLex.Currline,
               --#                                          SPARK_IO.File_Sys,
               --#                                          TemporaryFile,
               --#                                          TheHeap,
               --#                                          Token;
               is
                  DottedSimpleName, Identifier : EStrings.T;
                  Export                       : LexTokenManager.LexString;
                  NoExportsExcluded            : Boolean;
               begin

                  OK := True;

                  --  CFR1753: Check to see if an export has been excluded.
                  NoExportsExcluded := EStrings.IsEmpty
                     (SparkFormatCommandLineData.Content.ExcludeExport);

                  --  if NoExportsExcluded then
                  --     SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --        "ParseExportList: No Exclude Export Specified",
                  --        0);
                  --  else
                  --    SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --       "ParseExportList: Excluded Export specified",
                  --       0);
                  -- end if;


                  case Token is
                     when SPSymbols.identifier =>
                        loop
                           LexTokenManager.LexStringToString (LexVal.TokenStr, DottedSimpleName);
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                           loop
                              exit when Token /= SPSymbols.point;
                              EStrings.AppendString (DottedSimpleName, ".");
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token = SPSymbols.identifier then
                                 LexTokenManager.LexStringToString (LexVal.TokenStr, Identifier);
                                 EStrings.AppendExaminerString (DottedSimpleName, Identifier);
                                 File_IO.Lex (TemporaryFile, Token, LexVal);
                              else
                                 OK := False;
                              end if;
                              exit when not OK;
                           end loop;
                           exit when not OK;

                           --  CFR 1753: Check to see if this export has to be
                           --  excluded.
                           if NoExportsExcluded or else not EStrings.EqString
                                 (DottedSimpleName,
                                  SparkFormatCommandLineData.Content.ExcludeExport)
                           then
                              --  This export is not excluded.
                              --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              --     "ParseExportList: Export is included:",
                              --     0);
                              --  EStrings.PutLine (SPARK_IO.Standard_Output,
                              --     DottedSimpleName);
                              LexTokenManager.InsertExaminerString (DottedSimpleName, Export);
                              SeqAlgebra.AddMember (TheHeap, Exports, LexTokenManager.LexStringRef (Export));
                              SeqAlgebra.AddMember (TheHeap, SimilarExports, LexTokenManager.LexStringRef (Export));
                           --  else
                           --     SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                           --        "ParseExportList: Export is excluded:",
                           --        0);
                           --     EStrings.PutLine (SPARK_IO.Standard_Output,
                           --        DottedSimpleName);
                           end if;
                           exit when Token = SPSymbols.RWfrom;
                           if Token = SPSymbols.comma then
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token /= SPSymbols.identifier then
                                 OK := False;
                              end if;
                           else
                              OK := False;
                           end if;
                           exit when not OK;
                        end loop;
                     when SPSymbols.RWnull =>
                        SeqAlgebra.AddMember (TheHeap, Exports, RWNullToken);
                        SeqAlgebra.AddMember (TheHeap, SimilarExports, RWNullToken);
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                     when SPSymbols.semicolon =>
                        null;
                     when others =>
                        OK := False;
                  end case;

               end ParseExportList;

               procedure ParseImportList (TemporaryFile        : in     SPARK_IO.File_Type;
                                          Token                : in out SPSymbols.SPTerminal;
                                          LexVal               : in out LexTokenManager.LexValue;
                                          TheHeap              : in out Heap.HeapRecord;
                                          SimilarExports       : in     SeqAlgebra.Seq;
                                          Imports              : in     SeqAlgebra.Seq;
                                          ImportsForThisExport : in     SeqAlgebra.Seq;
                                          OK                   :    out Boolean)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in     MultiplyToken;
               --#        in out ErrorHandler.ErrorContext;
               --#        in out LexTokenManager.StringTable;
               --#        in out SparkLex.Currline;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives ErrorHandler.ErrorContext,
               --#         LexTokenManager.StringTable,
               --#         LexVal,
               --#         OK,
               --#         SparkLex.Currline,
               --#         SPARK_IO.File_Sys,
               --#         Token                       from CommandLineData.Content,
               --#                                          Dictionary.Dict,
               --#                                          ErrorHandler.ErrorContext,
               --#                                          LexTokenManager.StringTable,
               --#                                          LexVal,
               --#                                          SimilarExports,
               --#                                          SparkLex.Currline,
               --#                                          SPARK_IO.File_Sys,
               --#                                          TemporaryFile,
               --#                                          TheHeap,
               --#                                          Token &
               --#         Statistics.TableUsage,
               --#         TheHeap                     from *,
               --#                                          CommandLineData.Content,
               --#                                          Dictionary.Dict,
               --#                                          ErrorHandler.ErrorContext,
               --#                                          Imports,
               --#                                          ImportsForThisExport,
               --#                                          LexTokenManager.StringTable,
               --#                                          LexVal,
               --#                                          MultiplyToken,
               --#                                          SimilarExports,
               --#                                          SparkLex.Currline,
               --#                                          SPARK_IO.File_Sys,
               --#                                          TemporaryFile,
               --#                                          TheHeap,
               --#                                          Token;
               is
                  DottedSimpleName : EStrings.T;
                  Identifier       : EStrings.T;
                  Export           : SeqAlgebra.MemberOfSeq;
                  ImportName       : LexTokenManager.LexString;
                  SomeExports      : Boolean;
               begin

                  OK := True;

                  --  CFR1753: There may be no exports in the export list if an
                  --  excluded export was the only variable in the list.
                  SomeExports := not SeqAlgebra.IsEmptySeq (TheHeap, SimilarExports);
                  --  if SomeExports then
                  --     SPARK_IO.Put_line (SPARK_IO.Standard_Output,
                  --                  "ParseImportList: There are some exports",
                  --                  0);
                  --  end if;


                  loop
                     exit when Token = SPSymbols.semicolon or Token = SPSymbols.ampersand;
                     case Token is
                        when SPSymbols.identifier =>
                           LexTokenManager.LexStringToString (LexVal.TokenStr, DottedSimpleName);
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                           loop
                              exit when Token /= SPSymbols.point;
                              EStrings.AppendString (DottedSimpleName, ".");
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token = SPSymbols.identifier then
                                 LexTokenManager.LexStringToString (LexVal.TokenStr, Identifier);
                                 EStrings.AppendExaminerString (DottedSimpleName, Identifier);
                                 File_IO.Lex (TemporaryFile, Token, LexVal);
                              else
                                 OK := False;
                              end if;
                              exit when not OK;
                           end loop;
                           --  CFR1753: Ensure that the export list is non-empty
                           --  Otherwise do not add imports to the import sets.
                           if OK and SomeExports then
                              --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                              --     "ParseImporList: Adding import:",
                              --     0);
                              --  EStrings.PutLine (SPARK_IO.Standard_Output,
                              --     DottedSimpleName);

                              LexTokenManager.InsertExaminerString (DottedSimpleName, ImportName);
                              SeqAlgebra.AddMember (TheHeap, Imports, LexTokenManager.LexStringRef (ImportName));
                              SeqAlgebra.AddMember (TheHeap,
                                                    ImportsForThisExport,
                                                    LexTokenManager.LexStringRef (ImportName));
                           --  else
                           --     SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                           --        "ParseImporList: discarding import:",
                           --        0);
                           --     EStrings.PutLine (SPARK_IO.Standard_Output,
                           --        DottedSimpleName);
                           end if;
                        when SPSymbols.multiply =>
                           --  CFR1753: Ensure that the export list is non-empty
                           --  Otherwise do not add imports to the import sets.
                           --# accept F, 41, "Used to simplify loop structure";
                           if SomeExports then
                              Export := SeqAlgebra.FirstMember (TheHeap, SimilarExports);
                              loop
                                 exit when SeqAlgebra.IsNullMember (Export);
                                 --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                                 --     "ParseImporList: Adding import from *:",
                                 --     0);
                                 --  EStrings.PutLine (SPARK_IO.Standard_Output,
                                 --  DottedSimpleName);
                                 SeqAlgebra.AddMember (TheHeap, Imports, SeqAlgebra.ValueOfMember (TheHeap, Export));
                                 Export := SeqAlgebra.NextMember (TheHeap, Export);
                              end loop;
                              SeqAlgebra.AddMember (TheHeap, ImportsForThisExport, MultiplyToken);
                           --  else
                              --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                              --     "ParseImporList: No exports discarding *",
                              --     0);
                           end if;
                           --# end accept;
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                        when SPSymbols.comma =>
                           null;
                        when others =>
                           OK := False;
                     end case;
                     exit when not OK or Token = SPSymbols.semicolon or Token = SPSymbols.ampersand;
                     if Token = SPSymbols.comma then
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                     else
                        OK := False;
                     end if;
                  end loop;

               end ParseImportList;

               procedure StoreDependencies (TheHeap      : in out Heap.HeapRecord;
                                            Exports      : in     SeqAlgebra.Seq;
                                            Imports      : in     SeqAlgebra.Seq;
                                            Dependencies : in     RelationAlgebra.Relation)
               --# global in     MultiplyToken;
               --#        in out Statistics.TableUsage;
               --# derives Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Dependencies,
               --#                                    Exports,
               --#                                    Imports,
               --#                                    MultiplyToken,
               --#                                    TheHeap;
               is
                  Export, Import       : SeqAlgebra.MemberOfSeq;
                  ExportVal, ImportVal : Natural;
               begin
                  Export := SeqAlgebra.FirstMember (TheHeap, Exports);
                  loop
                     exit when SeqAlgebra.IsNullMember (Export);
                     ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);
                     Import := SeqAlgebra.FirstMember (TheHeap, Imports);
                     loop
                        exit when SeqAlgebra.IsNullMember (Import);
                        ImportVal := SeqAlgebra.ValueOfMember (TheHeap, Import);
                        if ExportVal = ImportVal then
                           ImportVal := MultiplyToken;
                        end if;
                        RelationAlgebra.InsertPair (TheHeap, Dependencies, ExportVal, ImportVal);
                        Import := SeqAlgebra.NextMember (TheHeap, Import);
                     end loop;
                     Export := SeqAlgebra.NextMember (TheHeap, Export);
                  end loop;
               end StoreDependencies;

            begin

               OK := True;
               NextLexVal := LexVal;

               loop
                  exit when Token = SPSymbols.semicolon;
                  SeqAlgebra.CreateSeq (TheHeap, SimilarExports);
                  ParseExportList (TemporaryFile, Token, NextLexVal, TheHeap, Exports, SimilarExports, OK);
                  if OK then
                     --  CFR1753: There may be no exports in the export list if an
                     --  excluded export was the only variable in the list,
                     --  however, we still need to parse the input list.
                     SomeExports := not SeqAlgebra.IsEmptySeq (TheHeap, SimilarExports);
                     --  if SomeExports then
                     --     SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --        "ParseDependencies: There is at least one export",
                     --        0);
                     --  else
                     --     SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --        "ParseDependencies: There are no exports associated with this import list",
                     --        0);
                     -- end if;

                     if Token = SPSymbols.RWfrom then
                        File_IO.Lex (TemporaryFile, Token, NextLexVal);
                        SeqAlgebra.CreateSeq (TheHeap, ImportsForThisExport);
                        ParseImportList (TemporaryFile, Token,
                                         NextLexVal,
                                         TheHeap,
                                         SimilarExports,
                                         Imports,
                                         ImportsForThisExport,
                                         OK);
                        --  CFR1753: Only add imports and store the dependencies
                        --  if there are some exported variables associated
                        --  this import list.
                        if SomeExports then
                           --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                           --     "ParseDependencies: Adding imports associated with this non-null import list",
                           --     0);
                           SeqAlgebra.AugmentSeq (TheHeap, Imports, ImportsForThisExport);
                           StoreDependencies (TheHeap, SimilarExports, ImportsForThisExport, Dependencies);
                        -- else
                           --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                           --     "ParseDependencies: Null export list - discard imports",
                           --     0);
                        end if;
                        SeqAlgebra.DisposeOfSeq (TheHeap, ImportsForThisExport);
                        if OK then
                           case Token is
                              when SPSymbols.ampersand =>
                                 File_IO.Lex (TemporaryFile, Token, NextLexVal);
                              when SPSymbols.semicolon =>
                                 null;
                              when others =>
                                 OK := False;
                           end case;
                        end if;
                     else
                        OK := False;
                     end if;
                  end if;
                  SeqAlgebra.DisposeOfSeq (TheHeap, SimilarExports);
                  exit when not OK;
               end loop;

            end ParseDependencyClauses;

         begin

            File_IO.Reset (TemporaryFile);
            SparkLex.ClearLineContext;
            SeqAlgebra.CreateSeq (TheHeap, Exports);
            SeqAlgebra.CreateSeq (TheHeap, Imports);
            RelationAlgebra.CreateRelation (TheHeap, Dependencies);
            File_IO.Lex (TemporaryFile, Token, LexVal);

            if Token = SPSymbols.annotation_start then
               StartCol := LexVal.Position.StartPos; -- we know StartCol can't be zero because the line can't be empty
               --# accept Flow_Message, 10, LexVal, "The next symbol should be 'derives', so we don't need the LexVal";
               File_IO.Lex (TemporaryFile, Token, LexVal);
               --# end accept;
               if Token = SPSymbols.RWderives then
                  File_IO.Lex (TemporaryFile, Token, LexVal);
                  ParseDependencyClauses (TemporaryFile,
                                          Token,
                                          LexVal,
                                          TheHeap,
                                          Exports,
                                          Imports,
                                          Dependencies,
                                          OK);
                  OK := OK and Token = SPSymbols.semicolon;
               else
                  OK := False;
               end if;
            else
               OK := False;
            end if;
            --# accept Flow_Message, 602, StartCol, StartCol, "StartCol may not be set if not OK";
         end Parse;

         procedure Reformat (OK            : in     Boolean;
                            TemporaryFile : in out SPARK_IO.File_Type;
                            Output        : in     SPARK_IO.File_Type;
                            StartCol      : in     EStrings.Positions;
                            TheHeap       : in out Heap.HeapRecord;
                            Exports       : in     SeqAlgebra.Seq;
                            Dependencies  : in     RelationAlgebra.Relation)
         --# global in     Annotation.StartString;
         --#        in     LexTokenManager.StringTable;
         --#        in     MultiplyToken;
         --#        in     RWNullToken;
         --#        in     SparkFormatCommandLineData.Content;
         --#        in out SPARK_IO.File_Sys;
         --#        in out Statistics.TableUsage;
         --# derives SPARK_IO.File_Sys     from *,
         --#                                    Annotation.StartString,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.StringTable,
         --#                                    MultiplyToken,
         --#                                    OK,
         --#                                    Output,
         --#                                    RWNullToken,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    StartCol,
         --#                                    TemporaryFile,
         --#                                    TheHeap &
         --#         Statistics.TableUsage,
         --#         TheHeap               from *,
         --#                                    Dependencies,
         --#                                    Exports,
         --#                                    LexTokenManager.StringTable,
         --#                                    MultiplyToken,
         --#                                    OK,
         --#                                    RWNullToken,
         --#                                    SparkFormatCommandLineData.Content,
         --#                                    TheHeap &
         --#         TemporaryFile         from *,
         --#                                    OK;
         is

            FirstExport : Boolean;
            ExportWidth : EStrings.Lengths;

            function DerivesCol (StartCol : EStrings.Positions) return EStrings.Positions
            is
            begin
               return StartCol + 4; -- length of "--# "
            end DerivesCol;

            -- writes "derives " in correct position
            procedure WriteDerives (Output   : in SPARK_IO.File_Type;
                                    StartCol : in EStrings.Positions)
            --# global in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol;
            is
            begin
               SPARK_IO.Set_Col (Output, DerivesCol (StartCol));
               SPARK_IO.Put_String (Output, "derives", 0);
               -- If ExportIndent is not Inline, then don't put out a trailing space,
               -- since this will be rejected by GNAT -gnatyb style check mode.
               if SparkFormatCommandLineData.Content.ExportIndent = SparkFormatCommandLineData.Inline then
                  SPARK_IO.Put_String (Output, " ", 0);
               end if;
            end WriteDerives;

            function ExportCol (StartCol : EStrings.Positions) return EStrings.Positions
            --# global in SparkFormatCommandLineData.Content;
            is
               ExportColumn : EStrings.Positions;
            begin
               if SparkFormatCommandLineData.Content.ExportIndent = SparkFormatCommandLineData.Inline then
                  ExportColumn := DerivesCol (StartCol) + 8; -- length of "derives "
               else
                  ExportColumn := DerivesCol (StartCol) + (SparkFormatCommandLineData.Content.ExportIndent - 1);
               end if;
               return ExportColumn;
            end ExportCol;

            -- only called when there is a value for separator indent, returns starting column
            function SeparatorCol (StartCol : EStrings.Positions) return EStrings.Positions
            --# global in SparkFormatCommandLineData.Content;
            is
            begin
               return DerivesCol (StartCol) + (SparkFormatCommandLineData.Content.SeparatorIndent - 1); -- StartCol "is --# " but indent value is given from "--#"
            end SeparatorCol;

            procedure WriteExport (FirstExport : in out Boolean;
                                   Output      : in     SPARK_IO.File_Type;
                                   StartCol    : in     EStrings.Positions;
                                   TheHeap     : in     Heap.HeapRecord;
                                   Export      : in     SeqAlgebra.MemberOfSeq)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives FirstExport       from * &
            --#         SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                Export,
            --#                                FirstExport,
            --#                                LexTokenManager.StringTable,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol,
            --#                                TheHeap;
            is
               ExportVal    : Natural;
               ExportString : EStrings.T;

            begin

               if FirstExport then
                  FirstExport := False;
                  if SparkFormatCommandLineData.Content.ExportIndent /= SparkFormatCommandLineData.Inline then
                     Annotation.Write (Output, StartCol);
                  end if;
               else
                  if SparkFormatCommandLineData.Content.SeparatorIndent = SparkFormatCommandLineData.Inline then
                     SPARK_IO.Put_Line (Output, " &", 0);
                  else
                     Annotation.Write (Output, StartCol);
                     SPARK_IO.Set_Col (Output, SeparatorCol (StartCol));
                     SPARK_IO.Put_Line (Output, "&", 0);
                  end if;
                  Annotation.Write (Output, StartCol);
               end if;

               SPARK_IO.Set_Col (Output, ExportCol (StartCol));
               ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);
               LexTokenManager.LexStringToString (LexTokenManager.ConvertLexStringRef (ExportVal), ExportString);
               EStrings.PutString (Output, ExportString);

            end WriteExport;

            procedure WriteSimilarExport (Output   : in     SPARK_IO.File_Type;
                                          StartCol : in     EStrings.Positions;
                                          TheHeap  : in     Heap.HeapRecord;
                                          Export   : in     SeqAlgebra.MemberOfSeq)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                Export,
            --#                                LexTokenManager.StringTable,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol,
            --#                                TheHeap;
            is
               ExportVal    : Natural;
               ExportString : EStrings.T;
            begin
               SPARK_IO.Put_Line (Output, ",", 0);
               Annotation.Write (Output, StartCol);
               SPARK_IO.Set_Col (Output, ExportCol (StartCol));
               ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);
               LexTokenManager.LexStringToString (LexTokenManager.ConvertLexStringRef (ExportVal), ExportString);
               EStrings.PutString (Output, ExportString);
            end WriteSimilarExport;

            procedure WriteNull (FirstExport : in Boolean;
                                 Output      : in SPARK_IO.File_Type;
                                 StartCol    : in EStrings.Positions)
            --# global in     Annotation.StartString;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                FirstExport,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol;
            is
            begin

               if not FirstExport then
                  SPARK_IO.Put_Line (Output, " &", 0);
                  Annotation.Write (Output, StartCol);
               elsif SparkFormatCommandLineData.Content.ExportIndent /= SparkFormatCommandLineData.Inline then
                  Annotation.Write (Output, StartCol);
               end if;

               SPARK_IO.Set_Col (Output, ExportCol (StartCol));
               SPARK_IO.Put_String (Output, "null", 0);

            end WriteNull;

            function LongestExport (TheHeap : Heap.HeapRecord; Exports : SeqAlgebra.Seq) return EStrings.Lengths
            --# global in LexTokenManager.StringTable;
            is

               MaxLength    : Natural;
               Export       : SeqAlgebra.MemberOfSeq;
               ExportVal    : Natural;
               ExportString : EStrings.T;

               function Max (A, B : Natural) return Natural
               is
                  Result : Natural;
               begin

                  if A >= B then
                     Result := A;
                  else
                     Result := B;
                  end if;

                  return Result;

               end Max;

            begin

               MaxLength := 0;
               Export := SeqAlgebra.FirstMember (TheHeap, Exports);

               loop
                  exit when SeqAlgebra.IsNullMember (Export);
                  ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);
                  LexTokenManager.LexStringToString (LexTokenManager.ConvertLexStringRef (ExportVal), ExportString);
                  MaxLength := Max (MaxLength, ExportString.Length);
                  Export := SeqAlgebra.NextMember (TheHeap, Export);
               end loop;

               return MaxLength;

            end LongestExport;

            function FromCol (StartCol    : EStrings.Positions;
                              ExportWidth : EStrings.Lengths) return EStrings.Positions
            --# global in SparkFormatCommandLineData.Content;
            is
               FromColumn : EStrings.Positions;
            begin
               if SparkFormatCommandLineData.Content.SeparatorIndent = SparkFormatCommandLineData.Inline then
                  FromColumn := (ExportCol (StartCol) + ExportWidth) + 1;
               else
                  FromColumn := SeparatorCol (StartCol);
               end if;
               return FromColumn;
            end FromCol;

            procedure WriteFrom (Output      : in SPARK_IO.File_Type;
                                 StartCol    : in EStrings.Positions;
                                 ExportWidth : in EStrings.Lengths)
            --# global in     Annotation.StartString;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                ExportWidth,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol;
            is
            begin
               if SparkFormatCommandLineData.Content.SeparatorIndent /= SparkFormatCommandLineData.Inline then
                  Annotation.Write (Output, StartCol);
               end if;
               SPARK_IO.Set_Col (Output, FromCol (StartCol, ExportWidth));
               SPARK_IO.Put_String (Output, "from", 0);
               -- If ImportIndent is not Inline, then don't put out a trailing space,
               -- since this will be rejected by GNAT -gnatyb style check mode.
               if SparkFormatCommandLineData.Content.ImportIndent = SparkFormatCommandLineData.Inline then
                  SPARK_IO.Put_String (Output, " ", 0);
               end if;
            end WriteFrom;

            function ImportCol (StartCol    : EStrings.Positions;
                                ExportWidth : EStrings.Lengths) return EStrings.Positions
            --# global in SparkFormatCommandLineData.Content;
            is
               ImportColumn : EStrings.Positions;
            begin
               if SparkFormatCommandLineData.Content.ImportIndent = SparkFormatCommandLineData.Inline then
                  ImportColumn := FromCol (StartCol, ExportWidth) + 5; -- length of "from "
               else
                  ImportColumn := DerivesCol (StartCol) + (SparkFormatCommandLineData.Content.ImportIndent - 1);
               end if;
               return ImportColumn;
            end ImportCol;

            procedure WriteImport (FirstImport : in out Boolean;
                                   Output      : in     SPARK_IO.File_Type;
                                   StartCol    : in     EStrings.Positions;
                                   ExportWidth : in     EStrings.Lengths;
                                   TheHeap     : in     Heap.HeapRecord;
                                   Import      : in     SeqAlgebra.MemberOfSeq)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives FirstImport       from * &
            --#         SPARK_IO.File_Sys from *,
            --#                                Annotation.StartString,
            --#                                ExportWidth,
            --#                                FirstImport,
            --#                                Import,
            --#                                LexTokenManager.StringTable,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol,
            --#                                TheHeap;
            is
               ImportVal    : Natural;
               ImportString : EStrings.T;
            begin

               if FirstImport then
                  FirstImport := False;
                  if SparkFormatCommandLineData.Content.ImportIndent /= SparkFormatCommandLineData.Inline then
                     Annotation.Write (Output, StartCol);
                  end if;
               else
                  SPARK_IO.Put_Line (Output, ",", 0);
                  Annotation.Write (Output, StartCol);
               end if;

               SPARK_IO.Set_Col (Output, ImportCol (StartCol, ExportWidth));
               ImportVal := SeqAlgebra.ValueOfMember (TheHeap, Import);
               LexTokenManager.LexStringToString (LexTokenManager.ConvertLexStringRef (ImportVal), ImportString);
               EStrings.PutString (Output, ImportString);

            end WriteImport;

            procedure WriteImports (Output       : in     SPARK_IO.File_Type;
                                    StartCol     : in     EStrings.Positions;
                                    ExportWidth  : in     EStrings.Lengths;
                                    Alphabetic   : in     Boolean;
                                    TheHeap      : in out Heap.HeapRecord;
                                    Imports      : in     SeqAlgebra.Seq)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     MultiplyToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Alphabetic,
            --#                                Annotation.StartString,
            --#                                ExportWidth,
            --#                                Imports,
            --#                                LexTokenManager.StringTable,
            --#                                MultiplyToken,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol,
            --#                                TheHeap &
            --#         TheHeap           from *,
            --#                                Imports,
            --#                                MultiplyToken;

            is
               Import        : SeqAlgebra.MemberOfSeq;
               FirstImport   : Boolean;
               TheIterator   : SPARKProgram.Iteration.Iterator;

            begin

               Import := SeqAlgebra.FirstMember (TheHeap, Imports);
               FirstImport := True;

               -- Imports might contain a "*", but it might not be the first
               -- member of the sequence.  SPARK _does_ require the "*" to appear
               -- first in the import list, so we scan for it alone on the first pass,
               -- write it out, then remove it from the sequence.
               -- Note that the sequence is disposed of after writing imports so it is
               -- OK to modify it here.

               loop
                  exit when SeqAlgebra.IsNullMember (Import);
                  if SeqAlgebra.ValueOfMember (TheHeap, Import) = MultiplyToken then
                     WriteImport (FirstImport, Output, StartCol, ExportWidth, TheHeap, Import);
                     SeqAlgebra.RemoveMember (TheHeap, Imports, MultiplyToken);
                     exit;
                  end if;
                  Import := SeqAlgebra.NextMember (TheHeap, Import);
               end loop;

               -- If '*' was the only import in the seq then the seq is now empty so we don't
               -- need to write any more imports. Need to guard against this case as the
               -- Initialise routine has an implicit precondition that the sequence is not
               -- empty.

               if not SeqAlgebra.IsEmptySeq (TheHeap, Imports) then

                  SPARKProgram.Iteration.Initialise (TheHeap, Imports, Alphabetic, TheIterator);
                  WriteImport (FirstImport,
                               Output,
                               StartCol,
                               ExportWidth,
                               TheHeap,
                               SPARKProgram.Iteration.CurrentMember (TheIterator));

                  loop

                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                     exit when SPARKProgram.Iteration.Complete (TheIterator);

                     WriteImport (FirstImport,
                                  Output,
                                  StartCol,
                                  ExportWidth,
                                  TheHeap,
                                  SPARKProgram.Iteration.CurrentMember (TheIterator));

                  end loop;

               end if;

            end WriteImports;

            procedure SubstituteSelfForMultiply (TheHeap                     : in out Heap.HeapRecord;
                                                 Exports                     : in     SeqAlgebra.Seq;
                                                 DependenciesWithMultiply    : in     RelationAlgebra.Relation;
                                                 DependenciesWithoutMultiply : in     RelationAlgebra.Relation)
            --# global in     MultiplyToken;
            --#        in out Statistics.TableUsage;
            --# derives Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    DependenciesWithMultiply,
            --#                                    DependenciesWithoutMultiply,
            --#                                    Exports,
            --#                                    MultiplyToken,
            --#                                    TheHeap;
            is
               Imports              : SeqAlgebra.Seq;
               Export, Import       : SeqAlgebra.MemberOfSeq;
               ExportVal, ImportVal : Natural;
            begin

               Export := SeqAlgebra.FirstMember (TheHeap, Exports);

               loop

                  exit when SeqAlgebra.IsNullMember (Export);

                  ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);
                  RelationAlgebra.RowExtraction (TheHeap, DependenciesWithMultiply, ExportVal, Imports);
                  Import := SeqAlgebra.FirstMember (TheHeap, Imports);

                  loop

                     exit when SeqAlgebra.IsNullMember (Import);

                     ImportVal := SeqAlgebra.ValueOfMember (TheHeap, Import);

                     if ImportVal = MultiplyToken then
                        ImportVal := ExportVal;
                     end if;

                     RelationAlgebra.InsertPair (TheHeap, DependenciesWithoutMultiply, ExportVal, ImportVal);
                     Import := SeqAlgebra.NextMember (TheHeap, Import);

                  end loop;

                  SeqAlgebra.DisposeOfSeq (TheHeap, Imports);
                  Export := SeqAlgebra.NextMember (TheHeap, Export);

               end loop;

            end SubstituteSelfForMultiply;

            procedure WriteSimilarClauses (Output       : in     SPARK_IO.File_Type;
                                           StartCol     : in     EStrings.Positions;
                                           ExportWidth  : in     EStrings.Lengths;
                                           FirstExport  : in out Boolean;
                                           Alphabetic   : in     Boolean;
                                           TheHeap      : in out Heap.HeapRecord;
                                           Exports      : in     SeqAlgebra.Seq;
                                           Dependencies : in     RelationAlgebra.Relation)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     MultiplyToken;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives FirstExport,
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Alphabetic,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    LexTokenManager.StringTable,
            --#                                    MultiplyToken,
            --#                                    RWNullToken,
            --#                                    TheHeap &
            --#         SPARK_IO.File_Sys     from *,
            --#                                    Alphabetic,
            --#                                    Annotation.StartString,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    ExportWidth,
            --#                                    FirstExport,
            --#                                    LexTokenManager.StringTable,
            --#                                    MultiplyToken,
            --#                                    Output,
            --#                                    RWNullToken,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    StartCol,
            --#                                    TheHeap;
            is

               SimilarExports, Imports : SeqAlgebra.Seq;
               Export, NextExport      : SeqAlgebra.MemberOfSeq;
               ExportVal               : Natural;
               CompressedExports       : RelationAlgebra.Relation;
               CompressedImports       : RelationAlgebra.Relation;
               CompressedIndex         : Natural;

               procedure FindSimilarExports (TheHeap        : in out Heap.HeapRecord;
                                             Export         : in     SeqAlgebra.MemberOfSeq;
                                             Dependencies   : in     RelationAlgebra.Relation;
                                             SimilarExports : in     SeqAlgebra.Seq;
                                             Imports        : in     SeqAlgebra.Seq)
               --# global in     RWNullToken;
               --#        in out Statistics.TableUsage;
               --# derives Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Dependencies,
               --#                                    Export,
               --#                                    Imports,
               --#                                    RWNullToken,
               --#                                    SimilarExports,
               --#                                    TheHeap;
               is

                  OtherImports   : SeqAlgebra.Seq;
                  OtherExport    : SeqAlgebra.MemberOfSeq;
                  OtherExportVal : Natural;

                  function IdenticalSequences (TheHeap : Heap.HeapRecord;
                                               S       : SeqAlgebra.Seq;
                                               T       : SeqAlgebra.Seq) return Boolean
                  is
                     M, N      : SeqAlgebra.MemberOfSeq;
                     Identical : Boolean;
                  begin

                     M := SeqAlgebra.FirstMember (TheHeap, S);
                     N := SeqAlgebra.FirstMember (TheHeap, T);
                     Identical := True;

                     loop

                        exit when SeqAlgebra.IsNullMember (M) or SeqAlgebra.IsNullMember (N);

                        if SeqAlgebra.ValueOfMember (TheHeap, M) /= SeqAlgebra.ValueOfMember (TheHeap, N) then
                           Identical := False;
                           exit;
                        end if;

                        M := SeqAlgebra.NextMember (TheHeap, M);
                        N := SeqAlgebra.NextMember (TheHeap, N);

                     end loop;

                     return Identical and SeqAlgebra.IsNullMember (M) and SeqAlgebra.IsNullMember (N);

                  end IdenticalSequences;

               begin

                  OtherExport := SeqAlgebra.NextMember (TheHeap, Export);

                  loop

                     exit when SeqAlgebra.IsNullMember (OtherExport);

                     OtherExportVal := SeqAlgebra.ValueOfMember (TheHeap, OtherExport);

                     if OtherExportVal /= RWNullToken then

                        RelationAlgebra.RowExtraction (TheHeap, Dependencies, OtherExportVal, OtherImports);

                        if IdenticalSequences (TheHeap, Imports, OtherImports) then
                           SeqAlgebra.AddMember (TheHeap, SimilarExports, OtherExportVal);
                        end if;

                        SeqAlgebra.DisposeOfSeq (TheHeap, OtherImports);
                     end if;

                     OtherExport := SeqAlgebra.NextMember (TheHeap, OtherExport);

                  end loop;

               end FindSimilarExports;

               -- Non-alphabetic version of WriteDependencies. Difference is that it
               -- takes Export and SimilarExports as separate parameters, whereas the
               -- alphabetic version just takes the whole lot as one sequence.
               -- This is only here to make sure that we preserve the old behaviour
               -- when non-alphabetic output is chosen. Should be possible to modify the
               -- other WriteDependencies so that it behaves in the same way as this
               -- when non-alphabetic output is requested.
               procedure WriteDependenciesNonAlpha (Output         : in     SPARK_IO.File_Type;
                                                    StartCol       : in     EStrings.Positions;
                                                    ExportWidth    : in     EStrings.Lengths;
                                                    FirstExport    : in out Boolean;
                                                    TheHeap        : in out Heap.HeapRecord;
                                                    Export         : in     SeqAlgebra.MemberOfSeq;
                                                    SimilarExports : in     SeqAlgebra.Seq;
                                                    Imports        : in     SeqAlgebra.Seq)
               --# global in     Annotation.StartString;
               --#        in     LexTokenManager.StringTable;
               --#        in     MultiplyToken;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out SPARK_IO.File_Sys;
               --# derives FirstExport       from * &
               --#         SPARK_IO.File_Sys from *,
               --#                                Annotation.StartString,
               --#                                Export,
               --#                                ExportWidth,
               --#                                FirstExport,
               --#                                Imports,
               --#                                LexTokenManager.StringTable,
               --#                                MultiplyToken,
               --#                                Output,
               --#                                SimilarExports,
               --#                                SparkFormatCommandLineData.Content,
               --#                                StartCol,
               --#                                TheHeap &
               --#         TheHeap           from *,
               --#                                Imports,
               --#                                MultiplyToken;
               is
                  SimilarExport : SeqAlgebra.MemberOfSeq;
               begin

                  WriteExport (FirstExport, Output, StartCol, TheHeap, Export);
                  SimilarExport := SeqAlgebra.FirstMember (TheHeap, SimilarExports);

                  loop
                     exit when SeqAlgebra.IsNullMember (SimilarExport);
                     WriteSimilarExport (Output, StartCol, TheHeap, SimilarExport);
                     SimilarExport := SeqAlgebra.NextMember (TheHeap, SimilarExport);
                  end loop;

                  WriteFrom (Output, StartCol, ExportWidth);
                  WriteImports (Output, StartCol, ExportWidth, False, TheHeap, Imports);

               end WriteDependenciesNonAlpha;

               procedure RemoveMembers (TheHeap : in out Heap.HeapRecord;
                                        A, B    : in     SeqAlgebra.Seq)
               --# derives TheHeap from *,
               --#                      A,
               --#                      B;
               is
                  M, Next  : SeqAlgebra.MemberOfSeq;
                  ValueOfM : Natural;
               begin
                  M := SeqAlgebra.FirstMember (TheHeap, A);
                  loop
                     exit when SeqAlgebra.IsNullMember (M);
                     ValueOfM := SeqAlgebra.ValueOfMember (TheHeap, M);
                     if SeqAlgebra.IsMember (TheHeap, B, ValueOfM) then
                        Next := SeqAlgebra.NextMember (TheHeap, M);
                        SeqAlgebra.RemoveMember (TheHeap, A, ValueOfM);
                        M := Next;
                     else
                        M := SeqAlgebra.NextMember (TheHeap, M);
                     end if;
                  end loop;
               end RemoveMembers;

               procedure SwapSequence (A, B : in out SeqAlgebra.Seq)
               --# derives A from B &
               --#         B from A;
               is
                  Temp : SeqAlgebra.Seq;
               begin
                  Temp := A;
                  A := B;
                  B := Temp;
               end SwapSequence;

               function SequenceLength (TheHeap : Heap.HeapRecord; S : SeqAlgebra.Seq) return Natural
               is
                  M      : SeqAlgebra.MemberOfSeq;
                  Length : Natural;
               begin

                  M := SeqAlgebra.FirstMember (TheHeap, S);
                  Length := 0;

                  loop
                     exit when SeqAlgebra.IsNullMember (M);
                     Length := Length + 1;
                     M := SeqAlgebra.NextMember (TheHeap, M);
                  end loop;

                  return Length;

               end SequenceLength;

               procedure TryAddingSelfDependencies (TheHeap        : in out Heap.HeapRecord;
                                                    Export         : in     SeqAlgebra.MemberOfSeq;
                                                    Dependencies   : in     RelationAlgebra.Relation;
                                                    SimilarExports : in out SeqAlgebra.Seq;
                                                    Imports        : in out SeqAlgebra.Seq)
               --# global in     MultiplyToken;
               --#        in     RWNullToken;
               --#        in out Statistics.TableUsage;
               --# derives Imports,
               --#         SimilarExports,
               --#         Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Dependencies,
               --#                                    Export,
               --#                                    Imports,
               --#                                    MultiplyToken,
               --#                                    RWNullToken,
               --#                                    SimilarExports,
               --#                                    TheHeap;
               is

                  OtherExport : SeqAlgebra.MemberOfSeq;

                  procedure TryAddingSelfDependency (TheHeap        : in out Heap.HeapRecord;
                                                     Export         : in     SeqAlgebra.MemberOfSeq;
                                                     OtherExport    : in     SeqAlgebra.MemberOfSeq;
                                                     Dependencies   : in     RelationAlgebra.Relation;
                                                     SimilarExports : in out SeqAlgebra.Seq;
                                                     Imports        : in out SeqAlgebra.Seq)
                  --# global in     RWNullToken;
                  --#        in out Statistics.TableUsage;
                  --# derives Imports,
                  --#         TheHeap               from Dependencies,
                  --#                                    Export,
                  --#                                    Imports,
                  --#                                    OtherExport,
                  --#                                    RWNullToken,
                  --#                                    SimilarExports,
                  --#                                    TheHeap &
                  --#         SimilarExports,
                  --#         Statistics.TableUsage from *,
                  --#                                    Dependencies,
                  --#                                    Export,
                  --#                                    OtherExport,
                  --#                                    RWNullToken,
                  --#                                    SimilarExports,
                  --#                                    TheHeap;
                  is
                     ExportVal, OtherExportVal         : Natural;
                     TrialSimilarExports, TrialImports : SeqAlgebra.Seq;
                     TrialDependencies                 : RelationAlgebra.Relation;
                  begin

                     ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);
                     OtherExportVal := SeqAlgebra.ValueOfMember (TheHeap, OtherExport);

                     RelationAlgebra.CreateRelation (TheHeap, TrialDependencies);
                     RelationAlgebra.AugmentRelation (TheHeap, TrialDependencies, Dependencies);
                     RelationAlgebra.InsertPair (TheHeap, TrialDependencies, OtherExportVal, OtherExportVal);

                     SeqAlgebra.CreateSeq (TheHeap, TrialSimilarExports);
                     RelationAlgebra.RowExtraction (TheHeap, TrialDependencies, ExportVal, TrialImports);

                     FindSimilarExports (TheHeap, Export, TrialDependencies, TrialSimilarExports, TrialImports);

                     if SequenceLength (TheHeap, TrialSimilarExports) > SequenceLength (TheHeap, SimilarExports) then
                        SwapSequence (SimilarExports, TrialSimilarExports);
                        SwapSequence (Imports, TrialImports);
                        RelationAlgebra.InsertPair (TheHeap, Dependencies, OtherExportVal, OtherExportVal);
                     end if;

                     SeqAlgebra.DisposeOfSeq (TheHeap, TrialSimilarExports);
                     SeqAlgebra.DisposeOfSeq (TheHeap, TrialImports);
                     RelationAlgebra.DisposeOfRelation (TheHeap, TrialDependencies);

                  end TryAddingSelfDependency;

               begin

                  if SeqAlgebra.IsMember (TheHeap, Imports, MultiplyToken) then

                     TryAddingSelfDependency (TheHeap, Export, Export, Dependencies, SimilarExports, Imports);
                     OtherExport := SeqAlgebra.NextMember (TheHeap, Export);

                     loop
                        exit when SeqAlgebra.IsNullMember (OtherExport);
                        TryAddingSelfDependency (TheHeap, Export, OtherExport, Dependencies, SimilarExports, Imports);
                        OtherExport := SeqAlgebra.NextMember (TheHeap, OtherExport);
                     end loop;

                  end if;

               end TryAddingSelfDependencies;

               procedure TryWithoutMultiply (TheHeap        : in out Heap.HeapRecord;
                                             Export         : in     SeqAlgebra.MemberOfSeq;
                                             Exports        : in     SeqAlgebra.Seq;
                                             Dependencies   : in     RelationAlgebra.Relation;
                                             SimilarExports : in out SeqAlgebra.Seq;
                                             Imports        : in out SeqAlgebra.Seq)
               --# global in     MultiplyToken;
               --#        in     RWNullToken;
               --#        in out Statistics.TableUsage;
               --# derives Imports,
               --#         TheHeap               from Dependencies,
               --#                                    Export,
               --#                                    Exports,
               --#                                    Imports,
               --#                                    MultiplyToken,
               --#                                    RWNullToken,
               --#                                    SimilarExports,
               --#                                    TheHeap &
               --#         SimilarExports,
               --#         Statistics.TableUsage from *,
               --#                                    Dependencies,
               --#                                    Export,
               --#                                    Exports,
               --#                                    MultiplyToken,
               --#                                    RWNullToken,
               --#                                    TheHeap;
               is
                  TrialSimilarExports, TrialImports : SeqAlgebra.Seq;
                  ExportVal                         : Natural;
                  DependenciesWithoutMultiply       : RelationAlgebra.Relation;
               begin

                  ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);

                  RelationAlgebra.CreateRelation (TheHeap, DependenciesWithoutMultiply);
                  SubstituteSelfForMultiply (TheHeap, Exports, Dependencies, DependenciesWithoutMultiply);
                  SeqAlgebra.CreateSeq (TheHeap, TrialSimilarExports);
                  RelationAlgebra.RowExtraction (TheHeap, DependenciesWithoutMultiply, ExportVal, TrialImports);
                  FindSimilarExports (TheHeap, Export, DependenciesWithoutMultiply, TrialSimilarExports, TrialImports);

                  if SequenceLength (TheHeap, TrialSimilarExports) > SequenceLength (TheHeap, SimilarExports) then
                     SwapSequence (SimilarExports, TrialSimilarExports);
                     SwapSequence (Imports, TrialImports);
                  end if;

                  SeqAlgebra.DisposeOfSeq (TheHeap, TrialSimilarExports);
                  SeqAlgebra.DisposeOfSeq (TheHeap, TrialImports);
                  RelationAlgebra.DisposeOfRelation (TheHeap, DependenciesWithoutMultiply);

               end TryWithoutMultiply;

               procedure BuildCompressedDependencies (TheHeap           : in out Heap.HeapRecord;
                                                      Dependencies      : in     RelationAlgebra.Relation;
                                                      Exports           : in     SeqAlgebra.Seq;
                                                      CompressedExports :    out RelationAlgebra.Relation;
                                                      CompressedImports :    out RelationAlgebra.Relation;
                                                      CompressedIndex   :    out Natural)

               --# global in     MultiplyToken;
               --#        in     RWNullToken;
               --#        in out Statistics.TableUsage;
               --# derives CompressedExports,
               --#         CompressedImports     from TheHeap &
               --#         CompressedIndex       from Dependencies,
               --#                                    Exports,
               --#                                    MultiplyToken,
               --#                                    RWNullToken,
               --#                                    TheHeap &
               --#         Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Dependencies,
               --#                                    Exports,
               --#                                    MultiplyToken,
               --#                                    RWNullToken,
               --#                                    TheHeap;

               is

                  ExportVal      : Natural;
                  Export         : SeqAlgebra.MemberOfSeq;
                  NextExport     : SeqAlgebra.MemberOfSeq;
                  SimilarExports : SeqAlgebra.Seq;
                  Imports        : SeqAlgebra.Seq;

               begin
                  -- Create relations to store sequences of exports and their imports.
                  -- These two relations will be indexed by CompressedIndex, so for any given value
                  -- of CompressedIndex one relation gives the sequence of exports and the other
                  -- provides the sequence of imports from which they are derived.
                  RelationAlgebra.CreateRelation (TheHeap, CompressedExports);
                  RelationAlgebra.CreateRelation (TheHeap, CompressedImports);
                  CompressedIndex := Natural'First;

                  Export := SeqAlgebra.FirstMember (TheHeap, Exports);

                  -- Loop condition avoids potential RTE. CompressedIndex can never exceed bounds of
                  -- Natural in practice because it will never exceed the number of exports.
                  while CompressedIndex < Natural'Last loop

                     exit when SeqAlgebra.IsNullMember (Export);

                     -- Look for other exports with the same dependency list, and store them
                     -- in the sequence SimilarExports.
                     ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);

                     if ExportVal /= RWNullToken then -- null derives comes last so ignore it here

                        SeqAlgebra.CreateSeq (TheHeap, SimilarExports);
                        RelationAlgebra.RowExtraction (TheHeap, Dependencies, ExportVal, Imports);
                        FindSimilarExports (TheHeap, Export, Dependencies, SimilarExports, Imports);
                        TryAddingSelfDependencies (TheHeap, Export, Dependencies, SimilarExports, Imports);
                        TryWithoutMultiply (TheHeap, Export, Exports, Dependencies, SimilarExports, Imports);

                        -- Rather than write the dependencies out directly here (in whatever order they
                        -- happen to be in) we build a new, compressed, dependency relation to be written
                        -- out (in a given order) once it is complete.
                        --
                        -- Note that if an export has no similar exports it *is* added to the compressed
                        -- sequence but it is *not* removed from the sequence of exports. (The implementations
                        -- of FindSimilarExports, TryAddingSelfDependencies and TryWithoutMultiply depend on
                        -- this being the case.) Consequently, the sequence of exports should not be used
                        -- after calling this subprogram.

                        CompressedIndex := CompressedIndex + 1;
                        RelationAlgebra.InsertPair (TheHeap, CompressedExports, CompressedIndex, ExportVal);
                        RelationAlgebra.AddRow (TheHeap, CompressedImports, CompressedIndex, Imports);

                        if SeqAlgebra.IsEmptySeq (TheHeap, SimilarExports) then
                           Export := SeqAlgebra.NextMember (TheHeap, Export);
                        else
                           RelationAlgebra.AddRow (TheHeap, CompressedExports, CompressedIndex, SimilarExports);
                           RemoveMembers (TheHeap, Exports, SimilarExports);
                           NextExport := SeqAlgebra.NextMember (TheHeap, Export);
                           SeqAlgebra.RemoveMember (TheHeap, Exports, ExportVal);
                           Export := NextExport;
                        end if;

                        SeqAlgebra.DisposeOfSeq (TheHeap, SimilarExports);
                        SeqAlgebra.DisposeOfSeq (TheHeap, Imports);

                     else
                        Export := SeqAlgebra.NextMember (TheHeap, Export); -- if null token move on to next item
                     end if;

                  end loop;

               end BuildCompressedDependencies;

               procedure WriteDependenciesAlphabetic (TheHeap           : in out Heap.HeapRecord;
                                                      Output            : in     SPARK_IO.File_Type;
                                                      StartCol          : in     EStrings.Positions;
                                                      ExportWidth       : in     EStrings.Lengths;
                                                      FirstExport       : in out Boolean;
                                                      Alphabetic        : in     Boolean;
                                                      CompressedExports : in     RelationAlgebra.Relation;
                                                      CompressedImports : in     RelationAlgebra.Relation;
                                                      CompressedIndex   : in     Natural)

               --# global in     Annotation.StartString;
               --#        in     LexTokenManager.StringTable;
               --#        in     MultiplyToken;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives FirstExport,
               --#         Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Alphabetic,
               --#                                    CompressedExports,
               --#                                    CompressedImports,
               --#                                    CompressedIndex,
               --#                                    LexTokenManager.StringTable,
               --#                                    MultiplyToken,
               --#                                    TheHeap &
               --#         SPARK_IO.File_Sys     from *,
               --#                                    Alphabetic,
               --#                                    Annotation.StartString,
               --#                                    CompressedExports,
               --#                                    CompressedImports,
               --#                                    CompressedIndex,
               --#                                    ExportWidth,
               --#                                    FirstExport,
               --#                                    LexTokenManager.StringTable,
               --#                                    MultiplyToken,
               --#                                    Output,
               --#                                    SparkFormatCommandLineData.Content,
               --#                                    StartCol,
               --#                                    TheHeap;

               is

                  RelationsWritten : SeqAlgebra.Seq;
                  LocalIndex       : Natural;
                  RelationCount    : Natural;
                  FirstAlphaIndex  : Natural;
                  FirstAlphaExport : LexTokenManager.LexString;
                  ExportsToWrite   : SeqAlgebra.Seq;
                  ImportsToWrite   : SeqAlgebra.Seq;
                  TheIterator      : SPARKProgram.Iteration.Iterator;
                  CurrentString    : LexTokenManager.LexString;

                  procedure WriteDependencies (Output         : in     SPARK_IO.File_Type;
                                               StartCol       : in     EStrings.Positions;
                                               ExportWidth    : in     EStrings.Lengths;
                                               FirstExport    : in out Boolean;
                                               Alphabetic     : in     Boolean;
                                               TheHeap        : in out Heap.HeapRecord;
                                               Exports        : in     SeqAlgebra.Seq;
                                               Imports        : in     SeqAlgebra.Seq)
                  --# global in     Annotation.StartString;
                  --#        in     LexTokenManager.StringTable;
                  --#        in     MultiplyToken;
                  --#        in     SparkFormatCommandLineData.Content;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives FirstExport       from * &
                  --#         SPARK_IO.File_Sys from *,
                  --#                                Alphabetic,
                  --#                                Annotation.StartString,
                  --#                                Exports,
                  --#                                ExportWidth,
                  --#                                FirstExport,
                  --#                                Imports,
                  --#                                LexTokenManager.StringTable,
                  --#                                MultiplyToken,
                  --#                                Output,
                  --#                                SparkFormatCommandLineData.Content,
                  --#                                StartCol,
                  --#                                TheHeap &
                  --#         TheHeap           from *,
                  --#                                Imports,
                  --#                                MultiplyToken;
                  is

                     TheIterator   : SPARKProgram.Iteration.Iterator;

                  begin

                     -- Dependency consists of <exports> from <imports>, so start with
                     -- the exports.
                     SPARKProgram.Iteration.Initialise (TheHeap, Exports, Alphabetic, TheIterator);

                     WriteExport (FirstExport,
                                  Output,
                                  StartCol,
                                  TheHeap,
                                  SPARKProgram.Iteration.CurrentMember (TheIterator));

                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                     -- if there are more exports derived from the same imports, write them out
                     while not SPARKProgram.Iteration.Complete (TheIterator) loop

                        WriteSimilarExport (Output,
                                            StartCol,
                                            TheHeap,
                                            SPARKProgram.Iteration.CurrentMember (TheIterator));

                        SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                     end loop;

                     -- from
                     WriteFrom (Output, StartCol, ExportWidth);

                     -- the imports
                     WriteImports (Output      => Output,
                                   StartCol    => StartCol,
                                   ExportWidth => ExportWidth,
                                   Alphabetic  => Alphabetic,
                                   TheHeap     => TheHeap,
                                   Imports     => Imports);

                  end WriteDependencies;

               begin

                  -- Need to find the alphabetical first member out of all the export sequences in
                  -- the ExportsToWrite relation. The corresponding dependency can then be written
                  -- out, after which it can be disposed of. Repeat until all relations have been
                  -- written.

                  SeqAlgebra.CreateSeq (TheHeap, RelationsWritten);
                  RelationCount := CompressedIndex; -- Keep a record of the total number of relations

                  loop -- until all exports have been written

                     FirstAlphaIndex := Natural'First;
                     LocalIndex := RelationCount;
                     FirstAlphaExport := LexTokenManager.NullString;

                     -- Go through all the export sequences (skipping those we have already written) looking
                     -- for the first export in alphabetical order.
                     while LocalIndex > Natural'First loop

                        if not SeqAlgebra.IsMember (TheHeap, RelationsWritten, LocalIndex) then

                           RelationAlgebra.RowExtraction (TheHeap, CompressedExports, LocalIndex, ExportsToWrite);

                           -- Find the first member of the current export sequence.
                           -- In this case we use initialise to find us the first member but we never go on to iterate
                           -- over the list.
                           SPARKProgram.Iteration.Initialise (TheHeap,
                                                              ExportsToWrite,
                                                              Alphabetic,
                                                              TheIterator);

                           -- If this is the first iteration then the item just returned must be the best match so far.
                           -- Otherwise, if it is better than our current best match then this export sequence becomes the one to write

                           CurrentString := SPARKProgram.Iteration.CurrentString (TheIterator);

                           if FirstAlphaIndex = Natural'First or else -- always do it on first iteration
                              LexTokenManager.LexStringCaseInsensitiveCompare (CurrentString, FirstAlphaExport) = LexTokenManager.StrFirst then

                              FirstAlphaExport := CurrentString;
                              FirstAlphaIndex := LocalIndex;

                           end if;

                        end if;

                        LocalIndex := LocalIndex - 1;

                     end loop;

                     if FirstAlphaIndex = Natural'First then -- defensive - should always be false
                        exit;
                     end if;

                     -- FirstAlphaIndex gives us the relation index for the next dependency to be written
                     -- Write it out, then add it to the sequence of relations that have been written.
                     RelationAlgebra.RowExtraction (TheHeap, CompressedExports, FirstAlphaIndex, ExportsToWrite);
                     RelationAlgebra.RowExtraction (TheHeap, CompressedImports, FirstAlphaIndex, ImportsToWrite);

                     WriteDependencies (Output      => Output,
                                        StartCol    => StartCol,
                                        ExportWidth => ExportWidth,
                                        FirstExport => FirstExport,
                                        Alphabetic  => Alphabetic,
                                        TheHeap     => TheHeap,
                                        Exports     => ExportsToWrite,
                                        Imports     => ImportsToWrite);

                     SeqAlgebra.DisposeOfSeq (TheHeap, ExportsToWrite);
                     SeqAlgebra.DisposeOfSeq (TheHeap, ImportsToWrite);

                     SeqAlgebra.AddMember (TheHeap, RelationsWritten, FirstAlphaIndex);

                     -- Exit if all relations have now been added to RelationsWritten
                     exit when SeqAlgebra.Length (TheHeap, RelationsWritten) >= RelationCount;

                  end loop; -- until all relations written

                  SeqAlgebra.DisposeOfSeq (TheHeap, RelationsWritten);

               end WriteDependenciesAlphabetic;

            begin -- WriteSimilarClauses

               if Alphabetic then

                  -- Compress the dependencies if possible. Store the results in CompressedExports
                  -- and CompressedImports, indexed by CompressedIndex.
                  BuildCompressedDependencies (TheHeap           => TheHeap,
                                               Dependencies      => Dependencies,
                                               Exports           => Exports,
                                               CompressedExports => CompressedExports,
                                               CompressedImports => CompressedImports,
                                               CompressedIndex   => CompressedIndex);

                  -- Extract the compressed dependencies and write them out.
                  WriteDependenciesAlphabetic (TheHeap           => TheHeap,
                                               Output            => Output,
                                               StartCol          => StartCol,
                                               ExportWidth       => ExportWidth,
                                               FirstExport       => FirstExport,
                                               Alphabetic        => Alphabetic, -- redundant at present
                                               CompressedExports => CompressedExports,
                                               CompressedImports => CompressedImports,
                                               CompressedIndex   => CompressedIndex);

               else -- non alphabetic case

                  Export := SeqAlgebra.FirstMember (TheHeap, Exports);

                  loop

                     exit when SeqAlgebra.IsNullMember (Export);

                     ExportVal := SeqAlgebra.ValueOfMember (TheHeap, Export);
                     SeqAlgebra.CreateSeq (TheHeap, SimilarExports);
                     RelationAlgebra.RowExtraction (TheHeap, Dependencies, ExportVal, Imports);
                     FindSimilarExports (TheHeap, Export, Dependencies, SimilarExports, Imports);
                     TryAddingSelfDependencies (TheHeap, Export, Dependencies, SimilarExports, Imports);
                     TryWithoutMultiply (TheHeap, Export, Exports, Dependencies, SimilarExports, Imports);

                     -- The difference between this and the alphabetic version above is that here we have an
                     -- export and a sequence of similar exports, whereas above we just have a single sequence
                     -- of similar exports (which includes whatever 'the' export happened to be).

                     if SeqAlgebra.IsEmptySeq (TheHeap, SimilarExports) then
                        Export := SeqAlgebra.NextMember (TheHeap, Export);
                     else
                        WriteDependenciesNonAlpha (Output,
                                                   StartCol,
                                                   ExportWidth,
                                                   FirstExport,
                                                   TheHeap,
                                                   Export,
                                                   SimilarExports,
                                                   Imports);
                        RemoveMembers (TheHeap, Exports, SimilarExports);
                        NextExport := SeqAlgebra.NextMember (TheHeap, Export);
                        SeqAlgebra.RemoveMember (TheHeap, Exports, ExportVal);
                        Export := NextExport;
                     end if;

                     SeqAlgebra.DisposeOfSeq (TheHeap, SimilarExports);
                     SeqAlgebra.DisposeOfSeq (TheHeap, Imports);

                  end loop;

               end if;

            end WriteSimilarClauses;

            procedure WriteRemainingClauses (Output       : in     SPARK_IO.File_Type;
                                             StartCol     : in     EStrings.Positions;
                                             ExportWidth  : in     EStrings.Lengths;
                                             FirstExport  : in out Boolean;
                                             Alphabetic   : in     Boolean;
                                             TheHeap      : in out Heap.HeapRecord;
                                             Exports      : in     SeqAlgebra.Seq;
                                             Dependencies : in     RelationAlgebra.Relation)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     MultiplyToken;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives FirstExport,
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Alphabetic,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    LexTokenManager.StringTable,
            --#                                    MultiplyToken,
            --#                                    RWNullToken,
            --#                                    TheHeap &
            --#         SPARK_IO.File_Sys     from *,
            --#                                    Alphabetic,
            --#                                    Annotation.StartString,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    ExportWidth,
            --#                                    FirstExport,
            --#                                    LexTokenManager.StringTable,
            --#                                    MultiplyToken,
            --#                                    Output,
            --#                                    RWNullToken,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    StartCol,
            --#                                    TheHeap;
            is

               ExportVal   : Natural;
               Imports     : SeqAlgebra.Seq;
               TheIterator : SPARKProgram.Iteration.Iterator;

            begin

               -- Write out the rest of the dependency clauses (ie those in which there is
               -- just a single 'normal' export in each clause, rather than a group of exports or
               -- a '*' or 'null'.

               -- This will initialize TheIterator to the first item in the sequence of exports
               SPARKProgram.Iteration.Initialise (TheHeap,
                                                  Exports,
                                                  Alphabetic,
                                                  TheIterator);

               loop

                  exit when SPARKProgram.Iteration.Complete (TheIterator);

                  ExportVal := SeqAlgebra.ValueOfMember (TheHeap,
                                                         SPARKProgram.Iteration.CurrentMember (TheIterator));

                  -- Null dependency clause is done separately at the end so guard against it here
                  if ExportVal /= RWNullToken then

                     -- export
                     WriteExport (FirstExport,
                                  Output,
                                  StartCol,
                                  TheHeap,
                                  SPARKProgram.Iteration.CurrentMember (TheIterator));

                     WriteFrom (Output, StartCol, ExportWidth); -- from

                     -- imports
                     RelationAlgebra.RowExtraction (TheHeap, Dependencies, ExportVal, Imports);
                     WriteImports (Output      => Output,
                                   StartCol    => StartCol,
                                   ExportWidth => ExportWidth,
                                   Alphabetic  => Alphabetic,
                                   TheHeap     => TheHeap,
                                   Imports     => Imports);
                     SeqAlgebra.DisposeOfSeq (TheHeap, Imports);

                  end if;

                  SPARKProgram.Iteration.Next (TheHeap, TheIterator);

               end loop;

            end WriteRemainingClauses;

            procedure WriteClauses (Output       : in     SPARK_IO.File_Type;
                                    StartCol     : in     EStrings.Positions;
                                    ExportWidth  : in     EStrings.Lengths;
                                    FirstExport  : in out Boolean;
                                    Alphabetic   : in     Boolean;
                                    TheHeap      : in out Heap.HeapRecord;
                                    Exports      : in     SeqAlgebra.Seq;
                                    Dependencies : in     RelationAlgebra.Relation)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     MultiplyToken;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives FirstExport,
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Alphabetic,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    LexTokenManager.StringTable,
            --#                                    MultiplyToken,
            --#                                    RWNullToken,
            --#                                    TheHeap &
            --#         SPARK_IO.File_Sys     from *,
            --#                                    Alphabetic,
            --#                                    Annotation.StartString,
            --#                                    Dependencies,
            --#                                    Exports,
            --#                                    ExportWidth,
            --#                                    FirstExport,
            --#                                    LexTokenManager.StringTable,
            --#                                    MultiplyToken,
            --#                                    Output,
            --#                                    RWNullToken,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    StartCol,
            --#                                    TheHeap;
            is

               Imports                     : SeqAlgebra.Seq;
               ExportVal                   : Natural;
               DependenciesWithoutMultiply : RelationAlgebra.Relation;
               TheIterator                 : SPARKProgram.Iteration.Iterator;

            begin

               -- This writes out the clauses in uncompressed form, ie no lists of exports sharing the
               -- same list of imports, and no '*'.

               RelationAlgebra.CreateRelation (TheHeap, DependenciesWithoutMultiply);
               SubstituteSelfForMultiply (TheHeap, Exports, Dependencies, DependenciesWithoutMultiply);

               SPARKProgram.Iteration.Initialise (TheHeap,
                                                  Exports,
                                                  Alphabetic,
                                                  TheIterator);

               while not SPARKProgram.Iteration.Complete (TheIterator) loop

                  ExportVal := SeqAlgebra.ValueOfMember (TheHeap,
                                                         SPARKProgram.Iteration.CurrentMember (TheIterator));

                  -- Null dependency clause must come last so don't write it here
                  if ExportVal /= RWNullToken then

                     -- export
                     WriteExport (FirstExport,
                                  Output,
                                  StartCol,
                                  TheHeap,
                                  SPARKProgram.Iteration.CurrentMember (TheIterator));

                     WriteFrom (Output, StartCol, ExportWidth); --from

                     -- imports
                     RelationAlgebra.RowExtraction (TheHeap, DependenciesWithoutMultiply, ExportVal, Imports);
                     WriteImports (Output      => Output,
                                   StartCol    => StartCol,
                                   ExportWidth => ExportWidth,
                                   Alphabetic  => Alphabetic,
                                   TheHeap     => TheHeap,
                                   Imports     => Imports);
                     SeqAlgebra.DisposeOfSeq (TheHeap, Imports);
                  end if;

                  SPARKProgram.Iteration.Next (TheHeap, TheIterator);

               end loop;

               RelationAlgebra.DisposeOfRelation (TheHeap, DependenciesWithoutMultiply);

            end WriteClauses;

            procedure WriteNullImportList (Output       : in     SPARK_IO.File_Type;
                                           StartCol     : in     EStrings.Positions;
                                           ExportWidth  : in     EStrings.Lengths;
                                           FirstExport  : in     Boolean;
                                           Alphabetic   : in     Boolean;
                                           TheHeap      : in out Heap.HeapRecord;
                                           Dependencies : in     RelationAlgebra.Relation)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     RWNullToken;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives SPARK_IO.File_Sys     from *,
            --#                                    Alphabetic,
            --#                                    Annotation.StartString,
            --#                                    Dependencies,
            --#                                    ExportWidth,
            --#                                    FirstExport,
            --#                                    LexTokenManager.StringTable,
            --#                                    Output,
            --#                                    RWNullToken,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    StartCol,
            --#                                    TheHeap &
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Dependencies,
            --#                                    RWNullToken,
            --#                                    TheHeap;
            is
               Imports     : SeqAlgebra.Seq;
               Import      : SeqAlgebra.MemberOfSeq;
               FirstImport : Boolean;
               TheIterator : SPARKProgram.Iteration.Iterator;

            begin

               RelationAlgebra.RowExtraction (TheHeap, Dependencies, RWNullToken, Imports);

               -- This procedure can be called even if there is no null import list. If the
               -- sequence is null then don't write it out.
               Import := SeqAlgebra.FirstMember (TheHeap, Imports);

               if not SeqAlgebra.IsNullMember (Import) then

                  SPARKProgram.Iteration.Initialise (TheHeap,
                                                     Imports,
                                                     Alphabetic,
                                                     TheIterator);

                  WriteNull (FirstExport, Output, StartCol); -- null
                  WriteFrom (Output, StartCol, ExportWidth); -- from

                  -- Now write out the list of imports in the requested order
                  FirstImport := True;

                  WriteImport (FirstImport,
                               Output,
                               StartCol,
                               ExportWidth,
                               TheHeap,
                               SPARKProgram.Iteration.CurrentMember (TheIterator));

                  SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  while not SPARKProgram.Iteration.Complete (TheIterator) loop

                     WriteImport (FirstImport,
                                  Output,
                                  StartCol,
                                  ExportWidth,
                                  TheHeap,
                                  SPARKProgram.Iteration.CurrentMember (TheIterator));

                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  end loop;

               end if;

               SeqAlgebra.DisposeOfSeq (TheHeap, Imports);

            end WriteNullImportList;

         begin

            if OK then

               Annotation.Write (Output, StartCol);
               WriteDerives (Output, StartCol);
               FirstExport := True;
               ExportWidth := LongestExport (TheHeap, Exports);

               case SparkFormatCommandLineData.Content.Operation is

                  when SparkFormatCommandLineData.Compress =>

                     WriteSimilarClauses (Output,
                                          StartCol,
                                          ExportWidth,
                                          FirstExport,
                                          SparkFormatCommandLineData.Content.AlphabeticOrdering,
                                          TheHeap,
                                          Exports,
                                          Dependencies);

                     -- If output is alphabetic then all clauses will have been written out by
                     -- WriteSimilarClauses so there will be no remaining clauses.
                     if not SparkFormatCommandLineData.Content.AlphabeticOrdering then
                        WriteRemainingClauses (Output,
                                               StartCol,
                                               ExportWidth,
                                               FirstExport,
                                               SparkFormatCommandLineData.Content.AlphabeticOrdering, -- redundant
                                               TheHeap,
                                               Exports,
                                               Dependencies);
                     end if;

                  when SparkFormatCommandLineData.Expand =>

                     WriteClauses (Output,
                                   StartCol,
                                   ExportWidth,
                                   FirstExport,
                                   SparkFormatCommandLineData.Content.AlphabeticOrdering,
                                   TheHeap,
                                   Exports,
                                   Dependencies);

               end case;

               WriteNullImportList (Output,
                                    StartCol,
                                    ExportWidth,
                                    FirstExport,
                                    SparkFormatCommandLineData.Content.AlphabeticOrdering,
                                    TheHeap,
                                    Dependencies);

               SPARK_IO.Put_Line (Output, ";", 0);

            else

               File_IO.Reset (TemporaryFile);
               SPARKProgram.Copy (TemporaryFile, Output);

            end if;

            SeqAlgebra.DisposeOfSeq (TheHeap, Exports);
            RelationAlgebra.DisposeOfRelation (TheHeap, Dependencies);
            File_IO.Delete (TemporaryFile);

         end Reformat;

      end DerivesList;

      package body DeclareList
      is
         -- "declare" has 7 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(7,
                                        EStrings.Contents'('d', 'e', 'c', 'l', 'a', 'r', 'e', others => ' '));

         -- checks it's at the start of a line up to "--# declare"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

         -- checks for ";" at end of line
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result : Boolean;
         begin

            Result := False;

            for I in EStrings.Lengths range EStrings.Positions'First .. InputLine.Length loop
               if InputLine.Content (I) = ';' then
                  Result := True;
                  exit;
               end if;
            end loop;

            return Result;

         end IsEnd;

      end DeclareList;

      package body ProofList
      is
         -- "pre" has 3 characters.
         KeywordPre : constant EStrings.T :=
                                EStrings.T'(3,
                                            EStrings.Contents'('p', 'r', 'e', others => ' '));

         -- "post" has 4 characters.
         KeywordPost : constant EStrings.T :=
                                EStrings.T'(4,
                                            EStrings.Contents'('p', 'o', 's', 't', others => ' '));

         -- "return" has 6 characters.
         KeywordReturn : constant EStrings.T :=
                                  EStrings.T'(6,
                                              EStrings.Contents'('r', 'e', 't', 'u', 'r', 'n', others => ' '));


         -- checks it's at the start of a line up to "--# pre" or "--# post" or "--# return"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, KeywordPre) or
                   SPARKKeywordMatching.MatchAtStart (InputLine, KeywordPost) or
                   SPARKKeywordMatching.MatchAtStart (InputLine, KeywordReturn);
         end IsStart;

         -- The 'IsEnd' for proof annotations below is not called anywhere, and thus has
         -- not been tested.

         -- checks for ";" at end of line
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result : Boolean;
         begin

            Result := False;

            for I in EStrings.Lengths range EStrings.Positions'First .. InputLine.Length loop
               if InputLine.Content (I) = ';' then
                  Result := True;
                  exit;
               end if;
            end loop;

            return Result;

         end IsEnd;

      end ProofList;

      package body GlobalsList
      is
         -- "global" has 6 characters.
         Keyword : constant EStrings.T :=
                            EStrings.T'(6,
                                        EStrings.Contents'('g', 'l', 'o', 'b', 'a', 'l', others => ' '));

         ConsecutiveEmptyLines : Natural := 0;

         -- checks it's at the start of a line up to "--# global"
         function IsStart (InputLine : EStrings.T) return Boolean
         is
         begin
            return SPARKKeywordMatching.MatchAtStart (InputLine, Keyword);
         end IsStart;

         --Here the end of a list of global declarations is detected.
         --Unfortunately, the SPARK syntax for globals does not have a unique terminator.
         --The ';' is used as a separator between different modes or as a terminator. Thus,
         --The end of a list of globals is detected by the end of a SPARK comment block, or
         --the start of a new comment item. Every possible following SPARK annotation item
         --needs to be considered, otherwise the real end of a global declaration may not be
         --detected. See SEPR: 2674
         function IsEnd (InputLine : EStrings.T) return Boolean
         is
            Result            : Boolean;
            IsAnnotationStart : Boolean;
            Index             : EStrings.Lengths;
         begin

            if DerivesList.IsStart (InputLine)  or
               DeclareList.IsStart (InputLine)  or
               ProofList.IsStart (InputLine)    or
               AcceptList.IsStart (InputLine)   or
               EndList.IsStart (InputLine)      or
               ForList.IsStart (InputLine)      or
               FunctionList.IsStart (InputLine) or
               TypeList.IsStart (InputLine)     or
               AssertList.IsStart (InputLine)   then
               Result := True;
            else
               Index  := 1;
               --# accept Flow_Message, 10, Index, "Final value of Index is not required";
               Annotation.IsStart (InputLine, Index, IsAnnotationStart);
               --# end accept;
               Result := not (IsAnnotationStart or
                              EStrings.IsEmpty (EStrings.Trim (InputLine)));
            end if;

            return Result;

         end IsEnd;

         procedure IncrementConsecutiveEmptyLines
         is
         begin
            ConsecutiveEmptyLines := ConsecutiveEmptyLines + 1;
         end IncrementConsecutiveEmptyLines;

         procedure ResetConsecutiveEmptyLines
         is
         begin
            ConsecutiveEmptyLines := 0;
         end ResetConsecutiveEmptyLines;

         function NoEmptyLines return Boolean
         is
         begin
            return ConsecutiveEmptyLines = 0;
         end NoEmptyLines;

         procedure Parse (TemporaryFile   : in out SPARK_IO.File_Type;
                          StartCol        :    out EStrings.Positions;
                          TheHeap         : in out Heap.HeapRecord;
                          GlobalVariables :    out GlobalVariablesType;
                          OK              :    out Boolean)
         is

            Token  : SPSymbols.SPTerminal;
            LexVal : LexTokenManager.LexValue;

            procedure Create (TheHeap         : in out Heap.HeapRecord;
                              GlobalVariables :    out GlobalVariablesType)
            --# global in out Statistics.TableUsage;
            --# derives GlobalVariables       from TheHeap &
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    TheHeap;
            is
               UnmodedGlobalVariables, InGlobalVariables, InOutGlobalVariables, OutGlobalVariables : SeqAlgebra.Seq;
            begin
               SeqAlgebra.CreateSeq (TheHeap, UnmodedGlobalVariables);
               SeqAlgebra.CreateSeq (TheHeap, InGlobalVariables);
               SeqAlgebra.CreateSeq (TheHeap, InOutGlobalVariables);
               SeqAlgebra.CreateSeq (TheHeap, OutGlobalVariables);
               GlobalVariables := GlobalVariablesType'(Unmoded   => UnmodedGlobalVariables,
                                                       InMode    => InGlobalVariables,
                                                       InOutMode => InOutGlobalVariables,
                                                       OutMode   => OutGlobalVariables);
            end Create;

            procedure ParseGlobalVariableClauses (TemporaryFile   : in     SPARK_IO.File_Type;
                                                  Token           : in     SPSymbols.SPTerminal;
                                                  LexVal          : in     LexTokenManager.LexValue;
                                                  TheHeap         : in out Heap.HeapRecord;
                                                  GlobalVariables : in     GlobalVariablesType;
                                                  OK              :    out Boolean)
            --# global in     CommandLineData.Content;
            --#        in     Dictionary.Dict;
            --#        in out ErrorHandler.ErrorContext;
            --#        in out LexTokenManager.StringTable;
            --#        in out SparkLex.Currline;
            --#        in out SPARK_IO.File_Sys;
            --#        in out Statistics.TableUsage;
            --# derives ErrorHandler.ErrorContext,
            --#         LexTokenManager.StringTable,
            --#         OK,
            --#         SparkLex.Currline,
            --#         SPARK_IO.File_Sys           from CommandLineData.Content,
            --#                                          Dictionary.Dict,
            --#                                          ErrorHandler.ErrorContext,
            --#                                          LexTokenManager.StringTable,
            --#                                          LexVal,
            --#                                          SparkLex.Currline,
            --#                                          SPARK_IO.File_Sys,
            --#                                          TemporaryFile,
            --#                                          Token &
            --#         Statistics.TableUsage,
            --#         TheHeap                     from *,
            --#                                          CommandLineData.Content,
            --#                                          Dictionary.Dict,
            --#                                          ErrorHandler.ErrorContext,
            --#                                          GlobalVariables,
            --#                                          LexTokenManager.StringTable,
            --#                                          LexVal,
            --#                                          SparkLex.Currline,
            --#                                          SPARK_IO.File_Sys,
            --#                                          TemporaryFile,
            --#                                          TheHeap,
            --#                                          Token;
            is

               NextToken  : SPSymbols.SPTerminal;
               NextLexVal : LexTokenManager.LexValue;

               procedure ParseGlobalVariableList (TemporaryFile   : in     SPARK_IO.File_Type;
                                                  Token           : in out SPSymbols.SPTerminal;
                                                  LexVal          : in out LexTokenManager.LexValue;
                                                  TheHeap         : in out Heap.HeapRecord;
                                                  GlobalVariables : in     GlobalVariablesType;
                                                  OK              :    out Boolean)
               --# global in     CommandLineData.Content;
               --#        in     Dictionary.Dict;
               --#        in out ErrorHandler.ErrorContext;
               --#        in out LexTokenManager.StringTable;
               --#        in out SparkLex.Currline;
               --#        in out SPARK_IO.File_Sys;
               --#        in out Statistics.TableUsage;
               --# derives ErrorHandler.ErrorContext,
               --#         LexTokenManager.StringTable,
               --#         LexVal,
               --#         OK,
               --#         SparkLex.Currline,
               --#         SPARK_IO.File_Sys,
               --#         Token                       from CommandLineData.Content,
               --#                                          Dictionary.Dict,
               --#                                          ErrorHandler.ErrorContext,
               --#                                          LexTokenManager.StringTable,
               --#                                          LexVal,
               --#                                          SparkLex.Currline,
               --#                                          SPARK_IO.File_Sys,
               --#                                          TemporaryFile,
               --#                                          Token &
               --#         Statistics.TableUsage,
               --#         TheHeap                     from *,
               --#                                          CommandLineData.Content,
               --#                                          Dictionary.Dict,
               --#                                          ErrorHandler.ErrorContext,
               --#                                          GlobalVariables,
               --#                                          LexTokenManager.StringTable,
               --#                                          LexVal,
               --#                                          SparkLex.Currline,
               --#                                          SPARK_IO.File_Sys,
               --#                                          TemporaryFile,
               --#                                          TheHeap,
               --#                                          Token;
               is
                  Mode             : Modes;
                  DottedSimpleName : EStrings.T;
                  Identifier       : EStrings.T;
                  GlobalVariable   : LexTokenManager.LexString;
               begin

                  OK := True;
                  Mode := Unmoded;

                  case Token is
                     when SPSymbols.RWin =>
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                        case Token is
                           when SPSymbols.RWout =>
                              Mode := InOutMode;
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                           when SPSymbols.identifier =>
                              Mode := InMode;
                           when others =>
                              OK := False;
                        end case;
                     when SPSymbols.RWout =>
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                        Mode := OutMode;
                     when SPSymbols.identifier =>
                        Mode := Unmoded;
                     when others =>
                        OK := False;
                  end case;

                  loop
                     exit when not OK;
                     case Token is
                        when SPSymbols.identifier =>
                           LexTokenManager.LexStringToString (LexVal.TokenStr, DottedSimpleName);
                           File_IO.Lex (TemporaryFile, Token, LexVal);
                           loop
                              exit when Token /= SPSymbols.point;
                              EStrings.AppendString (DottedSimpleName, ".");
                              File_IO.Lex (TemporaryFile, Token, LexVal);
                              if Token = SPSymbols.identifier then
                                 LexTokenManager.LexStringToString (LexVal.TokenStr, Identifier);
                                 EStrings.AppendExaminerString (DottedSimpleName, Identifier);
                                 File_IO.Lex (TemporaryFile, Token, LexVal);
                              else
                                 OK := False;
                              end if;
                              exit when not OK;
                           end loop;
                           if OK then
                              LexTokenManager.InsertExaminerString (DottedSimpleName, GlobalVariable);
                              SeqAlgebra.AddMember (TheHeap,
                                                    GlobalVariables (Mode),
                                                    LexTokenManager.LexStringRef (GlobalVariable));
                           end if;
                        when SPSymbols.comma =>
                           null;
                        when others =>
                           OK := False;
                     end case;
                     exit when not OK or Token = SPSymbols.semicolon;
                     if Token = SPSymbols.comma then
                        File_IO.Lex (TemporaryFile, Token, LexVal);
                     else
                        OK := False;
                     end if;
                  end loop;

               end ParseGlobalVariableList;

            begin

               OK := True;
               NextToken := Token;
               NextLexVal := LexVal;

               loop
                  exit when NextToken = SPSymbols.annotation_end;
                  ParseGlobalVariableList (TemporaryFile, NextToken, NextLexVal, TheHeap, GlobalVariables, OK);
                  if OK then
                     File_IO.Lex (TemporaryFile, NextToken, NextLexVal);
                     case NextToken is
                        when SPSymbols.annotation_end | SPSymbols.RWin | SPSymbols.RWout | SPSymbols.identifier =>
                           null;
                        when others =>
                           OK := False;
                     end case;
                  end if;
                  exit when not OK;
               end loop;

            end ParseGlobalVariableClauses;

         begin

            File_IO.Reset (TemporaryFile);
            SparkLex.ClearLineContext;
            Create (TheHeap, GlobalVariables);
            File_IO.Lex (TemporaryFile, Token, LexVal);

            if Token = SPSymbols.annotation_start then
               StartCol := LexVal.Position.StartPos; -- we know StartCol can't be zero because the line can't be empty
               --# accept Flow_Message, 10, LexVal, "The next symbol should be 'global', so we don't need the LexVal";
               File_IO.Lex (TemporaryFile, Token, LexVal); -- Assignment to LexVal is ineffective
               --# end accept;
               if Token = SPSymbols.RWglobal then
                  File_IO.Lex (TemporaryFile, Token, LexVal);
                  ParseGlobalVariableClauses (TemporaryFile, Token, LexVal, TheHeap, GlobalVariables, OK);
               else
                  OK := False;
               end if;
            else
               OK := False;
            end if;
            --# accept Flow_Message, 602, StartCol, StartCol, "StartCol may not be set if not OK";
         end Parse;

         procedure Reformat (OK                   : in     Boolean;
                             TemporaryFile        : in out SPARK_IO.File_Type;
                             Output               : in     SPARK_IO.File_Type;
                             StartCol             : in     EStrings.Positions;
                             TheHeap              : in out Heap.HeapRecord;
                             FunctionOrProcedure  : in     Subprogram.SubprogramType;
                             GlobalVariables      : in     GlobalVariablesType;
                             Imports              : in     SeqAlgebra.Seq;
                             Exports              : in     SeqAlgebra.Seq)
         is

            RevisedGlobalVariables : GlobalVariablesType;
            FirstGlobalVariable    : Boolean;
            AnExportIsExcluded     : Boolean;

            function GlobalCol (StartCol : EStrings.Positions) return EStrings.Positions
            is
            begin
               return StartCol + 4; -- length of "--# "
            end GlobalCol;

            procedure WriteGlobal (Output   : in SPARK_IO.File_Type;
                                   StartCol : in EStrings.Positions)
            --# global in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives SPARK_IO.File_Sys from *,
            --#                                Output,
            --#                                SparkFormatCommandLineData.Content,
            --#                                StartCol;
            is
            begin
               SPARK_IO.Set_Col (Output, GlobalCol (StartCol));
               SPARK_IO.Put_String (Output, "global", 0);
               -- If GlobalIndent is not Inline, then don't put out a trailing space,
               -- since this will be rejected by GNAT -gnatyb style check mode.
               if SparkFormatCommandLineData.Content.GlobalIndent = SparkFormatCommandLineData.Inline then
                  SPARK_IO.Put_String (Output, " ", 0);
               end if;
            end WriteGlobal;

            procedure RemoveExcludedExport (TheHeap             : in out Heap.HeapRecord;
                                            Imports             : in     SeqAlgebra.Seq;
                                            Exports             : in     SeqAlgebra.Seq;
                                            GlobalVariables     : in     GlobalVariablesType)
            --# global in out Statistics.TableUsage;
            --# derives Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Exports,
            --#                                    GlobalVariables,
            --#                                    Imports,
            --#                                    TheHeap;
            is
               GlobalVariable    : SeqAlgebra.MemberOfSeq;
               GlobalVariableVal : Natural;
               --  GlobalVariableStr : EStrings.T;
            begin
               -- SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
               --   "Entering RemoveExcludedExport",
               --    0);

               --  Iterate through all global variable modes to remove
               --  excluded export and all imported variables exclusively
               --  related to the excluded exported variable.

               --  An out mode global which is no longer exported should be removed
               GlobalVariable := SeqAlgebra.FirstMember (TheHeap, GlobalVariables (OutMode));

               while not SeqAlgebra.IsNullMember (GlobalVariable) loop
                  -- SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --   "RemoveExcludedExport: Checking for out mode globals",
                  --   0);
                  GlobalVariableVal := SeqAlgebra.ValueOfMember (TheHeap, GlobalVariable);
                  GlobalVariable := SeqAlgebra.NextMember (TheHeap, GlobalVariable);

                  if not SeqAlgebra.IsMember (TheHeap, Exports, GlobalVariableVal) then
                     --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --     "RemoveExcludedExport: Removing variable from out mode globals",
                     --     0);
                     SeqAlgebra.RemoveMember (TheHeap, GlobalVariables (OutMode), GlobalVariableVal);
                  end if;
               end loop;

               --  An in out mode global which is no longer exported should
               --  be removed from the in out mode globals and if it is still
               --  and imported variable it should be added to the in mode globals
               GlobalVariable := SeqAlgebra.FirstMember (TheHeap, GlobalVariables (InOutMode));

               while not SeqAlgebra.IsNullMember (GlobalVariable) loop
                  --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --     "RemoveExcludedExport: Checking for in out mode globals",
                  --     0);
                  GlobalVariableVal := SeqAlgebra.ValueOfMember (TheHeap, GlobalVariable);
                  GlobalVariable := SeqAlgebra.NextMember (TheHeap, GlobalVariable);

                  if not SeqAlgebra.IsMember (TheHeap, Exports, GlobalVariableVal) then
                     --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --     "RemoveExcludedExport: Removing variable from in out mode globals",
                     --     0);
                     SeqAlgebra.RemoveMember (TheHeap, GlobalVariables (InOutMode), GlobalVariableVal);
                     if SeqAlgebra.IsMember (TheHeap, Imports, GlobalVariableVal) then
                        --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                        --     "RemoveExcludedExport: Adding variable to in mode globals",
                        --     0);
                        SeqAlgebra.AddMember (TheHeap, GlobalVariables (InMode), GlobalVariableVal);
                     end if;
                  end if;
               end loop;

               --  An in mode global which is no longer imported should
               --  be removed from the in mode globals.
               GlobalVariable := SeqAlgebra.FirstMember (TheHeap, GlobalVariables (InMode));

               while not SeqAlgebra.IsNullMember (GlobalVariable) loop
                  --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  --     "RemoveExcludedExport: Checking for in mode globals",
                  --     0);
                  GlobalVariableVal := SeqAlgebra.ValueOfMember (TheHeap, GlobalVariable);
                  GlobalVariable := SeqAlgebra.NextMember (TheHeap, GlobalVariable);

                  if not SeqAlgebra.IsMember (TheHeap, Imports, GlobalVariableVal) then
                     --  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     --     "RemoveExcludedExport: Removing variable from in mode globals",
                     --     0);
                     SeqAlgebra.RemoveMember (TheHeap, GlobalVariables (InMode), GlobalVariableVal);
                  end if;
               end loop;

               --  An unmoded global which is no longer exported or imported should
               --  be removed from the unmoded globals.
               GlobalVariable := SeqAlgebra.FirstMember (TheHeap, GlobalVariables (UnModed));

               while not SeqAlgebra.IsNullMember (GlobalVariable) loop
                  --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                  --       "RemoveExcludedExport: Checking for unmoded globals: ",
                  --       0);

                  GlobalVariableVal := SeqAlgebra.ValueOfMember (TheHeap, GlobalVariable);
                  GlobalVariable := SeqAlgebra.NextMember (TheHeap, GlobalVariable);

                  --  LexTokenManager.LexStringToString (
                  --     LexTokenManager.ConvertLexStringRef (GlobalVariableVal),
                  --     GlobalVariableStr);

                  --  EStrings.PutLine (SPARK_IO.Standard_Output, GlobalVariableStr);

                  if not SeqAlgebra.IsMember (TheHeap, Exports, GlobalVariableVal) and then
                     not SeqAlgebra.IsMember (TheHeap, Imports, GlobalVariableVal)
                  then
                     --  SPARK_IO.Put_String (SPARK_IO.Standard_Output,
                     --       "RemoveExcludedExport: Removing variable from unmoded globals: ",
                     --       0);
                     --  EStrings.PutLine (SPARK_IO.Standard_Output, GlobalVariableStr);
                     SeqAlgebra.RemoveMember (TheHeap, GlobalVariables (UnModed), GlobalVariableVal);
                  end if;
               end loop;
            end RemoveExcludedExport;

            procedure AddModes (TheHeap             : in out Heap.HeapRecord;
                                FunctionOrProcedure : in     Subprogram.SubprogramType;
                                Imports             : in     SeqAlgebra.Seq;
                                Exports             : in     SeqAlgebra.Seq;
                                GlobalVariables     : in out GlobalVariablesType)
            --# global in     SparkFormatCommandLineData.Content;
            --#        in out Statistics.TableUsage;
            --# derives GlobalVariables       from *,
            --#                                    FunctionOrProcedure,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    TheHeap &
            --#         Statistics.TableUsage,
            --#         TheHeap               from *,
            --#                                    Exports,
            --#                                    FunctionOrProcedure,
            --#                                    GlobalVariables,
            --#                                    Imports,
            --#                                    SparkFormatCommandLineData.Content,
            --#                                    TheHeap;
            is

               procedure AddProcedureModes (TheHeap             : in out Heap.HeapRecord;
                                            Imports             : in     SeqAlgebra.Seq;
                                            Exports             : in     SeqAlgebra.Seq;
                                            GlobalVariables     : in     GlobalVariablesType)
               --# global in out Statistics.TableUsage;
               --# derives Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    Exports,
               --#                                    GlobalVariables,
               --#                                    Imports,
               --#                                    TheHeap;
               is
                  GlobalVariable    : SeqAlgebra.MemberOfSeq;
                  GlobalVariableVal : Natural;
                  RevisedMode       : Modes;
               begin

                  GlobalVariable := SeqAlgebra.FirstMember (TheHeap, GlobalVariables (UnModed));

                  loop

                     GlobalVariableVal := SeqAlgebra.ValueOfMember (TheHeap, GlobalVariable);

                     if SeqAlgebra.IsMember (TheHeap, Imports, GlobalVariableVal) then
                        if SeqAlgebra.IsMember (TheHeap, Exports, GlobalVariableVal) then
                           RevisedMode := InOutMode;
                        else
                           RevisedMode := InMode;
                        end if;
                     elsif SeqAlgebra.IsMember (TheHeap, Exports, GlobalVariableVal) then
                        RevisedMode := OutMode;
                     else
                        RevisedMode := Unmoded;
                     end if;

                     GlobalVariable := SeqAlgebra.NextMember (TheHeap, GlobalVariable);

                     if RevisedMode /= Unmoded then
                        SeqAlgebra.RemoveMember (TheHeap, GlobalVariables (Unmoded), GlobalVariableVal);
                        SeqAlgebra.AddMember (TheHeap, GlobalVariables (RevisedMode), GlobalVariableVal);
                     end if;

                     exit when SeqAlgebra.IsNullMember (GlobalVariable);

                  end loop;

               end AddProcedureModes;

               procedure AddFunctionModes (TheHeap         : in out Heap.HeapRecord;
                                           GlobalVariables : in out GlobalVariablesType)
               --# global in     SparkFormatCommandLineData.Content;
               --#        in out Statistics.TableUsage;
               --# derives GlobalVariables,
               --#         Statistics.TableUsage,
               --#         TheHeap               from *,
               --#                                    GlobalVariables,
               --#                                    SparkFormatCommandLineData.Content,
               --#                                    TheHeap;
               is

                  procedure Copy (TheHeap         : in out Heap.HeapRecord;
                                  GlobalVariables : in out GlobalVariablesType;
                                  FromMode        : in Modes;
                                  ToMode          : in Modes)
                  --# global in out Statistics.TableUsage;
                  --# derives GlobalVariables,
                  --#         Statistics.TableUsage,
                  --#         TheHeap               from *,
                  --#                                    FromMode,
                  --#                                    GlobalVariables,
                  --#                                    TheHeap,
                  --#                                    ToMode;
                  is
                     FromList    : SeqAlgebra.Seq;
                     ToList      : SeqAlgebra.Seq;
                     RevisedList : SeqAlgebra.Seq;
                  begin
                     FromList := GlobalVariables (FromMode);
                     ToList := GlobalVariables (ToMode);
                     SeqAlgebra.Union (TheHeap, FromList, ToList, RevisedList);
                     SeqAlgebra.DisposeOfSeq (TheHeap, ToList);
                     SeqAlgebra.Reduction (TheHeap, FromList, RevisedList);
                     GlobalVariables (FromMode) := FromList;
                     GlobalVariables (ToMode) := RevisedList;
                  end Copy;

               begin

                  case SparkFormatCommandLineData.Content.DefaultFunctionMode is
                     when SparkFormatCommandLineData.Unmoded =>
                        Copy (TheHeap, GlobalVariables, InMode, Unmoded);
                     when SparkFormatCommandLineData.InMode =>
                        Copy (TheHeap, GlobalVariables, Unmoded, InMode);
                  end case;

               end AddFunctionModes;

            begin
               case FunctionOrProcedure is
                  when Subprogram.IsUnknown | Subprogram.IsProcedure =>
                     AddProcedureModes (TheHeap, Imports, Exports, GlobalVariables);
                  when Subprogram.IsFunction =>
                     AddFunctionModes (TheHeap, GlobalVariables);
               end case;
            end AddModes;

            procedure WriteGlobalVariableClauses (Output              : in     SPARK_IO.File_Type;
                                                  StartCol            : in     EStrings.Positions;
                                                  FirstGlobalVariable : in out Boolean;
                                                  Alphabetic          : in     Boolean;
                                                  TheHeap             : in     Heap.HeapRecord;
                                                  GlobalVariables     : in     GlobalVariablesType;
                                                  Mode                : in     Modes)
            --# global in     Annotation.StartString;
            --#        in     LexTokenManager.StringTable;
            --#        in     SparkFormatCommandLineData.Content;
            --#        in out SPARK_IO.File_Sys;
            --# derives FirstGlobalVariable from *,
            --#                                  Alphabetic,
            --#                                  GlobalVariables,
            --#                                  LexTokenManager.StringTable,
            --#                                  Mode,
            --#                                  TheHeap &
            --#         SPARK_IO.File_Sys   from *,
            --#                                  Alphabetic,
            --#                                  Annotation.StartString,
            --#                                  FirstGlobalVariable,
            --#                                  GlobalVariables,
            --#                                  LexTokenManager.StringTable,
            --#                                  Mode,
            --#                                  Output,
            --#                                  SparkFormatCommandLineData.Content,
            --#                                  StartCol,
            --#                                  TheHeap;
            is

               GlobalVariable : SeqAlgebra.MemberOfSeq;
               TheIterator    : SPARKProgram.Iteration.Iterator;

               procedure WriteGlobalVariable (Output              : in     SPARK_IO.File_Type;
                                              StartCol            : in     EStrings.Positions;
                                              FirstGlobalVariable : in out Boolean;
                                              TheHeap             : in     Heap.HeapRecord;
                                              GlobalVariable      : in     SeqAlgebra.MemberOfSeq;
                                              GlobalVariables     : in     GlobalVariablesType;
                                              Mode                : in     Modes)
               --# global in     Annotation.StartString;
               --#        in     LexTokenManager.StringTable;
               --#        in     SparkFormatCommandLineData.Content;
               --#        in out SPARK_IO.File_Sys;
               --# derives FirstGlobalVariable from * &
               --#         SPARK_IO.File_Sys   from *,
               --#                                  Annotation.StartString,
               --#                                  FirstGlobalVariable,
               --#                                  GlobalVariable,
               --#                                  GlobalVariables,
               --#                                  LexTokenManager.StringTable,
               --#                                  Mode,
               --#                                  Output,
               --#                                  SparkFormatCommandLineData.Content,
               --#                                  StartCol,
               --#                                  TheHeap;
               is

                  GlobalVariableVal    : Natural;
                  GlobalVariableString : EStrings.T;

                  procedure WriteMode (Output          : in SPARK_IO.File_Type;
                                       StartCol        : in EStrings.Positions;
                                       TheHeap         : in Heap.HeapRecord;
                                       GlobalVariables : in GlobalVariablesType;
                                       Mode            : in Modes)
                  --# global in     SparkFormatCommandLineData.Content;
                  --#        in out SPARK_IO.File_Sys;
                  --# derives SPARK_IO.File_Sys from *,
                  --#                                GlobalVariables,
                  --#                                Mode,
                  --#                                Output,
                  --#                                SparkFormatCommandLineData.Content,
                  --#                                StartCol,
                  --#                                TheHeap;
                  is

                     function ModeCol (StartCol : EStrings.Positions) return EStrings.Positions
                     --# global in SparkFormatCommandLineData.Content;
                     is
                        ModeColumn : EStrings.Positions;
                     begin
                        if SparkFormatCommandLineData.Content.GlobalIndent >= 1 then
                           ModeColumn := GlobalCol (StartCol) + (SparkFormatCommandLineData.Content.GlobalIndent - 1); -- StartCol "is --# " but indent value is given from "--#"
                        else
                           ModeColumn := GlobalCol (StartCol) + 7; -- length of "global "
                        end if;
                        return ModeColumn;
                     end ModeCol;

                  begin

                     SPARK_IO.Set_Col (Output, ModeCol (StartCol));

                     case Mode is
                        when InMode =>
                           SPARK_IO.Put_String (Output, "in ", 0);
                           if not (SeqAlgebra.IsEmptySeq (TheHeap, GlobalVariables (InOutMode)) and
                                   SeqAlgebra.IsEmptySeq (TheHeap, GlobalVariables (OutMode))) then
                              SPARK_IO.Put_String (Output, "    ", 0);
                           end if;
                        when OutMode =>
                           if not (SeqAlgebra.IsEmptySeq (TheHeap, GlobalVariables (InMode)) and
                                   SeqAlgebra.IsEmptySeq (TheHeap, GlobalVariables (InOutMode))) then
                              SPARK_IO.Put_String (Output, "   ", 0);
                           end if;
                           SPARK_IO.Put_String (Output, "out ", 0);
                        when InOutMode =>
                           SPARK_IO.Put_String (Output, "in out ", 0);
                        when Unmoded =>
                           null;
                     end case;

                  end WriteMode;

               begin  -- WriteGlobalVariable

                  if FirstGlobalVariable then
                     FirstGlobalVariable := False;
                     if SparkFormatCommandLineData.Content.GlobalIndent >= 1 then
                        Annotation.Write (Output, StartCol);
                     end if;
                  else
                     case Mode is
                        when Unmoded =>
                           SPARK_IO.Put_Line (Output, ",", 0);
                        when InMode | OutMode | InOutMode =>
                           SPARK_IO.Put_Line (Output, ";", 0);
                     end case;
                     Annotation.Write (Output, StartCol);
                  end if;

                  WriteMode (Output, StartCol, TheHeap, GlobalVariables, Mode);
                  GlobalVariableVal := SeqAlgebra.ValueOfMember (TheHeap, GlobalVariable);
                  LexTokenManager.LexStringToString (LexTokenManager.ConvertLexStringRef (GlobalVariableVal),
                                                     GlobalVariableString);
                  EStrings.PutString (Output, GlobalVariableString);

               end WriteGlobalVariable;

            begin -- WriteGlobalVariableClauses

               -- This subprogram is called for each mode in turn.
               -- Each sequence of global variables could be null, in which case we don't write anything
               -- for that mode.
               GlobalVariable := SeqAlgebra.FirstMember (TheHeap, GlobalVariables (Mode));

               if not SeqAlgebra.IsNullMember (GlobalVariable) then

                  SPARKProgram.Iteration.Initialise (TheHeap,
                                                     GlobalVariables (Mode),
                                                     Alphabetic,
                                                     TheIterator);

                  WriteGlobalVariable (Output,
                                       StartCol,
                                       FirstGlobalVariable,
                                       TheHeap,
                                       SPARKProgram.Iteration.CurrentMember (TheIterator),
                                       GlobalVariables,
                                       Mode);

                  SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  while not SPARKProgram.Iteration.Complete (TheIterator) loop

                     WriteGlobalVariable (Output,
                                          StartCol,
                                          FirstGlobalVariable,
                                          TheHeap,
                                          SPARKProgram.Iteration.CurrentMember (TheIterator),
                                          GlobalVariables,
                                          Mode);

                     SPARKProgram.Iteration.Next (TheHeap, TheIterator);

                  end loop;

               end if;

            end WriteGlobalVariableClauses;

            procedure Dispose (TheHeap         : in out Heap.HeapRecord;
                               GlobalVariables : in     GlobalVariablesType)
            --# derives TheHeap from *,
            --#                      GlobalVariables;
            is
            begin
               for Mode in Modes loop
                  SeqAlgebra.DisposeOfSeq (TheHeap, GlobalVariables (Mode));
               end loop;
            end Dispose;

         begin -- GlobalList.Reformat

            RevisedGlobalVariables := GlobalVariables;

            if OK then

               --  CFR1753: Check to see if an export has been excluded.
               AnExportIsExcluded := not EStrings.IsEmpty
                  (SparkFormatCommandLineData.Content.ExcludeExport);
               case FunctionOrProcedure is
                  when Subprogram.IsProcedure | Subprogram.IsUnknown =>
                     if AnExportIsExcluded then
                        RemoveExcludedExport (TheHeap, Imports, Exports, RevisedGlobalVariables);
                     end if;
                  when Subprogram.IsFunction =>
                     null;
               end case;

               if not AnExportIsExcluded or else
                  not SeqAlgebra.IsEmptySeq (TheHeap, RevisedGlobalVariables (OutMode)) or else
                  not SeqAlgebra.IsEmptySeq (TheHeap, RevisedGlobalVariables (InOutMode)) or else
                  not SeqAlgebra.IsEmptySeq (TheHeap, RevisedGlobalVariables (InMode)) or else
                  not SeqAlgebra.IsEmptySeq (TheHeap, RevisedGlobalVariables (UnModed))
               then
                  Annotation.Write (Output, StartCol);
                  WriteGlobal (Output, StartCol);

                  if SparkFormatCommandLineData.Content.AddModes then
                     AddModes (TheHeap, FunctionOrProcedure, Imports, Exports, RevisedGlobalVariables);
                  end if;

                  FirstGlobalVariable := True;

                  for Mode in Modes loop
                     WriteGlobalVariableClauses (Output,
                                                 StartCol,
                                                 FirstGlobalVariable,
                                                 SparkFormatCommandLineData.Content.AlphabeticOrdering,
                                                 TheHeap,
                                                 RevisedGlobalVariables,
                                                 Mode);
                  end loop;

                  SPARK_IO.Put_Line (Output, ";", 0);
               end if;
            else
               File_IO.Reset (TemporaryFile);
               SPARKProgram.Copy (TemporaryFile, Output);
            end if;

            Dispose (TheHeap, RevisedGlobalVariables);
            File_IO.Delete (TemporaryFile);

            if not NoEmptyLines then
               SPARK_IO.New_Line (Output, ConsecutiveEmptyLines);
            end if;
            ResetConsecutiveEmptyLines;
         end Reformat;

      end GlobalsList;

      function ContainsFunctionSpecification (InputLine : EStrings.T) return Boolean
      is
         Index  : EStrings.Positions;
      begin

         Index := EStrings.Positions'First;
         WhiteSpace.Skip (InputLine, Index);

         return InputLine.Content (Index) = 'f' and
                InputLine.Content (Index + 1) = 'u' and
                InputLine.Content (Index + 2) = 'n' and
                InputLine.Content (Index + 3) = 'c' and
                InputLine.Content (Index + 4) = 't' and
                InputLine.Content (Index + 5) = 'i' and
                InputLine.Content (Index + 6) = 'o' and
                InputLine.Content (Index + 7) = 'n';

      end ContainsFunctionSpecification;

      function ContainsProcedureSpecification (InputLine : EStrings.T) return Boolean
      is
         Index  : EStrings.Positions;
      begin

         Index := EStrings.Positions'First;
         WhiteSpace.Skip (InputLine, Index);

         return InputLine.Content (Index) = 'p' and
                InputLine.Content (Index + 1) = 'r' and
                InputLine.Content (Index + 2) = 'o' and
                InputLine.Content (Index + 3) = 'c' and
                InputLine.Content (Index + 4) = 'e' and
                InputLine.Content (Index + 5) = 'd' and
                InputLine.Content (Index + 6) = 'u' and
                InputLine.Content (Index + 7) = 'r' and
                InputLine.Content (Index + 8) = 'e';

      end ContainsProcedureSpecification;

   begin  -- ReformatAnnotations

      State := ParsingSPARKCode;
      GlobalsListFile := SPARK_IO.Null_File;
      InheritClauseFile := SPARK_IO.Null_File;
      InitializeSpecFile := SPARK_IO.Null_File;
      OwnVarClauseFile := SPARK_IO.Null_File;
      DerivesListFile := SPARK_IO.Null_File;
      FunctionOrProcedure := Subprogram.IsUnknown;

      Heap.Initialize (TheHeap);
      SparkLex.ClearLineContext;
      Annotation.Initialize;
      DerivesList.Initialize;

      Annotations.Initialize (InheritAnno, "inherit", "main_program",
                              SparkFormatCommandLineData.Content.InheritIndent);

      Annotations.Initialize (InitializesAnno, "initializes", "",
                              SparkFormatCommandLineData.Content.InitializationIndent);

      Annotations.Initialize (OwnVarAnno, "own", "initializes",
                              SparkFormatCommandLineData.Content.OwnIndent);


      if not SPARK_IO.End_Of_File (Input) then
         EStrings.GetLine (Input, InputLine);
         File_IO_Required := False;

         while not (File_IO_Required and then SPARK_IO.End_Of_File (Input)) loop

            if File_IO_Required then
               EStrings.GetLine (Input, InputLine);
            else
               File_IO_Required := True;
            end if;

            case State is
               when ParsingSPARKCode =>
                  if ContainsFunctionSpecification (InputLine) then
                     EStrings.PutLine (Output, InputLine);
                     FunctionOrProcedure := Subprogram.IsFunction;
                  elsif ContainsProcedureSpecification (InputLine) then
                     EStrings.PutLine (Output, InputLine);
                     FunctionOrProcedure := Subprogram.IsProcedure;
                  elsif GlobalsList.IsStart (InputLine) then
                     File_IO.Create (GlobalsListFile);
                     EStrings.PutLine (GlobalsListFile, InputLine);
                     if SPARK_IO.End_Of_File (Input) then
                        -- This route only taken when global alone at end of file
                        GlobalsList.Parse (GlobalsListFile, StartCol, TheHeap, GlobalVariables, GlobalsListOK);
                        SeqAlgebra.CreateSeq (TheHeap, Imports);
                        SeqAlgebra.CreateSeq (TheHeap, Exports);
                        GlobalsList.Reformat (GlobalsListOK,
                                           GlobalsListFile,
                                           Output,
                                           StartCol,
                                           TheHeap,
                                           FunctionOrProcedure,
                                           GlobalVariables,
                                           Imports,
                                           Exports);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                     else
                        -- Normal route
                        State := ParsingGlobalsList;
                     end if;
                  elsif DerivesList.IsStart (InputLine) then
                     File_IO.Create (DerivesListFile);
                     EStrings.PutLine (DerivesListFile, InputLine);
                     if DerivesList.IsEnd (InputLine) then
                        --# accept Flow_Message, 10, Imports, "There is no globals list, so no need for the list of imports.";
                        DerivesList.Parse (DerivesListFile,
                                           StartCol,
                                           TheHeap,
                                           Imports,
                                           Exports,
                                           Dependencies,
                                           DerivesListOK);
                        --# end accept;
                        DerivesList.Reformat (DerivesListOK,
                                              DerivesListFile,
                                              Output,
                                              StartCol,
                                              TheHeap,
                                              Exports,
                                              Dependencies);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                     elsif SPARK_IO.End_Of_File (Input) then
                        File_IO.Reset (DerivesListFile);
                        Copy (DerivesListFile, Output);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                     else
                        State := ParsingDerivesList;
                     end if;
                  elsif Annotations.IsStartOf (InheritAnno, InputLine) then
                     File_IO.Create (InheritClauseFile);
                     State := ParsingInheritClause;
                     File_IO_Required := False;
                  elsif Annotations.IsStartOf (InitializesAnno, InputLine) then
                     File_IO.Create (InitializeSpecFile);
                     State := ParsingInitializationSpecification;
                     File_IO_Required := False;
                  elsif Annotations.IsStartOf (OwnVarAnno, InputLine) then
                     File_IO.Create (OwnVarClauseFile);
                     State := ParsingOwnVariableClause;
                     File_IO_Required := False;
                  else
                     EStrings.PutLine (Output, InputLine);
                  end if;
               when ParsingGlobalsList =>
                  if GlobalsList.IsEnd (InputLine) then
                     if DerivesList.IsStart (InputLine) then
                        if GlobalsList.NoEmptyLines then
                           File_IO.Create (DerivesListFile);
                           EStrings.PutLine (DerivesListFile, InputLine);
                           if DerivesList.IsEnd (InputLine) then
                              --# accept Flow_Message, 10, SparkLex.CurrLine, "Not used when both global and derives lists present" &
                              --#        Flow_Message, 10, StartCol, "If both globals and derives lists present, StartCol value from derives list is used";
                              GlobalsList.Parse (GlobalsListFile,
                                                 StartCol,
                                                 TheHeap,
                                                 GlobalVariables,
                                                 GlobalsListOK);
                              --# end accept;
                              DerivesList.Parse (DerivesListFile,
                                                 StartCol,
                                                 TheHeap,
                                                 Imports,
                                                 Exports,
                                                 Dependencies,
                                                 DerivesListOK);
                              GlobalsList.Reformat (GlobalsListOK,
                                                    GlobalsListFile,
                                                    Output,
                                                    StartCol,
                                                    TheHeap,
                                                    FunctionOrProcedure,
                                                    GlobalVariables,
                                                    Imports,
                                                    Exports);
                              DerivesList.Reformat (DerivesListOK,
                                                    DerivesListFile,
                                                    Output,
                                                    StartCol,
                                                    TheHeap,
                                                    Exports,
                                                    Dependencies);
                              FunctionOrProcedure := Subprogram.IsUnknown;
                              State := ParsingSPARKCode;
                           else
                              State := ParsingDerivesList;
                           end if;
                        else
                           GlobalsList.Parse (GlobalsListFile, StartCol, TheHeap, GlobalVariables, GlobalsListOK);
                           SeqAlgebra.CreateSeq (TheHeap, Imports);
                           SeqAlgebra.CreateSeq (TheHeap, Exports);
                           GlobalsList.Reformat (GlobalsListOK,
                                                 GlobalsListFile,
                                                 Output,
                                                 StartCol,
                                                 TheHeap,
                                                 FunctionOrProcedure,
                                                 GlobalVariables,
                                                 Imports,
                                                 Exports);
                           State := ParsingSPARKCode;
                           File_IO_Required := False;
                        end if;
                     else
                        GlobalsList.Parse (GlobalsListFile, StartCol, TheHeap, GlobalVariables, GlobalsListOK);
                        SeqAlgebra.CreateSeq (TheHeap, Imports);
                        SeqAlgebra.CreateSeq (TheHeap, Exports);
                        GlobalsList.Reformat (GlobalsListOK,
                                              GlobalsListFile,
                                              Output,
                                              StartCol,
                                              TheHeap,
                                              FunctionOrProcedure,
                                              GlobalVariables,
                                              Imports,
                                              Exports);
                        FunctionOrProcedure := Subprogram.IsUnknown;
                        State := ParsingSPARKCode;
                        File_IO_Required := False; -- Have another look at line in case is a function
                     end if;
                  elsif SPARK_IO.End_Of_File (Input) then
                     EStrings.PutLine (GlobalsListFile, InputLine);
                     GlobalsList.Parse (GlobalsListFile, StartCol, TheHeap, GlobalVariables, GlobalsListOK);
                     SeqAlgebra.CreateSeq (TheHeap, Imports);
                     SeqAlgebra.CreateSeq (TheHeap, Exports);
                     GlobalsList.Reformat (GlobalsListOK,
                                           GlobalsListFile,
                                           Output,
                                           StartCol,
                                           TheHeap,
                                           FunctionOrProcedure,
                                           GlobalVariables,
                                           Imports,
                                           Exports);
                     FunctionOrProcedure := Subprogram.IsUnknown;
                     State := ParsingSPARKCode;
                  else
                     EStrings.PutLine (GlobalsListFile, InputLine);
                     if EStrings.IsEmpty (EStrings.Trim (InputLine)) then
                        GlobalsList.IncrementConsecutiveEmptyLines;
                     else
                        GlobalsList.ResetConsecutiveEmptyLines;
                     end if;
                  end if;
               when ParsingDerivesList =>
                  EStrings.PutLine (DerivesListFile, InputLine);
                  if DerivesList.IsEnd (InputLine) then
                     DerivesList.Parse (DerivesListFile, StartCol, TheHeap, Imports, Exports, Dependencies, DerivesListOK);
                     if SPARK_IO.Is_Open (GlobalsListFile) then
                        GlobalsList.Parse (GlobalsListFile, StartCol, TheHeap, GlobalVariables, GlobalsListOK);
                        GlobalsList.Reformat (GlobalsListOK,
                                              GlobalsListFile,
                                              Output,
                                              StartCol,
                                              TheHeap,
                                              FunctionOrProcedure,
                                              GlobalVariables,
                                              Imports,
                                              Exports);
                     end if;
                     DerivesList.Reformat (DerivesListOK, DerivesListFile, Output, StartCol, TheHeap, Exports, Dependencies);
                     FunctionOrProcedure := Subprogram.IsUnknown;
                     State := ParsingSPARKCode;
                  elsif SPARK_IO.End_Of_File (Input) then
                     if SPARK_IO.Is_Open (GlobalsListFile) then
                        GlobalsList.Parse (GlobalsListFile, StartCol, TheHeap, GlobalVariables, GlobalsListOK);
                        SeqAlgebra.CreateSeq (TheHeap, Imports);
                        SeqAlgebra.CreateSeq (TheHeap, Exports);
                        GlobalsList.Reformat (GlobalsListOK,
                                              GlobalsListFile,
                                              Output,
                                              StartCol,
                                              TheHeap,
                                              FunctionOrProcedure,
                                              GlobalVariables,
                                              Imports,
                                              Exports);
                     end if;
                     File_IO.Reset (DerivesListFile);
                     Copy (DerivesListFile, Output);
                     FunctionOrProcedure := Subprogram.IsUnknown;
                     State := ParsingSPARKCode;
                  end if;
               when ParsingInheritClause =>
                  EStrings.PutLine (InheritClauseFile, InputLine);

                  if Annotations.IsEndOf (InheritAnno, InputLine) or else
                     SPARK_IO.End_Of_File (Input)
                  then
                     Reformatter.Initialise (InheritAnno,
                                             SparkFormatCommandLineData.Content.AlphabeticOrdering,
                                             Reformatter.Allow_Dotted_Names,
                                             TheHeap,
                                             InheritClause);
                     Reformatter.Parse (InheritClause, TheHeap, InheritClauseFile);
                     Reformatter.Reformat (InheritClause,
                                           TheHeap,
                                           InheritClauseFile,
                                           Output,
                                           ReformatSuccessful);
                     --# accept Flow_Message, 10, InheritClause, "The state variable has been 'disposed' and should not be reused without initialisation";
                     Reformatter.Finalise (InheritClause, TheHeap);
                     --# end accept;
                     File_IO.Delete (InheritClauseFile);
                     State := ParsingSPARKCode;
                     File_IO_Required := not ReformatSuccessful;
                  end if;
               when ParsingInitializationSpecification =>
                  EStrings.PutLine (InitializeSpecFile, InputLine);

                  if Annotations.IsEndOf (InitializesAnno, InputLine) or else
                     SPARK_IO.End_Of_File (Input)
                  then
                     Reformatter.Initialise (InitializesAnno,
                                             SparkFormatCommandLineData.Content.AlphabeticOrdering,
                                             Reformatter.Disallow_Dotted_Names,
                                             TheHeap,
                                             InitializationSpec);
                     Reformatter.Parse (InitializationSpec, TheHeap, InitializeSpecFile);
                     Reformatter.Reformat (InitializationSpec,
                                           TheHeap,
                                           InitializeSpecFile,
                                           Output,
                                           ReformatSuccessful);
                     --# accept Flow_Message, 10, InitializationSpec, "The state variable has been 'disposed' and should not be reused without initialisation";
                     Reformatter.Finalise (InitializationSpec, TheHeap);
                     --# end accept;
                     File_IO.Delete (InitializeSpecFile);
                     State := ParsingSPARKCode;
                     File_IO_Required := not ReformatSuccessful;
                  end if;
               when ParsingOwnVariableClause =>
                  EStrings.PutLine (OwnVarClauseFile, InputLine);

                  if Annotations.IsEndOf (OwnVarAnno, InputLine) or else
                     SPARK_IO.End_Of_File (Input)
                  then
                     Reformatter.Initialise (OwnVarAnno,
                                             SparkFormatCommandLineData.Content.AlphabeticOrdering,
                                             Reformatter.Disallow_Dotted_Names,
                                             TheHeap,
                                             OwnVariableClause);
                     Reformatter.Parse (OwnVariableClause, TheHeap, OwnVarClauseFile);
                     Reformatter.Reformat (OwnVariableClause,
                                           TheHeap,
                                           OwnVarClauseFile,
                                           Output,
                                           ReformatSuccessful);
                     --# accept Flow_Message, 10, OwnVariableClause, "The state variable has been 'disposed' and should not be reused without initialisation";
                     Reformatter.Finalise (OwnVariableClause, TheHeap);
                     --# end accept;
                     File_IO.Delete (OwnVarClauseFile);
                     State := ParsingSPARKCode;
                     File_IO_Required := not ReformatSuccessful;
                  end if;

               when  ParsingRefinementDefinition =>
                  -- To be completed
                  State := ParsingSPARKCode;

            end case;

         end loop;
      end if;

   end ReformatAnnotations;

end SPARKProgram;
