-- $Id: sparkprogram-reformatter-simplelex.adb 11892 2008-12-12 15:55:16Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (SparkProgram.Reformatter)
package body SimpleLex is

   function IsWhiteSpace (Char : Character) return Boolean
   is
   begin
      return Char = Ada.Characters.Latin_1.Space or Char = Ada.Characters.Latin_1.HT;
   end IsWhiteSpace;

   procedure SkipWhiteSpace (InputLine  : in     EStrings.T;
                             Index      : in out ELStrings.Positions;
                             StillWhite :    out Boolean)
   --# derives Index,
   --#         StillWhite from Index,
   --#                         InputLine;
   is
      InWhiteSpace : Boolean;
   begin
      InWhiteSpace := True;

      while InWhiteSpace and Index <= InputLine.Length loop
         if IsWhiteSpace (InputLine.Content (Index)) then
            Index := Index + 1;
         else
            InWhiteSpace := False;
         end if;
      end loop;

      StillWhite := InWhiteSpace;
   end SkipWhiteSpace;

   procedure ExtendedSkipWhiteSpace (InputFile    : in     SPARK_IO.File_Type;
                                     Anno         : in     Annotations.Anno_Type;
                                     InAnnotation : in out Boolean;
                                     InputLine    : in out EStrings.T;
                                     Index        : in out ELStrings.Positions)
   --# global in out SPARK_IO.File_Sys;
   --# derives InAnnotation,
   --#         Index,
   --#         InputLine,
   --#         SPARK_IO.File_Sys from Anno,
   --#                                InAnnotation,
   --#                                Index,
   --#                                InputFile,
   --#                                InputLine,
   --#                                SPARK_IO.File_Sys;
   is
      InWhiteSpace      : Boolean;
      IsAnnotationStart : Boolean;
   begin
      InWhiteSpace := True;

      while InWhiteSpace loop
         if Index > InputLine.Length then
            if SPARK_IO.End_Of_File (InputFile) then
               InWhiteSpace := False;
               InAnnotation := False;
            else
               EStrings.GetLine (InputFile, InputLine);
               Index := 1;
               -- Expect an annotation start
               Annotations.IsAnnoStart (Anno, InputLine, Index, IsAnnotationStart);
               if InAnnotation and not IsAnnotationStart then
                  InWhiteSpace := False;
                  InAnnotation := False;
               elsif not InAnnotation and IsAnnotationStart then
                  InWhiteSpace := False;
                  InAnnotation := True;
               else
                  -- otherwise (InAnnotation and IsAnnotationStart) or
                  --           (not InAnnotation and not IsAnnotationStart)
                  -- Treat as whitespace
                  null;
               end if;
            end if;
         end if;

         if InWhiteSpace then
            SkipWhiteSpace (InputLine, Index, InWhiteSpace);
         end if;
      end loop;

      if not InAnnotation then
         Index := 1;
         InputLine.Length := 0;
      end if;

   end ExtendedSkipWhiteSpace;

   procedure GetPropertyList (InputFile : in     SPARK_IO.File_Type;
                              Anno      : in     Annotations.Anno_Type;
                              InputLine : in out EStrings.T;
                              Index     : in out ELStrings.Positions;
                              TokenRec  :    out Token_Record)
   --# global in     CommandLineData.Content;
   --#        in out SPARK_IO.File_Sys;
   --# derives Index,
   --#         InputLine,
   --#         SPARK_IO.File_Sys,
   --#         TokenRec          from Anno,
   --#                                CommandLineData.Content,
   --#                                Index,
   --#                                InputFile,
   --#                                InputLine,
   --#                                SPARK_IO.File_Sys;
   is
      AnnoContinuation : Boolean;
      StartString : EStrings.T;
      Unused : Boolean;
   begin
      TokenRec.Token := property_list;
      --# accept Flow_Message, 23, TokenRec.TokenValue.Content, "Length = 1, so only Content (1) need be defined";
      TokenRec.TokenValue.Length := 1;
      TokenRec.TokenValue.Content (1) := InputLine.Content (Index);
      --# end accept;
      AnnoContinuation := True;
      Index := Index + 1;

      loop
         if InputLine.Length < Index then
            EStrings.GetLine (InputFile, InputLine);
            Index := 1;
            Annotations.IsAnnoStart (Anno, InputLine, Index, AnnoContinuation);
            if AnnoContinuation then

               EStrings.CopyString (StartString, "--");
               --# accept Flow_Message, 10, Unused, "The variable Unused is not required" &
               --#        Flow_Message, 33, Unused, "The variable Unused is not required";
               EStrings.AppendChar (StartString,
                                           CommandLineData.Content.AnnoChar,
                                           Unused);
               EStrings.AppendChar (StartString,
                                           ' ',
                                           Unused);
               -- we know there's always room

               -- Add symbol to value string to denote coninuation of property_list
               ELStrings.AppendExaminerString (TokenRec.TokenValue, StartString);
               WhiteSpace.Skip (InputLine, Index);
            end if;
         end if;

         exit when not AnnoContinuation or else InputLine.Length < Index or else
              InputLine.Content (Index) = ';';

         if TokenRec.TokenValue.Length < TokenRec.TokenValue.Content'Last then
            TokenRec.TokenValue.Length := TokenRec.TokenValue.Length + 1;
            TokenRec.TokenValue.Content (TokenRec.TokenValue.Length) :=
               InputLine.Content (Index);
            Index := Index + 1;
         else
            Index := Index + 1;  -- not sure what we do here yet - exceeded length of ExaminerLongString!
         end if;

      end loop;
      --# accept Flow_Message, 602, SPARK_IO.File_Sys, TokenRec.TokenValue.Content, "Content array is always sufficiently initialised" &
      --#        Flow_Message, 602, InputLine,         TokenRec.TokenValue.Content, "Content array is always sufficiently initialised" &
      --#        Flow_Message, 602, Index,             TokenRec.TokenValue.Content, "Content array is always sufficiently initialised" &
      --#        Flow_Message, 602, TokenRec,          TokenRec.TokenValue.Content, "Content array is always sufficiently initialised";
   end GetPropertyList;

   function Is_Alphanumeric (Ch : Character) return Boolean
   is
   begin
      return Ada.Characters.Handling.Is_Letter (Ch) or else
         Ada.Characters.Handling.Is_Digit (Ch);
   end Is_Alphanumeric;

   procedure GetIdentifier (InputLine : in     EStrings.T;
                            Index     : in out ELStrings.Positions;
                            TokenRec  :    out Token_Record)
   --# derives Index,
   --#         TokenRec from Index,
   --#                       InputLine;
   is
      StartPos  : EStrings.Positions;
      Searching : Boolean;
   begin
      StartPos := Index;
      Searching := True;
      Index := Index + 1; -- The first cheracter is alphanumeric

      while Searching and Index <= InputLine.Length loop
         if Is_Alphanumeric (InputLine.Content (Index)) then
            Index := Index + 1;
         elsif InputLine.Content (Index) = '_' then
            if Index < InputLine.Length and then
               Is_Alphanumeric (InputLine.Content (Index + 1))
            then
               Index := Index + 2;
            else
               Searching := False;
            end if;
         else
            Searching := False;
         end if;
      end loop;

      --# accept Flow_Message, 23, Tokenrec.TokenValue.Content, "The TokenValue.Length is set to the initialised part of the array";
      TokenRec.TokenValue.Length := Index - StartPos;
      for I in EStrings.Positions range StartPos .. Index - 1 loop
         TokenRec.TokenValue.Content ((I - StartPos) + 1) := InputLine.Content (I);
      end loop;
      --# end accept;

      TokenRec.Token := identifier;
      --# accept Flow_Message, 602, TokenRec, TokenRec.TokenValue.Content, "The Content array is defined up to TokenValue.Length";
   end GetIdentifier;

   procedure CheckForRW (RWord   : in     String;
                         Symbol  : in     Token_Type;
                         TokRec  : in out Token_Record)
   --# derives TokRec from *,
   --#                     RWord,
   --#                     Symbol;
   is
   begin
      if ELStrings.Eq1String (TokRec.TokenValue, RWord) then
         TokRec.Token := Symbol;
         ELStrings.CopyString (TokRec.TokenValue, RWord);
      end if;
   end CheckForRW;

   procedure CheckForAReservedWord (TokRec : in out Token_Record)
   --# derives TokRec from *;
   is
   begin
      case TokRec.TokenValue.Content (1) is
         when 'd' | 'D' =>
            CheckForRW ("derives", RWderives, TokRec);
         when 'f' | 'F' =>
            CheckForRW ("from", RWfrom, TokRec);
         when 'g' | 'G' =>
            CheckForRW ("global", RWglobal, TokRec);
         when 'i' | 'I' =>
            if TokRec.TokenValue.Length > 1 then
               case TokRec.TokenValue.Content (2) is
                  when 'n' | 'N' =>
                     if TokRec.TokenValue.Length > 2 then
                        case TokRec.TokenValue.Content (3) is
                           when 'h' | 'H' =>
                              CheckForRW ("inherit", RWinherit, TokRec);
                           when 'i' | 'I' =>
                              CheckForRW ("initializes", RWinitializes, TokRec);
                           when others =>
                              null;
                        end case;
                     else
                        CheckForRW ("in", RWin, TokRec);
                     end if;
                  when 's' | 'S' =>
                     CheckForRW ("is", RWis, TokRec);
                  when others =>
                     null;
               end case;
            end if;
         when 'm' | 'M' =>
            CheckForRW ("main_program", RWmain_program, TokRec);
         when 'o' | 'O' =>
            if TokRec.TokenValue.Length > 1 then
               case TokRec.TokenValue.Content (2) is
                  when 'u' | 'U' =>
                     CheckForRW ("out", RWout, TokRec);
                  when 'w' | 'W' =>
                     CheckForRW ("own", RWown, TokRec);
                  when others =>
                     null;
               end case;
            end if;
         when 'p' | 'P' =>
            CheckForRW ("protected", RWprotected, TokRec);
         when 't' | 'T' =>
            CheckForRW ("task", RWtask, TokRec);
         when others =>
            null;
      end case;
   end CheckForAReservedWord;

   procedure GetPunctuation (InputLine : in     EStrings.T;
                             Index     : in out ELStrings.Positions;
                             Token     : in     Token_Type;
                             TokenRec  :    out Token_Record)
   --# derives Index    from * &
   --#         TokenRec from Index,
   --#                       InputLine,
   --#                       Token;
   is
   begin
      TokenRec.Token := Token;
      --# accept Flow_Message, 23, TokenRec.TokenValue.Content, "Length = 1, so only Content (1) need be defined";
      TokenRec.TokenValue.Length := 1;
      TokenRec.TokenValue.Content (1) := InputLine.Content (Index);
      --# end accept;
      Index := Index + 1;
      --# accept Flow_Message, 602, TokenRec, TokenRec.TokenValue.Content, "Content array is always sufficiently initialised";
   end GetPunctuation;

   procedure Initialise (InputFile : in      SPARK_IO.File_Type;
                         Anno      : in      Annotations.Anno_Type;
                         LexState  :     out State)
   is
   begin
      LexState := State'(File  => InputFile,
                         Anno         => Anno,
                         Line         => EStrings.EmptyString,
                         Index        => ELStrings.Positions'First,
                         InAnnotation => False);
   end Initialise;

   procedure Next (This     : in out State;
                   TokenRec :    out Token_Record)
   is
      Index           : ELStrings.Positions;
      InputFile       : SPARK_IO.File_Type;
      Anno            : Annotations.Anno_Type;
      NowInAnnotation : Boolean;
   begin
      Index := This.Index;
      InputFile := This.File;
      Anno := This.Anno;
      NowInAnnotation := This.InAnnotation;

      ExtendedSkipWhiteSpace (InputFile, Anno, NowInAnnotation, This.Line, Index);

      if not NowInAnnotation then
         This.InAnnotation := False;
         TokenRec := Token_Record'(Token      => annotation_end,
                                   TokenValue => ELStrings.EmptyString);
      elsif not This.InAnnotation then -- and NowInAnnotation
         This.InAnnotation := True;
         TokenRec := Token_Record'(Token      => annotation_start,
                                   TokenValue => ELStrings.EmptyString);
      else
         if Is_Alphanumeric (This.Line.Content (Index)) then
            GetIdentifier (This.Line, Index, TokenRec);
            CheckForAReservedWord (TokenRec);
         else
            case This.Line.Content (Index) is
               when ':' =>
                  GetPunctuation (This.Line, Index, colon, TokenRec);
               when ',' =>
                  GetPunctuation (This.Line, Index, comma, TokenRec);
               when '(' =>
                  GetPropertyList (InputFile, Anno, This.Line, Index, TokenRec);
               when '.' =>
                  GetPunctuation (This.Line, Index, point, TokenRec);
               when ';' =>
                  GetPunctuation (This.Line, Index, semicolon, TokenRec);
               when others =>
                  GetPunctuation (This.Line, Index, other_punct, TokenRec);
            end case;
         end if;
      end if;

      This.Index := Index;
   end Next;

   function GetColNo (This : State) return EStrings.Positions
   is
   begin
      return This.Index;
   end GetColNo;

end SimpleLex;
