-- $Id: sparkprogram-iteration.adb 11376 2008-10-08 11:09:31Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


separate (SparkProgram)
package body Iteration is

   procedure FindNextAlphabetic (TheHeap     : in     Heap.HeapRecord;
                                 TheIterator : in out Iterator)
   --# global in LexTokenManager.StringTable;
   --# derives TheIterator from *,
   --#                          LexTokenManager.StringTable,
   --#                          TheHeap;
   is
      FirstMember : SeqAlgebra.MemberOfSeq;
      Placeholder : LexTokenManager.LexString;
      SeqComplete : Boolean;
      NextItem    : SeqAlgebra.MemberOfSeq;

      ThisMember    : SeqAlgebra.MemberOfSeq;
      MemberVal     : Natural;
      ThisLexString : LexTokenManager.LexString;
      NextItemLex   : LexTokenManager.LexString;

   begin

      ---------------------------------------------------------------------------------------
      -- We have a sequence of (lex) strings in no particular order. To return them
      -- in alphabetical order we go through the whole sequence looking for the first
      -- item in alphabetical order, return it, then start again looking for the next
      -- item and so on. To do this we need to use a placeholder to tell us what the last
      -- thing we returned was so that the state of the search is preserved between calls.
      --
      -- Each time this routine is called it loops over the whole sequence, comparing each
      -- item with the placeholder to try and find the next best match.
      -- (Note that it may be possible, and more efficient, to do this by deleting items
      -- from the sequence once they have been returned, but need to be sure that sequences
      -- are never re-used, eg when several exports have the same set of imports.)
      -- Consider doing this later if performance is an issue.
      --
      -- We know we have finished when we have traversed the whole sequence without finding
      -- a better match.
      --
      -- Note:
      --    The sequence is very likely to be in alphabetical order already. If it is then
      --    we can just write it straight out. If SPARKFormat needs to be made faster then
      --    this subprogram could check whether the sequence is already sorted on the first
      --    pass through (easy to check). If it is then it could just be written out in the
      --    order in which items occur in the sequence.
      ---------------------------------------------------------------------------------------

      FirstMember := TheIterator.FirstMember;
      Placeholder := TheIterator.Placeholder;

      ThisMember := FirstMember;
      SeqComplete := True;

      -- If this is the first call then initialize NextItemLex to first item in sequence.
      -- Otherwise, the best match so far is the last thing that was written.
      if Placeholder = LexTokenManager.NullString then
         MemberVal := SeqAlgebra.ValueOfMember (TheHeap, ThisMember);
         NextItemLex := LexTokenManager.ConvertLexStringRef (MemberVal);
         NextItem := ThisMember;
      else
         NextItemLex := Placeholder;
         NextItem := ThisMember;
      end if;

      loop

         exit when SeqAlgebra.IsNullMember (ThisMember);

         MemberVal := SeqAlgebra.ValueOfMember (TheHeap, ThisMember);
         ThisLexString := LexTokenManager.ConvertLexStringRef (MemberVal);

         -- For this to be the next item to write it has to come strictly after the last item that was written
         -- (Note that this test will fail in the case of MultiplyToken so we don't need a separate test to avoid
         -- writing it out in the middle of a list.)
         if LexTokenManager.LexStringCaseInsensitiveCompare (ThisLexString, Placeholder) = LexTokenManager.StrSecond then

            -- If NextItemLex = Placeholder it indicates that we haven't updated NextItemLex on this
            -- pass, so NextItemLex becomes the current item (provided current item is after Placeholder).
            -- Or, if this item is before (or equal to) the current best match then it becomes the new best match.
            if (NextItemLex = Placeholder) or else
               (LexTokenManager.LexStringCaseInsensitiveCompare (NextItemLex, ThisLexString) /= LexTokenManager.StrFirst)
            then
               NextItemLex := ThisLexString;
               NextItem := ThisMember;
               SeqComplete := False;
            end if;

         end if;

         ThisMember := SeqAlgebra.NextMember (TheHeap, ThisMember);

      end loop;

      TheIterator.Placeholder := LexTokenManager.ConvertLexStringRef (SeqAlgebra.ValueOfMember (TheHeap, NextItem));
      TheIterator.CurrentMember := NextItem;
      TheIterator.Complete := SeqComplete;

   end FindNextAlphabetic;

   procedure Initialise (TheHeap            : in     Heap.HeapRecord;
                         TheSeq             : in     SeqAlgebra.Seq;
                         AlphabeticOrdering : in     Boolean;
                         TheIterator        :    out Iterator)
   is
   begin
      TheIterator := Iterator'(FirstMember        => SeqAlgebra.FirstMember (TheHeap, TheSeq),
                               CurrentMember      => SeqAlgebra.FirstMember (TheHeap, TheSeq),
                               AlphabeticOrdering => AlphabeticOrdering,
                               Placeholder        => LexTokenManager.NullString,
                               Complete           => SeqAlgebra.IsEmptySeq (TheHeap, TheSeq));

      if AlphabeticOrdering then
         FindNextAlphabetic (TheHeap, TheIterator);
      end if;

   end Initialise;

   procedure Next (TheHeap     : in     Heap.HeapRecord;
                   TheIterator : in out Iterator)
   is
   begin
      if not SeqAlgebra.IsNullMember (TheIterator.CurrentMember) then
         if TheIterator.AlphabeticOrdering then
            FindNextAlphabetic (TheHeap, TheIterator);
         else
            TheIterator.CurrentMember :=
              SeqAlgebra.NextMember (TheHeap,
                                     TheIterator.CurrentMember);
            if SeqAlgebra.IsNullMember (TheIterator.CurrentMember) then
               TheIterator.Complete := True;
            else
               TheIterator.Complete := False;
               TheIterator.Placeholder := LexTokenManager.ConvertLexStringRef (SeqAlgebra.ValueOfMember (TheHeap, TheIterator.CurrentMember));
            end if;
         end if;
      else
         -- This indicates that CurrentMember has not changed.
         TheIterator.Complete := True;
      end if;
   end Next;

   function Complete (TheIterator : Iterator) return Boolean is
   begin
      return TheIterator.Complete;
   end Complete;

   function CurrentString (TheIterator : Iterator)
                           return LexTokenManager.LexString
   is
   begin
      return TheIterator.Placeholder;
   end CurrentString;

   function CurrentMember (TheIterator : Iterator)
                           return SeqAlgebra.MemberOfSeq
   is
   begin
      return TheIterator.CurrentMember;
   end CurrentMember;

end Iteration;
