-- $Id: sparkprogram-annotations.adb 11892 2008-12-12 15:55:16Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

separate (SparkProgram)
package body Annotations is
   procedure Initialize (This       :    out Anno_Type;
                         AnnoIntro  : in     String;
                         AnnoSucc   : in     String;
                         AnnoIndent : in     Natural)
   is
      StartStr : AnnoStart_Type;
      LocIntro : EStrings.T;
      LocSucc  : EStrings.T;
   begin
      StartStr := AnnoStart_Type'(1 | 2 => '-',
                                  3 => CommandLineData.Content.AnnoChar);
      EStrings.CopyString (LocIntro, AnnoIntro);

      if AnnoSucc = "" then
         LocSucc := EStrings.EmptyString;
      else
         EStrings.CopyString (LocSucc, AnnoSucc);
      end if;

      This := Anno_Type'(AnnoStart   => StartStr,
                         AnnoIntro   => LocIntro,
                         AnnoSucc    => LocSucc,
                         AnnoIndent  => AnnoIndent);
   end Initialize;

   function Intro (This : Anno_Type) return EStrings.T
   is
   begin
      return This.AnnoIntro;
   end Intro;

   function Indent (This : Anno_Type) return Natural
   is
   begin
      return This.AnnoIndent;
   end Indent;

   procedure IsAnnoStart (This      : in     Anno_Type;
                          InputLine : in     EStrings.T;
                          Index     : in out EStrings.Lengths;
                          OK        :    out Boolean)
   is
   begin
      WhiteSpace.Skip (InputLine, Index);

      if InputLine.Length >= Index + 2 and then
         InputLine.Content (Index)     = This.AnnoStart (1) and then
         InputLine.Content (Index + 1) = This.AnnoStart (2) and then
         InputLine.Content (Index + 2) = This.AnnoStart (3)
      then
         Index := Index + 3;
         OK := True;
      else
         OK := False;
      end if;

   end IsAnnoStart;

   function StartsWith (StartStr     : EStrings.T;
                        InputLine    : EStrings.T;
                        InitialIndex : EStrings.Positions)
                        return Boolean
   is
      Index : EStrings.Positions;
      LocOK : Boolean;
   begin
      Index := InitialIndex;
      WhiteSpace.Skip (InputLine, Index);

      LocOK := InputLine.Length > 0 and then
         ((InputLine.Length - Index) + 1) >= StartStr.Length;

      if LocOK then
         for I in EStrings.Lengths range 1 .. StartStr.Length
         loop
            LocOK := Ada.Characters.Handling.To_Lower (StartStr.Content (I)) =
              Ada.Characters.Handling.To_Lower
                (InputLine.Content ((I + Index) - 1));
            exit when not LocOK;
         end loop;
      end if;

      return LocOK;

   end StartsWith;

   function IsStartOf (This        : Anno_Type;
                       InputLine : EStrings.T)
                          return Boolean
   is
      Index  : EStrings.Positions;
      Result : Boolean;
   begin
      Index := 1;
      IsAnnoStart (This, InputLine, Index, Result);
      Result := Result and then StartsWith (This.AnnoIntro, InputLine, Index);
      return Result;
   end IsStartOf;

   function IsEndOf (This        : Anno_Type;
                     InputLine : EStrings.T) return Boolean
   is
      Index  : EStrings.Positions;
      LocOK  : Boolean;
      Result : Boolean;
   begin
      Index := 1;
      IsAnnoStart (This, InputLine, Index, LocOK);

      if not LocOK then
         -- No longer in an annotation and therfore ended
         Result := True;
      else
         -- Otherwise only ended if this is the start of a successive annotation
         if This.AnnoSucc.Length > 0 then
            Result := StartsWith (This.AnnoSucc, InputLine, Index);
         else
            Result := False;
         end if;
      end if;

      return Result;

   end IsEndOf;

   function ProperStartCol (StartCol : EStrings.Positions)
      return EStrings.Positions
   is
   begin
      return (StartCol + AnnoStart_Type'Length) + 1;
   end ProperStartCol;


   procedure Write (This      : in Anno_Type;
                    Output    : in SPARK_IO.File_Type;
                    StartCol  : in EStrings.Positions)
   is
   begin
      SPARK_IO.Set_Col (Output, StartCol);
      SPARK_IO.Put_String (Output, This.AnnoStart, 0);
   end Write;

   procedure WriteIntro (This     : in Anno_Type;
                         Output   : in SPARK_IO.File_Type;
                         StartCol : in EStrings.Positions)
   is
   begin
      Write (This, Output, StartCol);
      SPARK_IO.Set_Col (Output, ProperStartCol (StartCol));
      EStrings.PutString (Output, This.AnnoIntro);
      -- If Indent is not Inline, then don't put out a trailing space,
      -- since this will be rejected by GNAT -gnatyb style check mode.
      if This.AnnoIndent = SparkFormatCommandLineData.Inline then
         SPARK_IO.Put_String (Output, " ", 0);
      end if;
   end WriteIntro;

   function Name1StartCol (This     : in Anno_Type;
                           StartCol : in EStrings.Positions)
                           return EStrings.Positions
   is
      NameColumn : EStrings.Positions;
   begin
      if This.AnnoIndent /= SparkFormatCommandLineData.Inline then
         -- StartCol "is --# " but indent value is given from "--#"
         NameColumn := ProperStartCol (StartCol) + (This.AnnoIndent - 1);
      else
         NameColumn := ProperStartCol (StartCol) + (This.AnnoIntro.Length + 1);
      end if;
      return NameColumn;
   end Name1StartCol;

end Annotations;
