-- $Id$
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with Ada.Exceptions,
     Ada.Command_Line,
     GNAT.Traceback.Symbolic,
     Heap,
     ScreenEcho,
     SPARKFormatMain,
     Version;

procedure SPARKFormat
is
   ---------------------------------------------------------------
   -- Some hosts (notably Apple's OSX/Darwin )place an arbitrary
   -- and unchangeable limit on the stack usage of the environment
   -- task, which is not sufficient to run SPARKFormat, where
   -- the flow analyser Heap can be large.
   --
   -- Therefore, we start a local task to do all the
   -- hard work, since such tasks are not subject to such
   -- limits.
   ---------------------------------------------------------------

   function Calculate_Storage_Size return Natural
   is
      Page_Size : constant := 2**12; -- 4 kbytes

      --                                    MBytes
      Other_Stack_Usage : constant := 64 * (2 ** 20);

      Total_Size : Natural;
   begin
      Total_Size := Heap.HeapSize + Other_Stack_Usage;
      return ((Total_Size + Page_Size) / Page_Size) * Page_Size;
   end Calculate_Storage_Size;

   task RunSPARKFormat is
      pragma Storage_Size (Calculate_Storage_Size);
   end RunSPARKFormat;

   task body RunSPARKFormat
   is
   begin
      SPARKFormatMain;
   exception
      when E : others =>
         ScreenEcho.New_Line (1);
         ScreenEcho.Put_Line ("Unexpected internal error in SPARKFormat");
         ScreenEcho.New_Line (1);
         ScreenEcho.Put_Line (Version.Toolset_Support_Line1);
         ScreenEcho.Put_Line (Version.Toolset_Support_Line2);
         ScreenEcho.Put_Line (Version.Toolset_Support_Line3);
         ScreenEcho.Put_Line (Version.Toolset_Support_Line4);
         ScreenEcho.New_Line (1);
         ScreenEcho.Put_Line ("Exception information:");
         ScreenEcho.Put_Line (Ada.Exceptions.Exception_Information (E));
         ScreenEcho.Put_Line ("Traceback:");
         ScreenEcho.Put_Line (GNAT.Traceback.Symbolic.Symbolic_Traceback (E));
         Ada.Command_Line.Set_Exit_Status (9);
   end RunSPARKFormat;

begin
   null;
end SPARKFormat;
