%  $Id: userules.pro 12833 2009-03-31 10:31:59Z Dean Kuo $
%-------------------------------------------------------------------------------
%  (C) Praxis High Integrity Systems Limited
%-------------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
%===============================================================================

%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Support facilities for applying any user defined proof rules.
%###############################################################################

%###############################################################################
% MODULE
%###############################################################################

%###############################################################################
% DEPENDENCIES
%###############################################################################

%###############################################################################
% TYPES
%###############################################################################

%###############################################################################
% DATA
%###############################################################################

%###############################################################################
% PREDICATES
%###############################################################################















































apply_user_defined_proof_rules :-
        i_am_using_rule(udr0),
        retractall(fact_from_rule(_,_,_,_,_,_)),
        retractall(subgoal_from_rule(_,_,_,_,_,_,_)),
        retractall(used_unique_reference(_,_)),
        retractall(proved_by_user_rules),
        fail.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr1),
        get_conc(Goal, x, N),
        \+ get_proved_conc(N),
        try_to_prove_via_inference_rule(Goal),
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr2),
        get_conc(Goal, x, N),
        \+ get_proved_conc(N),
        try_to_prove_via_rewrite_rule(Goal),
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr3),
        create_new_facts_from_hypotheses,
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr4),
        get_conc(Goal, x, N),
        \+ get_proved_conc(N),
        try_to_prove_via_inference_rule(Goal),
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr5),
        get_conc(Goal, x, N),
        \+ get_proved_conc(N),
        try_to_prove_via_rewrite_rule(Goal),
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr6),
        create_new_facts_from_existing_facts,
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr7),
        get_conc(Goal, x, N),
        \+ get_proved_conc(N),
        try_to_prove_via_inference_rule(Goal),
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

apply_user_defined_proof_rules :-
        i_am_using_rule(udr8),
        get_conc(Goal, x, N),
        \+ get_proved_conc(N),
        try_to_prove_via_rewrite_rule(Goal),
        fail.

apply_user_defined_proof_rules :-
        record_any_proved_conclusions,
        proved_all_conclusions,
        !.

% always succeeds (eventually)
apply_user_defined_proof_rules :-
    !.

%###############################################################################
% record_any_proved_conclusions:
% Determines which (if any) conclusions have been proved by the previous
% application of user rules. Records the conclusions as proved and promotes any
% rules used to prove side conditions of rules to hypotheses.
% This clause should be invoked after each phase of simplification in
% apply_user_defined_proof_rules to avoid the unnecessary application of rules
% to conclusions which are already proved.
%###############################################################################

record_any_proved_conclusions :-
        i_am_using_rule(udr9),
        get_conc(Goal, x, N),
        \+ get_proved_conc(N),
        is_a_fact_from_rule(_Ref, From, Goal, Name, RuleMatchDetails, RuleSort),
        record_conclusion_proved_by_rules(N, From, Goal, Name,
                                          RuleMatchDetails, RuleSort),
        fail.

record_any_proved_conclusions :-
    !.

%-------------------------------------------------------------------------------

record_conclusion_proved_by_rules(N, From, Goal, Name, RuleMatchDetails, RuleSort) :-
        issue_message('Applied rule ', Name),
        add_proved_conc(N),
        issue_proved_message(N),
        determine_facts_and_promote_to_hypotheses(From, RuleMatchDetails, NewRuleMatchDetails),
        assert_log_fact(rule_proved_conc, [N, Goal, Name, NewRuleMatchDetails, RuleSort]),
        mark_whether_proved_by_user_rules(RuleSort),
        !.

%-------------------------------------------------------------------------------

mark_whether_proved_by_user_rules(user_inference_rule) :-
        !,
        maybe_add(proved_by_user_rules).

mark_whether_proved_by_user_rules(user_rewrite_rule) :-
        !,
        maybe_add(proved_by_user_rules).

mark_whether_proved_by_user_rules(_) :- !.

% --------------------------------------------------------------------------
%                  FIND INFERENCE AND REWRITE RULE MATCHES.
% --------------------------------------------------------------------------

try_to_prove_via_inference_rule(Goal) :-
    % Goal is ground.
    inference_rule_match(Name, Goal, Conditions, RuleSort),
    var_free(Conditions), /* and give priority to matches with ground Conditions */
    fulfil(Conditions, ProvedConditions, UnsatisfiedConditions),
    var_free(ProvedConditions),
    (
        UnsatisfiedConditions = [],
        % success: don't backtrack and retry
        !,
        record_rule_success(conc, Goal, Name, RuleSort, [], [inference, ProvedConditions], _)
    ;
        fail
    ).

try_to_prove_via_inference_rule(Goal) :-
    % Goal is ground.
    inference_rule_match(Name, Goal, Conditions, RuleSort),
    % while the Conditions are not fully ground.
    \+ var_free(Conditions),
    fulfil(Conditions, ProvedConditions, UnsatisfiedConditions),
    var_free(ProvedConditions),
    (
        UnsatisfiedConditions = [],
        % success: don't backtrack and retry
        !,
        record_rule_success(conc, Goal, Name, RuleSort, [], [inference, ProvedConditions], _)
    ;
        % for now; add subgoal otherwise
        fail
    ).

%-------------------------------------------------------------------------------

try_to_prove_via_rewrite_rule(Goal) :-
    % Goal is ground.
    rewrite_rule_match(Name, Goal, NewGoal, Conditions, RuleSort),
    % and give priority to matches with ground Conditions
    var_free(Conditions),
    fulfil([NewGoal|Conditions], ProvedConditions, UnsatisfiedConditions),
    % once the side-conditions have been met
    novars(NewGoal),
    var_free(ProvedConditions),
    (
        UnsatisfiedConditions = [],
        % success: don't backtrack and retry */
        !,
        record_rule_success(conc, Goal, Name, RuleSort, [], [rewrite, NewGoal, ProvedConditions], _)
    ;
        fail
    ).

try_to_prove_via_rewrite_rule(Goal) :-
    % Goal is ground.
    rewrite_rule_match(Name, Goal, NewGoal, Conditions, RuleSort),
    % Conditions are not fully ground.
    \+ var_free(Conditions),
    fulfil([NewGoal|Conditions], ProvedConditions, UnsatisfiedConditions),
    % once the side-conditions have been met
    novars(NewGoal),
    var_free(ProvedConditions),
    (
        UnsatisfiedConditions = [],
        % success: don't backtrack and retry
        !,
        record_rule_success(conc, Goal, Name, RuleSort, [], [rewrite, NewGoal, ProvedConditions], _)
    ;
        fail
    ).

%-------------------------------------------------------------------------------

create_new_facts_from_hypotheses :-
        get_hyp(Formula, x, N),
        \+ know_eliminated(N),
        rewrite_rule_match(Name, Formula, NewFormula, Conditions, RuleSort),
        fulfil(Conditions, ProvedConditions, UnsatisfiedConditions),
        nonvar(NewFormula),
        dont_already_know(NewFormula),
        var_free(ProvedConditions),
        (
           UnsatisfiedConditions = [],
           record_rule_success(hyp(N), NewFormula, Name, RuleSort, [], [rewrite, Formula, ProvedConditions], _)
        ;
           fail
        ).

%-------------------------------------------------------------------------------

create_new_facts_from_existing_facts :-
        copy_existing_facts,
        candidate_fact(N, Formula),
        rewrite_rule_match(Name, Formula, NewFormula, Conditions, RuleSort),
        fulfil(Conditions, ProvedConditions, UnsatisfiedConditions),
        nonvar(NewFormula),
        dont_already_know(NewFormula),
        var_free(ProvedConditions),
        (
           UnsatisfiedConditions = [],
           record_rule_success(fact(N), NewFormula, Name, RuleSort, [], [rewrite, Formula, ProvedConditions], _)
        ;
           fail
        ).

%-------------------------------------------------------------------------------

copy_existing_facts :-
        retractall(candidate_fact(_,_)),
        fail.

copy_existing_facts :-
        fact_from_rule(N, _, Fact, _, _, _),
        Fact \= true,
        assertz(candidate_fact(N, Fact)),
        fail.

copy_existing_facts :- !.

%-------------------------------------------------------------------------------

% Find a user-defined inference rule and provide its generalised form.
inference_rule_match(Name, Goal, Conditions, RuleSort) :-
        nonvar(Goal),
        Goal =.. [OP|Args],
        make_up(Goal1, OP, Args),
        !,
        Goal1 =.. [_|Args1],
        (
            inference_rule(Name, Goal1, Conditions1),
            RuleSort = inference_rule
        ;
            user_inference_rule(Name, Goal1, Conditions1),
            RuleSort = user_inference_rule
        ),
        add_conds(Args=Args1, Conditions1, Conditions).

%-------------------------------------------------------------------------------

%*RECORD_RULE_SUCCESS(
%      From,                   -- Operation for which the rule was applied
%      Goal,                   -- Goal satisfied
%      Name,                   -- Name of applied rule
%      UnprovedConditions,     -- Side conditions not proved
%      RuleMatchDetails,       -- How the rule was applied
%      FactNumber              -- The fact number corresponding to
%                              -- the proof of the Goal
%      )

record_rule_success(_, Goal, _Name, _RuleSort, [], _RuleMatchDetails, FactNumber) :-
        fact_from_rule(FactNumber, _, Goal, _, _, _),   /* already known: don't add again */
        !.
record_rule_success(From, Goal, Name, RuleSort, [], RuleMatchDetails, Ref) :-
        generate_new_unique_reference(fact, Ref),
        assertz(fact_from_rule(Ref, From, Goal, Name, RuleMatchDetails, RuleSort)),
        !.

%-------------------------------------------------------------------------------

% REWRITE_RULE_MATCH(Name, Goal, NewGoal, Conditions, RuleSort):
% find a user-defined rewrite rule that matches and
% provide its generalised form.

% User-defined rewrite rules OR non-ground rewrite rules coming
% from the RLS file are both applicable here.

rewrite_rule_match(Name, Goal, NewGoal, NewConditions, RuleSort) :-
        nonvar(Goal),
        (
            user_rewrite_rule(Name, OldExp, NewExp, Conditions),
            RuleSort = user_rewrite_rule
        ;
            nonground_replace_rule(Name, OldExp, NewExp, Conditions),
            RuleSort = nonground_replace_rule
        ),

        pattern_match_rule(Goal, OldExp, NewExp, NewGoal, Conditions, NewConditions).

rewrite_rule_match(Name, Goal, NewGoal, NewConditions, RuleSort) :-
        nonvar(Goal),
        % User-defined rewrite rules OR non-ground rewrite rules coming
        % from the RLS file are both applicable here,
        (
            user_rewrite_rule(Name, NewExp, OldExp, Conditions),
            RuleSort = user_rewrite_rule
        ;
            nonground_replace_rule(Name, NewExp, OldExp, Conditions),
            RuleSort = nonground_replace_rule
        ),
        % don't use f(X) may_be_replaced_by X rules in reverse
        nonvar(OldExp),
        pattern_match_rule(Goal, OldExp, NewExp, NewGoal, Conditions, NewConditions).

%-------------------------------------------------------------------------------

pattern_match_rule(Goal, OldExp, NewExp, NewGoal, Conditions, Conditions) :-
        Goal=OldExp,
        NewGoal=NewExp.

pattern_match_rule(Goal, OldExp, NewExp, NewGoal, Conditions, NewConditions) :-
        ground(Goal),
        \+ atomic(Goal),
        Goal =.. [Op|Args],
        pattern_match_rule_list(Args, OldExp, NewExp, NewArgs, Conditions, NewConditions),
        NewGoal =.. [Op|NewArgs].

%-------------------------------------------------------------------------------

pattern_match_rule_list([OldArg|Rest], OldExp, NewExp, [NewArg|Rest], Conditions, NewConditions) :-
        find_pattern_match(OldArg, OldExp, NewExp, NewArg, [], ExtraConditions),
        append(ExtraConditions, Conditions, NewConditions).

pattern_match_rule_list([Arg|OldRest], OldExp, NewExp, [Arg|NewRest], Conditions, NewConditions) :-
        pattern_match_rule_list(OldRest, OldExp, NewExp, NewRest, Conditions, NewConditions).

%-------------------------------------------------------------------------------

find_pattern_match(OldArg, OldExp, NewExp, NewArg, Conditions, NewConditions) :-
        pattern_match_rule(OldArg, OldExp, NewExp, NewArg, Conditions, NewConditions).

find_pattern_match(OldArg, OldExp, _NewExp, NewArg, Conditions, NewConditions) :-
        ground(OldArg),
        OldArg =.. [Op|OldArgs],
        nonvar(OldExp),
        % have the same principal functor
        OldExp =.. [Op|_OldExps],
        find_pattern_match_list(OldArgs, NewArgs, Conditions, NewConditions),
        NewArg =.. [Op|NewArgs].

%-------------------------------------------------------------------------------

find_pattern_match_list([Arg|Args], [Arg|Exps], Conditions, NewConditions) :-
        !,
        find_pattern_match_list(Args, Exps, Conditions, NewConditions).

find_pattern_match_list([Arg|Args], [Exp|Args], Conditions, [Arg=Exp|Conditions]) :-
        !.

find_pattern_match_list([], [], Conditions, Conditions).


%-------------------------------------------------------------------------------

% MAKE_UP(F1, OP, Args) - make F1 have functor OP & most general arguments list.
make_up(F1, OP, Args) :-
        !,
        generalise(Args, Blanks),
        !,
        F1 =.. [OP|Blanks],
        !.

% GENERALISE(A, Blanks) - create list Blank of vars, of the same length as A.
generalise([], []) :- !.
generalise([_|T], [_|U]) :- !, generalise(T, U).


% ADD_CONDS(Xs=Ys, OldList, NewList) - unify lists Xs & Ys, creating NewList from OldList
add_conds([]=[],J,J) :- !.
add_conds([C1|T1]=[C1|T2], OldList, NewList) :- !, add_conds(T1=T2, OldList, NewList).
add_conds([C1|T1]=[C2|T2], OldList, [C1=C2|NewList]) :- !, add_conds(T1=T2, OldList, NewList).


% -------------------------------------------------------------------------
%  SOLVE A RULE MATCH'S SIDE CONDITIONS (HANDLING GROUND CONDITIONS FIRST).
% -------------------------------------------------------------------------

% INFERENCE RULE STRATEGY (based on that in the Proof Checker):
% F may_be_deduced_from GOALS.
%
% 1. Split GOALS into fully-instantiated-goals (i.e. primary goals) and
%    partially-instantiated-goals (i.e. secondary goals).
%
% 2. Attempt to satisfy all primary goals.  Cut & branch point.  Either:
%
%   a. All were satisfied.  Then attempt to satisfy secondary goals.
%      As soon as one becomes satisfied, split the remainder into primaries
%      and secondaries and attempt to satisfy them in the same way (i.e.
%      recursively).
%
%   b. At least one can be shown to be false.  This rule-match will never
%      succeed in its current instantiation, so cut and fail.
%
%   c. Some were not satisfied.  Leave secondary goals and backtrack.


fulfil([], [], []) :- !.

fulfil(Goals, ProvedGoals, UnsatisfiedGoals) :-
        split(Goals, Primaries, Secondaries),
        try_to_satisfy(Primaries, ProvedPrimaries, UnsatisfiedPrimaries),
        (
           UnsatisfiedPrimaries = [],
           match_up(Secondaries, ProvedSecondaries, UnsatisfiedSecondaries)
        ;
           UnsatisfiedPrimaries \= [],
           ProvedSecondaries = [],
           UnsatisfiedSecondaries = Secondaries
        ),
        var_free(UnsatisfiedSecondaries),       /* all conditions must be ground */
        append(ProvedPrimaries, ProvedSecondaries, ProvedGoals),
        append(UnsatisfiedPrimaries, UnsatisfiedSecondaries, UnsatisfiedGoals).

%-------------------------------------------------------------------------------

split([], [], []) :- !.

split([G|Gs], [G|Ps], Ss) :-
        novars(G),
        !,
        split(Gs, Ps, Ss),
        !.

split([G|Gs], Ps, [G|Ss]) :-
        split(Gs, Ps, Ss),
        !.

%-------------------------------------------------------------------------------

try_to_satisfy([], [], []) :- !.

try_to_satisfy([G|Gs], [proved(G,[],[])|Ps], Us) :-
        nonvar(G),
        G=goal(D),
        !,
        evaluate_immediate_condition(D),
        !,
        try_to_satisfy(Gs, Ps, Us),
        !.

try_to_satisfy([G|_], _, _) :-
        % stop immediately if so
        does_not_typecheck_as_boolean(G),
        !,
        fail.

try_to_satisfy([G|Gs], [proved(G,Hs,Fs)|Ps], Us) :-
        rule_phase_infer(G, Hs, Fs),
        !,
        try_to_satisfy(Gs, Ps, Us),
        !.

try_to_satisfy([G|_], _, _) :-
        simplify(not G, NotG),
        % if a ground goal is deducibly false, stop now
        infer(NotG, _),
        !,
        fail.

try_to_satisfy([G|Gs], Ps, [G|Us]) :-
        try_to_satisfy(Gs, Ps, Us),
        !.

%-------------------------------------------------------------------------------

match_up([], [], []) :- !.

match_up(Goals, ProvedGoals, UnsatisfiedGoals) :-
        seek_solutions([], Goals, Proved, Unsatisfied),
        % ensure list of unsatisfied goals is shrinking
        Goals \= Unsatisfied,
        split(Unsatisfied, Primaries, Secondaries),
        try_to_satisfy(Primaries, ProvedPrimaries, UnsatisfiedPrimaries),
        append(Proved, ProvedPrimaries, ProvedSoFar),
        match_up(Secondaries, ProvedSecondaries, UnsatisfiedSecondaries),
        append(ProvedSoFar, ProvedSecondaries, ProvedGoals),
        append(UnsatisfiedPrimaries, UnsatisfiedSecondaries, UnsatisfiedGoals).

%-------------------------------------------------------------------------------

seek_solutions(Passed, [G|Goals], [proved(G,Hs,Fs)|ProvedGoals], UnsatisfiedGoals) :-
        do_satisfy_goal(G, Hs, Fs),
        seek_solutions(Passed, Goals, ProvedGoals, UnsatisfiedGoals).

seek_solutions(Passed, [G|Goals], ProvedGoals, UnsatisfiedGoals) :-
        append(Passed, [G], NowPassed),
        seek_solutions(NowPassed, Goals, ProvedGoals, UnsatisfiedGoals).

seek_solutions(UnsatisfiedGoals, [], [], UnsatisfiedGoals) :- !.

%-------------------------------------------------------------------------------

do_satisfy_goal(G, [], []) :-
        nonvar(G),
        G = goal(Pred),
        !,
        evaluate_immediate_condition(Pred),
        !.

do_satisfy_goal(G, Hs, Fs) :-
        retractall(current_sat_goal(_)),
        asserta(current_sat_goal(G)),
        !,
        try_satisfy_goal(G, [], Hs, Fs).

%-------------------------------------------------------------------------------

try_satisfy_goal(G, InstanceList, H, F) :-
        current_sat_goal(Goal),
        satisfy_goal(Goal, Hs, Fs),
        \+ is_in(Goal, InstanceList),
        !,
        (
           G = Goal,
           H = Hs,
           F = Fs
        ;
           try_satisfy_goal(G, [Goal|InstanceList], H, F)
        ).

%-------------------------------------------------------------------------------

satisfy_goal(A=B, Hs, Fs) :-
        novars(A),
        simplify(A,X),
        A\=X,
        !,
        satisfy_goal(X=B, Hs, Fs).

satisfy_goal(A=B, Hs, Fs) :-
        novars(B),
        simplify(B,X),
        B\=X,
        !,
        satisfy_goal(A=X, Hs, Fs).

satisfy_goal(A<>B, Hs, Fs) :-
        novars(A),
        simplify(A,X),
        A\=X,
        !,
        satisfy_goal(X<>B, Hs, Fs).

satisfy_goal(A<>B, Hs, Fs) :-
        novars(B),
        simplify(B,X),
        B\=X,
        !,
        satisfy_goal(A<>X, Hs, Fs).

satisfy_goal(A>=B, Hs, Fs) :-
        novars(A),
        simplify(A,X),
        A\=X,
        !,
        satisfy_goal(X>=B, Hs, Fs).

satisfy_goal(A>=B, Hs, Fs) :-
        novars(B),
        simplify(B,X),
        B\=X,
        !,
        satisfy_goal(A>=X, Hs, Fs).

satisfy_goal(A<=B, Hs, Fs) :-
        novars(A),
        simplify(A,X),
        A\=X,
        !,
        satisfy_goal(X<=B, Hs, Fs).

satisfy_goal(A<=B, Hs, Fs) :-
        novars(B),
        simplify(B,X),
        B\=X,
        !,
        satisfy_goal(A<=X, Hs, Fs).

satisfy_goal(A>B, Hs, Fs) :-
        novars(A),
        simplify(A,X),
        A\=X,
        !,
        satisfy_goal(X>B, Hs, Fs).

satisfy_goal(A>B, Hs, Fs) :-
        novars(B),
        simplify(B,X),
        B\=X,
        !,
        satisfy_goal(A>X, Hs, Fs).

satisfy_goal(A<B, Hs, Fs) :-
        novars(A),
        simplify(A,X),
        A\=X,
        !,
        satisfy_goal(X<B, Hs, Fs).

satisfy_goal(A<B, Hs, Fs) :-
        novars(B),
        simplify(B,X),
        B\=X,
        !,
        satisfy_goal(A<X, Hs, Fs).

satisfy_goal(G, Hs, Fs) :-
        var(G),
        !,
        (
           get_hyp(G, x, H),
           integer(H),
           Hs = [H],
           Fs = []
        ;
           fact_from_rule(F, _, G, _, _, _),
           G \= true,
           Hs = [],
           Fs = [F]
        ).

satisfy_goal(G, Hs, []) :-
        fact(G, Hs).

satisfy_goal(Goal, [], [F]) :-
        fact_from_rule(F, _, Goal, _, _, _).

satisfy_goal(not G, Hs, Fs) :-
        nonvar(G),
        (
           G=(not H)
        ;
           G=(A=B),
           H=(A<>B)
        ;
           G=(A<>B),
           H=(A=B)
        ;
           G=(A>B),
           H=(A<=B)
        ;
           G=(A<B),
           H=(A>=B)
        ;
           G=(A>=B),
           H=(A<B)
        ;
           G=(A<=B),
           H=(A>B)
        ),
        satisfy_goal(H, Hs, Fs).

satisfy_goal(A and B, Hs, Fs) :-
        novars(A),
        !,
        rule_phase_infer(A, H1, F1),
        satisfy_goal(B, H2, F2),
        merge_sort(H1, H2, Hs),
        merge_sort(F1, F2, Fs).

satisfy_goal(A and B, Hs, Fs) :-
        novars(B),
        !,
        rule_phase_infer(B, H1, F1),
        satisfy_goal(A, H2, F2),
        merge_sort(H1, H2, Hs),
        merge_sort(F1, F2, Fs).

satisfy_goal(A and B, Hs, Fs) :-
        satisfy_goal(A, H1, F1),
        satisfy_goal(B, H2, F2),
        merge_sort(H1, H2, Hs),
        merge_sort(F1, F2, Fs).

satisfy_goal(A or B, Hs, Fs) :-
        novars(A),
        !,
        (
           rule_phase_infer(A, Hs, Fs)
        ;
           satisfy_goal(B, Hs, Fs)
        ).

satisfy_goal(A or B, Hs, Fs) :-
        novars(B),
        !,
        (
           rule_phase_infer(B, Hs, Fs)
        ;
           satisfy_goal(A, Hs, Fs)
        ).

satisfy_goal(A or _B, Hs, Fs) :-
        satisfy_goal(A, Hs, Fs).

satisfy_goal(_A or B, Hs, Fs) :-
        satisfy_goal(B, Hs, Fs).

satisfy_goal(A -> B, Hs, Fs) :-
        novars(A),
        !,
        (
           simplify(not A, NotA),
           rule_phase_infer(NotA, Hs, Fs)
        ;
           satisfy_goal(B, Hs, Fs)
        ).

satisfy_goal(A -> B, Hs, Fs) :-
        novars(B),
        !,
        (
           rule_phase_infer(B, Hs, Fs)
        ;
           satisfy_goal(not A, Hs, Fs)
        ).

satisfy_goal(A -> B, Hs, Fs) :-
        (
           satisfy_goal(not A, Hs, Fs)
        ;
           satisfy_goal(B, Hs, Fs)
        ).

satisfy_goal(A <-> B, [], []) :-
        (
           novars(B),
           var(A),
           simplify(B, B1),
           A=B1
        ;
           novars(A),
           var(B),
           simplify(A, A1),
           B=A1
        ).

satisfy_goal(A <-> B, Hs, Fs) :-
        satisfy_goal(A -> B, H1, F1),
        satisfy_goal(B -> A, H2, F2),
        merge_sort(H1, H2, Hs),
        merge_sort(F1, F2, Fs).

satisfy_goal(A=B, [], []) :-
        (
           novars(B),
           (
              var(A),
              rational_expression(B),
              \+ base_rational(B),
              evaluate_rational_expression(B, A)
           ;
              A=B
           )
        ;
           novars(A),
           (
              var(B),
              rational_expression(A),
              \+ base_rational(A),
              evaluate_rational_expression(A, B)
           ;
              B=A
           )
        ).

satisfy_goal(A<>B, Hs, []) :-
        (
           fact(A>B, Hs)
        ;
           fact(A<B, Hs)
        ).

satisfy_goal(A<>B, [], [F]) :-
        (
           fact_from_rule(F, _, A>B, _, _, _)
        ;
           fact_from_rule(F, _, B<A, _, _, _)
        ;
           fact_from_rule(F, _, A<B, _, _, _)
        ;
           fact_from_rule(F, _, B>A, _, _, _)
        ).

satisfy_goal(A>B, Hs, Fs) :-
        satisfy_goal(A>=B, H1, F1),
        novars(A<>B),
        rule_phase_infer(A<>B, H2, F2),
        merge_sort(H1, H2, Hs),
        merge_sort(F1, F2, Fs).

satisfy_goal(A<B, Hs, Fs) :-
        satisfy_goal(A<=B, H1, F1),
        novars(A<>B),
        rule_phase_infer(A<>B, H2, F2),
        merge_sort(H1, H2, Hs),
        merge_sort(F1, F2, Fs).

satisfy_goal(A>=B, Hs, Fs) :-
        (
           (
              fact(A>B, Hs)
           ;
              fact(B<A, Hs)
           ),
           Fs = []
        ;
           (
              fact_from_rule(F, _, A>B, _, _, _)
           ;
              fact_from_rule(F, _, B<A, _, _, _)
           ),
           Fs = [F],
           Hs = []
        ;
           satisfy_goal(A=B, Hs, Fs)
        ).

satisfy_goal(A<=B, Hs, Fs) :-
        (
           (
              fact(A<B, Hs)
           ;
              fact(B>A, Hs)
           ),
           Fs = []
        ;
           (
              fact_from_rule(F, _, A<B, _, _, _)
           ;
              fact_from_rule(F, _, B>A, _, _, _)
           ),
           Fs = [F],
           Hs = []
        ;
           satisfy_goal(A=B, Hs)
        ).

satisfy_goal(X=A+B, []) :-
        novars(X),
        (
           novars(B),
           A=X-B
        ;
           novars(A),
           B=X-A
        ).

satisfy_goal(X=A-B, []) :-
        novars(X),
        (
           novars(B),
           A=X+B
        ;
           novars(A),
           B=A-X
        ).

%-------------------------------------------------------------------------------

% If Goal is ground and can be directly inferred from the
% (also ground) Condition of an existing inference rule
% then try to infer that Condition directly
%
% In particular, if we're trying to prove a conclusion
%
%     s__tail(s) >= 0
%
% and we have a hypothesis
%
%     word__always_valid(s)
%
% and 2 rules
%     X >= 0 may_be_deduced_from [ word__always_valid(X) ].
%     word__always_valid(s__tail(s)) may_be_deduced_from [word__always_valid(s)].
%
% Then this special case allows the relevant reasoning to
% be established.
%
% This is only a partial solution, though.  In future,
% a more general solution that allows multiple __tail()'s
% to be "stripped away" should be sought.

rule_phase_infer(Goal, Hs, [Fact]) :-
        ground(Goal),
        inference_rule(Name, Goal, [Condition]),
        ground(Condition),
        infer(Condition, Hs),
        record_rule_success(rule_phase_inference, Goal, Name, inference_rule,
                        [], [inference, [proved(Condition, Hs, [])]], Fact),
        !.

rule_phase_infer(Goal, Hs, [Fact]) :-
        ground(Goal),
        user_inference_rule(Name, Goal, [Condition]),
        ground(Condition),
        infer(Condition, Hs),
        record_rule_success(rule_phase_inference, Goal, Name, user_inference_rule,
                        [], [inference, [proved(Condition, Hs, [])]], Fact),
        !.

% More generally.
rule_phase_infer(Goal, Hs, []) :-
        infer(Goal, Hs),
        !.
rule_phase_infer(Goal, [], [F]) :-
        is_a_fact_from_rule(F, _, Goal, _, _, _),
        !.

%-------------------------------------------------------------------------------

is_a_fact_from_rule(F, From, Goal, Name, Details, RuleSort) :-
    fact_from_rule(F, From, Goal, Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X=Y,  Name, Details, RuleSort) :-
    fact_from_rule(F, From, Y=X,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<=Y, Name, Details, RuleSort) :-
    fact_from_rule(F, From, X<Y,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<=Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y>=X, Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<=Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y>X,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X>=Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, X>Y,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X>=Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y<=X, Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X>=Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y<X,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<Y,  Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y>X,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X>Y,  Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y<X,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<>Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y<>X, Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<>Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, X<Y,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<>Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, X>Y,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<>Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y<X,  Name, Details, RuleSort).

is_a_fact_from_rule(F, From, X<>Y, Name, Details, RuleSort) :-
     fact_from_rule(F, From, Y>X,  Name, Details, RuleSort).

%-------------------------------------------------------------------------------

dont_already_know(Formula) :-
        % discard as a candidate new fact, if so
        rule_phase_infer(Formula, _, _),
        !,
        fail.

dont_already_know(_Formula) :-
    % otherwise succeed
    !.

%-------------------------------------------------------------------------------

does_not_typecheck_as_boolean(G) :-
        % fail, if G is an ok boolean expression
        checkhastype(G, boolean),
        !,
        fail.

% otherwise succeed
does_not_typecheck_as_boolean(_).

%-------------------------------------------------------------------------------

% evaluate_immediate_condition(Cond): succeed if the condition can be satisfied, fail otherwise.
evaluate_immediate_condition(X) :- var(X), !, fail.

evaluate_immediate_condition((X,Y)) :-          % and
        evaluate_immediate_condition(X),
        !,
        evaluate_immediate_condition(Y).

evaluate_immediate_condition((X;_)) :-          % or: lhs
        evaluate_immediate_condition(X).

evaluate_immediate_condition((_;Y)) :-          % or: rhs
        evaluate_immediate_condition(Y).

evaluate_immediate_condition(integer(N)) :-
        novars(N),
        signed_integer(N).

evaluate_immediate_condition(integer(N)) :-
        integer(N).

evaluate_immediate_condition(intexp(N)) :-
        novars(N),
        intexp(N).

evaluate_immediate_condition(checktype(E,T)) :-
        novars(E),
        (
           var(T)
        ;
           atom(T)
        ),
        !,
        checktype(E,T).

evaluate_immediate_condition(simplify(E,V)) :-
        novars(E),
        (
           var(V)
        ;
           novars(V)
        ),
        !,
        simplify(E,V).

% --------------------------------------------------------------------------
%      PROMOTE FACTS INTO THE HYPOTHESES AND CONVERT REFERENCES TO THEM
% --------------------------------------------------------------------------

determine_facts_and_promote_to_hypotheses(From, RuleMatchDetails, NewRuleMatchDetails) :-
        (
           RuleMatchDetails    = [inference, Conds],
           NewRuleMatchDetails = [inference, NewConds]
        ;
           RuleMatchDetails    = [rewrite, Goal, Conds],
           NewRuleMatchDetails = [rewrite, Goal, NewConds]
        ),
        !,
        gather_facts_list([from(From)|Conds], FactNos),
        sort(FactNos, SortedFactNos),
        promote_to_hypotheses(SortedFactNos),
        convert_conditions_list(Conds, NewConds).

%-------------------------------------------------------------------------------

gather_facts_list([H|T], Facts) :-
        gather_facts(H, F1),
        !,
        gather_facts_list(T, F2),
        !,
        append(F1, F2, Facts).

gather_facts_list([], []).

%-------------------------------------------------------------------------------

gather_facts(from(fact(N)), [N]).
gather_facts(from(_), []).
gather_facts(proved(_, _, F), F).
gather_facts(_, []).

%-------------------------------------------------------------------------------

promote_to_hypotheses([F|Fs]) :-
        promote_fact_to_hypothesis(F),
        !,
        promote_to_hypotheses(Fs).

promote_to_hypotheses([]).

%-------------------------------------------------------------------------------











%Add hyp for fact not seen before.
promote_fact_to_hypothesis(F) :-
    fact_from_rule(F, From, Fact, Name, RuleMatchDetails, RuleSort),
        (
         RuleMatchDetails    = [inference, Conds],
         NewRuleMatchDetails = [inference, NewConds]
         ;
         RuleMatchDetails    = [rewrite, Goal, Conds],
         NewRuleMatchDetails = [rewrite, Goal, NewConds]
         ),
    !,
    convert_conditions_list(Conds, NewConds),
    add_hyp(Fact, x, HypNo),
    issue_message('Applied rule ', Name),
    assert_log_fact(applied_rule, [HypNo, Fact, Name, NewRuleMatchDetails, RuleSort]),
    !,
    %Replace the processed fact, with a form that records the mapping.  The
    %mapping form has empty RuleMatchDetails, and thus will not trigger the
    %adding of a hypothesis again.
    retract(fact_from_rule(F, From, Fact, Name, RuleMatchDetails, RuleSort)),
    assertz(fact_from_rule(F, From, true, hyp(HypNo), [], RuleSort)),
    % record the mapping
    !.

%Silently skip over a fact that has been seen before.
promote_fact_to_hypothesis(F) :-
    fact_from_rule(F, _From, true, hyp(_HypNo), [], _RuleSort),
    !.




%-------------------------------------------------------------------------------

convert_conditions_list([Cond|Conds], [NewCond|NewConds]) :-
        convert_condition(Cond, NewCond),
        !,
        convert_conditions_list(Conds, NewConds).

convert_conditions_list([], []).

%-------------------------------------------------------------------------------

convert_condition(proved(Formula, H1, Fs), proved(Formula, NewHs, [])) :-
        lookup_fact_mappings(Fs, H2),
        append(H1, H2, Hs),
        !,
        sort(Hs, NewHs).

%-------------------------------------------------------------------------------

lookup_fact_mappings([F|Fs], [H|Hs]) :-
        fact_from_rule(F, _, true, hyp(H), [], _),
        !,
        lookup_fact_mappings(Fs, Hs).
lookup_fact_mappings([F|Fs], [H|Hs]) :-
        fact_from_rule(F, _, Fact, _, _, _),
        get_hyp(Fact, _, H),
        !,
        lookup_fact_mappings(Fs, Hs).

lookup_fact_mappings([], []).

% --------------------------------------------------------------------------
%                             UTILITY PREDICATES
% --------------------------------------------------------------------------

% Generate new unique reference
generate_new_unique_reference(Label, N) :-
        retract(used_unique_reference(Label, M)),
        !,
        N is M+1,
        asserta(used_unique_reference(Label, N)),
        !.

generate_new_unique_reference(Label, 1) :-
        asserta(used_unique_reference(Label, 1)),
        !.

%-------------------------------------------------------------------------------

% MAYBE_ADD(FACT) -- add fact if not already known
maybe_add(X) :-
        call(X),
        !.

maybe_add(X) :-
        assertz(X),
        !.

%###############################################################################
% END-OF-FILE
