%  $Id: datavcg.pro 12106 2009-01-13 10:11:30Z Bill Ellis $
%-------------------------------------------------------------------------------
%  (C) Praxis High Integrity Systems Limited
%-------------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
%===============================================================================


%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Provides access to all information related to vcs. This information will
% be retrieved from the provided vcg file. 
%###############################################################################


%###############################################################################
% MODULE
%###############################################################################

:- module(datavcg, [get_vcg_file_handle/1,
                    add_vcg_file_handle/1,

                    get_vcg_seen_vc/0,
                    add_vcg_seen_vc/0,
                    get_vcg_trace/2,
                    add_vcg_trace/2,
                    get_vcg_vc/5,
                    add_vcg_vc/5,
                    get_vcg_hypothesis/3,
                    add_vcg_hypothesis/3,
                    get_vcg_conclusion/3,
                    add_vcg_conclusion/3,

                    prune_vcg_file_handle/0,
                    prune_all_vcg_vc/0,
                    prune_all_vcg_hypothesis/0,
                    prune_all_vcg_conclusion/0,

                    save_datavcg/0]).


%###############################################################################
% DEPENDENCIES
%###############################################################################

:- use_module('dataformats.pro',
              [add_state/2,
               add_type/2]).
:- use_module('ioutilities.pro',
              [write_terms_to_file/2]).
:- use_module('datadatafiles.pro',
              [must_get_datafiles_debug/2]).


%###############################################################################
% TYPES
%###############################################################################

% For path(s) from XXX to YYY:
% For checks of refinement integrity:
:- add_type('VCTrace',
            [traverseCutpoints('VCCutpointFrom', 'VCCutpointTo'),
             checkRefinementIntegrity,
             subclassInheritanceIntegrity]).

% start
% assertion of line NN
% ZZZ assertion of line NN
:- add_type('VCCutpointFrom',
            [start,
             assertion('AssertionKind', 'Line_Int')]).

% finish
% assertion of line NN
% default assertion of line NN
% check associated with statement of line NN
% run-time check associated with statement of line NN
% precondition check associated with statement of line NN
:- add_type('VCCutpointTo',
            [finish,
             assertion('AssertionKind', 'Line_Int'),
             check('CheckKind', 'Line_Int')]).

:- add_type('AssertionKind',
            [userprovided,
             default]).

:- add_type('CheckKind',
            [userprovided,
             runtime,
             precondition]).


%###############################################################################
% DATA
%###############################################################################

% VCs are loaded one-at-a-time. Thus, need a location to store the VCG file
% handle between each VC load.
:- add_state(get_vcg_file_handle,
             get_vcg_file_handle('VcgFile_Stream')).
:- dynamic(get_vcg_file_handle/1).

% Record that a vc has been seen.
:- add_state(get_vcg_seen_vc,
             get_vcg_seen_vc).
:- dynamic(get_vcg_seen_vc/0).

% A vcg file has a number of trace lines.
% Each trace line has a number of VCs.
% Each VC has a number of hypotheses and a number of conclusions.
:- add_state(get_vcg_trace,
             get_vcg_trace('TraceId_Atom', 'VCTrace')).
:- dynamic(get_vcg_trace/2).

:- add_state(get_vcg_vc,
             get_vcg_vc('VCId_Atom',
                        'Order_Int',
                        'Name_Atom',
                        'Number_Int',
                        'ParentVCTraceId_Atom')).
:- dynamic(get_vcg_vc/5).

:- add_state(get_vcg_hypothesis,
             get_vcg_hypothesis('Number_Int', 'Hyp_Term', 'ParentVCId_Atom')).
:- dynamic(get_vcg_hypothesis/3).

:- add_state(get_vcg_conclusion,
             get_vcg_conclusion('Number_Int', 'Conc_Term', 'ParentVCId_Atom')).
:- dynamic(get_vcg_conclusion/3).


%###############################################################################
% PREDICATES
%###############################################################################


%===============================================================================
% Add.
%===============================================================================

add_vcg_file_handle(VcgFile_Stream):-
    assert(get_vcg_file_handle(VcgFile_Stream)),
    !.

add_vcg_seen_vc:-
    assert(get_vcg_seen_vc),
    !.

add_vcg_trace(TraceId_Atom, VCTrace):-
    assert(get_vcg_trace(TraceId_Atom, VCTrace)),
    !.

add_vcg_vc(VCId_Atom, Order_Int, Name_Atom, Number_Int, ParentVCTraceId_Atom):-
    assert(get_vcg_vc(VCId_Atom, Order_Int, Name_Atom, Number_Int, ParentVCTraceId_Atom)),
    !.

add_vcg_hypothesis(Number_Int, Hyp_Term, ParentVCId_Atom):-
    assert(get_vcg_hypothesis(Number_Int, Hyp_Term, ParentVCId_Atom)),
    !.

add_vcg_conclusion(Number_Int, Conc_Term, ParentVCId_Atom):-
    assert(get_vcg_conclusion(Number_Int, Conc_Term, ParentVCId_Atom)),
    !.
%===============================================================================


%===============================================================================
% Prune.
%===============================================================================

prune_vcg_file_handle:-
    retractall(get_vcg_file_handle(_VcgFile_Stream)),
    !.

prune_all_vcg_vc:-
    retractall(get_vcg_vc(_VCId_Atom, _Order_Int, _Name_Atom, _Number_Int, _ParentVCTraceId_Atom)),
    !.

prune_all_vcg_hypothesis:-
    retractall(get_vcg_hypothesis(_Number_Int, _Hyp_Term, _ParentVCId_Atom)),
    !.

prune_all_vcg_conclusion:-
    retractall(get_vcg_conclusion(_Number_Int, _Conc_Term, _ParentVCId_Atom)),
    !.
%===============================================================================


%===============================================================================
% save_datavcg.
%===============================================================================

save_datavcg:-
    must_get_datafiles_debug(datavcg, DebugFile_Atom),
    write_terms_to_file(DebugFile_Atom,
                        [datavcg:get_vcg_file_handle/1,
                         datavcg:get_vcg_seen_vc/0,
                         datavcg:get_vcg_trace/2,
                         datavcg:get_vcg_vc/5,
                         datavcg:get_vcg_hypothesis/3,
                         datavcg:get_vcg_conclusion/3]),
    !.
%===============================================================================


%###############################################################################
% END-OF-FILE
