%  $Id: datahyp.pro 12637 2009-03-06 16:35:56Z dean kuo $
%-------------------------------------------------------------------------------
%  (C) Praxis High Integrity Systems Limited
%-------------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
%===============================================================================

%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Stores, retrieves and replaces hypothesis into the database. 





%###############################################################################

%###############################################################################
% MODULE
%###############################################################################

:- module(datahyp, [add_hyp/3,
                    add_hyp_min_id/4,
                    add_hyp_with_id/3,
                    get_hyp/3,
                    replace_hyp/4,
                    prune_hyp/3,
                    prune_all_hyps/3]).

%###############################################################################
% DEPENDENCIES
%###############################################################################

:- use_module('dataformats.pro', 
              [add_state/2,
              add_type/2]).

:- use_module('dataprovenance.pro', 
              [path_functions/0]).

:- use_module('ioutilities.pro',
              [show_error/2]).

:- use_module('opdeclar.pro',
              [declare_operators/0,
               hide_operators/0]).

:- use_module(library(lists), 
              [remove_dups/2]).

:- declare_operators.

%###############################################################################
% TYPES
%###############################################################################

:- add_type('HypothesisType', 
            [x, 
             ss, 
             [s, 'TYPE']]).

%###############################################################################
% DATA
%###############################################################################

:- add_state(get_hyp,
             get_hyp('Hypothesis_Term', 'HypothesisType', 'Id_Int')).

:- dynamic(get_hyp/3).

%###############################################################################
% PREDICATES
%###############################################################################

%===============================================================================
% add_hyp(+Hyp_Term, +HypType, -Id_Int)
%
% The predicate will determinate and always succeeds. The identfier of the 
% inserted hypothesis is bound to Id_Int;
%
% If the hypothesis is a duplicate then it is not inserted into the 
% database and its identifier is bound to Id_Int;
%
% No hypothesis is added to the database if the hyp is "true" and 
% it is not a path function.




%===============================================================================

add_hyp(true,_,0) :- 
    \+ path_functions, 
    !.

add_hyp(Hyp_Term, _HypType_1, Id_Int) :-
    get_hyp(Hyp_Term, _HypType_2, Id_Int),
    user:assert_log_fact(duplicate_hyp, [Id_Int, Hyp_Term]),
    !.

add_hyp(Hyp_Term, HypType, Id_Int) :-
    next_hyp_id(Id_Int),
    assertz(get_hyp(Hyp_Term, HypType, Id_Int)),
    retractall(user:could_not_infer(_)),
    !.

%===============================================================================

%===============================================================================
% add_hyp_min_id(+Hyp_Term, +HypType, +Min_Id_Int, -Id_Int)
%
% Similiar to add_hyp with an additional argument that specifies the 
% minimum identifier for the new hypothesis and that duplicates are 
% inserted into the database. 
%
% An error is raised if MinId_Int is less than 1 - see predicate
% for next_hyp_id_with_min.




%===============================================================================

add_hyp_min_id(true, _, _, 0) :-
    \+ path_functions, 
    !.

add_hyp_min_id(Hyp_Term, HypType, MinId_Int, Id_Int) :-
    next_hyp_id_with_min(MinId_Int, Id_Int),
    assertz(get_hyp(Hyp_Term, HypType, Id_Int)),
    retractall(user:could_not_infer(_)),
    !.

%===============================================================================

%===============================================================================
% add_hyp_with_id(+Hyp_Term, +HypType, +Id_Int)
%
% Adds a hypothesis into the database where all arguments are bounded.
%
% The predicate performs no check for duplicates or if the identifier
%    is already used.






%===============================================================================

add_hyp_with_id(Hyp_Term, HypType, Id_Int) :-
    assertz(get_hyp(Hyp_Term, HypType, Id_Int)).

%===============================================================================

%===============================================================================
% replace_hyp(+OldHyp_Term, +HypType, +HypId_Int, +NewHyp_Term)
%
% Replace a hypothesis in the database with a new hypothesis. 
%
% The predicate fails if OldHyp_Term, HypType, HypId_Ind is not 
% in the database.
%===============================================================================
    
replace_hyp(OldHyp_Term, HypType, HypId_Int, NewHyp_Term):-
    prune_hyp(OldHyp_Term, HypType, HypId_Int),
    add_hyp_with_id(NewHyp_Term, HypType, HypId_Int).

%===============================================================================

%===============================================================================
% prune_hyp(?Hyp_Term, ?HypType, ?HypId_Int)
%
% Predicate retracts a hypthesis from the database. 
%
% Predicate fails if nothing is deleted from the database.
%===============================================================================

prune_hyp(Hyp_Term, HypType, HypId_Int):-
    retract(get_hyp(Hyp_Term, HypType, HypId_Int)). 

%===============================================================================

%===============================================================================
% prune_all_hyps(?Hyp_Term, ?HypType, ?HypId_Int)
%
% Retracts all hypothesises from the database that matches the 
% arguments.
%
% Predicate always succeeds independent of whether anything is deleted 
% from the database.
%===============================================================================

prune_all_hyps(Hyp_Term, HypType, HypId_Int):-
    retractall(get_hyp(Hyp_Term, HypType, HypId_Int)).

%===============================================================================

%===============================================================================
% next_hyp_id(?FreeId_Int) : private
%
% The next free Id.




%===============================================================================

next_hyp_id(FreeId_Int):-
    next_hyp_id_with_min_x(1, FreeId_Int),
    !.

%===============================================================================

%===============================================================================
% next_hyp_id_with_min(+MinId_Int, -FreeId_Int) : private
%
% FreeId_Int is the next free Id that is greater or equal to MinId_Int.
% Program is halted should the minimum identifier be less than 1. 
%


%===============================================================================

next_hyp_id_with_min(MinId_Int, FreeId_Int) :-
    MinId_Int >= 1, 
    next_hyp_id_with_min_x(MinId_Int, FreeId_Int), 
    !.

next_hyp_id_with_min(MinId_Int, _FreeId_Int) :-
    % Display error message and halt if Min_Id_Int is less than 1.
    % See the predicate above.
    show_error('Can not find next hyopthesis id where its minimum ~a is less than 1.', 
               [MinId_Int]).

%===============================================================================

%===============================================================================
% next_hyp_id_with_min_x(+MinId_Int, ?FreeId_Int) : private
%
% Secondary predicate to next_hyp_id_with_min and next_hyp_id. 
% The predicate assume that its minimum identifier is greater than or equal
% to 1.
%===============================================================================
    
%
% If there are no hypothesis in the database and MinId_Int = 1 then 
% FreeId_Int is 1.
%  

next_hyp_id_with_min_x(MinId_Int, FreeId_Int) :-
    get_hyp(_, _, MinId_Int),
    !,
    NextId_Int is MinId_Int + 1,
    next_hyp_id_with_min_x(NextId_Int, FreeId_Int), 
    !. 

next_hyp_id_with_min_x(FreeId_Int, FreeId_Int) :-
    % No hyp with identifier FreeId_Int
    !.

%===============================================================================

%###############################################################################
% END-OF-FILE
