%  $Id: datahyp.pro 12637 2009-03-06 16:35:56Z dean kuo $
%-------------------------------------------------------------------------------
%  (C) Praxis High Integrity Systems Limited
%-------------------------------------------------------------------------------
% 
%  The SPARK toolset is free software; you can redistribute it and/or modify it
%  under terms of the GNU General Public License as published by the Free
%  Software Foundation; either version 3, or (at your option) any later
%  version. The SPARK toolset is distributed in the hope that it will be
%  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
%  Public License for more details. You should have received a copy of the GNU
%  General Public License distributed with the SPARK toolset; see file
%  COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
%  the license.
% 
%===============================================================================

%###############################################################################
% PURPOSE
%-------------------------------------------------------------------------------
% Stores, retrieves and replaces conclusions into the database. 
%###############################################################################

%###############################################################################
% MODULE
%###############################################################################

:- module(dataconc, [add_conc/3,
                     add_conc_min_id/4,
                     add_conc_with_id/3,
                     get_conc/3,
                     replace_conc/4,
                     prune_conc/3,
                     prune_all_concs/3]).

%###############################################################################
% DEPENDENCIES
%###############################################################################

:- use_module('dataformats.pro', 
              [add_state/2,
               add_type/2]).

:- use_module('ioutilities.pro',
              [show_error/2]).

%###############################################################################
% TYPES
%###############################################################################

:- add_type('ConclusionType', 
            [x, 
             ss, 
             [s, 'TYPE']]).

%###############################################################################
% DATA
%###############################################################################

:- add_state(get_conc,
             get_conc('Conclusion_Term', 'ConclusionType', 'Id_Int')).

:- dynamic(get_conc/3).

%###############################################################################
% PREDICATES
%###############################################################################

%===============================================================================
% add_conc(+Conc_Term, +ConcType, -Id_Int)
%
% The conclusion Conc_Term of type ConcType is added to the conclusion 
% database and Id_Int is bound to the conclusion's identifier.


%===============================================================================

add_conc(Conc_Term, ConcType, Id_Int) :-
    next_conc_id(Id_Int),
    assertz(get_conc(Conc_Term, ConcType, Id_Int)),
    !.

%===============================================================================

%===============================================================================
% add_conc_with_id(+Conc_Term, +ConcType, +Id_Int)
%
% Adds a conclusion into the database where all arguments are bounded.
%===============================================================================

add_conc_with_id(Conc_Term, ConcType, Id_Int) :-
    assertz(get_conc(Conc_Term, ConcType, Id_Int)).

%===============================================================================

%===============================================================================
% replace_conc(+OldConc_Term, +ConcType, +ConcId_Int, +ConcHyp_Term)
%
% Replace a conclusion in the database with a new conclusion. 
%
% The predicate fails if OldConc_Term, ConcType, ConcId_Ind is not 
% in the database.
%===============================================================================

replace_conc(OldConc_Term, ConcType, ConcId_Int, NewConc_Term):-
    prune_conc(OldConc_Term, ConcType, ConcId_Int),
    add_conc_with_id(NewConc_Term, ConcType, ConcId_Int).

%===============================================================================

%===============================================================================
% prune_conc(?Conc_Term, ?ConcType, ?ConcId_Int)
%
% Predicate retracts a conclusion from the database and fails if no conclusion
% is deleted.
%===============================================================================

prune_conc(Conc_Term, ConcType, ConcId_Int):-
    retract(get_conc(Conc_Term, ConcType, ConcId_Int)). 

%===============================================================================

%===============================================================================
% prune_all_concs(?Conc_Term, ?ConcType, ?ConcId_Int)
%
% Retracts all conclusions from the database that matche the 
% bounded arguments.
%
% Predicate always succeeds independent of whether anything is deleted.
%===============================================================================

prune_all_concs(Conc_Term, ConcType, ConcId_Int):-
    retractall(get_conc(Conc_Term, ConcType, ConcId_Int)).

%===============================================================================

%===============================================================================
% next_conc_id(?FreeId_Int) : private
%
% Finds the next free conclusion identifier.
%===============================================================================

next_conc_id(FreeId_Int):-
    next_conc_id_with_min_x(1, FreeId_Int),
    !.

%===============================================================================
% next_conc_id_with_min_x(+MinId_Int, ?FreeId_Int) : private
%
% Secondary predicate to next_conc_id_with_min and next_conc_id. 
% The predicate assume that its minimum identifier is greater than or equal
% to 1.
%===============================================================================
    
%
% If there are no conclusions in the database and MinId_Int = 1 then 
% FreeId_Int is 1, that is:
% 
% next_conc_id_with_min_x(1, 1):-
%    !.
%  

next_conc_id_with_min_x(MinId_Int, FreeId_Int) :-
    get_conc(_, _, MinId_Int),
    !,
    NextId_Int is MinId_Int + 1,
    next_conc_id_with_min_x(NextId_Int, FreeId_Int), 
    !. 

next_conc_id_with_min_x(FreeId_Int, FreeId_Int) :-
    % No conc with identifier FreeId_Int - see above.
    !.

%===============================================================================

%###############################################################################
% END-OF-FILE
