-- $Id: xmlsummary.adb 12255 2009-01-22 09:28:46Z Robin Messer $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


---------------------------------------------------------------------------------
-- Package for generating an XML version of the summary file.                  --
---------------------------------------------------------------------------------
with ELStrings,
     PathFormatter;

package body XMLSummary
--# own State is Schema, SchemaState, TagDepths, TagIDs, LastSummarySection;
is

   type Tags is (TPogs,
                 TType,
                 TTypes,
                 TDirectory,
                 TProduced,
                 TResults,
                 TFile,
                 TError,
                 TVc,
                 TSummary,
                 TSubprograms,
                 TSubprogramsInError,
                 TVcs,
                 TTools,
                 TAssert_Post,
                 TCheck,
                 TInheritance,
                 TPrecond,
                 TRefinement,
                 TRuntime,
                 TTotals,
                 TPercentages,
                 TExaminer,
                 TSimplifier,
                 TSimplifierUserRule,
                 TChecker,
                 TReview,
                 TFalse,
                 TContra,
                 TUser,
                 TUndetermined,
                 TUndischarged,
                 TUndetSection
                );

   type TagDepthArray is array (Tags) of SPARK_XML.TagDepth;

   type TagIDArray is array (Tags) of SPARK_XML.TagID;


   -- Package State
   TagDepths : TagDepthArray;
   TagIDs : TagIDArray;
   Schema : SPARK_XML.SchemaRecord;
   SchemaState : SPARK_XML.SchemaStateRecord;

   LastSummarySection : Tags;


   --------------------
   -- Error handling --
   --------------------

   procedure HandleError (Status : in SPARK_XML.SchemaStatus)
   --# global in Schema;
   --#        in SchemaState;
   --# derives null from Schema,
   --#                   SchemaState,
   --#                   Status;
   is
      --# hide HandleError;
   begin
      if SPARK_XML.IsError (Status) then
         SPARK_XML.PrintSchemaError (Status);
         SPARK_XML.PrintWorkingState (Schema, SchemaState);
      end if;
   end HandleError;


   procedure HandleSchemaError (Success : in Boolean;
                                Msg     : in String)
   --# derives null from Msg,
   --#                   Success;
   is
      --# hide HandleSchemaError;
   begin
      if not Success then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output, Msg, 0);
         SPARK_XML.PrintWorkingState (Schema, SchemaState);
      end if;
   end HandleSchemaError;





   function XStr (Str : in String) return XMLString
   is
   begin
      return SPARK_XML.XStr (Str);
   end XStr;

   procedure Init
   --# global out LastSummarySection;
   --#        out Schema;
   --#        out SchemaState;
   --#        out TagDepths;
   --#        out TagIDs;
   --# derives LastSummarySection,
   --#         Schema,
   --#         SchemaState,
   --#         TagDepths,
   --#         TagIDs             from ;
   is

      -- Load the tag definitions in to the Schema
      procedure LoadTags
      --# global in out Schema;
      --#           out TagIDs;
      --# derives Schema,
      --#         TagIDs from Schema;
      is

         type XStringArrayType is array (Tags) of XMLString;
         TagStrings : XStringArrayType;
         TmpTag : SPARK_XML.TagID;

      begin

         TagStrings := XStringArrayType'(
                                       TPogs               => XStr ("pogs"),
                                       TType               => XStr ("type"),
                                       TTypes              => XStr ("types"),
                                       TDirectory          => XStr ("directory"),
                                       TProduced           => XStr ("produced"),
                                       TResults            => XStr ("results"),
                                       TFile               => XStr ("file"),
                                       TError              => XStr ("error"),
                                       TVc                 => XStr ("vc"),
                                       TSummary            => XStr ("summary"),
                                       TSubprograms        => XStr ("subprograms"),
                                       TSubprogramsInError => XStr ("subprograms_in_error"),
                                       TVcs                => XStr ("vcs"),
                                       TTools              => XStr ("tools"),
                                       TAssert_Post        => XStr ("assert_post"),
                                       TCheck              => XStr ("check"),
                                       TInheritance        => XStr ("inheritance"),
                                       TPrecond            => XStr ("precond"),
                                       TRefinement         => XStr ("refinement"),
                                       TRuntime            => XStr ("runtime"),
                                       TTotals             => XStr ("totals"),
                                       TPercentages        => XStr ("percentages"),
                                       TExaminer           => XStr ("examiner"),
                                       TSimplifier         => XStr ("simplifier"),
                                       TSimplifierUserRule => XStr ("simplifier_user_rule"),
                                       TChecker            => XStr ("checker"),
                                       TReview             => XStr ("review"),
                                       TFalse              => XStr ("false"),
                                       TContra             => XStr ("contradiction"),
                                       TUser               => XStr ("user"),
                                       TUndetermined       => XStr ("undetermined"),
                                       TUndischarged       => XStr ("undischarged"),
                                       TUndetSection       => XStr ("undetermined_summary")
                                      );

         TagIDs := TagIDArray'(others => SPARK_XML.NullTag);

         for I in Tags loop
            SPARK_XML.AddTag (Schema, TagStrings (I), TmpTag);
            TagIDs (I) := TmpTag;
            if SPARK_XML.IsNullTag (TmpTag) then
               -- Run out of room, so generate an error.
               HandleSchemaError (False, "Failure adding tag");
               exit;
            end if;
         end loop;

      end LoadTags;


      -- Load the attribute definitions
      procedure LoadAttributes
      --# global in     TagIDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     TagIDs;
      is
         TmpAttrib : SPARK_XML.AttributeID;
         TmpSuccess : Boolean;

         type Attribute is record
            ID : SPARK_XML.TagID;
            Name : SPARK_XML.XMLString;
            Typ : SPARK_XML.AttributeType;
            Req : Boolean;
         end record;

         -- Set TotalAttributes to the total number of attributes to be added
         -- to tags in the Schema.
         TotalAttributes : constant Integer := 26;

         type AttributeIndex is range 1 .. TotalAttributes;
         type AttributeList is array (AttributeIndex) of Attribute;
         Attributes : AttributeList;

      begin
         -----------------------------------|TAG                  |Attribute name |Type                |Required?|
         ---------------------------------------------------------|---------------|--------------------|---------|
         Attributes :=
           AttributeList'(Attribute'(TagIDs (TPogs),         XStr ("report_name"),  SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TPogs),         XStr ("version"),      SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TPogs),         XStr ("licensee"),     SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TPogs),         XStr ("ignoredates"),  SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TFile),         XStr ("name"),         SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TFile),         XStr ("procedure"),    SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TFile),         XStr ("generated"),    SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TFile),         XStr ("simplified"),   SPARK_XML.ATString,  False),
                          Attribute'(TagIDs (TFile),         XStr ("checked"),      SPARK_XML.ATString,  False),
                          Attribute'(TagIDs (TVc),           XStr ("vcnumber"),     SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TVc),           XStr ("from"),         SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TVc),           XStr ("to"),           SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TVc),           XStr ("topointtype"),  SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TVc),           XStr ("status"),       SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (TSubprograms),  XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TSubprogramsInError),  XStr ("total"), SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TTools),        XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TAssert_Post),  XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TCheck),        XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TInheritance),  XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TPrecond),      XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TRefinement),   XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TRuntime),      XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TUndetSection), XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TTotals),       XStr ("total"),        SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (TError),        XStr ("filetype"),     SPARK_XML.ATString,  True)
                         );

         for I in AttributeIndex loop
            --# accept F, 10, TmpAttrib, "TmpAttrib unused here";
            SPARK_XML.AddAttributeToTag (Schema,
                                         Attributes (I).ID,
                                         Attributes (I).Name,
                                         Attributes (I).Typ,
                                         Attributes (I).Req,
                                         TmpAttrib,
                                         TmpSuccess);
            --# end accept;
            HandleSchemaError (TmpSuccess, "Failure adding attribute to tag");
            exit when not TmpSuccess;
         end loop;

         --# accept F, 33, TmpAttrib, "TmpAttrib unused here";
      end LoadAttributes;



      -- Load the hierarchy definitions
      procedure BuildHierarchy
      --# global in     TagIDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     TagIDs;
      is
         type TagRel is record
            Parent : SPARK_XML.TagID;
            Child : SPARK_XML.TagID;
            Required : Boolean;
         end record;

         MaxRelations : constant Integer := 96;

         subtype RelIndex is Integer range 1 .. MaxRelations;

         type RelArray is array (RelIndex) of TagRel;
         Relations : RelArray;

         TmpSuccess : Boolean;

      begin

         Relations :=
           RelArray'(
                     -- Base tag
                     TagRel'(SPARK_XML.NullTag, TagIDs (TPogs), False),
                     -- Pogs
                     TagRel'(TagIDs (TPogs), TagIDs (TTypes), False),
                     TagRel'(TagIDs (TPogs), TagIDs (TDirectory), False),
                     TagRel'(TagIDs (TPogs), TagIDs (TProduced), False),
                     TagRel'(TagIDs (TPogs), TagIDs (TResults), False),
                     -- Types
                     TagRel'(TagIDs (TTypes), TagIDs (TType), False),
                     -- Results
                     TagRel'(TagIDs (TResults), TagIDs (TFile), False),
                     TagRel'(TagIDs (TResults), TagIDs (TSummary), False),
                     --File
                     TagRel'(TagIDs (TFile), TagIDs (TVc), False),
                     TagRel'(TagIDs (TFile), TagIDs (TError), False),
                     -- Summary
                     TagRel'(TagIDs (TSummary), TagIDs (TSubprograms), False),
                     TagRel'(TagIDs (TSummary), TagIDs (TSubprogramsInError), False),
                     TagRel'(TagIDs (TSummary), TagIDs (TVcs), False),
                     --Subprograms
                     TagRel'(TagIDs (TSubprograms), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TReview), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TContra), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TUser), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TSubprograms), TagIDs (TUndetermined), False),
                     --Subprograms in error
                     TagRel'(TagIDs (TSubprogramsInError), TagIDs (TUndetermined), False),
                     --VCs
                     TagRel'(TagIDs (TVcs), TagIDs (TAssert_Post), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TPrecond), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TCheck), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TRuntime), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TRefinement), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TInheritance), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TUndetSection), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TTotals), False),
                     TagRel'(TagIDs (TVcs), TagIDs (TPercentages), False),
                     -- Assert_post
                     TagRel'(TagIDs (TAssert_Post), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TAssert_Post), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TAssert_Post), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TAssert_Post), TagIDs (TReview), False),
                     TagRel'(TagIDs (TAssert_Post), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TAssert_Post), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TAssert_Post), TagIDs (TUndetermined), False),
                     -- Precondition
                     TagRel'(TagIDs (TPrecond), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TPrecond), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TPrecond), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TPrecond), TagIDs (TReview), False),
                     TagRel'(TagIDs (TPrecond), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TPrecond), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TPrecond), TagIDs (TUndetermined), False),
                     -- Check
                     TagRel'(TagIDs (TCheck), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TCheck), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TCheck), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TCheck), TagIDs (TReview), False),
                     TagRel'(TagIDs (TCheck), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TCheck), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TCheck), TagIDs (TUndetermined), False),
                     -- Runtime
                     TagRel'(TagIDs (TRuntime), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TRuntime), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TRuntime), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TRuntime), TagIDs (TReview), False),
                     TagRel'(TagIDs (TRuntime), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TRuntime), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TRuntime), TagIDs (TUndetermined), False),
                     -- Refinement
                     TagRel'(TagIDs (TRefinement), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TRefinement), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TRefinement), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TRefinement), TagIDs (TReview), False),
                     TagRel'(TagIDs (TRefinement), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TRefinement), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TRefinement), TagIDs (TUndetermined), False),
                     -- Inheritance
                     TagRel'(TagIDs (TInheritance), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TInheritance), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TInheritance), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TInheritance), TagIDs (TReview), False),
                     TagRel'(TagIDs (TInheritance), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TInheritance), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TInheritance), TagIDs (TUndetermined), False),
                     -- Undetermined_summary
                     TagRel'(TagIDs (TUndetSection), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TUndetSection), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TUndetSection), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TUndetSection), TagIDs (TReview), False),
                     TagRel'(TagIDs (TUndetSection), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TUndetSection), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TUndetSection), TagIDs (TUndetermined), False),
                     -- Totals
                     TagRel'(TagIDs (TTotals), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TTotals), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TTotals), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TTotals), TagIDs (TReview), False),
                     TagRel'(TagIDs (TTotals), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TTotals), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TTotals), TagIDs (TUndetermined), False),
                     -- Percentages
                     TagRel'(TagIDs (TPercentages), TagIDs (TExaminer), False),
                     TagRel'(TagIDs (TPercentages), TagIDs (TSimplifier), False),
                     TagRel'(TagIDs (TPercentages), TagIDs (TSimplifierUserRule), False),
                     TagRel'(TagIDs (TPercentages), TagIDs (TChecker), False),
                     TagRel'(TagIDs (TPercentages), TagIDs (TReview), False),
                     TagRel'(TagIDs (TPercentages), TagIDs (TFalse), False),
                     TagRel'(TagIDs (TPercentages), TagIDs (TUndischarged), False),
                     TagRel'(TagIDs (TPercentages), TagIDs (TUndetermined), False)
                    );




         for I in RelIndex loop
            SPARK_XML.AddChildTag (Schema,
                                   Relations (I).Parent,
                                   Relations (I).Child,
                                   Relations (I).Required,
                                   TmpSuccess);
            HandleSchemaError (TmpSuccess, "Failure loading tag hierarchy");
            exit when not TmpSuccess;
         end loop;

      end BuildHierarchy;

   begin
      LastSummarySection := TAssert_Post;

      -- Zero the tag depths.
      TagDepths := TagDepthArray'(others => 0);

      SPARK_XML.InitSchema (Schema);
      SPARK_XML.InitSchemaState (SchemaState);

      -- Load the tags
      LoadTags;
      LoadAttributes;
      BuildHierarchy;

   end Init;

   --
   -- Generalised procedures for starting simple container tags, and closing contianer tags.
   --



   procedure StartSimpleContainer (TagIdent : Tags;
                                   Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIdent,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIdent,
   --#                                TagIDs;
   is

      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TagIdent),
                                    Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      TagDepths (TagIdent) := Depth;

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag

   end StartSimpleContainer;

   procedure EndContainer (TagIdent : in Tags;
                           Report   : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths,
   --#                                TagIdent &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths,
   --#                                TagIdent;
   is

      XML : XMLString;
      Stat : SPARK_XML.SchemaStatus;

   begin

      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          TagDepths (TagIdent),
                          XML,
                          Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

   end EndContainer;


   procedure EndTag (Depth  : in SPARK_XML.TagDepth;
                     Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Depth &
   --#         SPARK_IO.File_Sys from *,
   --#                                Depth,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState;
   is

      XML : XMLString;
      Stat : SPARK_XML.SchemaStatus;

   begin

      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          Depth,
                          XML,
                          Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

   end EndTag;






   procedure StartPogs (ReportName  : in XMLString;
                        PogsVersion : in XMLString;
                        Licensee    : in XMLString;
                        IgnoreDates : in Boolean;
                        Report      : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                IgnoreDates,
   --#                                Licensee,
   --#                                PogsVersion,
   --#                                ReportName,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                IgnoreDates,
   --#                                Licensee,
   --#                                PogsVersion,
   --#                                Report,
   --#                                ReportName,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is

      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TPogs),
                                    Stat);
      HandleError (Stat);


      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("report_name"),
                                 ReportName,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("version"),
                                 PogsVersion,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("licensee"),
                                 Licensee,
                                 Stat);
      HandleError (Stat);

      if IgnoreDates then
         SPARK_XML.AddAttributeStr (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("ignoredates"),
                                    XStr ("true"),
                                    Stat);
      else
         SPARK_XML.AddAttributeStr (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("ignoredates"),
                                    XStr ("false"),
                                    Stat);
      end if;

      HandleError (Stat);


      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);

      HandleError (Stat);

      TagDepths (TPogs) := Depth;

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag

   end StartPogs;



   procedure EndPogs (Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TPogs, Report);
   end EndPogs;



   procedure StartTypes (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is

   begin
      StartSimpleContainer (TTypes, Report);
   end StartTypes;






   procedure EndTypes (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TTypes, Report);
   end EndTypes;


   procedure StartResults (Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      StartSimpleContainer (TResults, Report);
   end StartResults;



   procedure EndResults (Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TResults, Report);
   end EndResults;



   procedure StartSummary (Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      StartSimpleContainer (TSummary, Report);
   end StartSummary;


   procedure EndSummary (Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TSummary, Report);
   end EndSummary;


   procedure StartSubprogramSummary (TotalSubprogs : in Natural;
                                     Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                TotalSubprogs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                TotalSubprogs;
   is
      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TSubprograms),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("total"),
                                 TotalSubprogs,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      TagDepths (TSubprograms) := Depth;

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag


   end StartSubprogramSummary;


   procedure EndSubprogramSummary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TSubprograms, Report);
   end EndSubprogramSummary;


   procedure StartSubprogramsInErrorSummary (TotalSubprogsInError : in Natural;
                                             Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                TotalSubprogsInError &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                TotalSubprogsInError;
   is
      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TSubprogramsInError),
                                    Stat);

      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("total"),
                                 TotalSubprogsInError,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      TagDepths (TSubprogramsInError) := Depth;

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag


   end StartSubprogramsInErrorSummary;

   procedure EndSubprogramsInErrorSummary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TSubprogramsInError, Report);
   end EndSubprogramsInErrorSummary;



   procedure StartVCSummary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      StartSimpleContainer (TVcs, Report);
   end StartVCSummary;


   procedure EndVCSummary (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TVcs, Report);
   end EndVCSummary;



   ---------------------------------
   -- Container tags with attributes
   ---------------------------------


   -- For optional attributes, if the EmptyString is supplied then the attribute
   -- will not be added to the tag.

   procedure StartFile (Name       : in XMLLongString;
                        Proc       : in XMLString;
                        Generated  : in XMLString;
                        Simplified : in XMLString; -- Optional
                        Checked    : in XMLString; -- Optional
                        Report     : in SPARK_IO.File_Type)
   --# global in     CommandLine.Data;
   --#        in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --# derives SchemaState,
   --#         TagDepths         from *,
   --#                                Checked,
   --#                                CommandLine.Data,
   --#                                Generated,
   --#                                Name,
   --#                                Proc,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Simplified,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Checked,
   --#                                CommandLine.Data,
   --#                                Generated,
   --#                                Name,
   --#                                Proc,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Simplified,
   --#                                TagIDs;
   is
      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

      TruncatedName : EStrings.T;

   begin
      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TFile),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      -- quick fix to avoid a lot of work in SPARK_XML
      TruncatedName := ELStrings.ToExaminerString (PathFormatter.Format (Name));
      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("name"),
                                 TruncatedName,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("procedure"),
                                 Proc,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("generated"),
                                 Generated,
                                 Stat);
      HandleError (Stat);

      if not EStrings.EqString (Simplified, EStrings.EmptyString) then
         SPARK_XML.AddAttributeStr (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("simplified"),
                                    Simplified,
                                    Stat);
         HandleError (Stat);
      end if;

      if not EStrings.EqString (Checked, EStrings.EmptyString) then
         SPARK_XML.AddAttributeStr (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("checked"),
                                    Checked,
                                    Stat);
         HandleError (Stat);
      end if;

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      TagDepths (TFile) := Depth;

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag

   end StartFile;



   procedure EndFile (Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (TFile, Report);
   end EndFile;


   procedure FileError (Message : in EStrings.T;
                        FType   : in FileType;
                        Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                FType,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                FType,
   --#                                Message,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TError),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      case FType is
         when VCGFile => EStrings.CopyString (XML, "vcg");
         when SIVFile => EStrings.CopyString (XML, "siv");
         when PLGFile => EStrings.CopyString (XML, "plg");
         when REVFile => EStrings.CopyString (XML, "rev");
      end case;

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("filetype"),
                                 XML,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag

      EStrings.PutString (Report,
                                 Message);  -- Output the error message

      EndTag (Depth, Report); -- Close the tag.

   end FileError;



   procedure VC (VCNumber    : in XMLString;
                 FromPoint   : in XMLString;
                 ToPointType : in XMLString;
                 ToPoint     : in XMLString;
                 Status      : in VCStatus;
                 Report      : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                FromPoint,
   --#                                Schema,
   --#                                Status,
   --#                                TagIDs,
   --#                                ToPoint,
   --#                                ToPointType,
   --#                                VCNumber &
   --#         SPARK_IO.File_Sys from *,
   --#                                FromPoint,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Status,
   --#                                TagIDs,
   --#                                ToPoint,
   --#                                ToPointType,
   --#                                VCNumber;
   is

      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TVc),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("vcnumber"),
                                 VCNumber,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("from"),
                                 FromPoint,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("to"),
                                 ToPoint,
                                 Stat);
      HandleError (Stat);
      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("topointtype"),
                                 ToPointType,
                                 Stat);
      HandleError (Stat);

      case Status is
         when VCG    => EStrings.CopyString (XML, "vcg");
         when SIV    => EStrings.CopyString (XML, "siv");
         when PLG    => EStrings.CopyString (XML, "plg");
         when REV    => EStrings.CopyString (XML, "rev");
         when WRONG  => EStrings.CopyString (XML, "false");
         when CONTRA => EStrings.CopyString (XML, "contradiction");
         when USER   => EStrings.CopyString (XML, "user");
         when TODO   => EStrings.CopyString (XML, "todo");
      end case;

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("status"),
                                 XML,
                                 Stat);
      HandleError (Stat);


      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag

      EndTag (Depth, Report); -- Close the tag.

   end VC;


   procedure StartSummarySection (Section : in SummarySection;
                                  Total   : in Integer;
                                  Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#        in out TagDepths;
   --#           out LastSummarySection;
   --# derives LastSummarySection from Section &
   --#         SchemaState,
   --#         TagDepths          from *,
   --#                                 Schema,
   --#                                 SchemaState,
   --#                                 Section,
   --#                                 TagIDs,
   --#                                 Total &
   --#         SPARK_IO.File_Sys  from *,
   --#                                 Report,
   --#                                 Schema,
   --#                                 SchemaState,
   --#                                 Section,
   --#                                 TagIDs,
   --#                                 Total;
   is
      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      TmpTag : Tags;
   begin
      case Section is
         when SSTotals => TmpTag := TTotals;
         when SSPercentages => TmpTag := TPercentages;
         when SSAssert_Post => TmpTag := TAssert_Post;
         when SSCheck => TmpTag := TCheck;
         when SSInheritance => TmpTag := TInheritance;
         when SSPrecondition => TmpTag := TPrecond;
         when SSRefinement => TmpTag := TRefinement;
         when SSRuntime => TmpTag := TRuntime;
         when SSUndetermined => TmpTag := TUndetSection;
      end case;

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TmpTag),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      if not (TmpTag = TPercentages) then  -- Percentages don't need a total.
         SPARK_XML.AddAttributeInt (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("total"),
                                    Total,
                                    Stat);
         HandleError (Stat);
      end if;

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  TagDepths (TmpTag), -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);  -- Output the opening tag

      LastSummarySection := TmpTag;

   end StartSummarySection;



   -- Summary sections cannot enclose each other, so calls to this
   -- procedure will just close the last opened section.
   procedure EndSummarySection (Report  : in SPARK_IO.File_Type)
   --# global in     LastSummarySection;
   --#        in     Schema;
   --#        in     TagDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                LastSummarySection,
   --#                                TagDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                LastSummarySection,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagDepths;
   is
   begin
      EndContainer (LastSummarySection, Report);
   end EndSummarySection;



   -- Simple non-container tags.
   -- These either have just attributes or values

   procedure TextLongTag (Text   : in XMLLongString;
                      TTag   : in Tags;
                      Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs,
   --#                                TTag &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Text,
   --#                                TTag;
   is

      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin
      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TTag),
                                    Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);  -- Output the directory name

      ELStrings.PutString (Report,
                                 Text);  -- Output the directory name

      EndTag (Depth, Report);

   end TextLongTag;

   procedure TextTag (Text   : in XMLString;
                      TTag   : in Tags;
                      Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs,
   --#                                TTag &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Text,
   --#                                TTag;
   is

      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

   begin
      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TTag),
                                    Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);  -- Output the directory name

      EStrings.PutString (Report,
                                 Text);  -- Output the directory name

      EndTag (Depth, Report);

   end TextTag;

   procedure Directory (DirectoryName : in XMLLongString;
                        Report        : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                DirectoryName,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is

   begin
      TextLongTag (DirectoryName,
               TDirectory,
               Report);
   end Directory;

   procedure Produced (ProducedDate : in XMLString;
                       Report       : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                ProducedDate,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is

   begin
      TextTag (ProducedDate,
               TProduced,
               Report);
   end Produced;

   procedure SummaryItemStr (Item   : in ItemType;
                             Value  : in EStrings.T;
                             Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Item,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Value;
   is
      type IDLookupArray is array (ItemType) of Tags;

      IDTable : constant IDLookupArray := IDLookupArray'(ITExaminer => TExaminer,
                                                         ITSimplifier => TSimplifier,
                                                         ITSimplifierUserRule => TSimplifierUserRule,
                                                         ITChecker => TChecker,
                                                         ITReview => TReview,
                                                         ITFalse => TFalse,
                                                         ITContra => TContra,
                                                         ITUser => TUser,
                                                         ITUndischarged => TUndischarged,
                                                         ITUndetermined => TUndetermined
                                                        );

      XML : XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

      TmpTag : Tags;
   begin

      TmpTag := IDTable (Item);

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (TmpTag),
                                    Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report, XML); -- Output the opening tag

      EStrings.PutString (Report, Value);  -- Output the value

      EndTag (Depth, Report);
   end SummaryItemStr;

   procedure SummaryItemInt (Item   : in ItemType;
                             Value  : in Integer;
                             Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Item,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Value;
   is
      subtype StringLength is Integer range 1 .. 10;
      subtype TempString is String (StringLength);
      ValString : TempString;
      ValEString : EStrings.T;
   begin

      -- Convert the integer to a string.
      SPARK_IO.Put_Int_To_String (ValString, Value, 1, 10);
      EStrings.CopyString (ValEString, ValString);

      ValEString := EStrings.Trim (ValEString);  -- Remove spaces.

      SummaryItemStr (Item,
                      ValEString,
                      Report);
   end SummaryItemInt;

   procedure AnalysisType (Typ    : in XMLString;
                           Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Typ;
   is
   begin
      TextTag (Typ,
               TType,
               Report);
   end AnalysisType;


end XMLSummary;
