-- $Id: vcs.adb 12255 2009-01-22 09:28:46Z Robin Messer $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package to perform processing of VC files                                   --
--                                                                            --
--The method of collecting information from the VC files is described in the  --
--specification. Below is given the algorithm for writing the information to  --
--the report file.                                                            --
--                                                                            --
--In order to print the status of all VCs as a table, information about the   --
--VCs is now collected in the VCHeap.                                         --
--                                                                            --
--A record is put on the heap for each VC identified in the VCG file,         --
--indicating its name (used for indexing the records) and status              --
--(undischarged at this point).                                               --
--                                                                            --
--The status is subsequently updated for VCs identified when parsing the      --
--SIV and PLG files (if the VC has been recognised as fully simplified        --
--or fully proved).                                                           --
--                                                                            --
--When this parsing phase is complete, the content of the VCHeap is printed   --
--to the report file.                                                         --
--                                                                            --
--------------------------------------------------------------------------------
with VCDetails,
     EStrings,
     SPARK_Calendar,
     FatalErrors,
     Heap,
     OSFiling,
     VCHeap,
     CommandLine,
     Total,
     XMLSummary,
     PathFormatter,
     Ada.Characters.Handling;

use type EStrings.T;
use type SPARK_Calendar.Error_Code;

package body VCS
is

   subtype Range_10 is Integer range 1 .. 10;
   subtype String_10 is String (Range_10);

   -- All non-refinement and non-inheritance VCs have line numbers associated
   -- with their start and finish points.
   -- Sentinel values are used as follows:
   --    -1 is used for refinement and inheritance VCs, which do not have any
   --       line number associated with them.
   --     0 is used to mean the start of a subprogram (e.g. the precondition)
   -- 'Last is used to mean the end of a subprogram (e.g. the postcondition)

   subtype VCLineType is Integer range -1 .. Integer'Last;
   Refinement_Or_Inheritance_VC : constant VCLineType := -1;
   VCLineStart                  : constant VCLineType := 0;
   VCLineEnd                    : constant VCLineType := VCLineType'Last;

   -- Added to describe different forms of corrupt files.
   -- (This is known not to be a complete solution. The file parsing routines in
   -- pogs are quite basic, making it difficult to offer extensive well-formed
   -- file checks. Neverthless, the checks that are considered, are the most common,
   -- and are tracable to genuine user concerns.)
   type FileStatusT is (NotCorrupt,
                        CorruptEmptyFile,
                        CorruptUnknownSubprogram);

   type FileTypes is (SimplifiedVCFileType, StandardVCFileType);

   type ParsingStateType is (Initial, FirstRange, FirstVCName, NewRange, NewVCName);

   type VCInfoType is record
      StartLine            : VCLineType;
      EndLine              : VCLineType;
      EndLinePointType     : VCDetails.TerminalPointType;
      NumberOfVCs          : Natural;
      Valid                : Boolean;   -- used so that invalid data (at start of
      -- processing) is not printed
      FileType             : FileTypes; -- SIV / VCG / PLG
      AnyVCsPrinted        : Boolean;   -- used to print the 'UndischargedVCs' message
      ThisStartLinePrinted : Boolean;   -- used to decide whether to print the
      -- start line for the given range
   end record;


   subtype ErrorsIndex is Integer range 1 .. 1000;
   type ErrorsList is array (ErrorsIndex) of EStrings.T;

   type ReviewErrors is record
      Errors       : Boolean;     -- Have we recorded any errors?
      ErrorList    : ErrorsList;  -- The list of errors
      LastError    : ErrorsIndex; -- The last error
      ExcessCount  : Natural;     -- How many errors are unreported.
   end record;


   ReportWrapColumn : constant Positive := 70;

   -- Note that all the columns below refer to the line AFTER it has been
   -- trimmed of white space
   VCGFileDateTimeStartColumn : constant EStrings.Positions := 8;

   -- note, setting this to 20 ensures that the resolution is only down to
   -- the nearest second, for compatibility with the same information as
   -- extracted from the SIV file
   VCGFileDateTimeLength :  constant EStrings.Lengths := 20;

   SIVFileVCGenerationDateStartColumn : constant EStrings.Positions
     := 9;
   SIVFileVCGenerationDateLength : constant EStrings.Lengths
     := 11;
   SIVFileVCGenerationTimeStartColumn : constant EStrings.Positions
     := 22;
   SIVFileVCGenerationTimeLength : constant EStrings.Lengths
     := 8;
   SIVFileSimplificationDateStartColumn : constant EStrings.Positions
     := 43;
   SIVFileSimplificationDateLength :  constant EStrings.Lengths
     := 11;
   SIVFileSimplificationTimeStartColumn : constant EStrings.Positions
     := 56;
   SIVFileSimplificationTimeLength :  constant EStrings.Lengths
     := 8;

   -- Constants to extract Proof Log File date
   PLGFileVCProofDateStartColumn : constant EStrings.Positions := 8;
   PLGFileVCProofDateLength      : constant EStrings.Lengths := 11;
   PLGFileVCProofTimeStartColumn : constant EStrings.Positions := 28;
   PLGFileVCProofTimeLength      : constant EStrings.Lengths := 8;

   UnknownSIVDate : constant EStrings.T :=
     EStrings.T'
     (Length  => 21,
      Content => EStrings.Contents'('U', 'n', 'k', 'n', 'o', 'w', 'n', ' ',
                                           'S', 'I', 'V', ' ',
                                           'f', 'i', 'l', 'e', ' ',
                                           'd', 'a', 't', 'e', others => ' '));

   UnknownVCGDate : constant EStrings.T :=
     EStrings.T'
     (Length  => 21,
      Content => EStrings.Contents'('U', 'n', 'k', 'n', 'o', 'w', 'n', ' ',
                                           'V', 'C', 'G', ' ',
                                           'f', 'i', 'l', 'e', ' ',
                                           'd', 'a', 't', 'e', others => ' '));


   ----------------------------------------------------------------------------
   -- NB this trims the string given, then appends a space and a trimmed version
   -- of the next line from the input file
   procedure AppendNextLineFromFile
     (Line : in out EStrings.T;
      File : in     SPARK_IO.File_Type)
   --# global in out SPARK_IO.File_sys;
   --# derives Line,
   --#         SPARK_IO.File_sys from *,
   --#                                File,
   --#                                SPARK_IO.File_sys;
   is
      NextLine    : EStrings.T;
      TrimmedLine : EStrings.T;
   begin
      TrimmedLine := EStrings.Trim (Line);
      Line := TrimmedLine;
      EStrings.AppendStringTruncate (Line, " ");
      EStrings.GetLine (File, NextLine);
      TrimmedLine := EStrings.Trim (NextLine);
      EStrings.AppendExaminerStringTruncate (Line, TrimmedLine);
   end AppendNextLineFromFile;

   ---------------------------------------------------------------------------
   function ExtractLineNumberAtPosition
     (Line     : EStrings.T;
      StartPos : EStrings.Positions)
     return VCLineType
   is
      Result   : VCLineType := 0;
      Pos      : EStrings.Positions;
      Finished : Boolean := False;
   begin
      Pos := StartPos;

      while not Finished loop
         if Line.Content (Pos) >= '0' and Line.Content (Pos) <= '9' then
            Result := (10 * Result) +
              (Character'Pos (Line.Content (Pos)) - Character'Pos ('0'));
            if Pos < Line.Length then
               Pos := Pos + 1;
            else
               Finished := True;
            end if;
         else
            Finished := True;
         end if;
      end loop;

      return Result;
   end ExtractLineNumberAtPosition;

   ---------------------------------------------------------------------------
   function IsVCProofSuccessLine (Line : EStrings.T)
      return Boolean
   is
   begin -- IsVCProofSuccessLine
      return EStrings.Eq1String (EStrings.Section (Line, 1, 14),
                                        "*** PROVED VC ");
   end IsVCProofSuccessLine;

   ---------------------------------------------------------------------------
   function IsNewRangeLine (Line : EStrings.T)
      return Boolean
   is
   begin -- IsNewRangeLine
      return EStrings.Eq1String (EStrings.Section (Line, 1, 17),
                                        "For path(s) from ") or
             EStrings.Eq1String (EStrings.Section (Line, 1, 14),
                                        "For checks of ");
   end IsNewRangeLine;

   ---------------------------------------------------------------------------
   function IsNewVCLine (Line : EStrings.T)
      return Boolean
   is
      RetVal : Boolean;
   begin -- IsNewVCLine
      -- The shortest possible New VC Line is for a function that has
      -- a single letter identifier, followed by a full-stop e.g.
      --   function_g.
      -- which is 11 characters.
      if Line.Length >= 11 then
         RetVal :=
           EStrings.Eq1String
           (EStrings.Section (Line, 1, 10), "procedure_") or else
           EStrings.Eq1String (EStrings.Section (Line, 1, 9),
                                      "function_") or else
           EStrings.Eq1String (EStrings.Section (Line, 1, 10),
                                      "task_type_");
         if RetVal then
            for I in EStrings.Lengths range 9 .. Line.Length - 1 loop
               if not (Ada.Characters.Handling.Is_Alphanumeric (Line.Content (I)) or
                         Line.Content (I) = '_') then

                  RetVal := False;
                  exit;
               end if;
               --# assert I in 9 .. Line.Length - 1 and
               --#        Line = Line% and
               --#        Line.Length >= 11;
            end loop;

            if Line.Content (Line.Length) /= '.' then
               RetVal := False;
            end if;
         end if;
      else
         RetVal := False;
      end if;
      return RetVal;
   end IsNewVCLine;

   ---------------------------------------------------------------------------
   function IsTriviallyTrueVC (Line : EStrings.T)
      return Boolean
   is
   begin
      return EStrings.Eq1String
        (EStrings.Section (Line, 1, 10),
         "*** true .");
   end IsTriviallyTrueVC;

   ---------------------------------------------------------------------------
   function IsTriviallyFalseVC (Line : EStrings.T)
      return Boolean
   is
      VCProvedFalse : Boolean;
      UnusedPos     : EStrings.Positions;
   begin
      if Line.Length >= 14 then
         --# accept F, 10, UnusedPos, "UnusedPos unused here" &
         --#        F, 33, UnusedPos, "UnusedPos unused here";
         EStrings.FindSubString (Line, "   false .", VCProvedFalse, UnusedPos);
         -- It must be a conclusion, not a hypothesis!
         VCProvedFalse := VCProvedFalse and (Line.Content (1) = 'C');
      else
         VCProvedFalse := False;
      end if;
      return VCProvedFalse;
   end IsTriviallyFalseVC;

   ---------------------------------------------------------------------------
   function IsVCErrorMessage (Line : EStrings.T)
      return Boolean
   is
   begin -- IsVCErrorMessage
      return Line.Length > 0 and then
             Line.Content (1) = '!';
   end IsVCErrorMessage;

   ---------------------------------------------------------------------------
   -- see file header
   procedure WriteVCInfo (ReportFile              : in     SPARK_IO.File_Type;
                          VCInfo                  : in     VCInfoType;
                          AnythingPrintedThisTime :    out Boolean)
   --# global in out SPARK_IO.File_sys;
   --# derives AnythingPrintedThisTime from VCInfo &
   --#         SPARK_IO.File_sys       from *,
   --#                                      ReportFile,
   --#                                      VCInfo;
      is separate;
   pragma Unreferenced (WriteVCInfo);

   ---------------------------------------------------------------------------
   procedure PrintVCReport (VCFileName        : in ELStrings.T;
                            VCFileDateTime    : in EStrings.T;
                            SIVFileName       : in ELStrings.T;
                            SIVFileDateTime   : in EStrings.T;
                            PLGFileName       : in ELStrings.T;
                            PLGFileDateTime   : in EStrings.T;
                            SLGFileName       : in ELStrings.T;
                            VCError           : in Boolean;
                            VCErrorString     : in EStrings.T;
                            SIVError          : in Boolean;
                            SIVErrorString    : in EStrings.T;
                            PLGError          : in Boolean;
                            PLGErrorString    : in EStrings.T;
                            SLGError          : in Boolean;
                            SLGFileMissing    : in Boolean;
                            REVErrors         : in ReviewErrors;
                            ReportFile        : in SPARK_IO.File_Type;
                            TempFile          : in SPARK_IO.File_Type;
                            TempFalseFile     : in SPARK_IO.File_Type;
                            TempContraFile    : in SPARK_IO.File_Type;
                            TempUserFile      : in SPARK_IO.File_Type;
                            TempPRVerrFile    : in SPARK_IO.File_Type;
                            TempWarnErrorFile : in SPARK_IO.File_Type)
   --# global in     CommandLine.Data;
   --#        in     VCHeap.I_State;
   --#        in     VCHeap.State;
   --#        in out SPARK_IO.File_sys;
   --#        in out XMLSummary.State;
   --# derives SPARK_IO.File_sys from *,
   --#                                CommandLine.Data,
   --#                                PLGError,
   --#                                PLGErrorString,
   --#                                PLGFileDateTime,
   --#                                PLGFileName,
   --#                                ReportFile,
   --#                                REVErrors,
   --#                                SIVError,
   --#                                SIVErrorString,
   --#                                SIVFileDateTime,
   --#                                SIVFileName,
   --#                                SLGError,
   --#                                SLGFileMissing,
   --#                                SLGFileName,
   --#                                TempContraFile,
   --#                                TempFalseFile,
   --#                                TempFile,
   --#                                TempPRVerrFile,
   --#                                TempUserFile,
   --#                                TempWarnErrorFile,
   --#                                VCError,
   --#                                VCErrorString,
   --#                                VCFileDateTime,
   --#                                VCFileName,
   --#                                VCHeap.I_State,
   --#                                VCHeap.State,
   --#                                XMLSummary.State &
   --#         XMLSummary.State  from *,
   --#                                CommandLine.Data,
   --#                                PLGError,
   --#                                PLGFileDateTime,
   --#                                REVErrors,
   --#                                SIVError,
   --#                                SIVFileDateTime,
   --#                                VCError,
   --#                                VCFileDateTime,
   --#                                VCFileName,
   --#                                VCHeap.I_State,
   --#                                VCHeap.State;

      is separate;

   ---------------------------------------------------------------------------
   -- Process a line of type (1). Note that this can actually spread over two
   -- file lines, so the first action of this procedure is to read the next
   -- line from the file and append it onto the previous one. This has no
   -- unwanted side effects, as there is always a blank line between this and
   -- the start of the VC
   procedure ProcessNewRangeLine
     (Line   : in     EStrings.T;
      VCInfo : in out VCInfoType)
   --# derives VCInfo from *,
   --#                     Line;
      is separate;

   ---------------------------------------------------------------------------
   procedure ReadNextNonBlankLine
     (File     : in     SPARK_IO.File_Type;
      Success  :    out Boolean;
      FileLine :    out EStrings.T)
   --# global in out SPARK_IO.File_sys;
   --# derives FileLine,
   --#         SPARK_IO.File_sys,
   --#         Success           from File,
   --#                                SPARK_IO.File_sys;
   is
      Finished    : Boolean := False;
      Status      : Boolean := False;
      LineRead    : EStrings.T;
      TrimmedLine : EStrings.T := EStrings.EmptyString;
   begin -- ReadNextNonBlankLine
      while not Finished loop
         EStrings.GetLine (File, LineRead);
         TrimmedLine := EStrings.Trim (LineRead);

         if TrimmedLine.Length > 0 then
            Status := True;
            Finished := True;
         else
            if SPARK_IO.End_Of_File (File) then
               Status := False;
               Finished := True;
            end if;
         end if;
      end loop;
      Success  := Status;
      FileLine := TrimmedLine;
   end ReadNextNonBlankLine;

   ---------------------------------------------------------------------------
   procedure AnalyseVCFile
     (ReportFile          : in     SPARK_IO.File_Type;
      FileName            : in     ELStrings.T;
      ErrorInFile         :    out Boolean;
      FileError           :    out EStrings.T;
      FileDateTime        :    out EStrings.T)
   --# global in     CommandLine.Data;
   --#        in out FatalErrors.State;
   --#        in out SPARK_IO.File_sys;
   --#        in out VCHeap.I_State;
   --#        in out VCHeap.State;
   --# derives ErrorInFile,
   --#         FileError         from CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SPARK_IO.File_sys &
   --#         FatalErrors.State from *,
   --#                                CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SPARK_IO.File_sys,
   --#                                VCHeap.State &
   --#         FileDateTime      from FileName,
   --#                                SPARK_IO.File_sys &
   --#         SPARK_IO.File_sys,
   --#         VCHeap.I_State,
   --#         VCHeap.State      from *,
   --#                                CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SPARK_IO.File_sys;
   -- precondition to entering this procedure is that the VCG file exists
      is separate;

   ---------------------------------------------------------------------------
   procedure AnalyseSimplifiedVCFile
     (ReportFile        : in     SPARK_IO.File_Type;
      FileName          : in     ELStrings.T;
      VCFileDateTime    : in     EStrings.T;
      SIVFileDateTime   :    out EStrings.T;
      ErrorInSIVFile    :    out Boolean;
      FileError         :    out EStrings.T)
   --# global in     CommandLine.Data;
   --#        in out FatalErrors.State;
   --#        in out SPARK_IO.File_sys;
   --#        in out VCHeap.State;
   --# derives ErrorInSIVFile,
   --#         FileError,
   --#         SPARK_IO.File_sys from CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SPARK_IO.File_sys,
   --#                                VCFileDateTime &
   --#         FatalErrors.State,
   --#         VCHeap.State      from *,
   --#                                CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SPARK_IO.File_sys,
   --#                                VCFileDateTime,
   --#                                VCHeap.State &
   --#         SIVFileDateTime   from FileName,
   --#                                SPARK_IO.File_sys;
   -- precondition to entering this procedure is that the SIV file exists
      is separate;


   ---------------------------------------------------------------------------
   procedure AnalyseReviewFile
     (ReportFile      : in     SPARK_IO.File_Type;
      FileName        : in     ELStrings.T;
      Errors          :    out ReviewErrors)
   --# global in     VCHeap.I_State;
   --#        in out FatalErrors.State;
   --#        in out SPARK_IO.File_sys;
   --#        in out VCHeap.State;
   --# derives Errors            from FileName,
   --#                                ReportFile,
   --#                                SPARK_IO.File_sys,
   --#                                VCHeap.I_State,
   --#                                VCHeap.State &
   --#         FatalErrors.State,
   --#         SPARK_IO.File_sys,
   --#         VCHeap.State      from *,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SPARK_IO.File_sys,
   --#                                VCHeap.I_State,
   --#                                VCHeap.State;
   -- precondition to entering this procedure is that the PRV file exists
      is separate;

   ---------------------------------------------------------------------------
   procedure AnalyseProofLogFile
     (ReportFile      : in     SPARK_IO.File_Type;
      FileName        : in     ELStrings.T;
      SIVFileDateTime : in     EStrings.T;
      PLGFileDateTime :    out EStrings.T;
      ErrorInFile     :    out Boolean;
      FileError       :    out EStrings.T)
   --# global in     CommandLine.Data;
   --#        in out FatalErrors.State;
   --#        in out SPARK_IO.File_sys;
   --#        in out VCHeap.State;
   --# derives ErrorInFile,
   --#         FileError,
   --#         SPARK_IO.File_sys from CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SIVFileDateTime,
   --#                                SPARK_IO.File_sys &
   --#         FatalErrors.State,
   --#         VCHeap.State      from *,
   --#                                CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                SIVFileDateTime,
   --#                                SPARK_IO.File_sys,
   --#                                VCHeap.State &
   --#         PLGFileDateTime   from CommandLine.Data,
   --#                                FileName,
   --#                                SIVFileDateTime,
   --#                                SPARK_IO.File_sys;
   -- precondition to entering this procedure is that the SIV file exists
      is separate;

   procedure AnalyseSimpLogFile
     (ReportFile      : in     SPARK_IO.File_Type;
      FileName        : in     ELStrings.T;
      RuleFilesErrors : in out SPARK_IO.File_Type;
      RuleFilesUsed   : in out SPARK_IO.File_Type;
      SLGErrorInFile  :    out Boolean)
   --# global in     CommandLine.Data;
   --#        in out FatalErrors.State;
   --#        in out SPARK_IO.File_sys;
   --# derives FatalErrors.State,
   --#         SPARK_IO.File_sys from *,
   --#                                CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                RuleFilesErrors,
   --#                                RuleFilesUsed,
   --#                                SPARK_IO.File_sys &
   --#         RuleFilesErrors,
   --#         RuleFilesUsed     from *,
   --#                                CommandLine.Data,
   --#                                FileName,
   --#                                SPARK_IO.File_sys &
   --#         SLGErrorInFile    from CommandLine.Data,
   --#                                FileName,
   --#                                ReportFile,
   --#                                RuleFilesErrors,
   --#                                RuleFilesUsed,
   --#                                SPARK_IO.File_sys;
   -- precondition to entering this procedure is that the SLG file exists
   -- and XML output is not selected.
   -- and that the SIV file date time was valid
      is separate;


   ---------------------------------------------------------------------------
   procedure Analyse (ReportFile        : in     SPARK_IO.File_Type;
                      FileName          : in     ELStrings.T;
                      AnalyseProofLog   : in     Boolean;
                      TempFile          : in     SPARK_IO.File_Type;
                      TempFalseFile     : in     SPARK_IO.File_Type;
                      TempContraFile    : in     SPARK_IO.File_Type;
                      TempUserFile      : in     SPARK_IO.File_Type;
                      TempRluErrorFile  : in out SPARK_IO.File_Type;
                      TempRluUsedFile   : in out SPARK_IO.File_Type;
                      TempPRVerrFile    : in     SPARK_IO.File_Type;
                      TempWarnErrorFile : in     SPARK_IO.File_Type)
   is

      VCFileName               : ELStrings.T;
      VCFileDateTime           : EStrings.T;
      VCFileContainedError     : Boolean;
      VCFileError              : EStrings.T;

      SIVFileDateTime          : EStrings.T;
      SimplifiedVCFileName     : ELStrings.T;
      SIVFileContainedError    : Boolean;
      SIVFileError             : EStrings.T;

      SimplifierLogFileName    : ELStrings.T;
      SLGFileContainedError    : Boolean;
      SLGFileIsMissing         : Boolean;

      ProofLogFileName         : ELStrings.T;
      PLGFileDateTime          : EStrings.T;
      PLGFileError             : EStrings.T;
      PLGFileContainedError    : Boolean;

      PRVFileName              : ELStrings.T;
      PRVFileErrors            : ReviewErrors;


      ------------------------------------------------------------------------
      procedure SayFileNameBeingProcessed
        (FileName : in ELStrings.T)
      --# global in     CommandLine.Data;
      --#        in out SPARK_IO.File_sys;
      --# derives SPARK_IO.File_sys from *,
      --#                                CommandLine.Data,
      --#                                FileName;
      is
      begin
         SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Processing file ", 0);
         ELStrings.PutLine (SPARK_IO.Standard_Output, PathFormatter.Format (FileName));
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
      end SayFileNameBeingProcessed;

      -------------------------------------------------------------------------
   begin -- Analyse

      SIVFileDateTime       := EStrings.EmptyString;
      PLGFileDateTime       := EStrings.EmptyString;

      SIVFileError := EStrings.EmptyString;
      PLGFileError := EStrings.EmptyString;

      PRVFileErrors := ReviewErrors'(Errors => False,
                                     ErrorList => ErrorsList'(others => EStrings.EmptyString),
                                     LastError => ErrorsIndex'First,
                                     ExcessCount => 0);

      -- Initialise for VCG
      VCFileName := FileName;
      ELStrings.AppendExaminerString (VCFileName, OSFiling.VCFileExtension);

      -- If the vcg file does not exist, then don't process anything here.

      if OSFiling.IsFile (VCFileName) then

         -- Initilise the other files that may be seen.
         --
         -- The logic here is quite confusing.
         --
         -- These files are initialised to their full names if the vcg
         -- file exists. This initialisation is independent to the
         -- existence of these files, or their own dependencies.  The
         -- absence of this initialisation (null string, from default
         -- initialisations above) is detected and sometimes used. This
         -- null string test (hopefully) is only ever used to infer
         -- whether or not the vcg file exists. It might be a lot neater
         -- to just test to see if the vcg file exists.
         --
         -- Unfortunately, these empty strings may be processed in an
         -- unsafe manner. However, it seems that the potentially unsafe
         -- routines are only called when the vcg file is present, so
         -- these initialisations below must take place, and the
         -- potential errors are avoided.
         --
         -- Previously the code contained the following comment, prior
         -- to the empty string initialisation:
         --
         -- "These are optional, if there is no SIV or PLG file then we
         -- want to pass the empty string to XMLSummary."
         --
         -- This has never been the behaviour as coded. If there is an
         -- intention that empty returned strings should denote absence
         -- of file then the code that processes file names should test
         -- for these empty strings and behave accordingly (rather than
         -- raising constraint errors,  say).  Possible points to add
         -- this functionality are marked: EmptyStringHere?
         --
         -- Oh, note also that these are all local variables. But, this
         -- 'Analyse' procedure also contains a call to 'PrintVCReport'
         -- (it does analysis and printing), so the local variables have
         -- a wider impact than might be expected.

         -- Initialise for SIV
         SimplifiedVCFileName := FileName;
         ELStrings.AppendExaminerString (SimplifiedVCFileName,
                                                   OSFiling.SimplifiedVCFileExtension);
         SIVFileContainedError := False;

         -- Initialise for SLG
         SimplifierLogFileName := FileName;
         ELStrings.AppendExaminerString (SimplifierLogFileName,
                                                   OSFiling.SimplifierLogFileExtension);
         SLGFileContainedError := False;
         SLGFileIsMissing := False;

         -- Initialise for PLG
         ProofLogFileName := FileName;
         ELStrings.AppendExaminerString (ProofLogFileName,
                                                   OSFiling.ProofLogFileExtension);
         PLGFileContainedError := False;


         -- Initialise for PRV ("Proof Review")
         PRVFileName := FileName;
         ELStrings.AppendExaminerString (PRVFileName, OSFiling.ReviewFileExtension);


         -- Processing VCG file
         -- ###################
         -- If the VCG file exists: analyse it
         SayFileNameBeingProcessed (VCFileName);
         AnalyseVCFile (ReportFile,            -- in
                        VCFileName,            -- in
                        VCFileContainedError,  -- out
                        VCFileError,           -- out
                        VCFileDateTime);       -- out, for comparison with entry in SIV file

         if VCFileContainedError then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
               "*** Warning: VC file error: subprogram processing abandoned ***", 0);
            SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
            VCHeap.RaiseError (VCDetails.CorruptFile);
         end if;

         --# assert true;

         -- Processing SIV file
         -- ###################
         -- If VCG analysis is not in error then:
         -- If the SIV file exists: analyse it
         if not VCFileContainedError then
            if OSFiling.IsFile (SimplifiedVCFileName) then
               SayFileNameBeingProcessed (SimplifiedVCFileName);
               AnalyseSimplifiedVCFile
                 (ReportFile,                --in
                  SimplifiedVCFileName,      --in
                  VCFileDateTime,            --in
                  SIVFileDateTime,           --out
                  SIVFileContainedError,     --out
                  SIVFileError);             --out

               if SIVFileContainedError then
                  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     "*** Warning: SIV file error: subprogram processing abandoned ***", 0);
                  SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
                  VCHeap.RaiseError (VCDetails.CorruptFile);
               else

                  -- Processing SLG file
                  -- ###################
                  -- If VCG analysis is not in error then:
                  -- If the SIV analysis is not in error then:
                  -- if the SLG file exists: analyse it
                  -- (There is also some odd logic involving XML here - see 2263)
                  -- DEBUG If you're looking to fix SEPR: 2263, then this is a good place to start..
                  if not CommandLine.Data.XML then
                     if OSFiling.IsFile (SimplifierLogFileName) then
                        SayFileNameBeingProcessed (SimplifierLogFileName);
                        AnalyseSimpLogFile
                           (ReportFile,             --in
                            SimplifierLogFileName,  --in
                            TempRluErrorFile,       --in out
                            TempRluUsedFile,        --in out
                            SLGFileContainedError); --   out

                        if SLGFileContainedError then
                           SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                              "*** Warning: SLG file error: subprogram processing abandoned ***", 0);
                           SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
                           VCHeap.RaiseError (VCDetails.CorruptFile);
                        end if;

                     else
                        SPARK_IO.Put_String (SPARK_IO.Standard_Output, "Simplifier log file not found: ", 0);
                        ELStrings.PutLine (SPARK_IO.Standard_Output,
                           PathFormatter.Format (SimplifierLogFileName));
                        SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);

                        -- The absence of an SLG file is not treated as an error.
                        -- Some organisations may deliberately delete SLG files, to
                        -- save space. To support the use of POGS on such data sets,
                        -- missing SLG files can not be regarded as errors, as this
                        -- would lead to the analysis of these subprograms being
                        -- abandoned.

                        -- It is still necessary to report missing SLG files. To maintain
                        -- the distinction between missing SLG and error, an additional
                        -- variable is used.
                        SLGFileIsMissing := True;
                        VCHeap.RaiseError (VCDetails.MissingSLGFile);

                        -- See: EmptyStringHere?
                        -- SimplifierLogFileName := ELStrings.EmptyString;
                     end if;

                  end if;
               end if;
            else
               -- DEBUG If you're looking to fix SEPR: 2264, then this is a good place to start..

               -- otherwise say that VCs not simplified
               SPARK_IO.New_Line (ReportFile, 1);
               SPARK_IO.Put_Line (ReportFile, "VCs not simplified", 0);

               -- See: EmptyStringHere?
               --SimplifiedVCFileName := ELStrings.EmptyString;
            end if;
         end if;

         --# assert true;

         -- Processing PLG file
         -- ###################
         -- Unsimplified VCs might have been proven using the checker alone.
         -- If VCG analysis is not in error then:
         -- If any SIV analysis is not in error then:
         -- If any SLG analysis is not in error then:
         -- If the 'AnalyseProofLog ' is set:
         -- If the PLG file exists then: analyse it
         -- Analyse the PLG file
         if not VCFileContainedError and
            not SIVFileContainedError and
            not SLGFileContainedError and
            AnalyseProofLog then
            if OSFiling.IsFile (ProofLogFileName) then
               SayFileNameBeingProcessed (ProofLogFileName);
               AnalyseProofLogFile
                 (ReportFile,            --in
                  ProofLogFileName,      --in
                  SIVFileDateTime,       --in
                  PLGFileDateTime,       --out
                  PLGFileContainedError, --out
                  PLGFileError);         --out
               if PLGFileContainedError then
                  SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                     "*** Warning: PLG file error: subprogram processing abandoned ***", 0);
                  SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
                  VCHeap.RaiseError (VCDetails.CorruptFile);
               end if;
            else
               null;
               -- See: EmptyStringHere?
               --ProofLogFileName := ELStrings.EmptyString;
            end if;
         end if;

         --# assert True;

         -- Processing PRV file
         -- ###################

         -- PRV files may be present regardless of simplification or application of the checker.
         -- If VCG analysis is not in error then:
         -- If any SIV analysis is not in error then:
         -- If any SLG analysis is not in error then:
         -- If any PLG analysis is not in error then:
         -- If the PRV file exists then: analyse it

         if not VCFileContainedError and
            not SIVFileContainedError and
            not SLGFileContainedError and
            not PLGFileContainedError and
            OSFiling.IsFile (PRVFileName) then
            SayFileNameBeingProcessed (PRVFileName);
            AnalyseReviewFile
               (ReportFile,            --in
                PRVFileName,           --in
                PRVFileErrors);        --out

            if PRVFileErrors.Errors then
               SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                  "*** Warning: PRV file error: subprogram processing abandoned ***", 0);
               SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
               VCHeap.RaiseError (VCDetails.CorruptFile);
            end if;
         end if;

         Total.UpdateTotals;

         --# assert True;

         PrintVCReport (VCFileName,
                        VCFileDateTime,
                        SimplifiedVCFileName,
                        SIVFileDateTime,
                        ProofLogFileName,
                        PLGFileDateTime,
                        SimplifierLogFileName,
                        VCFileContainedError,
                        VCFileError,
                        SIVFileContainedError,
                        SIVFileError,
                        PLGFileContainedError,
                        PLGFileError,
                        SLGFileContainedError,
                        SLGFileIsMissing,
                        PRVFileErrors,
                        ReportFile,
                        TempFile,
                        TempFalseFile,
                        TempContraFile,
                        TempUserFile,
                        TempPRVerrFile,
                        TempWarnErrorFile);

         SPARK_IO.New_Line (ReportFile, 1);
      end if;

      --# accept Flow, 601, XMLSummary.State, TempRluErrorFile, "False coupling through SPARK_IO" &
      --#        Flow, 601, XMLSummary.State, TempRluUsedFile,  "False coupling through SPARK_IO" &
      --#        Flow, 601, VCHeap.State, TempRluErrorFile, "False coupling through SPARK_IO" &
      --#        Flow, 601, VCHeap.State, TempRluUsedFile,  "False coupling through SPARK_IO" &
      --#        Flow, 601, Total.State,  TempRluErrorFile, "False coupling through SPARK_IO" &
      --#        Flow, 601, Total.State,  TempRluUsedFile,  "False coupling through SPARK_IO" &
      --#        Flow, 601, TempRluErrorFile, CommandLine.Data, "False coupling through SPARK_IO" &
      --#        Flow, 601, TempRluUsedFile, CommandLine.Data,  "False coupling through SPARK_IO" &
      --#        Flow, 601, TempRluErrorFile, ReportFile, "False coupling through SPARK_IO" &
      --#        Flow, 601, TempRluUsedFile, ReportFile,  "False coupling through SPARK_IO";
   end Analyse;

end VCS;
