-- $Id: vcs-printvcreport.adb 11680 2008-11-12 15:53:18Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to print the VC report                                            --
--                                                                            --
--------------------------------------------------------------------------------
with PathFormatter,
     SPARK_IO;
separate (VCS)

--------------------------------------------------------------------------
procedure PrintVCReport (VCFileName        : in ELStrings.T;
                         VCFileDateTime    : in EStrings.T;
                         SIVFileName       : in ELStrings.T;
                         SIVFileDateTime   : in EStrings.T;
                         PLGFileName       : in ELStrings.T;
                         PLGFileDateTime   : in EStrings.T;
                         SLGFileName       : in ELStrings.T;
                         VCError           : in Boolean;
                         VCErrorString     : in EStrings.T;
                         SIVError          : in Boolean;
                         SIVErrorString    : in EStrings.T;
                         PLGError          : in Boolean;
                         PLGErrorString    : in EStrings.T;
                         SLGError          : in Boolean;
                         SLGFileMissing    : in Boolean;
                         REVErrors         : in ReviewErrors;
                         ReportFile        : in SPARK_IO.File_Type;
                         TempFile          : in SPARK_IO.File_Type;
                         TempFalseFile     : in SPARK_IO.File_Type;
                         TempContraFile    : in SPARK_IO.File_Type;
                         TempUserFile      : in SPARK_IO.File_Type;
                         TempPRVerrFile    : in SPARK_IO.File_Type;
                         TempWarnErrorFile : in SPARK_IO.File_Type)
is
   VCSuccess             : Boolean;
   ThisVC                : Heap.Atom;
   NextVC                : Heap.Atom;
   VCNumber              : EStrings.T;
   TableLine             : EStrings.T;
   VCNamePrefix          : EStrings.T;
   VCNameTabulation      : Integer;
   VCPathStartTabulation : Integer;
   VCPathEndTabulation   : Integer;
   ToDo                  : Boolean;
   ThisVCName            : EStrings.T;
   ThisVCPathStart       : EStrings.T;
   ThisVCPathEnd         : EStrings.T;
   ThisVCEndType         : VCDetails.TerminalPointType;

   ThisVCProvedByExaminer   : Boolean;
   ThisVCProvedBySimplifier : Boolean;
   ThisVCProvedByChecker    : Boolean;
   ThisVCProvedByReview     : Boolean;
   ThisVCProvedFalse        : Boolean;
   ThisVCProvedContra       : Boolean;
   ThisVCProvedUser         : Boolean;

   VCStat : XMLSummary.VCStatus;
   TmpErrorString : EStrings.T;

   NumUndischarged : Integer;
   NumFalse  : Integer;
   NumContra : Integer;
   NumUser   : Integer;

   Duplicated : Boolean;
   UnusedPos : Integer;
   NonDuplicatedErrors : Integer;

   --------------------------------------------------------------------------
   procedure AppendValue (Line  : in out EStrings.T;
                          IsSet : in     Boolean)
   --# derives Line from *,
   --#                   IsSet;
   is
   begin --AppendValue
      if IsSet then
         EStrings.AppendStringTruncate (Line, " YES ");
      else
         EStrings.AppendStringTruncate (Line, "     ");
      end if;
   end AppendValue;

   --------------------------------------------------------------------------
   procedure AddPadding (Line    : in out EStrings.T;
                         Length  : in     Integer;
                         Padding : in     String)

   --# derives Line from *,
   --#                   Length,
   --#                   Padding;
   is
   begin -- AddPadding

      while Line.Length < Length loop
         EStrings.AppendStringTruncate (Line, Padding);
      end loop;

   end AddPadding;

   function NaturalToString (Number : in Natural) return String_10
   is
      TmpString : String_10;
   begin
         SPARK_IO.Put_Int_To_String (TmpString,
                                     Number,
                                     1,
                                     10);
      return TmpString;
   end NaturalToString;

begin --PrintVCReport
   NumUndischarged := 0;
   NumFalse := 0;
   NumContra := 0;
   NumUser := 0;
   VCNamePrefix := VCHeap.GetVCNamePrefix;

   if CommandLine.Data.XML then
      -- Start the file
      XMLSummary.StartFile (VCFileName,
                            VCNamePrefix,
                            VCFileDateTime,
                            SIVFileDateTime,
                            PLGFileDateTime,
                            ReportFile);

      if VCError then
         XMLSummary.FileError (VCErrorString,
                               XMLSummary.VCGFile,
                               ReportFile);
      end if;

      --# assert True;

      if SIVError then
         XMLSummary.FileError (SIVErrorString,
                               XMLSummary.SIVFile,
                               ReportFile);
      end if;

      if PLGError then
         XMLSummary.FileError (PLGErrorString,
                               XMLSummary.PLGFile,
                               ReportFile);
      end if;

      --# assert True;
      if REVErrors.Errors then
         -- loop through the errors and generate a tag for each
         for CurrentError in Integer range ErrorsIndex'First .. REVErrors.LastError loop
            XMLSummary.FileError (REVErrors.ErrorList (CurrentError),
                                  XMLSummary.REVFile,
                                  ReportFile);
         end loop;

         -- Output a final message saying how many additional errors there were.
         if REVErrors.ExcessCount > 0 then
            EStrings.CopyString (TmpErrorString, "There were too many errors, ");
            EStrings.AppendStringTruncate (TmpErrorString,
                                          NaturalToString (REVErrors.ExcessCount));
            EStrings.AppendStringTruncate (TmpErrorString,
                                          " errors were not reported.");
            XMLSummary.FileError (TmpErrorString,
                                  XMLSummary.REVFile,
                                  ReportFile);
         end if;
      end if;


      --Only output detials if there were no errors for this proof object.
      if not (VCError or
              SIVError or
              PLGError or
              REVErrors.LastError > ErrorsIndex'First)
      then

         -- Now loop through the VCHeap and print one table line per VC
         ThisVC := VCHeap.FirstEntry;
         VCSuccess := True;

         while not Heap.IsNullPointer (ThisVC) and VCSuccess
         --#assert True;
         loop

            VCHeap.Details (ThisVC,
                            ThisVCName,
                            ThisVCPathStart,
                            ThisVCPathEnd,
                            ThisVCEndType,
                            ThisVCProvedByExaminer,
                            ThisVCProvedBySimplifier,
                            ThisVCProvedByChecker,
                            ThisVCProvedByReview,
                            ThisVCProvedFalse,
                            ThisVCProvedContra,
                            ThisVCProvedUser);

            -- trim the VC name prefix from the VCName to get VC number
            -- as we only print the VC number in the table
            VCNumber := EStrings.Section (ThisVCName,
                                                 VCNamePrefix.Length + 2,
                                                 ThisVCName.Length - (VCNamePrefix.Length + 1));

            --# assert True;

            -- Work out what the status of the vc is.
            if ThisVCProvedByExaminer then
               VCStat := XMLSummary.VCG;
            elsif ThisVCProvedBySimplifier then
               VCStat := XMLSummary.SIV;
            elsif ThisVCProvedByChecker then
               VCStat := XMLSummary.PLG;
            elsif ThisVCProvedByReview then
               VCStat := XMLSummary.REV;
            elsif ThisVCProvedFalse then
               VCStat := XMLSummary.WRONG;
            else
               VCStat := XMLSummary.TODO;
            end if;

            --# assert True;

            if ThisVCProvedContra then
               VCStat := XMLSummary.CONTRA;
            elsif ThisVCProvedUser then
               VCStat := XMLSummary.USER;
            end if;

            --# assert True;

            -- Generate the vc tag.
            XMLSummary.VC (VCNumber,
                           ThisVCPathStart,
                           XMLSummary.XStr (VCDetails.EndTypeImage (ThisVCEndType)),
                           ThisVCPathEnd,
                           VCStat,
                           ReportFile);

            VCHeap.Next (ThisVC,
                         VCSuccess,
                         NextVC);

            ThisVC := NextVC;
         end loop;
      end if;

      XMLSummary.EndFile (ReportFile);

   else

      -- If there any warnings or errors found in analysing the VCG,
      -- SIV or PLG file, record the filname of the errant file to a
      -- temporary warning and error file for reporting in the overall
      -- or short summary generated by the Total package.
      if VCError then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.PutLine (TempWarnErrorFile,
                                      PathFormatter.Format (VCFileName));
      end if;

      --# assert True;
      if SIVError then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.PutLine (TempWarnErrorFile,
                                      PathFormatter.Format (SIVFileName));
      end if;

      --# assert True;
      if PLGError then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.PutLine (TempWarnErrorFile,
                                      PathFormatter.Format (PLGFileName));
      end if;

      --# assert True;
      if (SLGError or SLGFileMissing) then
         SPARK_IO.Put_Integer (TempWarnErrorFile, 1, 4, 10);
         ELStrings.PutLine (TempWarnErrorFile,
                                      PathFormatter.Format (SLGFileName));
      end if;

      --# assert True;
      if REVErrors.LastError /= ErrorsIndex'First then
         --  don't count any errors which are duplicated in review file
         NonDuplicatedErrors := REVErrors.LastError - 1;
         for I in Integer range (ErrorsIndex'First + 1) .. REVErrors.LastError loop
            --# accept F, 10, UnusedPos, "UnusedPos unused here";
            EStrings.FindSubString
              (REVErrors.ErrorList (I), "duplicated", Duplicated, UnusedPos);
            --# end accept;

            if Duplicated then
               NonDuplicatedErrors := NonDuplicatedErrors - 1;
            end if;
         end loop;

         --# assert True;
         --  if there are any (nonduplicated) errors in the file, print number and VC name
         if NonDuplicatedErrors >= 1 then
            if REVErrors.ExcessCount > 0 then
               SPARK_IO.Put_Line (ReportFile, "There were too many errors", 0);
            else
               SPARK_IO.Put_Integer (TempPRVerrFile, NonDuplicatedErrors, 4, 10);
            end if;
            ELStrings.PutLine (TempPRVerrFile,
                                         PathFormatter.Format (VCFileName));
         end if;
      end if;

      -- Only display the table if there were no errors for this proof object.
      -- Note than a missing SLG file is not treated as an error.
      -- Some organisations may deliberately delete SLG files, to
      -- save space.
      if not (VCError or
              SIVError or
              SLGError or
              PLGError or
              REVErrors.LastError > ErrorsIndex'First)
      then

         SPARK_IO.New_Line (ReportFile, 1);
         EStrings.CopyString (TableLine, "VCs for ");
         EStrings.AppendExaminerStringTruncate (TableLine, VCNamePrefix);
         EStrings.AppendStringTruncate (TableLine, " :");
         EStrings.PutLine (ReportFile, TableLine);

         VCNameTabulation := VCHeap.GetLongestVCNameLength - VCNamePrefix.Length;
         -- Give at least 5 columns (including a leading space)
         -- for the VC number, so subprograms with up to 9999
         -- VCs all come out looking the same.
         if VCNameTabulation < 5 then
            VCNameTabulation := 5;
         end if;

         VCPathStartTabulation := 3 + (VCNameTabulation +
           VCHeap.GetLongestVCStartLength);
         VCPathEndTabulation := 3 + (VCPathStartTabulation +
           VCHeap.GetLongestVCEndLength);

         -- Print the table header
         EStrings.CopyString (TableLine, "-");
         AddPadding (TableLine, VCPathEndTabulation + 42, "-");
         EStrings.PutLine (ReportFile, TableLine);

         EStrings.CopyString (TableLine, "   ");
         AddPadding (TableLine, VCNameTabulation, " ");
         EStrings.AppendStringTruncate (TableLine, " |      ");
         AddPadding (TableLine, VCPathStartTabulation, " ");
         EStrings.AppendStringTruncate (TableLine, " |    ");
         AddPadding (TableLine, VCPathEndTabulation, " ");
         EStrings.AppendStringTruncate (TableLine, "  |  -----Proved In-----  |       |       |");
         EStrings.PutLine (ReportFile, TableLine);

         EStrings.CopyString (TableLine, " # ");
         AddPadding (TableLine, VCNameTabulation, " ");
         EStrings.AppendStringTruncate (TableLine, " | From ");
         AddPadding (TableLine, VCPathStartTabulation, " ");
         EStrings.AppendStringTruncate (TableLine, " | To ");
         AddPadding (TableLine, VCPathEndTabulation, " ");
         EStrings.AppendStringTruncate (TableLine, "  | vcg | siv | plg | prv | False | TO DO |");
         EStrings.PutLine (ReportFile, TableLine);

         EStrings.CopyString (TableLine, "-");
         AddPadding (TableLine, VCPathEndTabulation + 42, "-");
         EStrings.PutLine (ReportFile, TableLine);


         -- Now loop through the VCHeap and print one table line per VC
         ThisVC := VCHeap.FirstEntry;
         VCSuccess := True;

         while not Heap.IsNullPointer (ThisVC) and VCSuccess
         loop

            VCHeap.Details (ThisVC,
                            ThisVCName,
                            ThisVCPathStart,
                            ThisVCPathEnd,
                            ThisVCEndType,
                            ThisVCProvedByExaminer,
                            ThisVCProvedBySimplifier,
                            ThisVCProvedByChecker,
                            ThisVCProvedByReview,
                            ThisVCProvedFalse,
                            ThisVCProvedContra,
                            ThisVCProvedUser);

            -- trim the VC name prefix from the VCName to get VC number
            -- as we only print the VC number in the table

            VCNumber := EStrings.Section (ThisVCName,
                                                 VCNamePrefix.Length + 2,
                                                 ThisVCName.Length - (VCNamePrefix.Length + 1));


            -- Start composing the table line for this VC
            EStrings.CopyString (TableLine, " ");
            EStrings.AppendExaminerStringTruncate (TableLine, VCNumber);

            -- pad with spaces to longest VC number length
            AddPadding (TableLine, VCNameTabulation, " ");

            EStrings.AppendStringTruncate (TableLine, " | ");
            EStrings.AppendExaminerStringTruncate (TableLine, ThisVCPathStart);
            AddPadding (TableLine, VCPathStartTabulation, " ");

            EStrings.AppendStringTruncate (TableLine, " | ");

            EStrings.AppendStringTruncate (TableLine,
                                          VCDetails.EndTypeImage (ThisVCEndType));
            EStrings.AppendExaminerStringTruncate (TableLine, ThisVCPathEnd);
            AddPadding (TableLine, VCPathEndTabulation, " ");

            EStrings.AppendStringTruncate (TableLine, "  |");

            AppendValue (TableLine, ThisVCProvedByExaminer);
            EStrings.AppendStringTruncate (TableLine, "|");

            AppendValue (TableLine, ThisVCProvedBySimplifier);
            EStrings.AppendStringTruncate (TableLine, "|");

            AppendValue (TableLine, ThisVCProvedByChecker);
            EStrings.AppendStringTruncate (TableLine, "|");

            AppendValue (TableLine, ThisVCProvedByReview);
            EStrings.AppendStringTruncate (TableLine, "| ");

            AppendValue (TableLine, ThisVCProvedFalse);
            EStrings.AppendStringTruncate (TableLine, " | ");

            --# assert True;

            if ThisVCProvedFalse then
               NumFalse := NumFalse + 1;
            end if;

            --# assert True;

            ToDo := not (ThisVCProvedByExaminer or
                         ThisVCProvedBySimplifier or
                         ThisVCProvedByChecker or
                         ThisVCProvedByReview or
                         ThisVCProvedFalse);
            AppendValue (TableLine, ToDo);

            EStrings.AppendStringTruncate (TableLine, " | ");

            --# assert True;

            if ToDo then
               NumUndischarged := NumUndischarged + 1;
            end if;

            --# assert True;

            if ThisVCProvedContra then
               NumContra := NumContra + 1;
            end if;

            --# assert True;

            if ThisVCProvedUser then
               NumUser := NumUser + 1;
            end if;

            --# assert True;

            -- Now print the table line and get info for the next VC
            EStrings.PutLine (ReportFile, TableLine);

            VCHeap.Next (ThisVC,
                         VCSuccess,
                         NextVC);

            ThisVC := NextVC;

         end loop;

         --# assert True;
         if NumContra > 0 then
            SPARK_IO.Put_Integer (TempContraFile, NumContra, 4, 10);
            ELStrings.PutLine (TempContraFile,
                                         PathFormatter.Format (VCFileName));
         end if;

         --# assert True;
         if NumUser > 0 then
            SPARK_IO.Put_Integer (TempUserFile, NumUser, 4, 10);
            ELStrings.PutLine (TempUserFile,
                                         PathFormatter.Format (VCFileName));
         end if;

         --# assert True;
         if NumFalse > 0 then
            SPARK_IO.Put_Integer (TempFalseFile, NumFalse, 4, 10);
            ELStrings.PutLine (TempFalseFile,
                                         PathFormatter.Format (VCFileName));
         end if;

         --# assert True;
         -- if the subprogram contains any undischarged VCs add the name and number to TempFile
         if NumUndischarged > 0 then
            SPARK_IO.Put_Integer (TempFile, NumUndischarged, 4, 10);
            ELStrings.PutLine (TempFile,
                                         PathFormatter.Format (VCFileName));
         end if;

         -- Print the table footer if we are not doing XML.
         EStrings.CopyString (TableLine, "-");
         AddPadding (TableLine, VCPathEndTabulation + 42, "-");
         EStrings.PutLine (ReportFile, TableLine);

         SPARK_IO.New_Line (ReportFile, 1);
      end if;
   end if;

   --# accept F, 33, UnusedPos, "UnusedPos unused here";
end PrintVCReport;
