-- $Id: vcs-analysevcfile.adb 12146 2009-01-14 13:14:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .VCG file                                            --
--                                                                            --
--------------------------------------------------------------------------------
separate (VCS)
procedure AnalyseVCFile
  (ReportFile          : in     SPARK_IO.File_Type;
   FileName            : in     ELStrings.T;
   ErrorInFile         :    out Boolean;
   FileError           :    out EStrings.T;
   FileDateTime        :    out EStrings.T)
is
   DummyCloseStatus : SPARK_IO.File_Status;
   DateTime         : EStrings.T;
   FileLine         : EStrings.T;
   FinishedWithFile : Boolean;
   OpenStatus       : SPARK_IO.File_Status;
   ReadLineSuccess  : Boolean;
   SubProgramName   : EStrings.T;
   VCFile           : SPARK_IO.File_Type := SPARK_IO.Null_File;
   VCInfo           : VCInfoType;
   FileStatus       : FileStatusT;

   CurrentVCName    : EStrings.T;
   ParsingState     : ParsingStateType := Initial;
   TrimmedLine      : EStrings.T;

   ------------------------------------------------------------------------
   procedure ExtractVCFileDateTimeAndSubprogName
     (VCFile         : in     SPARK_IO.File_Type;
      FileDateTime   :    out EStrings.T;
      SubProgramName :    out EStrings.T;
      FileStatus     :    out FileStatusT)

   --# global in out SPARK_IO.File_sys;
   --# derives FileDateTime,
   --#         FileStatus,
   --#         SPARK_IO.File_sys,
   --#         SubProgramName    from SPARK_IO.File_sys,
   --#                                VCFile;
   is
      FileLine    : EStrings.T;
      TrimmedLine : EStrings.T;
      SubprogramFound : Boolean := False;
   begin
      FileStatus := NotCorrupt;
      FileDateTime := EStrings.EmptyString;
      SubProgramName := EStrings.EmptyString;

      -- Check for completely empty file.
      EStrings.GetLine (VCFile, FileLine);
      if EStrings.Eq1String (FileLine, "") and SPARK_IO.End_Of_File (VCFile) then
         FileStatus := CorruptEmptyFile;
      else
         --Keep on reading from this file, until the desired information is retrieved
         --or the end of the file is reached.
         loop
            TrimmedLine := EStrings.Trim (FileLine);

            -- find date
            -- (There is an implicit assumption that the date, if present, will appear
            --  before the subprogram name.)
            if EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 4),
                                          "DATE") then
               FileDateTime := EStrings.Section (TrimmedLine,
                                                        VCGFileDateTimeStartColumn,
                                                        VCGFileDateTimeLength);
            end if;

            -- find and set SubProgramName
            -- once this has been found go on to analyse the rest of the file
            -- Match against the whole string plus space or newline - to guard against VC
            -- headers ('procedure_x.') being wrongly detected as the subprogram name.
            if (EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 8), "FUNCTION") and
                (EStrings.Eq1String (EStrings.Section (TrimmedLine, 9, 1), " ") or
                 TrimmedLine.Length = 8)) or
               (EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 9), "PROCEDURE") and
                (EStrings.Eq1String (EStrings.Section (TrimmedLine, 10, 1), " ") or
                 TrimmedLine.Length = 9)) or
               (EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 4), "TASK") and
                (EStrings.Eq1String (EStrings.Section (TrimmedLine, 5, 1), " ") or
                 TrimmedLine.Length = 4)) then
                  SubProgramName := EStrings.Trim (FileLine);
                  SubprogramFound := True;
            end if;

            exit when (SubprogramFound or SPARK_IO.End_Of_File (VCFile));
            EStrings.GetLine (VCFile, FileLine);
         end loop;
      end if;

      if FileDateTime = EStrings.EmptyString then
            EStrings.CopyString (FileDateTime, "Unknown Date (for vc generation)");
      end if;

      if (FileStatus = NotCorrupt) and not (SubprogramFound) then
         FileStatus := CorruptUnknownSubprogram;
      end if;
   end ExtractVCFileDateTimeAndSubprogName;

   --------------------------------------------------------------------------
   function GetLineNumber (LineNumber : VCLineType)
      return EStrings.T
   is
      Number        : Integer;
      NumberString  : String_10;
      Result        : EStrings.T;
      TrimmedResult : EStrings.T;
   begin -- GetLineNumber

      if LineNumber = Refinement_Or_Inheritance_VC then
         EStrings.CopyString (TrimmedResult, "     ");
      elsif LineNumber = VCLineStart then
         EStrings.CopyString (TrimmedResult, "start");
      elsif LineNumber = VCLineEnd then
         EStrings.CopyString (TrimmedResult, "finish");
      else
         Number := LineNumber;
         SPARK_IO.Put_Int_To_String (NumberString,
                                     Number,
                                     1,
                                     10);
         EStrings.CopyString (Result, NumberString);
         TrimmedResult := EStrings.Trim (Result);
      end if;

      return TrimmedResult;

   end GetLineNumber;

   --------------------------------------------------------------------------

begin -- AnalyseVCFile
   CurrentVCName   := EStrings.EmptyString;

   -- open VC file
   SPARK_IO.Open (VCFile,
                  SPARK_IO.In_File,
                  FileName.Length,
                  FileName.Content,
                  "",
                  OpenStatus);
   if OpenStatus /= SPARK_IO.Ok then
      FatalErrors.Process (FatalErrors.CouldNotOpenInputFile, ELStrings.EmptyString);
   end if;

   --No errors, until discover otherwise.
   ErrorInFile := False;
   FileError := EStrings.EmptyString;

   ExtractVCFileDateTimeAndSubprogName (VCFile, DateTime, SubProgramName, FileStatus);

   --Report any error to standard out, store in XML summary structute, and
   --set error flag accordingly.
   --Note that XML summary structures seem to be generated regardless, even if XML is
   --not being used. (This is a little confusing and inefficient, but perfectly safe)
   case FileStatus is
      when NotCorrupt =>
         null;
      when CorruptEmptyFile =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* VC file corrupt: empty file ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         FileError := XMLSummary.XStr ("VC file corrupt: empty file");
         ErrorInFile := True;
      when CorruptUnknownSubprogram =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* VC file corrupt: missing subprogram name ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         FileError := XMLSummary.XStr ("VC file corrupt: missing subprogram name");
         ErrorInFile := True;
   end case;

   --Record the date regardless of errors. This may be a string of the form 'no date'.
   FileDateTime := DateTime;

   --Only continue working on this file if an error has not been seen.
   --(Previously POGS would attempt to work with corrupt files. This feature has the
   -- capacity to produce confusing and wrong results.)
   if not (ErrorInFile) then

      if not CommandLine.Data.XML then
         SPARK_IO.Put_String (ReportFile, "File ", 0);
         if CommandLine.Data.PlainOutput then
            ELStrings.PutLine
              (ReportFile, ELStrings.LowerCase (OSFiling.BaseFileName (FileName)));
         else
            ELStrings.PutLine (ReportFile, FileName);
         end if;

         EStrings.PutLine (ReportFile, SubProgramName);
         SPARK_IO.New_Line (ReportFile, 1);

         if CommandLine.Data.IgnoreDates then
            SPARK_IO.Put_Line (ReportFile, "*** Warning: VC date stamps ignored ***", 0);
         else
            SPARK_IO.Put_String (ReportFile, "VCs generated ", 0);
            EStrings.PutLine (ReportFile, DateTime);
         end if;
      end if;

      -- find first non blank line
      -- if we get to the end of the file first, flag a fatal error
      ReadNextNonBlankLine (VCFile, ReadLineSuccess, FileLine);

      if not ReadLineSuccess then
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* VC file corrupt: no data beyond header ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         FileError := XMLSummary.XStr ("VC file corrupt: no data beyond header");
         ErrorInFile := True;
      else
         -- if it's an error message then reproduce this and exit
         -- assumption: that if an error message is present it will be
         -- the first non blank line after the file header
         if IsVCErrorMessage (FileLine) then
            if not CommandLine.Data.XML then
               SPARK_IO.New_Line (ReportFile, 1);
               EStrings.PutString (ReportFile, FileLine);
            end if;
            FileError := FileLine;  -- Return the file error
            ErrorInFile := True;
         else
            ErrorInFile := False;
            FileError := EStrings.EmptyString;

            -- initialize the 'current information' structure
            VCInfo := VCInfoType'(StartLine            => VCLineStart,
                                  EndLine              => VCLineEnd,
                                  EndLinePointType     => VCDetails.Undetermined_Point,
                                  NumberOfVCs          => 0,
                                  ThisStartLinePrinted => False,
                                  FileType             => StandardVCFileType,
                                  AnyVCsPrinted        => False,
                                  Valid                => False);

            FinishedWithFile := False;

            -- process file line-by-line
            -- on entry to the loop there is already a valid line in the
            -- FileLine buffer
            while not FinishedWithFile loop
               -- examine line and act accordingly
               if IsNewRangeLine (FileLine) then
                  case ParsingState is
                     when Initial =>
                        ParsingState := FirstRange;
                     when FirstVCName =>
                        ParsingState := NewRange;
                     when NewVCName =>
                        ParsingState := NewRange;
                     when others =>
                        null;
                  end case;

                  AppendNextLineFromFile (FileLine, VCFile);

                  ProcessNewRangeLine (FileLine, VCInfo);

               elsif IsNewVCLine (FileLine) then
                  case ParsingState is
                     when FirstRange =>
                        -- Initialise VCHeap and store the first VC on the VCHeap
                        TrimmedLine := EStrings.Trim (FileLine);
                        CurrentVCName := EStrings.Section
                          (TrimmedLine, 1, TrimmedLine.Length - 1);

                        ParsingState := FirstVCName;
                        VCHeap.Reinitialize (CurrentVCName,
                                             GetLineNumber (VCInfo.StartLine),
                                             GetLineNumber (VCInfo.EndLine),
                                             VCInfo.EndLinePointType);
                     when FirstVCName =>
                        TrimmedLine := EStrings.Trim (FileLine);
                        CurrentVCName := EStrings.Section
                          (TrimmedLine, 1, TrimmedLine.Length - 1);
                        ParsingState := NewVCName;
                        VCHeap.Add (VCHeap.FirstEntry,
                                    CurrentVCName,
                                    GetLineNumber (VCInfo.StartLine),
                                    GetLineNumber (VCInfo.EndLine),
                                    VCInfo.EndLinePointType,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False);
                     when NewRange =>
                        -- Store a new VC on the VC Heap
                        TrimmedLine := EStrings.Trim (FileLine);
                        CurrentVCName := EStrings.Section
                          (TrimmedLine, 1, TrimmedLine.Length - 1);
                        --SPARK_IO.Put_Line(ReportFile,"NewVCNameFound - New range",0);
                        ParsingState := NewVCName;
                        VCHeap.Add (VCHeap.FirstEntry,
                                    CurrentVCName,
                                    GetLineNumber (VCInfo.StartLine),
                                    GetLineNumber (VCInfo.EndLine),
                                    VCInfo.EndLinePointType,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False);
                     when NewVCName =>
                        -- The range has not changed, but store a new VC on the VC Heap
                        TrimmedLine := EStrings.Trim (FileLine);
                        CurrentVCName := EStrings.Section
                          (TrimmedLine, 1, TrimmedLine.Length - 1);
                        --SPARK_IO.Put_Line(ReportFile,"NewVCNameFound - Same range2",0);
                        ParsingState := NewVCName;
                        VCHeap.Add (VCHeap.FirstEntry,
                                    CurrentVCName,
                                    GetLineNumber (VCInfo.StartLine),
                                    GetLineNumber (VCInfo.EndLine),
                                    VCInfo.EndLinePointType,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False,
                                    False);
                     when others =>
                        null;
                  end case;

                  VCInfo.NumberOfVCs := VCInfo.NumberOfVCs + 1;

               elsif IsTriviallyTrueVC (FileLine) then

                  VCHeap.MarkAsProvedByExaminer (CurrentVCName);

               elsif IsTriviallyFalseVC (FileLine) then

                  VCHeap.MarkAsProvedFalse (CurrentVCName);

               end if;

               -- read next line
               ReadNextNonBlankLine (VCFile, ReadLineSuccess, FileLine);

               -- if unsuccessful then check EOF
               -- and set FinishedWithFile accordingly
               if not ReadLineSuccess then
                  if SPARK_IO.End_Of_File (VCFile) then
                     FinishedWithFile := True;
                  else
                     FatalErrors.Process (FatalErrors.ProblemReadingFile, ELStrings.EmptyString);
                  end if;
               end if;
            end loop;


            -- write information for last VC
            -- two VCInfo parameters are necessary as WriteVCInfo compares them
            -- in deciding what to write (see definition of WriteVCInfo)
            if VCInfo.Valid then
               -- Reporting is now done as a table, so this has been commented out
               --WriteVCInfo( ReportFile, VCInfo, Dummy );
               null;
            else
               if not CommandLine.Data.XML then
                  SPARK_IO.Put_Line (ReportFile, "No VCs in file", 0);
               end if;
            end if;
         end if;

      end if;
   end if;

   --# accept F, 10, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 33, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 10, VCFile,           "VCFile unused here";
   SPARK_IO.Close (VCFile, DummyCloseStatus); -- Expect ineffective assigment

end AnalyseVCFile;
