-- $Id: vcs-analysesimplifiedvcfile.adb 12146 2009-01-14 13:14:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Procedure to analyse a .SIV file                                            --
--                                                                            --
--------------------------------------------------------------------------------
separate (VCS)
procedure AnalyseSimplifiedVCFile
  (ReportFile        : in     SPARK_IO.File_Type;
   FileName          : in     ELStrings.T;
   VCFileDateTime    : in     EStrings.T;
   SIVFileDateTime   :    out EStrings.T;
   ErrorInSIVFile    :    out Boolean;
   FileError         :    out EStrings.T)
is
   BadFileFormat       : Boolean := False;
   DummyCloseStatus    : SPARK_IO.File_Status;
   FileLine            : EStrings.T;
   FinishedWithFile    : Boolean;
   OpenStatus          : SPARK_IO.File_Status;
   ProcessSuccess      : Boolean;
   ReadLineSuccess     : Boolean;
   SimplifiedVCFile    : SPARK_IO.File_Type := SPARK_IO.Null_File;
   FileStatus          :  FileStatusT;

   SimplificationDateTime : EStrings.T;

   VCGenerationDateTimeFromSIVFile : EStrings.T;

   TrimmedLine : EStrings.T;
   CurrentVCName : EStrings.T;

   -------------------------------------------------------------------------
   -- NOTE, this procedure also removes the comma inserted in the string
   -- by the simplifier
   --
   procedure ExtractDatesAndTimesFromSimplifiedVCFile
     (SimplifiedVCFile       : in     SPARK_IO.File_Type;
      VCGenerationDateTime   :    out EStrings.T;
      SimplificationDateTime :    out EStrings.T;
      FileStatus             :    out FileStatusT)
   --# global in out SPARK_IO.File_sys;
   --# derives FileStatus,
   --#         SimplificationDateTime,
   --#         SPARK_IO.File_sys,
   --#         VCGenerationDateTime   from SimplifiedVCFile,
   --#                                     SPARK_IO.File_sys;
   is
      FileLine        : EStrings.T;
      TrimmedLine     : EStrings.T;
      GenDateTime     : EStrings.T;
      SimpDateTime    : EStrings.T;
      SubprogramFound : Boolean := False;

   begin -- ExtractDatesAndTimesFromSimplifiedVCFile

      FileStatus := NotCorrupt;
      VCGenerationDateTime := EStrings.EmptyString;
      SimplificationDateTime := EStrings.EmptyString;

      --Check for completly empty file.
      EStrings.GetLine (SimplifiedVCFile, FileLine);
      if EStrings.Eq1String (FileLine, "") and SPARK_IO.End_Of_File (SimplifiedVCFile) then
         FileStatus := CorruptEmptyFile;
      else
         --Keep on reading from this file, until the desired information is retrieved
         --or the end of the fikle is reached.
         loop
            TrimmedLine := EStrings.Trim (FileLine);

            -- find date
            -- (There is an implicit assumption that the date, if present, will appear
            --  before the subprogram name.)
            -- When the Examiner is in plain_output mode, the DATE line doesn't appear.
            if EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 7),
                                          "CREATED") then
               -- extract the VC generation date and time from the string
               GenDateTime := EStrings.Section
                 (TrimmedLine,
                  SIVFileVCGenerationDateStartColumn,
                  SIVFileVCGenerationDateLength);
               EStrings.AppendStringTruncate (GenDateTime, " ");
               EStrings.AppendExaminerStringTruncate
                 (GenDateTime,
                  EStrings.Section (TrimmedLine,
                                          SIVFileVCGenerationTimeStartColumn,
                                          SIVFileVCGenerationTimeLength));
               VCGenerationDateTime := GenDateTime;

               -- extract the simplification date and time from the string
               SimpDateTime := EStrings.Section
                 (TrimmedLine,
                  SIVFileSimplificationDateStartColumn,
                  SIVFileSimplificationDateLength);
               EStrings.AppendStringTruncate (SimpDateTime, " ");
               EStrings.AppendExaminerStringTruncate
                 (SimpDateTime,
                  EStrings.Section (TrimmedLine,
                                           SIVFileSimplificationTimeStartColumn,
                                           SIVFileSimplificationTimeLength));

               SimplificationDateTime := SimpDateTime;
            end if;

            -- don't do anything with the subprogram name, but this brings us to the
            -- correct point in the file
            -- Match against the whole string plus space or newline - to guard against VC
            -- headers ('procedure_x.') being wrongly detected as the subprogram name.
            if (EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 8), "FUNCTION") and
                (EStrings.Eq1String (EStrings.Section (TrimmedLine, 9, 1), " ") or
                 TrimmedLine.Length = 8)) or
               (EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 9), "PROCEDURE") and
                (EStrings.Eq1String (EStrings.Section (TrimmedLine, 10, 1), " ") or
                 TrimmedLine.Length = 9)) or
               (EStrings.Eq1String (EStrings.Section (TrimmedLine, 1, 4), "TASK") and
                (EStrings.Eq1String (EStrings.Section (TrimmedLine, 5, 1), " ") or
                 TrimmedLine.Length = 4)) then
                  SubprogramFound := True;
            end if;

            exit when (SubprogramFound or SPARK_IO.End_Of_File (SimplifiedVCFile));
            EStrings.GetLine (SimplifiedVCFile, FileLine);
         end loop;
      end if;

      if (FileStatus = NotCorrupt) and not (SubprogramFound) then
         FileStatus := CorruptUnknownSubprogram;
      end if;

      -- if date has not been found must be in plain output mode
      -- The above is a false assumption -- the file may just be corrupt. However, the
      -- effect below of setting the string as unknown date is reasonable for both cases.
      if VCGenerationDateTime = EStrings.EmptyString then
         EStrings.AppendExaminerStringTruncate (VCGenerationDateTime,   UnknownVCGDate);
         EStrings.AppendExaminerStringTruncate (SimplificationDateTime, UnknownSIVDate);
      end if;
   end ExtractDatesAndTimesFromSimplifiedVCFile;

   -------------------------------------------------------------------------
   -- look at the next non-blank line to see whether it starts
   -- "*** true". If so the VC has been discharged. Otherwise, increment
   -- the counter of undischarged VCs, and set the flag that an undischarged
   -- VC has been found
   procedure ProcessNewSimplifiedVCLine
     (SimplifiedVCFile  : in     SPARK_IO.File_Type;
      VCName            : in     EStrings.T;
      Success           :    out Boolean)
   --# global in out FatalErrors.State;
   --#        in out SPARK_IO.File_sys;
   --#        in out VCHeap.State;
   --# derives FatalErrors.State,
   --#         VCHeap.State      from *,
   --#                                SimplifiedVCFile,
   --#                                SPARK_IO.File_sys,
   --#                                VCHeap.State,
   --#                                VCName &
   --#         SPARK_IO.File_sys,
   --#         Success           from SimplifiedVCFile,
   --#                                SPARK_IO.File_sys;
   is
      FileLine        : EStrings.T;
      ReadLineSuccess : Boolean;

   begin -- ProcessNewSimplifiedVCLine
      ReadNextNonBlankLine (SimplifiedVCFile, ReadLineSuccess, FileLine);
      if not ReadLineSuccess then
         Success := False;
      else
         Success := True;
         if EStrings.Eq1String
           (EStrings.Section (FileLine, 1, 8), "*** true") then
            VCHeap.MarkAsProvedBySimplifier (VCName);
            if EStrings.Eq1String
           (EStrings.Section (FileLine, 15, 15), "* contradiction") then
               VCHeap.MarkAsProvedByContradiction (VCName);
            elsif EStrings.Eq1String
           (EStrings.Section (FileLine, 15, 14), "* proved using") then
               VCHeap.MarkAsProvedUsingUserProofRule (VCName);
            end if;
         end if;
      end if;
   end ProcessNewSimplifiedVCLine;

   --------------------------------------------------------------------------

begin -- AnalyseSimplifiedVCFile
      -- open simplified VC file
   SPARK_IO.Open (SimplifiedVCFile,
                  SPARK_IO.In_File,
                  FileName.Length,
                  FileName.Content,
                  "",
                  OpenStatus);
   if OpenStatus /= SPARK_IO.Ok then
      FatalErrors.Process (FatalErrors.CouldNotOpenInputFile, ELStrings.EmptyString);
   end if;

   --No errors, until discover otherwise.
   ErrorInSIVFile := False;
   FileError := EStrings.EmptyString;

   ExtractDatesAndTimesFromSimplifiedVCFile (SimplifiedVCFile,
                                             VCGenerationDateTimeFromSIVFile,
                                             SimplificationDateTime,
                                             FileStatus);
   case FileStatus is
      when NotCorrupt =>
         null;
      when CorruptEmptyFile =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* SIV file corrupt: empty file ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         FileError := XMLSummary.XStr ("SIV file corrupt: empty file");
         ErrorInSIVFile := True;
      when CorruptUnknownSubprogram =>
         SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                            "************* SIV file corrupt: missing subprogram name ************", 0);
         SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
         FileError := XMLSummary.XStr ("SIV file corrupt: missing subprogram name");
         ErrorInSIVFile := True;
   end case;

   --Only continue working on this file if an error has not been seen.
   --(Previously POGS would attempt to work with corrupt files. This feature has the
   -- capacity to produce confusing and wrong results.)
   if not (ErrorInSIVFile) then
      if CommandLine.Data.IgnoreDates or else
        EStrings.EqString (VCGenerationDateTimeFromSIVFile,
                                  VCFileDateTime) then

         if not CommandLine.Data.IgnoreDates and
            not CommandLine.Data.XML then
               SPARK_IO.New_Line (ReportFile, 1);
               SPARK_IO.Put_String (ReportFile, "VCs simplified ", 0);
               EStrings.PutLine (ReportFile, SimplificationDateTime);
         end if;

         -- find first non blank line
         -- if we get to the end of the file first, flag a fatal error
         ReadNextNonBlankLine (SimplifiedVCFile, ReadLineSuccess, FileLine);

         if not ReadLineSuccess then
            SPARK_IO.Put_Line (SPARK_IO.Standard_Output,
                               "************* SIV file corrupt: no data beyond header ************", 0);
            SPARK_IO.New_Line (SPARK_IO.Standard_Output, 1);
            BadFileFormat := True;
            FileError := XMLSummary.XStr ("SIV file corrupt: no data beyond header");
         else

            FinishedWithFile := False;
            FileError := EStrings.EmptyString;

            -- process file line-by-line
            -- on entry to the loop there is already a valid line in the
            -- FileLine buffer
            CurrentVCName := EStrings.EmptyString;
            while not FinishedWithFile loop
               -- examine line and act accordingly
               if IsNewRangeLine (FileLine) then
                  AppendNextLineFromFile (FileLine, SimplifiedVCFile);

               elsif IsNewVCLine (FileLine) then

                  TrimmedLine := EStrings.Trim (FileLine);
                  CurrentVCName := EStrings.Section
                    (TrimmedLine, 1, TrimmedLine.Length - 1);

                  ProcessNewSimplifiedVCLine
                    (SimplifiedVCFile,
                     CurrentVCName,
                     ProcessSuccess);

                  if not ProcessSuccess then
                     --# accept F, 41, "Expression is stable but cheap";
                     if not CommandLine.Data.XML then
                        SPARK_IO.Put_String
                          (ReportFile,
                           "*** Warning: Bad format in simplified VC file ***", 0);
                     end if;
                     --# end accept;
                     FinishedWithFile := True;
                     BadFileFormat := True;
                     FileError := XMLSummary.XStr ("Warning: Bad format in simplified VC file");
                  end if;
               end if;

               if not FinishedWithFile then
                  -- read next line and check if VC has been proved false
                  ReadNextNonBlankLine (SimplifiedVCFile, ReadLineSuccess, FileLine);
                  if IsTriviallyFalseVC (FileLine) then
                     VCHeap.MarkAsProvedFalse (CurrentVCName);
                  end if;
                  -- if unsuccessful then check EOF
                  -- and set FinishedWithFile accordingly
                  if not ReadLineSuccess then
                     if SPARK_IO.End_Of_File (SimplifiedVCFile) then
                        FinishedWithFile := True;
                     else
                        FatalErrors.Process (FatalErrors.ProblemReadingFile,
                                             ELStrings.EmptyString);
                     end if;
                  end if;
               end if;
            end loop;

         end if;

      else
         -- SIV file is out of date
         if not CommandLine.Data.XML then
            SPARK_IO.New_Line (ReportFile, 1);
            SPARK_IO.Put_Line (ReportFile,
                               "*** Warning: Simplified VC file out of date ***",
                            0);
            SPARK_IO.Put_String (ReportFile, "VCs Generated: ", 0);
            EStrings.PutString (ReportFile, VCFileDateTime);
            SPARK_IO.New_Line (ReportFile, 1);

            SPARK_IO.Put_String (ReportFile, "SIV File Date: ", 0);
            EStrings.PutString (ReportFile, VCGenerationDateTimeFromSIVFile);
            SPARK_IO.New_Line (ReportFile, 1);
         end if;

         BadFileFormat := True;
         FileError := XMLSummary.XStr ("Warning: Simplified VC file out of date");
      end if;
   end if;

   --# accept F, 10, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 33, DummyCloseStatus, "DummyCloseStatus unused here" &
   --#        F, 10, SimplifiedVCFile, "SimplifiedVCFile unused here";
   SPARK_IO.Close (SimplifiedVCFile, DummyCloseStatus);

   SIVFileDateTime := SimplificationDateTime;
   --Either an error being raised, or a 'BadFileFormat'
   --being detected is an error.
   ErrorInSIVFile  := ErrorInSIVFile or BadFileFormat;

end AnalyseSimplifiedVCFile;
