-- $Id: vcheap.ads 12146 2009-01-14 13:14:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package combining Heap and VCDetails to give an ordered list of VC details. --
--                                                                            --
--This is implemented as an abstract state machine. This is possible as there --
--is only a single occurrence of the VCHeap in the program.   It is necessary --
--to prevent unacceptable overheads of creating local copies of the embedded  --
--Heap and VCDetails   types to circumvent the entire variable rule. It would --
--also be possible to implement Heap and VCDetails   as ASMs but not          --
--necessarily desirable as this would affect the high level annotations of the--
--program.                                                                    --
--                                                                            --
--------------------------------------------------------------------------------
with EStrings;
with Heap;
with VCDetails;

use type Heap.Atom;

--# inherit ELStrings,
--#         EStrings,
--#         FatalErrors,
--#         Heap,
--#         HeapIndex,
--#         VCDetails;
package VCHeap
--# own I_State,
--#     State;
is

   -- StartIndex is a point in the linked list at which to start the
   -- search. This is used to start insertion at the first VC.
   -- If the name table is full, a fatal error is produced and Add does not
   -- return
   procedure Add (StartIndex         : in Heap.Atom;
                  NewName            : in EStrings.T;
                  PathStart          : in EStrings.T;
                  PathEnd            : in EStrings.T;
                  EndType            : in VCDetails.TerminalPointType;
                  ProvedByExaminer   : in Boolean;
                  ProvedBySimplifier : in Boolean;
                  ProvedByChecker    : in Boolean;
                  ProvedByReview     : in Boolean;
                  ProvedFalse        : in Boolean;
                  ProvedByContradiction    : in Boolean;
                  ProvedUsingUserProofRule : in Boolean);
   --# global in out FatalErrors.State;
   --#        in out I_State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                EndType,
   --#                                NewName,
   --#                                PathEnd,
   --#                                PathStart,
   --#                                ProvedByChecker,
   --#                                ProvedByContradiction,
   --#                                ProvedByExaminer,
   --#                                ProvedByReview,
   --#                                ProvedBySimplifier,
   --#                                ProvedFalse,
   --#                                ProvedUsingUserProofRule,
   --#                                StartIndex,
   --#                                State &
   --#         I_State           from *,
   --#                                NewName,
   --#                                PathEnd,
   --#                                PathStart;

   --------------------------------------------------------------------------
   -- this procedure returns the VC details for the specified entry in the
   -- linked list.
   procedure Details (ListIndex          : in     Heap.Atom;
                      VCName             :    out EStrings.T;
                      PathStart          :    out EStrings.T;
                      PathEnd            :    out EStrings.T;
                      EndType            :    out VCDetails.TerminalPointType;
                      ProvedByExaminer   :    out Boolean;
                      ProvedBySimplifier :    out Boolean;
                      ProvedByChecker    :    out Boolean;
                      ProvedByReview     :    out Boolean;
                      ProvedFalse        :    out Boolean;
                      ProvedByContradiction    : out Boolean;
                      ProvedUsingUserProofRule : out Boolean);
   --# global in State;
   --# derives EndType,
   --#         PathEnd,
   --#         PathStart,
   --#         ProvedByChecker,
   --#         ProvedByContradiction,
   --#         ProvedByExaminer,
   --#         ProvedByReview,
   --#         ProvedBySimplifier,
   --#         ProvedFalse,
   --#         ProvedUsingUserProofRule,
   --#         VCName                   from ListIndex,
   --#                                       State;

   --------------------------------------------------------------------------
   function FirstEntry return Heap.Atom;
   --# global in State;

   --------------------------------------------------------------------------
   procedure Initialize;
   --# global out I_State;
   --#        out State;
   --# derives I_State,
   --#         State   from ;

   --------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in VCDetails.ErrorType);
   --# global in out State;
   --# derives State from *,
   --#                    ErrorKind;

   --------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in VCDetails.ErrorType) return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   procedure Reinitialize (FirstElement   : in   EStrings.T;
                           FirstPathStart : in   EStrings.T;
                           FirstPathEnd   : in   EStrings.T;
                           FirstEndType   : in   VCDetails.TerminalPointType);
   --# global out I_State;
   --#        out State;
   --# derives I_State from FirstElement,
   --#                      FirstPathEnd,
   --#                      FirstPathStart &
   --#         State   from FirstElement,
   --#                      FirstEndType,
   --#                      FirstPathEnd,
   --#                      FirstPathStart;

   --------------------------------------------------------------------------
   -- this procedure returns the 'NextOne' ordered element in VCH after
   -- 'AfterThis'. It is successful if the NextOne is not a 'null' pointer
   procedure Next (AfterThis : in     Heap.Atom;
                   Success   :    out Boolean;
                   NextOne   :    out Heap.Atom);
   --# global in State;
   --# derives NextOne,
   --#         Success from AfterThis,
   --#                      State;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByExaminer (VCName : in EStrings.T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                State,
   --#                                VCName;

   --------------------------------------------------------------------------
   procedure MarkAsProvedBySimplifier (VCName : in EStrings.T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                State,
   --#                                VCName;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByChecker (VCName : in EStrings.T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                State,
   --#                                VCName;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByReview (VCName : in EStrings.T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                State,
   --#                                VCName;

   --------------------------------------------------------------------------
   procedure MarkAsProvedFalse (VCName : in EStrings.T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                State,
   --#                                VCName;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByContradiction (VCName : in EStrings.T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                State,
   --#                                VCName;

   --------------------------------------------------------------------------
   procedure MarkAsProvedUsingUserProofRule (VCName : in EStrings.T);
   --# global in out FatalErrors.State;
   --#        in out State;
   --# derives FatalErrors.State,
   --#         State             from *,
   --#                                State,
   --#                                VCName;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByExaminer (VCName : in EStrings.T)
      return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedBySimplifier (VCName : in EStrings.T)
      return Boolean;
   --# global in State;
   --------------------------------------------------------------------------
   function IsMarkedAsProvedByChecker (VCName : in EStrings.T)
      return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByReview (VCName : in EStrings.T)
      return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedFalse (VCName : in EStrings.T)
      return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByContradiction (VCName : in EStrings.T)
      return Boolean;
   --# global in State;

   function IsMarkedAsProvedUsingUserProofRule (VCName : in EStrings.T)
      return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   function Exists (VCName : EStrings.T) return Boolean;
   --# global in State;

   --------------------------------------------------------------------------
   procedure GetVCNameEndType
     (VCName : in     EStrings.T;
      VCType :    out VCDetails.TerminalPointType);
   --# global in     State;
   --#        in out FatalErrors.State;
   --# derives FatalErrors.State from *,
   --#                                State,
   --#                                VCName &
   --#         VCType            from State,
   --#                                VCName;

   --------------------------------------------------------------------------
   function GetLongestVCNameLength return Integer;
   --# global in I_State;

   --------------------------------------------------------------------------
   function GetLongestVCStartLength return Integer;
   --# global in I_State;

   --------------------------------------------------------------------------
   function GetLongestVCEndLength return Integer;
   --# global in I_State;

   --------------------------------------------------------------------------
   function GetVCNamePrefix return EStrings.T;
   --# global in I_State;

end VCHeap;
