-- $Id: vcheap.adb 12146 2009-01-14 13:14:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package combining Heap and VCDetails to give an ordered list of VC details. --
--                                                                            --
--------------------------------------------------------------------------------
with ELStrings,
     FatalErrors,
     HeapIndex;

use type HeapIndex.IndexType;

package body VCHeap
--# own State is ThePointers,
--#              TheDetails,
--#              StartOfPointersList &
--#     I_State is VCNamePrefix,
--#                LongestVCNameLength,
--#                LongestVCStartLength,
--#                LongestVCEndLength;
is

   ThePointers         : Heap.HeapRecord;
   TheDetails          : VCDetails.DataType;
   StartOfPointersList : Heap.Atom;

   -- this one is used to record the prefix name for the VCs
   VCNamePrefix         : EStrings.T;

   -- the following are used as VC are added to the heap to record information
   -- used later for displaying the results as a table
   LongestVCNameLength  : Integer;
   LongestVCStartLength : Integer;
   LongestVCEndLength   : Integer;


   function FirstEntry return Heap.Atom
   --# global in StartOfPointersList;
   is
   begin
      return StartOfPointersList;
   end FirstEntry;

   ------------------------------------------------------------------------
   procedure Add (StartIndex         : in Heap.Atom;
                  NewName            : in EStrings.T;
                  PathStart          : in EStrings.T;
                  PathEnd            : in EStrings.T;
                  EndType            : in VCDetails.TerminalPointType;
                  ProvedByExaminer   : in Boolean;
                  ProvedBySimplifier : in Boolean;
                  ProvedByChecker    : in Boolean;
                  ProvedByReview     : in Boolean;
                  ProvedFalse        : in Boolean;
                  ProvedByContradiction : in Boolean;
                  ProvedUsingUserProofRule : in Boolean)
   --# global in out FatalErrors.State;
   --#        in out LongestVCEndLength;
   --#        in out LongestVCNameLength;
   --#        in out LongestVCStartLength;
   --#        in out TheDetails;
   --#        in out ThePointers;
   --# derives FatalErrors.State,
   --#         TheDetails,
   --#         ThePointers          from *,
   --#                                   EndType,
   --#                                   NewName,
   --#                                   PathEnd,
   --#                                   PathStart,
   --#                                   ProvedByChecker,
   --#                                   ProvedByContradiction,
   --#                                   ProvedByExaminer,
   --#                                   ProvedByReview,
   --#                                   ProvedBySimplifier,
   --#                                   ProvedFalse,
   --#                                   ProvedUsingUserProofRule,
   --#                                   StartIndex,
   --#                                   TheDetails,
   --#                                   ThePointers &
   --#         LongestVCEndLength   from *,
   --#                                   PathEnd &
   --#         LongestVCNameLength  from *,
   --#                                   NewName &
   --#         LongestVCStartLength from *,
   --#                                   PathStart;
   is
      ExistingName       : EStrings.T;
      ExistingPathStart  : EStrings.T;
      ExistingPathEnd    : EStrings.T;
      ExistingEndType    : VCDetails.TerminalPointType;

      ExistingProvedByExaminer   : Boolean;
      ExistingProvedBySimplifier : Boolean;
      ExistingProvedByChecker    : Boolean;
      ExistingProvedByReview     : Boolean;
      ExistingProvedFalse        : Boolean;
      ExistingProvedByContradiction : Boolean;
      ExistingProvedUsingUserProofRule : Boolean;

      ListIndex       : Heap.Atom;
      LoopFinished    : Boolean := False;
      NextEntryInList : Heap.Atom;
      OrderResult     : EStrings.OrderTypes;
      RetrieveSuccess : Boolean;

      ------------------------------------------------------------------------
      procedure InsertInList
        (ListIndex          : in Heap.Atom;
         NextEntryInList    : in Heap.Atom;
         Name               : in EStrings.T;
         PathStart          : in EStrings.T;
         PathEnd            : in EStrings.T;
         EndType            : in VCDetails.TerminalPointType;
         ProvedByExaminer   : in Boolean;
         ProvedBySimplifier : in Boolean;
         ProvedByChecker    : in Boolean;
         ProvedByReview     : in Boolean;
         ProvedFalse        : in Boolean;
         ProvedByContradiction    : in Boolean;
         ProvedUsingUserProofRule : in Boolean)
      --# global in out FatalErrors.State;
      --#        in out TheDetails;
      --#        in out ThePointers;
      --# derives FatalErrors.State from *,
      --#                                TheDetails,
      --#                                ThePointers &
      --#         TheDetails        from *,
      --#                                EndType,
      --#                                Name,
      --#                                PathEnd,
      --#                                PathStart,
      --#                                ProvedByChecker,
      --#                                ProvedByContradiction,
      --#                                ProvedByExaminer,
      --#                                ProvedByReview,
      --#                                ProvedBySimplifier,
      --#                                ProvedFalse,
      --#                                ProvedUsingUserProofRule &
      --#         ThePointers       from *,
      --#                                ListIndex,
      --#                                NextEntryInList,
      --#                                TheDetails;
      is
         CreateAtomSuccess : Boolean;
         DetailsAddSuccess : Boolean;
         NewDetailsIndex   : HeapIndex.IndexType;
         NewPointersIndex  : Heap.Atom;
      begin -- InsertInList
         -- allocate heap atom
         Heap.CreateAtom (ThePointers,
                          NewPointersIndex,
                          CreateAtomSuccess);

         -- allocate file details entry
         VCDetails.Add (Details            => TheDetails,
                        Index              => NewDetailsIndex,
                        Success            => DetailsAddSuccess,
                        Name               => Name,
                        Path_Start         => PathStart,
                        Path_End           => PathEnd,
                        End_Type           => EndType,
                        ProvedByExaminer   => ProvedByExaminer,
                        ProvedBySimplifier => ProvedBySimplifier,
                        ProvedByChecker    => ProvedByChecker,
                        ProvedByReview     => ProvedByReview,
                        ProvedFalse        => ProvedFalse,
                        ProvedByContradiction => ProvedByContradiction,
                        ProvedUsingUserProofRule => ProvedUsingUserProofRule);

         if not (CreateAtomSuccess and DetailsAddSuccess) then
            FatalErrors.Process (FatalErrors.VCHeapFull, ELStrings.EmptyString);
         end if;

         -- point heap atom to file details entry
         Heap.UpdateAValue (ThePointers, NewPointersIndex, NewDetailsIndex);

         -- link heap atom into list
         Heap.UpdateAPointer (ThePointers, ListIndex, NewPointersIndex);
         Heap.UpdateAPointer (ThePointers, NewPointersIndex, NextEntryInList);

      end InsertInList;

      -------------------------------------------------------------------------
      function LongestOf
        (First_Length : Integer; Second_Length : Integer) return Integer
      is
         Result : Integer;
      begin -- LongestOf
         if First_Length > Second_Length then
            Result := First_Length;
         else
            Result := Second_Length;
         end if;

         return Result;
      end LongestOf;

      -------------------------------------------------------------------------
   begin -- Add
      -- start at point specified in linked list
      ListIndex := StartIndex;

      while not LoopFinished loop
         -- if current item is last then add after it
         NextEntryInList := Heap.APointer (ThePointers, ListIndex);

         if NextEntryInList = 0 then
            InsertInList (ListIndex,
                          NextEntryInList,
                          NewName,
                          PathStart,
                          PathEnd,
                          EndType,
                          ProvedByExaminer,
                          ProvedBySimplifier,
                          ProvedByChecker,
                          ProvedByReview,
                          ProvedFalse,
                          ProvedByContradiction,
                          ProvedUsingUserProofRule);
            LoopFinished := True;
         else
            -- otherwise get relative order of next item in list and the new item
            --# accept F, 10, ExistingPathStart, "ExistingPathStart not used here" &
            --#        F, 10, ExistingPathEnd,   "ExistingPathEnd not used here" &
            --#        F, 10, ExistingEndType,   "ExistingEndType not used here" &
            --#        F, 10, ExistingProvedByExaminer,         "ExistingProvedByExaminer not used here" &
            --#        F, 10, ExistingProvedBySimplifier,       "ExistingProvedBySimplifier not used here" &
            --#        F, 10, ExistingProvedByChecker,          "ExistingProvedByChecker not used here" &
            --#        F, 10, ExistingProvedByReview,           "ExistingProvedByReview not used here" &
            --#        F, 10, ExistingProvedFalse,              "ExistingProvedFalse not used here" &
            --#        F, 10, ExistingProvedByContradiction,    "ExistingProvedByContradiction not used here" &
            --#        F, 10, ExistingProvedUsingUserProofRule, "ExistingProvedUsingUserProofRule not used here";
            VCDetails.Retrieve (TheDetails,
                                Heap.AValue (ThePointers, NextEntryInList),
                                RetrieveSuccess,
                                ExistingName,
                                ExistingPathStart,
                                ExistingPathEnd,
                                ExistingEndType,
                                ExistingProvedByExaminer,
                                ExistingProvedBySimplifier,
                                ExistingProvedByChecker,
                                ExistingProvedByReview,
                                ExistingProvedFalse,
                                ExistingProvedByContradiction,
                                ExistingProvedUsingUserProofRule);
            --# end accept;

            if not RetrieveSuccess then
               FatalErrors.Process (FatalErrors.VCDataStructureInconsistency,
                                    ELStrings.EmptyString);
            end if;

            VCDetails.Order (ExistingName,
                             NewName,
                             OrderResult);

            case OrderResult is
               when EStrings.FirstOneFirst =>
                  -- next item in list is first, keep going down list
                  ListIndex := NextEntryInList;

               when EStrings.SecondOneFirst =>
                  -- new item is first, insert here
                  InsertInList (ListIndex,
                                NextEntryInList,
                                NewName,
                                PathStart,
                                PathEnd,
                                EndType,
                                ProvedByExaminer,
                                ProvedBySimplifier,
                                ProvedByChecker,
                                ProvedByReview,
                                ProvedFalse,
                                ProvedByContradiction,
                                ProvedUsingUserProofRule);
                  LoopFinished := True;

               when EStrings.NeitherFirst =>
                  -- items identical: do nothing
                  LoopFinished := True;

            end case;
         end if;
      end loop;

      LongestVCNameLength  := LongestOf (NewName.Length, LongestVCNameLength);
      LongestVCStartLength := LongestOf (PathStart.Length, LongestVCStartLength);

      LongestVCEndLength := LongestOf (PathEnd.Length +
                                       VCDetails.EndTypeImageLength,
                                       LongestVCEndLength);
      --# accept F, 33, ExistingPathStart, "ExistingPathStart not used here" &
      --#        F, 33, ExistingPathEnd,   "ExistingPathEnd not used here" &
      --#        F, 33, ExistingEndType,   "ExistingEndType not used here" &
      --#        F, 33, ExistingProvedByExaminer,         "ExistingProvedByExaminer not used here" &
      --#        F, 33, ExistingProvedBySimplifier,       "ExistingProvedBySimplifier not used here" &
      --#        F, 33, ExistingProvedByChecker,          "ExistingProvedByChecker not used here" &
      --#        F, 33, ExistingProvedByReview,           "ExistingProvedByReview not used here" &
      --#        F, 33, ExistingProvedFalse,              "ExistingProvedFalse not used here" &
      --#        F, 33, ExistingProvedByContradiction,    "ExistingProvedByContradiction not used here" &
      --#        F, 33, ExistingProvedUsingUserProofRule, "ExistingProvedUsingUserProofRule not used here";
   end Add;

   ----------------------------------------------------------------------------
   -- this procedure returns the file details for the specified entry in the
   -- linked list.
   procedure Details (ListIndex          : in     Heap.Atom;
                      VCName             :    out EStrings.T;
                      PathStart          :    out EStrings.T;
                      PathEnd            :    out EStrings.T;
                      EndType            :    out VCDetails.TerminalPointType;
                      ProvedByExaminer   :    out Boolean;
                      ProvedBySimplifier :    out Boolean;
                      ProvedByChecker    :    out Boolean;
                      ProvedByReview     :    out Boolean;
                      ProvedFalse        :    out Boolean;
                      ProvedByContradiction : out Boolean;
                      ProvedUsingUserProofRule : out Boolean)
   --# global in TheDetails;
   --#        in ThePointers;
   --# derives EndType,
   --#         PathEnd,
   --#         PathStart,
   --#         ProvedByChecker,
   --#         ProvedByContradiction,
   --#         ProvedByExaminer,
   --#         ProvedByReview,
   --#         ProvedBySimplifier,
   --#         ProvedFalse,
   --#         ProvedUsingUserProofRule,
   --#         VCName                   from ListIndex,
   --#                                       TheDetails,
   --#                                       ThePointers;
   is
      DetailsIndex : HeapIndex.IndexType;
      Dummy        : Boolean;
   begin -- Details
      -- dereference linked list pointer
      DetailsIndex := Heap.AValue (ThePointers, ListIndex);

      -- if not null pointer then follow it
      if DetailsIndex /= 0 then
         --# accept F, 10, Dummy, "Dummy not used here";
         VCDetails.Retrieve (TheDetails,
                             DetailsIndex,
                             Dummy,
                             VCName,
                             PathStart,
                             PathEnd,
                             EndType,
                             ProvedByExaminer,
                             ProvedBySimplifier,
                             ProvedByChecker,
                             ProvedByReview,
                             ProvedFalse,
                             ProvedByContradiction,
                             ProvedUsingUserProofRule);
         --# end accept;
      else
         -- if null pointer then return failure
         VCName     := EStrings.EmptyString;
         PathStart  := EStrings.EmptyString;
         PathEnd    := EStrings.EmptyString;
         EndType    := VCDetails.Undetermined_Point;
         ProvedByExaminer   := False;
         ProvedBySimplifier := False;
         ProvedByChecker    := False;
         ProvedByReview     := False;
         ProvedFalse        := False;
         ProvedByContradiction := False;
         ProvedUsingUserProofRule := False;
      end if;
      --# accept F, 33, Dummy, "Dummy not used here";
   end Details;

   --------------------------------------------------------------------------
   procedure Initialize
   --# global out LongestVCEndLength;
   --#        out LongestVCNameLength;
   --#        out LongestVCStartLength;
   --#        out StartOfPointersList;
   --#        out TheDetails;
   --#        out ThePointers;
   --#        out VCNamePrefix;
   --# derives LongestVCEndLength,
   --#         LongestVCNameLength,
   --#         LongestVCStartLength,
   --#         StartOfPointersList,
   --#         TheDetails,
   --#         ThePointers,
   --#         VCNamePrefix         from ;
   is
   begin
      Heap.Initialize (ThePointers);
      VCDetails.Initialize (TheDetails);
      StartOfPointersList := 0;

      VCNamePrefix := EStrings.EmptyString;
      LongestVCNameLength  := 0;
      LongestVCStartLength := 0;
      LongestVCEndLength   := 0;
   end Initialize;

   --------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in VCDetails.ErrorType)
   --# global in out TheDetails;
   --# derives TheDetails from *,
   --#                         ErrorKind;
   is
   begin
      VCDetails.RaiseError (ErrorKind, TheDetails);
   end RaiseError;

   --------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in VCDetails.ErrorType) return Boolean
   --# global in TheDetails;
   is
   begin
      return VCDetails.ErrorRaised (ErrorKind, TheDetails);
   end ErrorRaised;

   --------------------------------------------------------------------------
   procedure Reinitialize
     (FirstElement   : in EStrings.T;
      FirstPathStart : in EStrings.T;
      FirstPathEnd   : in EStrings.T;
      FirstEndType   : in VCDetails.TerminalPointType)
   --# global out LongestVCEndLength;
   --#        out LongestVCNameLength;
   --#        out LongestVCStartLength;
   --#        out StartOfPointersList;
   --#        out TheDetails;
   --#        out ThePointers;
   --#        out VCNamePrefix;
   --# derives LongestVCEndLength   from FirstPathEnd &
   --#         LongestVCNameLength,
   --#         VCNamePrefix         from FirstElement &
   --#         LongestVCStartLength from FirstPathStart &
   --#         StartOfPointersList,
   --#         ThePointers          from  &
   --#         TheDetails           from FirstElement,
   --#                                   FirstEndType,
   --#                                   FirstPathEnd,
   --#                                   FirstPathStart;
   is
      Dummy              : Boolean;
      FirstDetailsIndex  : HeapIndex.IndexType;
      FirstPointersIndex : Heap.Atom;
   begin -- Reinitialize
      Heap.Initialize (ThePointers);
      VCDetails.Initialize (TheDetails);

      -- insert first item
      --# accept F, 10, Dummy, "Dummy unused here";
      VCDetails.Add (TheDetails,
                     FirstDetailsIndex,
                     Dummy,
                     FirstElement,
                     FirstPathStart,
                     FirstPathEnd,
                     FirstEndType,
                     False,
                     False,
                     False,
                     False,
                     False,
                     False,
                     False);
      Heap.CreateAtom (ThePointers,
                       FirstPointersIndex,
                       Dummy);
      --# end accept;

      Heap.UpdateAValue (ThePointers, FirstPointersIndex, FirstDetailsIndex);
      Heap.UpdateAPointer (ThePointers, FirstPointersIndex, 0);

      StartOfPointersList := FirstPointersIndex;

      LongestVCNameLength  := FirstElement.Length;
      LongestVCStartLength := FirstPathStart.Length;
      LongestVCEndLength   := FirstPathEnd.Length + VCDetails.EndTypeImageLength;

      VCNamePrefix := EStrings.Section
        (FirstElement, 1, FirstElement.Length - 2);
      --# accept F, 33, Dummy, "Dummy unused here";
   end Reinitialize;

   ---------------------------------------------------------------------------
   -- this procedure returns the 'NextOne' ordered element in FH after
   -- 'AfterThis'. It is successful if the NextOne is not a 'null' pointer
   procedure Next (AfterThis : in     Heap.Atom;
                   Success   :    out Boolean;
                   NextOne   :    out Heap.Atom)
   --# global in ThePointers;
   --# derives NextOne,
   --#         Success from AfterThis,
   --#                      ThePointers;
   is
      NextInList : Heap.Atom;
   begin -- Next
      NextInList := Heap.APointer (ThePointers, AfterThis);
      if NextInList = 0 then
         Success := False;
         NextOne := 0;
      else
         Success := True;
         NextOne := NextInList;
      end if;
   end Next;



   procedure FindVCByName
     (VCName  : in     EStrings.T;
      VCIndex :    out HeapIndex.IndexType)
   --# global in     StartOfPointersList;
   --#        in     TheDetails;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --# derives FatalErrors.State from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName &
   --#         VCIndex           from StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      ListIndex    : Heap.Atom;
      Found        : Boolean;
      LoopFinished : Boolean;
      RetrieveSuccess     : Boolean;
      CurrentVCName       : EStrings.T;
      CurrentVCPathStart  : EStrings.T;
      CurrentVCPathEnd    : EStrings.T;
      CurrentVCEndType    : VCDetails.TerminalPointType;
      CurrentVCProvedByExaminer   : Boolean;
      CurrentVCProvedBySimplifier : Boolean;
      CurrentVCProvedByChecker    : Boolean;
      CurrentVCProvedByReview     : Boolean;
      CurrentVCProvedFalse        : Boolean;
      CurrentVCProvedByContradiction    : Boolean;
      CurrentVCProvedUsingUserProofRule : Boolean;
   begin
      ListIndex    := StartOfPointersList;
      Found        := False;
      LoopFinished := False;

      while not Heap.IsNullPointer (ListIndex) and not LoopFinished loop
         --# accept F, 10, CurrentVCPathStart, "CurrentVCPathStart not used here" &
         --#        F, 10, CurrentVCPathEnd,   "CurrentVCPathEnd not used here" &
         --#        F, 10, CurrentVCEndType,   "CurrentVCEndType not used here" &
         --#        F, 10, CurrentVCProvedByExaminer,         "CurrentVCProvedByExaminer not used here" &
         --#        F, 10, CurrentVCProvedBySimplifier,       "CurrentVCProvedBySimplifier not used here" &
         --#        F, 10, CurrentVCProvedByChecker,          "CurrentVCProvedByChecker not used here" &
         --#        F, 10, CurrentVCProvedByReview,           "CurrentVCProvedByReview not used here" &
         --#        F, 10, CurrentVCProvedFalse,              "CurrentVCProvedFalse not used here" &
         --#        F, 10, CurrentVCProvedByContradiction,    "CurrentVCProvedByContradiction not used here" &
         --#        F, 10, CurrentVCProvedUsingUserProofRule, "CurrentVCProvedUsingUserProofRule not used here";
         VCDetails.Retrieve (TheDetails,
                             Heap.AValue (ThePointers, ListIndex),
                             RetrieveSuccess,
                             CurrentVCName,
                             CurrentVCPathStart,
                             CurrentVCPathEnd,
                             CurrentVCEndType,
                             CurrentVCProvedByExaminer,
                             CurrentVCProvedBySimplifier,
                             CurrentVCProvedByChecker,
                             CurrentVCProvedByReview,
                             CurrentVCProvedFalse,
                             CurrentVCProvedByContradiction,
                             CurrentVCProvedUsingUserProofRule);
         --# end accept;
         if not RetrieveSuccess then
            FatalErrors.Process (FatalErrors.VCDataStructureInconsistency, ELStrings.EmptyString);
         end if;

         if EStrings.EqString (VCName, CurrentVCName) then
            Found        := True;
            LoopFinished := True;
         else
            ListIndex := Heap.APointer (ThePointers, ListIndex);
         end if;
      end loop;

      if Found then
         VCIndex := Heap.AValue (ThePointers, ListIndex);
      else
         VCIndex := 0;
      end if;
      --# accept F, 33, CurrentVCPathStart, "CurrentVCPathStart not used here" &
      --#        F, 33, CurrentVCPathEnd,   "CurrentVCPathEnd not used here" &
      --#        F, 33, CurrentVCEndType,   "CurrentVCEndType not used here" &
      --#        F, 33, CurrentVCProvedByExaminer,          "CurrentVCProvedByExaminer not used here" &
      --#        F, 33, CurrentVCProvedBySimplifier,       "CurrentVCProvedBySimplifier not used here" &
      --#        F, 33, CurrentVCProvedByChecker,          "CurrentVCProvedByChecker not used here" &
      --#        F, 33, CurrentVCProvedByReview,           "CurrentVCProvedByReview not used here" &
      --#        F, 33, CurrentVCProvedFalse,              "CurrentVCProvedFalse not used here" &
      --#        F, 33, CurrentVCProvedByContradiction,    "CurrentVCProvedByContradiction not used here" &
      --#        F, 33, CurrentVCProvedUsingUserProofRule, "CurrentVCProvedUsingUserProofRule not used here";
   end FindVCByName;


   ---------------------------------------------------------------------------
   procedure MarkAsProvedByExaminer (VCName : in EStrings.T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State,
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      if DetailsIndex /= 0 then
         VCDetails.MarkAsProvedByExaminer (TheDetails, DetailsIndex);
      end if;
   end MarkAsProvedByExaminer;

   ---------------------------------------------------------------------------
   procedure MarkAsProvedBySimplifier (VCName : in EStrings.T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State,
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      if DetailsIndex /= 0 then
         VCDetails.MarkAsProvedBySimplifier (TheDetails, DetailsIndex);
      end if;
   end MarkAsProvedBySimplifier;

   ---------------------------------------------------------------------------
   procedure MarkAsProvedByChecker (VCName : in EStrings.T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State,
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      if DetailsIndex /= 0 then
         VCDetails.MarkAsProvedByChecker (TheDetails, DetailsIndex);
      end if;
   end MarkAsProvedByChecker;

   ---------------------------------------------------------------------------
   procedure MarkAsProvedByReview (VCName : in EStrings.T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State,
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      if DetailsIndex /= 0 then
         VCDetails.MarkAsProvedByReview (TheDetails, DetailsIndex);
      end if;
   end MarkAsProvedByReview;

   ---------------------------------------------------------------------------
   procedure MarkAsProvedFalse (VCName : in EStrings.T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State,
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      if DetailsIndex /= 0 then
         VCDetails.MarkAsProvedFalse (TheDetails, DetailsIndex);
      end if;
   end MarkAsProvedFalse;

   ---------------------------------------------------------------------------
   procedure MarkAsProvedByContradiction (VCName : in EStrings.T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State,
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      if DetailsIndex /= 0 then
         VCDetails.MarkAsProvedByContradiction (TheDetails, DetailsIndex);
      end if;
   end MarkAsProvedByContradiction;

   ---------------------------------------------------------------------------
   procedure MarkAsProvedUsingUserProofRule (VCName : in EStrings.T)
   --# global in     StartOfPointersList;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --#        in out TheDetails;
   --# derives FatalErrors.State,
   --#         TheDetails        from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      if DetailsIndex /= 0 then
         VCDetails.MarkAsProvedUsingUserProofRule (TheDetails, DetailsIndex);
      end if;
   end MarkAsProvedUsingUserProofRule;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByExaminer (VCName : in EStrings.T)
      return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide IsMarkedAsProvedByExaminer;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      return (DetailsIndex /= 0) and then
              VCDetails.IsMarkedAsProvedByExaminer (TheDetails, DetailsIndex);
   end IsMarkedAsProvedByExaminer;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedBySimplifier (VCName : in EStrings.T)
      return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide IsMarkedAsProvedBySimplifier;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      return (DetailsIndex /= 0) and then
              VCDetails.IsMarkedAsProvedBySimplifier (TheDetails, DetailsIndex);
   end IsMarkedAsProvedBySimplifier;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByChecker (VCName : in EStrings.T)
      return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide IsMarkedAsProvedByChecker;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      return (DetailsIndex /= 0) and then
              VCDetails.IsMarkedAsProvedByChecker (TheDetails, DetailsIndex);
   end IsMarkedAsProvedByChecker;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByReview (VCName : in EStrings.T)
      return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide IsMarkedAsProvedByReview;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      return (DetailsIndex /= 0) and then
              VCDetails.IsMarkedAsProvedByReview (TheDetails, DetailsIndex);
   end IsMarkedAsProvedByReview;

      --------------------------------------------------------------------------
   function IsMarkedAsProvedFalse (VCName : in EStrings.T)
      return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide IsMarkedAsProvedFalse;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      return (DetailsIndex /= 0) and then
              VCDetails.IsMarkedAsProvedFalse (TheDetails, DetailsIndex);
   end IsMarkedAsProvedFalse;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByContradiction (VCName : in EStrings.T)
      return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide IsMarkedAsProvedByContradiction;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      return (DetailsIndex /= 0) and then
              VCDetails.IsMarkedAsProvedByContradiction (TheDetails, DetailsIndex);
   end IsMarkedAsProvedByContradiction;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedUsingUserProofRule (VCName : in EStrings.T)
      return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide IsMarkedAsProvedUsingUserProofRule;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);

      return (DetailsIndex /= 0) and then
              VCDetails.IsMarkedAsProvedUsingUserProofRule (TheDetails, DetailsIndex);
   end IsMarkedAsProvedUsingUserProofRule;

   ---------------------------------------------------------------------------
   function Exists (VCName : EStrings.T) return Boolean
   --# global in StartOfPointersList;
   --#        in TheDetails;
   --#        in ThePointers;
   is
      -- Hide this function to hide the (unfortunate and downright
      -- annoying) side-effect that FindVCByName can have on FatalErrors.State

      --# hide Exists;
      DetailsIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, DetailsIndex);
      return (DetailsIndex /= 0);
   end Exists;

   ---------------------------------------------------------------------------
   procedure GetVCNameEndType
     (VCName : in     EStrings.T;
      VCType :    out VCDetails.TerminalPointType)
   --# global in     StartOfPointersList;
   --#        in     TheDetails;
   --#        in     ThePointers;
   --#        in out FatalErrors.State;
   --# derives FatalErrors.State from *,
   --#                                StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName &
   --#         VCType            from StartOfPointersList,
   --#                                TheDetails,
   --#                                ThePointers,
   --#                                VCName;
   is
      VCIndex : HeapIndex.IndexType;
   begin
      FindVCByName (VCName, VCIndex);
      VCType := VCDetails.EndPointType (TheDetails, VCIndex);
   end GetVCNameEndType;

   ---------------------------------------------------------------------------
   function GetLongestVCNameLength return Integer
   --# global in LongestVCNameLength;
   is
   begin
      return LongestVCNameLength;
   end GetLongestVCNameLength;

   ---------------------------------------------------------------------------
   function GetLongestVCStartLength return Integer
   --# global in LongestVCStartLength;
   is
   begin
      return LongestVCStartLength;
   end GetLongestVCStartLength;

   ---------------------------------------------------------------------------
   function GetLongestVCEndLength return Integer
   --# global in LongestVCEndLength;
   is
   begin
      return LongestVCEndLength;
   end GetLongestVCEndLength;


   --------------------------------------------------------------------------
   function GetVCNamePrefix return EStrings.T
   --# global in VCNamePrefix;
   is
   begin
      return VCNamePrefix;
   end GetVCNamePrefix;

end VCHeap;
