-- $Id: vcdetails.ads 11367 2008-10-07 15:47:17Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package providing a structure in which to store VC details.                 --
--                                                                            --
--To be used in tandem with the Heap data structure, hence the use of         --
--Heap.Atom as the array range                                                --
--------------------------------------------------------------------------------
with EStrings;
with HeapIndex;

use type HeapIndex.IndexType;

--# inherit EStrings,
--#         HeapIndex;

package VCDetails
is
   type TerminalPointType is (Assert_Point,
                              Precondition_Check_Point,
                              Check_Statement_Point,
                              Runtime_Check_Point,
                              Refinement_VC_Point,
                              Inheritance_VC_Point,
                              Undetermined_Point);

   type ErrorType is (MissingSLGFile, CorruptFile);

   EndTypeImageLength : constant := 12;
   subtype EndTypeImageIndex is Positive range 1 .. EndTypeImageLength;
   subtype EndTypeImageString is String (EndTypeImageIndex);
   type EndTypeToImageArray is array (TerminalPointType) of EndTypeImageString;

   EndTypeImage : constant EndTypeToImageArray :=
     EndTypeToImageArray'
        (Assert_Point             => "   assert @ ",
         Precondition_Check_Point => "pre check @ ",
         Check_Statement_Point    => "check stm @ ",
         Runtime_Check_Point      => "rtc check @ ",
         Refinement_VC_Point      => "refinement  ",
         Inheritance_VC_Point     => "inheritance ",
         Undetermined_Point       => "            ");


   type DataType is private;

   --------------------------------------------------------------------------
   procedure Add (Details                      : in out DataType;
                  Index                        :    out HeapIndex.IndexType;
                  Success                      :    out Boolean;
                  Name                         : in     EStrings.T;
                  Path_Start                   : in     EStrings.T;
                  Path_End                     : in     EStrings.T;
                  End_Type                     : in     TerminalPointType;
                  ProvedByExaminer             : in     Boolean;
                  ProvedBySimplifier           : in     Boolean;
                  ProvedByChecker              : in     Boolean;
                  ProvedByReview               : in     Boolean;
                  ProvedFalse                  : in     Boolean;
                  ProvedByContradiction        : in     Boolean;
                  ProvedUsingUserProofRule : in     Boolean);
   --# derives Details from *,
   --#                      End_Type,
   --#                      Name,
   --#                      Path_End,
   --#                      Path_Start,
   --#                      ProvedByChecker,
   --#                      ProvedByContradiction,
   --#                      ProvedByExaminer,
   --#                      ProvedByReview,
   --#                      ProvedBySimplifier,
   --#                      ProvedFalse,
   --#                      ProvedUsingUserProofRule &
   --#         Index,
   --#         Success from Details;

   --------------------------------------------------------------------------
   procedure Initialize (Details : out DataType);
   --# derives Details from ;

   --------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in ErrorType;
                         Details   : in out DataType);
   --# derives Details from *,
   --#                      ErrorKind;

   --------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in ErrorType;
                         Details   : in DataType) return Boolean;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByExaminer (Details : in out DataType;
                                     Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   procedure MarkAsProvedBySimplifier (Details : in out DataType;
                                       Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByChecker (Details : in out DataType;
                                    Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByReview (Details : in out DataType;
                                   Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   procedure MarkAsProvedFalse (Details : in out DataType;
                                Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   procedure MarkAsProvedByContradiction (Details : in out DataType;
                                          Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   procedure MarkAsProvedUsingUserProofRule (Details : in out DataType;
                                                 Index   : in     HeapIndex.IndexType);
   --# derives Details from *,
   --#                      Index;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByExaminer (Details : DataType;
                                        Index   : HeapIndex.IndexType)
                                       return Boolean;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedBySimplifier (Details : DataType;
                                          Index   : HeapIndex.IndexType)
                                         return Boolean;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByChecker (Details : DataType;
                                       Index   : HeapIndex.IndexType)
      return Boolean;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByReview (Details : DataType;
                                      Index   : HeapIndex.IndexType)
                                      return Boolean;
   --------------------------------------------------------------------------
   function IsMarkedAsProvedFalse (Details : DataType;
                                   Index   : HeapIndex.IndexType)
                                   return Boolean;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedByContradiction (Details : DataType;
                                             Index   : HeapIndex.IndexType)
      return Boolean;

   --------------------------------------------------------------------------
   function IsMarkedAsProvedUsingUserProofRule (Details : DataType;
                                                    Index   : HeapIndex.IndexType)
      return Boolean;

   --------------------------------------------------------------------------
   -- this compares the information given and returns
   -- Result is as defined in section 5.4 of the specification
   -- it works directly on values rather than on indices into the Details
   -- structure so that information can be compared before insertion
   -- NOTE : the procedure is successful iff neither Type is Invalid
   --
   procedure Order (FirstName  : in     EStrings.T;
                    SecondName : in     EStrings.T;
                    Result     :    out EStrings.OrderTypes);
   --# derives Result from FirstName,
   --#                     SecondName;

   --------------------------------------------------------------------------
   procedure Retrieve (Details            : in     DataType;
                       Index              : in     HeapIndex.IndexType;
                       Success            :    out Boolean;
                       Name               :    out EStrings.T;
                       PathStart          :    out EStrings.T;
                       PathEnd            :    out EStrings.T;
                       EndType            :    out TerminalPointType;
                       ProvedByExaminer   :    out Boolean;
                       ProvedBySimplifier :    out Boolean;
                       ProvedByChecker    :    out Boolean;
                       ProvedByReview     :    out Boolean;
                       ProvedFalse        :    out Boolean;
                       ProvedByContradiction : out Boolean;
                       ProvedUsingUserProofRule : out Boolean);
   --# derives EndType,
   --#         Name,
   --#         PathEnd,
   --#         PathStart,
   --#         ProvedByChecker,
   --#         ProvedByContradiction,
   --#         ProvedByExaminer,
   --#         ProvedByReview,
   --#         ProvedBySimplifier,
   --#         ProvedFalse,
   --#         ProvedUsingUserProofRule,
   --#         Success                  from Details,
   --#                                       Index;

   --------------------------------------------------------------------------
   -- PathEndToPathType
   --
   -- Parses a Path End string to convert it into a Terminal Point Type
   -- The basic pattersn scanned for are:
   --   ... to check ...              -> check statement point
   --   ... to run-time check ...     -> run-time check point
   --   ... to precondition check ... -> precondition check point
   --   ... to assertion ...          -> assertion point
   --   ... refinement ...            -> refinement VC point
   --   ... inheritance  ...          -> inheritance VC point
   --   ... anything else ...         -> undetermined point
   --------------------------------------------------------------------------
   function PathEndToPathType
     (Line : EStrings.T)
     return TerminalPointType;

   function EndPointType
     (Details : in DataType;
      Index   : in HeapIndex.IndexType)
     return TerminalPointType;


private
   type DetailsElement is record
      Name                : EStrings.T;
      PathStart           : EStrings.T;
      PathEnd             : EStrings.T;
      EndType             : TerminalPointType;
      ProvedByExaminer    : Boolean;
      ProvedBySimplifier  : Boolean;
      ProvedByChecker     : Boolean;
      ProvedByReview      : Boolean;
      ProvedFalse         : Boolean;
      ProvedByContradiction         : Boolean;
      ProvedUsingUserProofRule  : Boolean;
   end record;

   NullDetailsElement : constant DetailsElement :=
     DetailsElement'(Name               => EStrings.EmptyString,
                     PathStart          => EStrings.EmptyString,
                     PathEnd            => EStrings.EmptyString,
                     EndType            => Undetermined_Point,
                     ProvedByExaminer   => False,
                     ProvedBySimplifier => False,
                     ProvedByChecker    => False,
                     ProvedByReview     => False,
                     ProvedFalse        => False,
                     ProvedByContradiction     => False,
                     ProvedUsingUserProofRule  => False);

   type ElementsArray is array (HeapIndex.IndexType) of DetailsElement;

   -- Use an array to store the different error kinds that may be associated
   -- with a subprogram. This construct may be extended by adding a new enumerated
   -- type, without having to extend the subprogram interfaces.
   type ErrorArray  is array (ErrorType) of Boolean;

   -- HighMark is the number of the highest allocated atom - zero if the
   -- structure is empty
   -- ErrorStatus records different kinds of errors that may be encountered
   -- during analysis.
   type DataType is record
      Details          : ElementsArray;
      HighMark         : HeapIndex.IndexType;
      ErrorStatus      : ErrorArray;
   end record;

end VCDetails;
