-- $Id: vcdetails.adb 11367 2008-10-07 15:47:17Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package providing a structure in which to store VC details.                 --
--                                                                            --
--To be used in tandem with the Heap data structure, hence the use of         --
--Heap.Atom as the array range                                                --
--------------------------------------------------------------------------------
package body VCDetails
is

   ---------------------------------------------------------------------------
   procedure Add (Details                      : in out DataType;
                  Index                        :    out HeapIndex.IndexType;
                  Success                      :    out Boolean;
                  Name                         : in     EStrings.T;
                  Path_Start                   : in     EStrings.T;
                  Path_End                     : in     EStrings.T;
                  End_Type                     : in     TerminalPointType;
                  ProvedByExaminer             : in     Boolean;
                  ProvedBySimplifier           : in     Boolean;
                  ProvedByChecker              : in     Boolean;
                  ProvedByReview               : in     Boolean;
                  ProvedFalse                  : in     Boolean;
                  ProvedByContradiction        : in     Boolean;
                  ProvedUsingUserProofRule     : in     Boolean)
   is
   begin
      if Details.HighMark < HeapIndex.IndexType'Last then
         Success := True;
         Details.HighMark := Details.HighMark + 1;
         Index := Details.HighMark;
         Details.Details (Details.HighMark) := DetailsElement'(Name,
                                                               Path_Start,
                                                               Path_End,
                                                               End_Type,
                                                               ProvedByExaminer,
                                                               ProvedBySimplifier,
                                                               ProvedByChecker,
                                                               ProvedByReview,
                                                               ProvedFalse,
                                                               ProvedByContradiction,
                                                               ProvedUsingUserProofRule);
      else
         Success := False;
         Index := 0;
      end if;
   end Add;

   --------------------------------------------------------------------------
   procedure Initialize (Details : out DataType)
   is
   begin
      -- Only set HighMark here.  Initializing the whole array is
      -- unnecessary here, and is VERY SLOW on machines with limited
      -- RAM likes VAXes, where the initialization causes massive
      -- VM thrashing.
      Details.HighMark := 0;
      -- Also set that no error has been seen.
      Details.ErrorStatus := ErrorArray'(others => False);

      --# accept F, 31,           Details.Details, "Partial initialization" &
      --#        F, 32,           Details.Details, "Partial initialization" &
      --#        F, 602, Details, Details.Details, "Partial initialization";
   end Initialize;

   -------------------------------------------------------------------------
   procedure RaiseError (ErrorKind : in ErrorType;
                         Details   : in out DataType)
   is
   begin
      Details.ErrorStatus (ErrorKind) := True;
   end RaiseError;

   -------------------------------------------------------------------------
   function ErrorRaised (ErrorKind : in ErrorType;
                         Details   : in DataType) return Boolean
   is
   begin
      return Details.ErrorStatus (ErrorKind);
   end ErrorRaised;

   -------------------------------------------------------------------------
   procedure MarkAsProvedByExaminer (Details : in out DataType;
                                     Index   : in     HeapIndex.IndexType)
   is
   begin
      Details.Details (Index).ProvedByExaminer := True;
   end MarkAsProvedByExaminer;

   -------------------------------------------------------------------------
   procedure MarkAsProvedBySimplifier (Details : in out DataType;
                                       Index   : in     HeapIndex.IndexType)
   is
   begin
      -- A True VC in a SIV file might have actually been already
      -- proven (trivially) by the Examiner, so in that case, we
      -- don't mark it as having been proven by the Simplifier as well...
      if not Details.Details (Index).ProvedByExaminer then
         -- The Simplifier CAN prove a VC that has a False conclusion
         -- by reducing one of the Hypotheses to False, so this takes
         -- precedence over ProvedFalse
         Details.Details (Index).ProvedBySimplifier := True;
         Details.Details (Index).ProvedFalse        := False;
      end if;
   end MarkAsProvedBySimplifier;

   -------------------------------------------------------------------------
   procedure MarkAsProvedByChecker (Details : in out DataType;
                                    Index   : in     HeapIndex.IndexType)
   is
   begin
      -- The Checker only ever offers the possibility of proving
      -- VCs that have definitely not been proven by the Examiner
      -- or Simplifier, so no need to defend this assignment...

      -- The Checker CAN prove a VC that has a False conclusion
      -- by reducing one of the Hypotheses to False, so this takes
      -- precedence over ProvedFalse
      Details.Details (Index).ProvedByChecker := True;
      Details.Details (Index).ProvedFalse     := False;
   end MarkAsProvedByChecker;

   -------------------------------------------------------------------------
   procedure MarkAsProvedByReview (Details : in out DataType;
                                   Index   : in     HeapIndex.IndexType)
   is
   begin
      -- The Review file only ever offers the possibility of proving
      -- VCs that have definitely not been proven by the Examiner
      -- or Simplifier, so no need to defend this assignment...

      -- A PRV file CAN prove a VC that has a False conclusion
      -- by reducing one of the Hypotheses to False, so this takes
      -- precedence over ProvedFalse
      Details.Details (Index).ProvedByReview := True;
      Details.Details (Index).ProvedFalse    := False;
   end MarkAsProvedByReview;

   -------------------------------------------------------------------------
   procedure MarkAsProvedFalse (Details : in out DataType;
                                Index   : in     HeapIndex.IndexType)
   is
   begin
      -- If proved false there is no danger of ticks in two different columns
      -- so no need to defend this assignment...
      Details.Details (Index).ProvedFalse := True;
   end MarkAsProvedFalse;

   -------------------------------------------------------------------------
   procedure MarkAsProvedByContradiction (Details : in out DataType;
                                          Index   : in     HeapIndex.IndexType)
   is
   begin
      -- Proved by contradiction takes precedence over a VC that has
      -- a False conclucion.  This is possible, since a VC might have
      -- a False Conclusion in the VCG file but contradictory
      -- Hypotheses that will be detected in the SIV file.
      Details.Details (Index).ProvedByContradiction := True;
      Details.Details (Index).ProvedFalse           := False;
   end MarkAsProvedByContradiction;

   -------------------------------------------------------------------------
   procedure MarkAsProvedUsingUserProofRule (Details : in out DataType;
                                             Index   : in     HeapIndex.IndexType)
   is
   begin
      Details.Details (Index).ProvedUsingUserProofRule := True;
   end MarkAsProvedUsingUserProofRule;

   function IsMarkedAsProvedByExaminer (Details : DataType;
                                        Index   : HeapIndex.IndexType)
                                       return Boolean
   is
   begin
      return Details.Details (Index).ProvedByExaminer;
   end IsMarkedAsProvedByExaminer;


   function IsMarkedAsProvedBySimplifier (Details : DataType;
                                          Index   : HeapIndex.IndexType)
                                         return Boolean
   is
   begin
      return Details.Details (Index).ProvedBySimplifier;
   end IsMarkedAsProvedBySimplifier;


   function IsMarkedAsProvedByChecker (Details : DataType;
                                       Index   : HeapIndex.IndexType)
      return Boolean
   is
   begin
      return Details.Details (Index).ProvedByChecker;
   end IsMarkedAsProvedByChecker;


   function IsMarkedAsProvedByReview (Details : DataType;
                                      Index   : HeapIndex.IndexType)
      return Boolean
   is
   begin
      return Details.Details (Index).ProvedByReview;
   end IsMarkedAsProvedByReview;


   function IsMarkedAsProvedFalse (Details : DataType;
                                   Index   : HeapIndex.IndexType)
      return Boolean
   is
   begin
      return Details.Details (Index).ProvedFalse;
   end IsMarkedAsProvedFalse;

   function IsMarkedAsProvedByContradiction (Details : DataType;
                                             Index   : HeapIndex.IndexType)
      return Boolean
   is
   begin
      return Details.Details (Index).ProvedByContradiction;
   end IsMarkedAsProvedByContradiction;

   function IsMarkedAsProvedUsingUserProofRule (Details : DataType;
                                                    Index   : HeapIndex.IndexType)
      return Boolean
   is
   begin
      return Details.Details (Index).ProvedUsingUserProofRule;
   end IsMarkedAsProvedUsingUserProofRule;


   -------------------------------------------------------------------------
   procedure Order (FirstName  : in     EStrings.T;
                    SecondName : in     EStrings.T;
                    Result     :    out EStrings.OrderTypes)
   is
   begin -- Order
      -- check which name comes first
      if FirstName.Length = SecondName.Length then
         Result := EStrings.LexOrder (FirstName, SecondName);

      elsif FirstName.Length < SecondName.Length then
         Result := EStrings.FirstOneFirst;

      else
         Result := EStrings.SecondOneFirst;
      end if;
   end Order;

   --------------------------------------------------------------------------
   procedure Retrieve (Details            : in     DataType;
                       Index              : in     HeapIndex.IndexType;
                       Success            :    out Boolean;
                       Name               :    out EStrings.T;
                       PathStart          :    out EStrings.T;
                       PathEnd            :    out EStrings.T;
                       EndType            :    out TerminalPointType;
                       ProvedByExaminer   :    out Boolean;
                       ProvedBySimplifier :    out Boolean;
                       ProvedByChecker    :    out Boolean;
                       ProvedByReview     :    out Boolean;
                       ProvedFalse        :    out Boolean;
                       ProvedByContradiction    : out Boolean;
                       ProvedUsingUserProofRule : out Boolean)
   is
   begin
      if Index <= Details.HighMark and Index /= 0 then
         Success    := True;
         Name       := Details.Details (Index).Name;
         PathStart  := Details.Details (Index).PathStart;
         PathEnd    := Details.Details (Index).PathEnd;
         EndType    := Details.Details (Index).EndType;
         ProvedByExaminer   := Details.Details (Index).ProvedByExaminer;
         ProvedBySimplifier := Details.Details (Index).ProvedBySimplifier;
         ProvedByChecker    := Details.Details (Index).ProvedByChecker;
         ProvedByReview     := Details.Details (Index).ProvedByReview;
         ProvedFalse        := Details.Details (Index).ProvedFalse;
         ProvedByContradiction := Details.Details (Index).ProvedByContradiction;
         ProvedUsingUserProofRule := Details.Details (Index).ProvedUsingUserProofRule;
      else
         Success    := False;
         Name       := EStrings.EmptyString;
         PathStart  := EStrings.EmptyString;
         PathEnd    := EStrings.EmptyString;
         EndType    := Undetermined_Point;
         ProvedByExaminer   := False;
         ProvedBySimplifier := False;
         ProvedByChecker    := False;
         ProvedByReview     := False;
         ProvedFalse        := False;
         ProvedByContradiction    := False;
         ProvedUsingUserProofRule := False;
      end if;
   end Retrieve;

   --------------------------------------------------------------------------
   function PathEndToPathType (Line : EStrings.T)
                              return TerminalPointType
   is
      DummyPosition    : EStrings.Positions;
      EndPosition      : EStrings.Positions;
      PointType        : TerminalPointType;
      RefinementFound  : Boolean;
      InheritanceFound : Boolean;
      ToFound          : Boolean;
      CheckFound       : Boolean;
      AssertFound      : Boolean;
      FinishFound      : Boolean;

      RuntimeCheckFound        : Boolean;
      PreconditionCheckFound   : Boolean;
   begin -- ExtractLastLinePointType

      --# accept F, 10, DummyPosition, "DummyPosition unused here";
      EStrings.FindSubString (Line,
                                     "inheritance",
                                     InheritanceFound,
                                     DummyPosition);

      EStrings.FindSubString (Line,
                                     "refinement",
                                     RefinementFound,
                                     DummyPosition);

      --# accept F, 10, ToFound, "ToFound unused here";
      EStrings.FindSubString (Line,
                                     " to ",
                                     ToFound,
                                     EndPosition);
      --# end accept;

      EStrings.FindSubStringAfter (Line,
                                          EndPosition,
                                          "check",
                                          CheckFound,
                                          DummyPosition);

      if InheritanceFound then

         PointType := Inheritance_VC_Point;

      elsif RefinementFound then

         PointType := Refinement_VC_Point;

      elsif CheckFound then
         EStrings.FindSubStringAfter (Line,
                                             EndPosition,
                                             "precondition",
                                             PreconditionCheckFound,
                                             DummyPosition);

         EStrings.FindSubStringAfter (Line,
                                             EndPosition,
                                             "run-time",
                                             RuntimeCheckFound,
                                             DummyPosition);

         if PreconditionCheckFound then
            PointType := Precondition_Check_Point;
         elsif RuntimeCheckFound then
            PointType := Runtime_Check_Point;
         else
            PointType := Check_Statement_Point;
         end if;

      else
         EStrings.FindSubStringAfter (Line,
                                             EndPosition,
                                             "assert",
                                             AssertFound,
                                             DummyPosition);

         EStrings.FindSubStringAfter (Line,
                                             EndPosition,
                                             "finish",
                                             FinishFound,
                                             DummyPosition);

         if AssertFound or FinishFound then
            PointType := Assert_Point;
         else
            PointType := Undetermined_Point;
         end if;
      end if;
      --# end accept;

      --# accept F, 33, DummyPosition,  "DummyPosition unused here" &
      --#        F, 33, ToFound,        "ToFound unused here";
      return PointType;
   end PathEndToPathType;


   --------------------------------------------------------------------------
   function EndPointType
     (Details : in DataType;
      Index   : in HeapIndex.IndexType)
     return TerminalPointType
   is
      Result : TerminalPointType;
   begin
      if Index <= Details.HighMark and Index /= 0 then

         Result := Details.Details (Index).EndType;
      else
         Result := Undetermined_Point;
      end if;

      return Result;
   end EndPointType;


end VCDetails;
