-- $Id: spark_io.ads 11367 2008-10-07 15:47:17Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


package SPARK_IO
--# own FILE_SYS;
--# initializes FILE_SYS;
is
   type File_Type is private;
   type File_Mode is (In_File, Out_File, Append_File);
   type File_Status is (Ok, Status_Error, Mode_Error, Name_Error, Use_Error,
                        Device_Error, End_Error,  Data_Error, Layout_Error);

   subtype Number_Base is Integer range 2 .. 16;

   function Standard_Input return File_Type;
   --# global in FILE_SYS;

   function Standard_Output return File_Type;
   --# global in FILE_SYS;

   Null_File : constant File_Type;

   -- File Management
   procedure Create (File         : in out File_Type;
                     Name_Length  : in     Natural;
                     Name_Of_File : in     String;
                     Form_Of_File : in     String;
                     Status       :    out File_Status);
   --# global in out FILE_SYS;
   --# derives FILE,
   --#         FILE_SYS,
   --#         STATUS   from FILE,
   --#                       FILE_SYS,
   --#                       FORM_OF_FILE,
   --#                       NAME_LENGTH,
   --#                       NAME_OF_FILE;

   procedure Open (File         : in out File_Type;
                   Mode_Of_File : in     File_Mode;
                   Name_Length  : in     Natural;
                   Name_Of_File : in     String;
                   Form_Of_File : in     String;
                   Status       :    out File_Status);
   --# global in out FILE_SYS;
   --# derives FILE,
   --#         FILE_SYS,
   --#         STATUS   from FILE,
   --#                       FILE_SYS,
   --#                       FORM_OF_FILE,
   --#                       MODE_OF_FILE,
   --#                       NAME_LENGTH,
   --#                       NAME_OF_FILE;

   procedure Close (File   : in out File_Type;
                    Status :    out File_Status);
   --# global in out FILE_SYS;
   --# derives FILE,
   --#         FILE_SYS from *,
   --#                       FILE &
   --#         STATUS   from FILE,
   --#                       FILE_SYS;

   procedure Delete (File   : in out File_Type;
                     Status :    out File_Status);
   --# global in out FILE_SYS;
   --# derives FILE,
   --#         FILE_SYS from *,
   --#                       FILE &
   --#         STATUS   from FILE,
   --#                       FILE_SYS;

   procedure Reset (File         : in out File_Type;
                    Mode_Of_File : in     File_Mode;
                    Status       :    out File_Status);
   --# global in out FILE_SYS;
   --# derives FILE,
   --#         FILE_SYS from *,
   --#                       FILE,
   --#                       MODE_OF_FILE &
   --#         STATUS   from FILE,
   --#                       FILE_SYS,
   --#                       MODE_OF_FILE;

   function Valid_File (File : in File_Type) return Boolean;
   --# global in FILE_SYS;

   function Mode (File : in File_Type) return File_Mode;
   --# global in FILE_SYS;

   procedure Name (File         : in     File_Type;
                   Name_Of_File :    out String;
                   Stop         :    out Natural);
   --# global in FILE_SYS;
   --# derives NAME_OF_FILE,
   --#         STOP         from FILE,
   --#                           FILE_SYS;

   procedure Form (File         : in     File_Type;
                   Form_Of_File :    out String;
                   Stop         :    out Natural);
   --# global in FILE_SYS;
   --# derives FORM_OF_FILE,
   --#         STOP         from FILE,
   --#                           FILE_SYS;

   function Is_Open (File : in File_Type) return Boolean;
   --# global in FILE_SYS;


   -- Control of default input and output files

   --
   -- Not supported in SPARK_IO
   --

   -- Specification of line and page lengths

   --
   -- Not supported in SPARK_IO
   --


   -- Column, Line and Page Control

   procedure New_Line (File    : in File_Type;
                       Spacing : in Positive);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       FILE,
   --#                       SPACING;

   procedure Skip_Line (File    : in File_Type;
                        Spacing : in Positive);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       FILE,
   --#                       SPACING;

   procedure New_Page (File : in File_Type);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       FILE;

   function End_Of_Line (File : in File_Type) return Boolean;
   --# global in FILE_SYS;

   function End_Of_File (File : in File_Type) return Boolean;
   --# global in FILE_SYS;

   procedure Set_Col (File : in File_Type;
                      Posn : in Positive);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       FILE,
   --#                       POSN;

   function Col (File : in File_Type) return Positive;
   --# global in FILE_SYS;

   function Line (File : in File_Type) return Positive;
   --# global in FILE_SYS;

   -- Character Input-Output

   procedure Get_Char (File : in     File_Type;
                       Item :    out Character);
   --# global in out FILE_SYS;
   --# derives FILE_SYS,
   --#         ITEM     from FILE,
   --#                       FILE_SYS;

   procedure Put_Char (File : in File_Type;
                       Item : in Character);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       FILE,
   --#                       ITEM;


   -- String Input-Output

   procedure Get_String (File : in     File_Type;
                         Item :    out String;
                         Stop :    out Natural);
   --# global in out FILE_SYS;
   --# derives FILE_SYS,
   --#         ITEM,
   --#         STOP     from FILE,
   --#                       FILE_SYS;

   procedure Put_String (File : in File_Type;
                         Item : in String;
                         Stop : in Natural);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       FILE,
   --#                       ITEM,
   --#                       STOP;

   procedure Get_Line (File : in     File_Type;
                       Item :    out String;
                       Stop :    out Natural);
   --# global in out FILE_SYS;
   --# derives FILE_SYS,
   --#         ITEM,
   --#         STOP     from FILE,
   --#                       FILE_SYS;

   procedure Put_Line (File : in File_Type;
                       Item : in String;
                       Stop : in Natural);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       FILE,
   --#                       ITEM,
   --#                       STOP;


   -- INTEGER Input-Output

   -- SPARK_IO only supports input-output of
   -- the built-in integer type INTEGER

   procedure Get_Integer (File  : in     File_Type;
                          Item  :    out Integer;
                          Width : in     Natural;
                          Read  :    out Boolean);
   --# global in out FILE_SYS;
   --# derives FILE_SYS,
   --#         ITEM,
   --#         READ     from FILE,
   --#                       FILE_SYS,
   --#                       WIDTH;

   procedure Put_Integer (File  : in File_Type;
                          Item  : in Integer;
                          Width : in Natural;
                          Base  : in Number_Base);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       BASE,
   --#                       FILE,
   --#                       ITEM,
   --#                       WIDTH;

   procedure Get_Int_From_String (Source  : in     String;
                                  Item    :    out Integer;
                                  Startpt : in     Positive;
                                  Stop    :    out Natural);
   --# derives ITEM,
   --#         STOP from Source,
   --#                   STARTpt;

   procedure Put_Int_To_String (Dest    :    out String;
                                Item    : in     Integer;
                                Startpt : in     Positive;
                                Base    : in     Number_Base);
   --# derives DEST from BASE,
   --#                   ITEM,
   --#                   STARTpt;


   -- FLOAT Input-Output

   -- SPARK_IO only supports input-output of
   -- the built-in real type FLOAT

   procedure Get_Float (File  : in     File_Type;
                        Item  :    out Float;
                        Width : in     Natural;
                        Read  :    out Boolean);
   --# global in out FILE_SYS;
   --# derives FILE_SYS,
   --#         ITEM,
   --#         READ     from FILE,
   --#                       FILE_SYS,
   --#                       WIDTH;

   procedure Put_Float (File : in File_Type;
                        Item : in Float;
                        Fore : in Natural;
                        Aft  : in Natural;
                        Exp  : in Natural);
   --# global in out FILE_SYS;
   --# derives FILE_SYS from *,
   --#                       AFT,
   --#                       EXP,
   --#                       FILE,
   --#                       FORE,
   --#                       ITEM;

   procedure Get_Float_From_String (Source  : in     String;
                                    Item    :    out Float;
                                    Startpt : in     Positive;
                                    Stop    :    out Natural);
   --# derives ITEM,
   --#         STOP from Source,
   --#                   STARTpt;

   procedure Put_Float_To_String (Dest    : in out String;
                                  Item    : in     Float;
                                  Startpt : in     Positive;
                                  Aft     : in     Natural;
                                  Exp     : in     Natural);
   --# derives DEST from *,
   --#                   AFT,
   --#                   EXP,
   --#                   ITEM,
   --#                   STARTpt;


   pragma Inline (Valid_File, End_Of_Line, End_Of_File, Get_Char);

private
   --# hide SPARK_IO
   type File_Descriptor;
   type File_Type is access File_Descriptor;
   Null_File : constant File_Type := null;
end SPARK_IO;
