-- $Id: spark_io.adb 11367 2008-10-07 15:47:17Z Bill Ellis $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Text_IO,
     Unchecked_Deallocation;

package body SPARK_IO
is
   --# hide SPARK_IO

   -- File Management

   type File_Descriptor is record
      File_Ref : Text_IO.File_Type;
   end record;

   type File_System is record
      Standard_Input  : File_Type;
      Standard_Output : File_Type;
   end record;

   File_Sys : File_System;

   function Standard_Input return File_Type
   is
   begin
      return File_Sys.Standard_Input;
   end Standard_Input;

   function Standard_Output return File_Type
   is
   begin
      return File_Sys.Standard_Output;
   end Standard_Output;

   procedure Dispose is new Unchecked_Deallocation (File_Descriptor, File_Type);

   procedure Create (File         : in out File_Type;
                     Name_Length  : in     Natural;
                     Name_Of_File : in     String;
                     Form_Of_File : in     String;
                     Status       :    out File_Status)
   is
   begin
      File := new File_Descriptor;
      Text_IO.Create (File.File_Ref,
                      Text_IO.Out_File,
                      Name_Of_File (Name_Of_File'First .. Name_Length),
                      Form_Of_File);
      Status := Ok;
   exception
      when Text_IO.Status_Error =>
         Status := Status_Error;
         Dispose (File);
      when Text_IO.Name_Error =>
         Status := Name_Error;
         Dispose (File);
      when Text_IO.Use_Error =>
         Status := Use_Error;
         Dispose (File);
      when Text_IO.Device_Error =>
         Status := Device_Error;
         Dispose (File);
   end Create;

   procedure Open (File         : in out File_Type;
                   Mode_Of_File : in     File_Mode;
                   Name_Length  : in     Natural;
                   Name_Of_File : in     String;
                   Form_Of_File : in     String;
                   Status       :    out File_Status)
   is
      FMODE : Text_IO.File_Mode;
   begin
      File := new File_Descriptor;
      case Mode_Of_File is
         when In_File  => FMODE := Text_IO.In_File;
         when Out_File => FMODE := Text_IO.Out_File;
         when Append_File => FMODE := Text_IO.Append_File;
      end case;
      Text_IO.Open (File.File_Ref, FMODE, Name_Of_File (Name_Of_File'First .. Name_Length), Form_Of_File);
      Status := Ok;
   exception
      when Text_IO.Status_Error =>
         Status := Status_Error;
         Dispose (File);
      when Text_IO.Name_Error =>
         Status := Name_Error;
         Dispose (File);
      when Text_IO.Use_Error =>
         Status := Use_Error;
         Dispose (File);
      when Text_IO.Device_Error =>
         Status := Device_Error;
         Dispose (File);
   end Open;

   procedure Close (File   : in out File_Type;
                    Status : out File_Status)
   is
   begin
      if File = null then
         Status := Status_Error;
      else
         Text_IO.Close (File.File_Ref);
         Dispose (File);
         Status := Ok;
      end if;
   exception
      when Text_IO.Status_Error => Status := Status_Error;
      when Text_IO.Device_Error => Status := Device_Error;
   end Close;

   procedure Delete (File   : in out File_Type;
                     Status : out File_Status)
   is
   begin
      if File = null then
         Status := Status_Error;
      else
         Text_IO.Delete (File.File_Ref);
         Dispose (File);
         Status := Ok;
      end if;
   exception
      when Text_IO.Status_Error => Status := Status_Error;
      when Text_IO.Use_Error    => Status := Use_Error;
      when Text_IO.Device_Error => Status := Device_Error;
   end Delete;

   procedure Reset (File         : in out File_Type;
                    Mode_Of_File : in File_Mode;
                    Status       : out File_Status)
   is
      FMODE : Text_IO.File_Mode;
   begin
      if File = null then
         Status := Status_Error;
      else
         case Mode_Of_File is
            when In_File     => FMODE := Text_IO.In_File;
            when Out_File    => FMODE := Text_IO.Out_File;
            when Append_File => FMODE := Text_IO.Append_File;
         end case;
         Text_IO.Reset (File.File_Ref, FMODE);
         Status := Ok;
      end if;
   exception
      when Text_IO.Status_Error => Status := Status_Error;
      when Text_IO.Use_Error    => Status := Use_Error;
      when Text_IO.Device_Error => Status := Device_Error;
   end Reset;

   function Valid_File (File : in File_Type) return Boolean
   is
   begin
      return File /= null;
   end Valid_File;

   function File_Ref (File : in File_Type) return Text_IO.File_Type
   is
   begin
      if File = File_Sys.Standard_Input then
         return Text_IO.Standard_Input;
      elsif File = File_Sys.Standard_Output then
         return Text_IO.Standard_Output;
      else
         pragma Warnings (Off);
         return File.File_Ref;
         pragma Warnings (On);
      end if;
   end File_Ref;
   pragma Inline (File_Ref);

   function Is_Open (File : in File_Type) return Boolean
   is
   begin
      return Valid_File (File) and then
         Text_IO.Is_Open (File_Ref (File));
   end Is_Open;

   function Mode (File : in File_Type) return File_Mode
   is
      FMode : File_Mode;
   begin
      if Is_Open (File) and then
         Text_IO.Is_Open (File_Ref (File)) then
         case Text_IO.Mode (File_Ref (File)) is
            when Text_IO.In_File     => FMode := In_File;
            when Text_IO.Out_File    => FMode := Out_File;
            when Text_IO.Append_File => FMode := Append_File;
         end case;
      else
         FMode := In_File;
      end if;
      return FMode;
   end Mode;

   function Is_In (File : in File_Type) return Boolean
   is
   begin
      return Is_Open (File) and then Mode (File) = In_File;
   end Is_In;

   function Is_Out (File : in File_Type) return Boolean
   is
   begin
      return Is_Open (File) and then (Mode (File) = Out_File or
                                      Mode (File) = Append_File);
   end Is_Out;

   procedure Name (File         : in File_Type;
                   Name_Of_File : out String;
                   Stop         : out Natural)
   is
   begin
      if Is_Open (File) then
         declare
            FN : constant String := Text_IO.Name (File_Ref (File));
         begin
            if Name_Of_File'Length >= FN'Length then
               Name_Of_File (FN'Range) := FN;
               Stop := FN'Length;
            else
               Name_Of_File := FN (Name_Of_File'Range);
               Stop := Name_Of_File'Length;
            end if;
         end;
      else
         Stop := Name_Of_File'First - 1;
      end if;
   exception
      when others => Stop := Name_Of_File'First - 1;
   end Name;

   procedure Form (File         : in File_Type;
                   Form_Of_File : out String;
                   Stop         : out Natural)
   is
   begin
      if Is_Open (File) then
         declare
            FM : constant String := Text_IO.Form (File_Ref (File));
         begin
            if Form_Of_File'Length >= FM'Length then
               Form_Of_File (FM'Range) := FM;
               Stop := FM'Length;
            else
               Form_Of_File := FM (Form_Of_File'Range);
               Stop := Form_Of_File'Length;
            end if;
         end;
      else
         Stop := Form_Of_File'First - 1;
      end if;
   exception
      when others => Stop := Form_Of_File'First - 1;
   end Form;

   -- Line and file terminator control

   function P_TO_PC (P : Positive) return Text_IO.Positive_Count
   is
   begin
      return Text_IO.Positive_Count (P);
   exception
      when others => return Text_IO.Positive_Count'Last;
   end P_TO_PC;

   function PC_TO_P (PC : Text_IO.Positive_Count) return Positive
   is
   begin
      return Positive (PC);
   exception
      when others => return Positive'Last;
   end PC_TO_P;

   procedure New_Line (File    : in File_Type;
                       Spacing : in Positive)
   is
      Gap    : Text_IO.Positive_Count;
   begin
      if Is_Out (File) then
         Gap := P_TO_PC (Spacing);
         Text_IO.New_Line (File_Ref (File), Gap);
      end if;
   exception
      when others => null;
   end New_Line;

   procedure Skip_Line (File    : in File_Type;
                        Spacing : in Positive)
   is
      Gap    : Text_IO.Positive_Count;
   begin
      if Is_In (File) then
         Gap := P_TO_PC (Spacing);
         Text_IO.Skip_Line (File_Ref (File), Gap);
      end if;
   exception
      when others => null;
   end Skip_Line;

   procedure New_Page  (File : in File_Type)
   is
   begin
      if Is_Out (File) then
         Text_IO.New_Page (File_Ref (File));
      end if;
   exception
      when others => null;
   end New_Page;

   function End_Of_Line (File : in File_Type) return Boolean
   is
      Eoln : Boolean;
   begin
      if Is_In (File) then
         Eoln := Text_IO.End_Of_Line (File_Ref (File));
      else
         Eoln := False;
      end if;
      return Eoln;
   end End_Of_Line;

   function End_Of_File (File : in File_Type) return Boolean
   is
      EOF : Boolean;
   begin
      if Is_In (File) then
         EOF := Text_IO.End_Of_File (File_Ref (File));
      else
         EOF := True;
      end if;
      return EOF;
   end End_Of_File;

   procedure Set_Col (File : in File_Type;
                      Posn : in Positive)
   is
      Col    : Text_IO.Positive_Count;
   begin
      if Is_Open (File) then
         Col := P_TO_PC (Posn);
         Text_IO.Set_Col (File_Ref (File), Col);
      end if;
   exception
      when others => null;
   end Set_Col;

   function Col (File : in File_Type) return Positive
   is
      Posn : Positive;
      Col  : Text_IO.Positive_Count;
   begin
      if Is_Open (File) then
         Col := Text_IO.Col (File_Ref (File));
         Posn := PC_TO_P (Col);
      else
         Posn := 1;
      end if;
      return Posn;
   exception
      when Text_IO.Status_Error => return 1;
      when Text_IO.Layout_Error => return PC_TO_P (Text_IO.Count'Last);
      when Text_IO.Device_Error => return 1;
   end Col;

   function Line (File : in File_Type) return Positive
   is
      POSN : Positive;
      Line  : Text_IO.Positive_Count;
   begin
      if Is_Open (File) then
         Line := Text_IO.Line (File_Ref (File));
         POSN := PC_TO_P (Line);
      else
         POSN := 1;
      end if;
      return POSN;
   exception
      when Text_IO.Status_Error => return 1;
      when Text_IO.Layout_Error => return PC_TO_P (Text_IO.Count'Last);
      when Text_IO.Device_Error => return 1;
   end Line;


   -- Character IO

   procedure Get_Char (File : in File_Type;
                       Item : out Character)
   is
   begin
      if Is_In (File) then
         Text_IO.Get (File_Ref (File), Item);
      end if;
   exception
      when others => null;
   end Get_Char;

   procedure Put_Char (File : in File_Type;
                       Item : in Character)
   is
   begin
      if Is_Out (File) then
         Text_IO.Put (File_Ref (File), Item);
      end if;
   exception
      when others => null;
   end Put_Char;


   -- String IO

   procedure Get_String (File : in File_Type;
                         Item : out String;
                         Stop : out Natural)
   is
      LSTP : Natural;
   begin
      if Is_In (File) then
         LSTP := Item'First - 1;
         loop
            exit when End_Of_File (File);
            LSTP := LSTP + 1;
            Get_Char (File, Item (LSTP));
            exit when LSTP = Item'Last;
         end loop;
         Stop := LSTP;
      else
         Stop := Item'First - 1;
      end if;
   end Get_String;

   procedure Put_String (File : in File_Type;
                         Item : in String;
                         Stop : in Natural)
   is
      ES : Positive;
   begin
      if Stop = 0 then
         ES := Item'Last;
      else
         ES := Stop;
      end if;
      if Is_Out (File) then
         Text_IO.Put (File_Ref (File), Item (Item'First .. ES));
      end if;
   exception
      when others => null;
   end Put_String;

   procedure Get_Line (File : in File_Type;
                       Item : out String;
                       Stop : out Natural)
   is
   begin
      if Is_In (File) then
         Text_IO.Get_Line (File_Ref (File), Item, Stop);
      else
         Stop := Item'First - 1;
      end if;
   exception
      when others => Stop := Item'First - 1;
   end Get_Line;

   procedure Put_Line (File : in File_Type;
                       Item : in String;
                       Stop : in Natural)
   is
      ES : Positive;
   begin
      if Stop = 0 then
         ES := Item'Last;
      else
         ES := Stop;
      end if;
      if Is_Out (File) then
         Text_IO.Put_Line (File_Ref (File), Item (Item'First .. ES));
      end if;
   exception
      when others => null;
   end Put_Line;


   -- Integer IO

   package Integer_IO is new Text_IO.Integer_IO (Integer);

   procedure Get_Integer (File  : in     File_Type;
                          Item  :    out Integer;
                          Width : in     Natural;
                          Read  :    out Boolean)
   is
   begin
      if Is_In (File) then
         Integer_IO.Get (File_Ref (File), Item, Width);
         Read := True;
      else
         Read := False;
      end if;
   exception
      when others => Read := False;
   end Get_Integer;

   procedure Put_Integer (File  : in File_Type;
                          Item  : in Integer;
                          Width : in Natural;
                          Base  : in Number_Base)
   is
   begin
      if Is_Out (File) then
         Integer_IO.Put (File_Ref (File), Item, Width, Base);
      end if;
   exception
      when others => null;
   end Put_Integer;

   procedure Get_Int_From_String (Source  : in     String;
                                  Item    :    out Integer;
                                  Startpt : in     Positive;
                                  Stop    :    out Natural)
   is
   begin
      Integer_IO.Get (Source (Startpt .. Source'Last), Item, Stop);
   exception
      when others => Stop := Startpt - 1;
   end Get_Int_From_String;

   procedure Put_Int_To_String (Dest    :    out String;
                                Item    : in     Integer;
                                Startpt : in     Positive;
                                Base    : in     Number_Base)
   is
   begin
      Integer_IO.Put (Dest (Startpt .. Dest'Last), Item, Base);
   exception
      when others => null;
   end Put_Int_To_String;


   -- Float IO

   package Real_IO is new Text_IO.Float_IO (Float);

   procedure Get_Float (File  : in     File_Type;
                        Item  :    out Float;
                        Width : in     Natural;
                        Read  :    out Boolean)
   is
   begin
      if Is_In (File) then
         Real_IO.Get (File_Ref (File), Item, Width);
         Read := True;
      else
         Read := False;
      end if;
   exception
      when others => Read := False;
   end Get_Float;

   procedure Put_Float (File : in File_Type;
                        Item : in Float;
                        Fore : in Natural;
                        Aft  : in Natural;
                        Exp  : in Natural)
   is
   begin
      if Is_Out (File) then
         Real_IO.Put (File_Ref (File), Item, Fore, Aft, Exp);
      end if;
   exception
      when others => null;
   end Put_Float;

   procedure Get_Float_From_String (Source  : in     String;
                                    Item    :    out Float;
                                    Startpt : in     Positive;
                                    Stop    :    out Natural)
   is
   begin
      Real_IO.Get (Source (Startpt .. Source'Last), Item, Stop);
   exception
      when others => Stop := Startpt - 1;
   end Get_Float_From_String;

   procedure Put_Float_To_String (Dest    : in out String;
                                  Item    : in     Float;
                                  Startpt : in     Positive;
                                  Aft     : in     Natural;
                                  Exp     : in     Natural)
   is
   begin
      Real_IO.Put (Dest (Startpt .. Dest'Last), Item, Aft, Exp);
   exception
      when others => null;
   end Put_Float_To_String;

begin
   File_Sys := File_System'(Standard_Input  => new File_Descriptor,
                            Standard_Output => new File_Descriptor);
end SPARK_IO;
