-- $Id: osfiling.adb 12312 2009-01-27 14:24:42Z Robin Messer $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Ada.Characters.Latin_1,
     FatalErrors,
     GNAT.Directory_Operations,
     GNAT.IO_Aux,
     GNAT.OS_Lib,
     CommandLine;

package body OSFiling
is

   ------------------------------------------------------------------------------
   --
   -- Given a path, this function changes directory to that path and returns a
   -- string containing the full path. Useful when you have a path such as "."
   -- or ".." and you want to get a meaningful directory name.
   --
   -- Note: has the side-effect of changing the current directory to be the
   -- specified directory so if that isn't what you want then save the current
   -- directory before calling this function, then cd back to it afterwards.
   --
   -- Note also that the returned directory name will have a trailing slash
   -- that you may have to deal with.
   --
   -- If the specified path doesn't exist then a fatal error is reported.
   --
   ------------------------------------------------------------------------------
   function CD_And_Get_Name (Path : ELStrings.T)
      return GNAT.Directory_Operations.Dir_Name_Str
   is
   begin
      GNAT.Directory_Operations.Change_Dir (Path.Content (1 .. Path.Length));
      return GNAT.Directory_Operations.Get_Current_Dir;
   exception
      when others =>
         -- Note: this call will NOT return so the return statement following
         -- it will never be executed, but is expected by the compiler.
         FatalErrors.Process (FatalErrors.ExpectedDirectoryMissing, Path);
         return "";
   end CD_And_Get_Name;

   ----------------------------------------------------------------------------

   function BaseDirName (Path : ELStrings.T)
                         return ELStrings.T
   is
      Result : ELStrings.T;
      Current_Dir : constant GNAT.Directory_Operations.Path_Name
         := GNAT.Directory_Operations.Get_Current_Dir; -- save for later
      Full_Path_With_Slash : constant String := CD_And_Get_Name (Path);
      Full_Path_No_Slash : constant String := Full_Path_With_Slash (1 .. Full_Path_With_Slash'Length - 1);
      Base_Name : constant String := GNAT.Directory_Operations.Base_Name (Full_Path_No_Slash, "");
   begin
      GNAT.Directory_Operations.Change_Dir (Current_Dir); -- return to saved dir
      Result.Content := (others => Ada.Characters.Latin_1.NUL);
      Result.Content (1 .. Base_Name'Length) := Base_Name;
      Result.Length := Base_Name'Length;
      return Result;
   exception
      when others =>
         -- Note: this call will NOT return so the return statement following
         -- it will never be executed, but is expected by the compiler.
         FatalErrors.Process (FatalErrors.ExpectedDirectoryMissing, Path);
         return Result;
   end BaseDirName;

   ----------------------------------------------------------------------------

   function BaseFileName (Path : ELStrings.T)
                         return ELStrings.T
   is
      Result : ELStrings.T;
      C : constant String := GNAT.Directory_Operations.Base_Name
                                (Path.Content (1 .. Path.Length), "");
   begin
      Result.Content := (others => Ada.Characters.Latin_1.NUL);
      Result.Content (1 .. C'Length) := C;
      Result.Length := C'Length;
      return Result;
   end BaseFileName;

   ----------------------------------------------------------------------------
   function DefaultReportExtn return EStrings.T
   --# global in CommandLine.Data;
   is
   begin
      if CommandLine.Data.XML then
         return EStrings.T'
           (Length => 4,
         Content => EStrings.Contents'('.', 's', 'm', 'l', others => ' '));
      else
         return EStrings.T'
           (Length => 4,
         Content => EStrings.Contents'('.', 's', 'u', 'm', others => ' '));
      end if;
   end DefaultReportExtn;

   ------------------------------------------------------------------------------
   -- this function combines the inputs to produce the path to the
   -- subdirectory
   function DownDirectory (Path         : ELStrings.T;
                           SubDirectory : ELStrings.T)
                          return ELStrings.T
   is
      Result : ELStrings.T;
      Success : Boolean;
   begin
      Result := Path;
      if Path.Content (Path.Length) /= GNAT.Directory_Operations.Dir_Separator then
         ELStrings.AppendChar (Result, GNAT.Directory_Operations.Dir_Separator, Success);  -- Success unused
      end if;
      ELStrings.AppendExaminerLongString (Result, SubDirectory);
      return Result;
   end DownDirectory;

   ------------------------------------------------------------------------------
   -- this function combines the inputs to produce a full file name
   function FullFileName (Path     : ELStrings.T;
                          FileName : ELStrings.T)
                         return ELStrings.T
   is
   begin
      return DownDirectory (Path, FileName);
   end FullFileName;

   ------------------------------------------------------------------------------
   function GetWorkingDirectory return ELStrings.T
   is
      --# hide GetWorkingDirectory;
      Res    : Natural;
      ResStr : ELStrings.T;
      Cwd    : ELStrings.T;
   begin
      Cwd := ELStrings.EmptyString;
      ResStr.Content := (others => Ada.Characters.Latin_1.NUL);
      GNAT.Directory_Operations.Get_Current_Dir (ResStr.Content, Res);
      for I in ELStrings.Positions loop
         exit when ResStr.Content (I) = Ada.Characters.Latin_1.NUL;
         Cwd.Length := I;
         Cwd.Content (I) := ResStr.Content (I);
      end loop;
      -- Get_Current_Dir can return a trailing '/' on NT..
      if Cwd.Length /= 0 and then Cwd.Content (Cwd.Length) = GNAT.Directory_Operations.Dir_Separator then
         Cwd.Content (Cwd.Length) := Ada.Characters.Latin_1.NUL;
         Cwd.Length := Cwd.Length - 1;
      end if;
      return Cwd;
   end GetWorkingDirectory;

   function IsDirectory (Name : ELStrings.T) return Boolean
   is
      StringName : constant String := Name.Content (1 .. Name.Length);
   begin
      return GNAT.OS_Lib.Is_Directory (StringName);
   end IsDirectory;

   -----------------------------------------------------------------------------
   function IsFile (Name : ELStrings.T) return Boolean
   is
      StringName : constant String := Name.Content (1 .. Name.Length);
   begin
      return GNAT.IO_Aux.File_Exists (StringName);
   end IsFile;

   ------------------------------------------------------------------------------
   function Order (FirstName, SecondName : ELStrings.T)
                  return ELStrings.OrderTypes
   is
   begin
      return ELStrings.LexOrder (FirstName, SecondName);
   end Order;

   ------------------------------------------------------------------------------
   procedure RemoveFileExtension (FileName : in out ELStrings.T)
   is
      DotPos : ELStrings.Positions;
   begin
      DotPos := FileName.Length;

      while DotPos > 1 and FileName.Content (DotPos) /= '.' loop
         DotPos := DotPos - 1;
      end loop;

      if FileName.Content (DotPos) = '.' then
         FileName.Length := DotPos - 1;
      end if;
   end RemoveFileExtension;

   ----------------------------------------------------------------------------
   function SimplifiedVCFileExtension return EStrings.T
   is
   begin
      return EStrings.T'
         (Length => 4,
          Content => EStrings.Contents'('.', 's', 'i', 'v', others => ' '));
   end SimplifiedVCFileExtension;

   ------------------------------------------------------------------------------

   -- this function removes the last directory name from the supplied string
   function DirName (Path : ELStrings.T)
                     return ELStrings.T
   is
      Result : ELStrings.T;
      C : constant String := GNAT.Directory_Operations.Dir_Name
                                (Path.Content (1 .. Path.Length));
   begin
      Result.Content := (others => Ada.Characters.Latin_1.NUL);
      Result.Content (1 .. C'Length) := C;
      Result.Length := C'Length;
      return Result;
   end DirName;

   ----------------------------------------------------------------------------
   function VCFileExtension return EStrings.T
   is
   begin
      return EStrings.T'
         (Length => 4,
          Content => EStrings.Contents'('.', 'v', 'c', 'g', others => ' '));
   end VCFileExtension;

   ----------------------------------------------------------------------------
   function ProofLogFileExtension return EStrings.T
   is
   begin
      return EStrings.T'
         (Length => 4,
          Content => EStrings.Contents'('.', 'p', 'l', 'g', others => ' '));
   end ProofLogFileExtension;

   ----------------------------------------------------------------------------
   function ReviewFileExtension return EStrings.T
   is
   begin
      return EStrings.T'
         (Length => 4,
          Content => EStrings.Contents'('.', 'p', 'r', 'v', others => ' '));
   end ReviewFileExtension;

   ----------------------------------------------------------------------------
   function SimplifierLogFileExtension return EStrings.T
   is
   begin
      return EStrings.T'
         (Length => 4,
          Content => EStrings.Contents'('.', 's', 'l', 'g', others => ' '));
   end SimplifierLogFileExtension;

   ----------------------------------------------------------------------------
   function DirectorySeparator return Character
   is
   begin
      return GNAT.Directory_Operations.Dir_Separator;
   end DirectorySeparator;

end OSFiling;
