-- $Id: osdirectory.adb 12146 2009-01-14 13:14:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


with Ada.Characters.Latin_1,
     ELStrings,
     FatalErrors,
     FileDetails,
     FileHeap,
     OSFiling,
     GNAT.Directory_Operations;

package body OSDirectory
is
   ------------------------------------------------------------------------
   -- this procedure reads the named directory using the C calls
   -- opendir(), readdir() and closedir()
   -- for each entry, it calls OSFiling.IsDirectory and then enters it in the
   -- linked list
   ------------------------------------------------------------------------
   procedure Scan (ListIndex : in Heap.Atom)
   is
      --# hide Scan;
      DetailsSuccess   : Boolean;
      FileType         : FileDetails.FileTypes;
      DirectoryIsResolved : Boolean;
      TempName    : String (1 .. 1024);
      BaseName,
      DirName,
      FileName    : ELStrings.T;
      Dir         : GNAT.Directory_Operations.Dir_Type;
      Last        : Natural;

      -- Convert a regular string into an Examiner string
      function CreateExaminerLongString (Str : in String)
                                    return ELStrings.T
      is
         EStr : ELStrings.T;
         Pos  : ELStrings.Positions := 1;
      begin
         while Str (Pos) /= Ada.Characters.Latin_1.NUL loop
            EStr.Content (Pos) := Str (Pos);
            Pos := Pos + 1;
         end loop;

         EStr.Length := Pos - 1;

         return EStr;
      end CreateExaminerLongString;

   --------------------------------------------------------------------------
   begin -- ScanDirectory
      -- retrieve directory details
      FileHeap.Details (ListIndex,
                        DetailsSuccess,
                        DirName,
                        FileType,
                        DirectoryIsResolved);

      if not DetailsSuccess then
         FatalErrors.Process (FatalErrors.DataStructureInconsistency, ELStrings.EmptyString);
      end if;

      -- block to trap exceptions from Open
      begin
         -- Read from DirName
         GNAT.Directory_Operations.Open
            (Dir => Dir,
             Dir_Name => DirName.Content (1 .. DirName.Length));
      exception
         when others =>
            -- note: this call will NOT return
            FatalErrors.Process (FatalErrors.ExpectedDirectoryMissing, DirName);
      end;

      -- Now repeatedly read a file from the directory
      loop
         -- read the file
         TempName := (others => Ada.Characters.Latin_1.NUL);
         GNAT.Directory_Operations.Read (Dir, TempName, Last);
         if Last = 0 then
            exit;
         end if;
         -- Ignore dot files; hidden Unix ones, and ., ..
         if TempName (1) /= '.' then
            -- Create a proper name
            BaseName := CreateExaminerLongString (TempName);
            FileName := OSFiling.DownDirectory (DirName, BaseName);
            -- Breadth-first search; add any directory to our
            --  to-do list
            if OSFiling.IsDirectory (FileName) then
               FileHeap.Add (ListIndex,
                             FileName,
                             FileDetails.Directory);
            else
               -- It's a vanilla file; get its full name
               FileName := OSFiling.FullFileName (DirName, BaseName);
               -- Remove the extension
               OSFiling.RemoveFileExtension (FileName);
               FileHeap.Add (ListIndex,
                             FileName,
                             FileDetails.PlainFile);
            end if; -- IsDirectory
         end if;  -- TempName(1) /= '.'
      end loop; -- infinite
      GNAT.Directory_Operations.Close (Dir);
   end Scan;
end OSDirectory;
