-- $Id: findfiles.adb 12146 2009-01-14 13:14:58Z Rod Chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Package for reading the file structure on the disc. This package is         --
--OS independent, and calls OSDirectory.Scan to perform the actual directory  --
--scan.                                                                       --
--                                                                            --
--------------------------------------------------------------------------------
with FatalErrors,
     FileDetails,
     FileHeap,
     Heap,
     OSDirectory;

use type FileDetails.FileTypes;

package body FindFiles
is

   ---------------------------------------------------------------------------
   procedure Scan (StartDirectory : in ELStrings.T)
   is
      DetailsSuccess      : Boolean;
      DirectoryIsResolved : Boolean;
      FileType            : FileDetails.FileTypes;
      ListIndex           : Heap.Atom;
      Name                : ELStrings.T;
      NextListIndex       : Heap.Atom;
      NextSuccess         : Boolean := True;
      UnResolvedDirFound  : Boolean := True;

   begin
      -- initialize the FileHeap
      FileHeap.Initialize (StartDirectory);

      while UnResolvedDirFound loop
         UnResolvedDirFound := False;

         -- start at beginning of linked list
         ListIndex := FileHeap.FirstEntry;

         -- while not end of list
         while NextSuccess loop
            -- get entry
            --# accept F, 10, Name, "Name unused here";
            FileHeap.Details (ListIndex,
                              DetailsSuccess,
                              Name,
                              FileType,
                              DirectoryIsResolved);
            --# end accept;

            if not DetailsSuccess then
               FatalErrors.Process (FatalErrors.DataStructureInconsistency, ELStrings.EmptyString);
            end if;

            -- if unresolved directory
            if FileType = FileDetails.Directory and not DirectoryIsResolved then

               -- read directory and insert items in list
               OSDirectory.Scan (ListIndex);

               -- mark directory as resolved
               FileHeap.MarkDirectoryResolved (ListIndex);

               UnResolvedDirFound := True;
            end if;

            -- go to next entry
            if DetailsSuccess then
               FileHeap.Next (ListIndex, NextSuccess, NextListIndex);
               ListIndex := NextListIndex;
            end if;
         end loop;
      end loop;
      --# accept F, 33, Name, "Name unused here";
   end Scan;
end FindFiles;
