-- $Id: fatalerrors.adb 12207 2009-01-19 15:47:06Z Robin Messer $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================


--------------------------------------------------------------------------------
--Synopsis:                                                                   --
--                                                                            --
--Output 'fatal' error messages                                               --
--Note that the Process procedure does not return to point of call. Instead   --
--it raises an exception which is trapped by the main program, which causes   --
--the program to stop politely.                                               --
--                                                                            --
--                                                                            --
--------------------------------------------------------------------------------
with EStrings,
     CommandLine,
     POGS_Exceptions,
     SPARK_IO;

package body FatalErrors
is
   --# hide FatalErrors;
   -- hidden to mask the use of exceptions and the non-initialization of
   -- FatalErrors.State

   procedure Process (Error   : in ErrorType;
                      Message : in ELStrings.T)
   is
      T : EStrings.T;
   begin
      case Error is
         when CouldNotOpenInputFile =>
            EStrings.CopyString
              (T, "Could not open expected input file " &
                 "- unexpected file system error");

         when CouldNotCreateReportFile =>
            EStrings.CopyString
              (T, "Could not create report file ");
            EStrings.PutString (SPARK_IO.Standard_Output, T);
            ELStrings.PutLine (SPARK_IO.Standard_Output, Message);
            EStrings.CopyString (T, "Check you have write permission for directory");

         when DataStructureInconsistency =>
            EStrings.CopyString
              (T, "Internal file table inconsistent.");

         when VCDataStructureInconsistency =>
            EStrings.CopyString
              (T, "Internal VC table inconsistent.");

         when ExpectedDirectoryMissing =>
            EStrings.CopyString
              (T, "Directory ");
            EStrings.PutString (SPARK_IO.Standard_Output, T);
            ELStrings.PutString (SPARK_IO.Standard_Output, Message);
            EStrings.CopyString (T, " not found or not readable.");

         when FileHeapFull =>
            EStrings.CopyString
              (T, "Internal file table too small.");

         when VCHeapFull =>
            EStrings.CopyString
              (T, "Internal VC table too small.");

         when InvalidCommandLine =>
            EStrings.CopyString
              (T, "Usage: ");
            EStrings.AppendExaminerStringTruncate
              (T, CommandLine.UsageString);

         when ProblemReadingFile =>
            EStrings.CopyString
              (T, "Could not read from input file " &
               "- unexpected file system error");

         when ProblemCreatingTempFile =>
            EStrings.CopyString
              (T, "Could not create temporary file.");

         when SubprogramTotalsInconsistent =>
            EStrings.CopyString
              (T, "Overall subprogram summary counts inconsistent.");

      end case;

      EStrings.PutLine (SPARK_IO.Standard_Output, T);

      raise POGS_Exceptions.Usage_Error;
   end Process;
end FatalErrors;
