-- $Id: xmlreport.adb 11674 2008-11-12 12:20:09Z rod chapman $
--------------------------------------------------------------------------------
-- (C) Praxis High Integrity Systems Limited
--------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--==============================================================================

with SystemErrors,
     FileSystem,
     CommandLineData;
package body XMLReport
--# own State is Schema, SchemaState, TagStrings, TagIDs, SectionDepths, FileDepth;
is


   -- Set the version of the report file schema.  This should be the CVS version of the
   -- sparkreport.xsd file.
   SchemaVersion : constant EStrings.T :=
     EStrings.T'
     (3,
      EStrings.Contents'('1', '.', '8',
                                others => ' '));



   type MyTag is (MTBrief_Justifications,
                  MTCommandline,
                  MTCompiler_Data,
                  MTCompiler_Item,
                  MTCyclic_Requirements,
                  MTFile,
                  MTFilename,
                  MTFull_Justification,
                  MTFull_Justifications,
                  MTIndex,
                  MTIndexes,
                  MTJustifications_Section,
                  MTListing,
                  MTMessage,
                  MTMessages,
                  MTMetafile,
                  MTMetafiles,
                  MTOption,
                  MTPrologue,
                  MTReport,
                  MTResults,
                  MTScope,
                  MTSuppressed,
                  MTSuppressedPragma,
                  MTSymbol,
                  MTTarget_Config,
                  MTUnit,
                  MTUnits_Notfound,
                  MTWarnings_Config,
                  MTUnitsInFile);

   type TagStringArray is array (MyTag) of XMLString;
   type TagIDArray is array (MyTag) of SPARK_XML.TagID;
   type SectionDepthArray is array (Sections) of SPARK_XML.TagDepth;


   -------------------
   -- Own Variables --
   -------------------

   -- Dictionary for the schema
   TagStrings : TagStringArray;
   TagIDs : TagIDArray;

   -- The schema
   Schema : SPARK_XML.SchemaRecord;
   SchemaState : SPARK_XML.SchemaStateRecord;

   -- Additional schema state info
   FileDepth : SPARK_XML.TagDepth;
   SectionDepths : SectionDepthArray;


   --------------------
   -- Error handling --
   --------------------

   procedure HandleError (Status : in SPARK_XML.SchemaStatus)
   --# global in Schema;
   --#        in SchemaState;
   --# derives null from Schema,
   --#                   SchemaState,
   --#                   Status;
   is
   begin
      if SPARK_XML.IsError (Status) then
         SPARK_XML.PrintSchemaError (Status);
         SPARK_XML.PrintWorkingState (Schema, SchemaState);

         case Status is
            when SPARK_XML.SSToManyAttributes =>
               SystemErrors.FatalError (SystemErrors.XMLGenerationError, "Exceeded Attribute Limit");
            when SPARK_XML.SSStackFull =>
               SystemErrors.FatalError (SystemErrors.XMLGenerationError, "Exceeded XML Stack limit");
            when SPARK_XML.SSTagIncomplete =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Attempted to gerenate incomplete tag");
            when SPARK_XML.SSInvalidDepth =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Invalid depth value for tag closure");
            when SPARK_XML.SSStackEmpty =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Tag stack empty");
            when SPARK_XML.SSWrongContentType =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Incorrect content type for attribute");
            when SPARK_XML.SSInvalidAttribute =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Invalid attribute for tag");
            when SPARK_XML.SSInvalidTag =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Invalid tag");
            when SPARK_XML.SSNoSuchTag =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "No such tag defined in schema");
            when SPARK_XML.SSTagNotFound =>
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt,
                                        "Could not find instance of the tag to close");
            when SPARK_XML.SSOK =>  -- We should never get to this case
               SystemErrors.FatalError (SystemErrors.IllegalXMLGenerationAttempt, "Everything is fine");
         end case;
      end if;

   end HandleError;


   procedure HandleSchemaError (Success : in Boolean;
                                Msg     : in String)
   --# derives null from Msg,
   --#                   Success;
   is
      --# hide HandleSchemaError;
   begin
      if not Success then
         SPARK_XML.PrintWorkingState (Schema, SchemaState);
         SystemErrors.FatalError (SystemErrors.XMLSchemaError, Msg);
      end if;
   end HandleSchemaError;



   function XStr (Str : in String) return EStrings.T
   is
      TmpStr : EStrings.T;
   begin
      EStrings.CopyString (TmpStr, Str);
      return TmpStr;
   end XStr;

   function FilterString (Str : in XMLString) return XMLString
   is
   begin
      return SPARK_XML.FilterString (Str);
   end FilterString;

   function FilterLongString (Str : in XMLLongString) return XMLLongString
   is
   begin
      return SPARK_XML.FilterLongString (Str);
   end FilterLongString;



   --------------------
   -- Initialisation --
   --------------------

   -- Build the dictionary, then create the schema representation for SPARK_XML

   procedure Init
   --# global out FileDepth;
   --#        out Schema;
   --#        out SchemaState;
   --#        out SectionDepths;
   --#        out TagIDs;
   --#        out TagStrings;
   --# derives FileDepth,
   --#         Schema,
   --#         SchemaState,
   --#         SectionDepths,
   --#         TagIDs,
   --#         TagStrings    from ;
   is

      -- Load the tag definitions in to the Schema
      procedure LoadTags
      --# global in out Schema;
      --#           out TagIDs;
      --#           out TagStrings;
      --# derives Schema,
      --#         TagIDs     from Schema &
      --#         TagStrings from ;
      is

         TmpTag : SPARK_XML.TagID;

      begin

         TagStrings := TagStringArray'(MTBrief_Justifications =>   XStr ("brief_justifications"),
                                       MTCommandline =>            XStr ("commandline"),
                                       MTCompiler_Data =>          XStr ("compiler_data"),
                                       MTCompiler_Item =>          XStr ("compiler_item"),
                                       MTCyclic_Requirements =>    XStr ("cyclic_requirements"),
                                       MTFile =>                   XStr ("file"),
                                       MTFilename =>               XStr ("filename"),
                                       MTFull_Justification =>     XStr ("full_justification"),
                                       MTFull_Justifications =>    XStr ("full_justifications"),
                                       MTIndex =>                  XStr ("index"),
                                       MTIndexes =>                XStr ("indexes"),
                                       MTJustifications_Section => XStr ("justifications_section"),
                                       MTListing =>                XStr ("listing"),
                                       MTMessage =>                XStr ("message"),
                                       MTMessages =>               XStr ("messages"),
                                       MTMetafile =>               XStr ("metafile"),
                                       MTMetafiles =>              XStr ("metafiles"),
                                       MTOption =>                 XStr ("option"),
                                       MTPrologue =>               XStr ("prologue"),
                                       MTReport =>                 XStr ("report"),
                                       MTResults =>                XStr ("results"),
                                       MTScope =>                  XStr ("scope"),
                                       MTSuppressed =>             XStr ("suppressed"),
                                       MTSuppressedPragma =>       XStr ("pragma"),
                                       MTSymbol =>                 XStr ("symbol"),
                                       MTTarget_Config =>          XStr ("target_config"),
                                       MTUnit =>                   XStr ("unit"),
                                       MTUnits_Notfound =>         XStr ("units_notfound"),
                                       MTWarnings_Config =>        XStr ("warnings_config"),
                                       MTUnitsInFile =>            XStr ("units"));

         TagIDs := TagIDArray'(others => SPARK_XML.NullTag);

         for I in MyTag loop
            SPARK_XML.AddTag (Schema, TagStrings (I), TmpTag);
            TagIDs (I) := TmpTag;
            if SPARK_XML.IsNullTag (TmpTag) then
               -- Run out of room, so generate an error.
               HandleSchemaError (False, "Failure adding tag");
               exit;
            end if;
         end loop;

      end LoadTags;


      -- Load the attribute definitions
      procedure LoadAttributes
      --# global in     TagIDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     TagIDs;
      is
         TmpAttrib : SPARK_XML.AttributeID;
         TmpSuccess : Boolean;

         type Attribute is record
            ID : SPARK_XML.TagID;
            Name : SPARK_XML.XMLString;
            Typ : SPARK_XML.AttributeType;
            Req : Boolean;
         end record;

         -- Set TotalAttributes to the total number of attributes to be added
         -- to tags in the Schema.
         TotalAttributes : constant Integer := 25;

         type AttributeIndex is range 1 .. TotalAttributes;
         type AttributeList is array (AttributeIndex) of Attribute;
         Attributes : AttributeList;

      begin

         -- This table encodes the legal attributes of each XML element, as specified
         -- in the SPARKReport Schema Description in sparkreport.xsd
         Attributes :=
           AttributeList'(Attribute'(TagIDs (MTReport),      XStr ("version"),  SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTSymbol),      XStr ("id"),       SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTOption),      XStr ("id"),       SPARK_XML.ATString,  False),  -- Remove?

                          Attribute'(TagIDs (MTUnit),        XStr ("name"),     SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTUnit),        XStr ("type"),     SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTUnit),        XStr ("status"),   SPARK_XML.ATString,  False),

                          Attribute'(TagIDs (MTMessage),     XStr ("class"),    SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTMessage),     XStr ("code"),     SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTMessage),     XStr ("line"),     SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTMessage),     XStr ("offset"),   SPARK_XML.ATInteger, True),

                          Attribute'(TagIDs (MTFull_Justification), XStr ("class"),      SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTFull_Justification), XStr ("code"),       SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTFull_Justification), XStr ("line_from"),  SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTFull_Justification), XStr ("line_to"),    SPARK_XML.ATString, True),
                          Attribute'(TagIDs (MTFull_Justification), XStr ("match_no"),   SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTFull_Justification), XStr ("match_line"), SPARK_XML.ATInteger, True),

                          Attribute'(TagIDs (MTBrief_Justifications), XStr ("matched"),   SPARK_XML.ATInteger, True),
                          Attribute'(TagIDs (MTBrief_Justifications), XStr ("unmatched"), SPARK_XML.ATInteger, False),

                          Attribute'(TagIDs (MTMetafile),    XStr ("name"),     SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTFile),        XStr ("unit"),     SPARK_XML.ATString,  False),
                          Attribute'(TagIDs (MTFile),        XStr ("type"),     SPARK_XML.ATString,  False),
                          Attribute'(TagIDs (MTFile),        XStr ("filename"), SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTCompiler_Item), XStr ("item"),   SPARK_XML.ATString,  True),

                          Attribute'(TagIDs (MTScope),       XStr ("id"),       SPARK_XML.ATString,  True),
                          Attribute'(TagIDs (MTScope),       XStr ("type"),     SPARK_XML.ATString,  True)
                         );

         for I in AttributeIndex loop
            --# accept Flow, 10, TmpAttrib, "Expected ineffective assignment to TmpAttrib";
            SPARK_XML.AddAttributeToTag (Schema,  -- Expect ineffective statement
                                         Attributes (I).ID,
                                         Attributes (I).Name,
                                         Attributes (I).Typ,
                                         Attributes (I).Req,
                                         TmpAttrib,
                                         TmpSuccess);
            --# end accept;
            HandleSchemaError (TmpSuccess, "Failure adding attribute to tag");
            exit when not TmpSuccess;
         end loop;

      --# accept Flow, 33, TmpAttrib, "Expected to be neither referenced nor exported";
      end LoadAttributes;  -- Expect "TmpAttrib neither referenced nor exported"



      -- Load the hierarchy definitions
      procedure BuildHierarchy
      --# global in     TagIDs;
      --#        in out Schema;
      --# derives Schema from *,
      --#                     TagIDs;
      is
         type TagRel is record
            Parent : SPARK_XML.TagID;
            Child : SPARK_XML.TagID;
            Required : Boolean;
         end record;

         MaxRelations : constant Integer := 37;

         subtype RelIndex is Integer range 1 .. MaxRelations;

         type RelArray is array (RelIndex) of TagRel;
         Relations : RelArray;

         TmpSuccess : Boolean;

      begin

         -- This table encodes the legal nesting of XML elements, as specified
         -- in the SPARKReport Schema Description in sparkreport.xsd
         Relations :=
           RelArray'(
                     TagRel'(SPARK_XML.NullTag,              TagIDs (MTReport),              False),

                     TagRel'(TagIDs (MTScope),               TagIDs (MTScope),               False),
                     TagRel'(TagIDs (MTScope),               TagIDs (MTMessage),             False),

                     TagRel'(TagIDs (MTMessage),             TagIDs (MTSymbol),              False),

                     TagRel'(TagIDs (MTReport),              TagIDs (MTPrologue),            False),
                     TagRel'(TagIDs (MTReport),              TagIDs (MTResults),             False),

                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTCommandline),         False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTIndexes),             False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTMetafiles),           False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTWarnings_Config),     False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTCompiler_Data),       False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTTarget_Config),       False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTUnits_Notfound),      False),
                     TagRel'(TagIDs (MTPrologue),            TagIDs (MTCyclic_Requirements), False),

                     TagRel'(TagIDs (MTCommandline),         TagIDs (MTFilename),            False),
                     TagRel'(TagIDs (MTCommandline),         TagIDs (MTOption),              False),

                     TagRel'(TagIDs (MTCompiler_Data),       TagIDs (MTCompiler_Item),       False),

                     TagRel'(TagIDs (MTCyclic_Requirements), TagIDs (MTUnit),                False),

                     TagRel'(TagIDs (MTIndexes),             TagIDs (MTIndex),               False),

                     TagRel'(TagIDs (MTTarget_Config),       TagIDs (MTFilename),            False),
                     TagRel'(TagIDs (MTTarget_Config),       TagIDs (MTMessages),            False),

                     TagRel'(TagIDs (MTMessages),            TagIDs (MTMessage),             False),

                     TagRel'(TagIDs (MTMetafiles),           TagIDs (MTMetafile),            False),

                     TagRel'(TagIDs (MTMetafile),            TagIDs (MTMetafile),            False),
                     TagRel'(TagIDs (MTMetafile),            TagIDs (MTFilename),            False),

                     TagRel'(TagIDs (MTUnits_Notfound),      TagIDs (MTUnit),                False),

                     TagRel'(TagIDs (MTWarnings_Config),     TagIDs (MTSuppressed),          False),
                     TagRel'(TagIDs (MTWarnings_Config),     TagIDs (MTSuppressedPragma),    False),

                     TagRel'(TagIDs (MTFile),                TagIDs (MTUnitsInFile),            False),
                     TagRel'(TagIDs (MTFile),                TagIDs (MTScope),                  False),
                     TagRel'(TagIDs (MTFile),                TagIDs (MTMessage),                False),
                     TagRel'(TagIDs (MTFile),                TagIDs (MTJustifications_Section), False),

                     TagRel'(TagIDs (MTJustifications_Section), TagIDs (MTFull_Justifications),  False),
                     TagRel'(TagIDs (MTJustifications_Section), TagIDs (MTBrief_Justifications), False),

                     TagRel'(TagIDs (MTFull_Justifications), TagIDs (MTFull_Justification),  False),

                     TagRel'(TagIDs (MTUnitsInFile),         TagIDs (MTUnit),                False),

                     TagRel'(TagIDs (MTResults),             TagIDs (MTFile),                False));


         for I in RelIndex loop
            SPARK_XML.AddChildTag (Schema,
                                   Relations (I).Parent,
                                   Relations (I).Child,
                                   Relations (I).Required,
                                   TmpSuccess);
            HandleSchemaError (TmpSuccess, "Failure loading tag hierarchy");
            exit when not TmpSuccess;
         end loop;

      end BuildHierarchy;


   begin
      SectionDepths := SectionDepthArray'(others => SPARK_XML.TagDepth'First);
      FileDepth := SPARK_XML.TagDepth'First;

      SPARK_XML.InitSchema (Schema);
      SPARK_XML.InitSchemaState (SchemaState);

      LoadTags;
      LoadAttributes;
      BuildHierarchy;

   end Init;



   ------------------------------------------------------------------------
   -- Tag producing functions                                            --
   ------------------------------------------------------------------------

   ------------------------------------------------------------------------
   -- Producers for simple container tags.                               --
   -- These have no attributes of their own and only contain other tags. --
   -- The only exception is the Listing tag, which contains large        --
   -- amounts of text, and is also included.                             --
   ------------------------------------------------------------------------

   procedure StartSection (Section : in Sections;
                           Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SectionDepths;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState,
   --#         SectionDepths     from *,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Section,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Section,
   --#                                TagIDs;
   is
      Tag : SPARK_XML.TagID;
      Status : SPARK_XML.SchemaStatus;
      Depth : SPARK_XML.TagDepth;

      XML : SPARK_XML.XMLString;

   begin

      case Section is
         when SReport             => Tag := TagIDs (MTReport);
         when SPrologue           => Tag := TagIDs (MTPrologue);
         when SCommandline        => Tag := TagIDs (MTCommandline);
         when SCompilerData       => Tag := TagIDs (MTCompiler_Data);
         when SCyclicRequirements => Tag := TagIDs (MTCyclic_Requirements);
         when SIndexes            => Tag := TagIDs (MTIndexes);
         when STargetConfig       => Tag := TagIDs (MTTarget_Config);
         when SMessages           => Tag := TagIDs (MTMessages);
         when SMetaFiles          => Tag := TagIDs (MTMetafiles);
         when SUnitsNotFound      => Tag := TagIDs (MTUnits_Notfound);
         when SWarningsConfig     => Tag := TagIDs (MTWarnings_Config);
         when SResults            => Tag := TagIDs (MTResults);
         when SListing            => Tag := TagIDs (MTListing);
         when SUnitsInFile        => Tag := TagIDs (MTUnitsInFile);
         when SJustifications     => Tag := TagIDs (MTJustifications_Section);
         when SFullJustifications => Tag := TagIDs (MTFull_Justifications);
      end case;

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    Tag,
                                    Status);

      HandleError (Status);

      if Section = SReport then
         SPARK_XML.AddAttributeStr (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("version"),
                                    SchemaVersion,
                                    Status);
         HandleError (Status);
      end if;


      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  Depth, -- Record the section depth.
                                  Status);
      SectionDepths (Section) := Depth;

      HandleError (Status);

      EStrings.PutString (Report,
                                 XML);

   end StartSection;


   procedure EndSection (Section : in Sections;
                         Report  : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     SectionDepths;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Section,
   --#                                SectionDepths &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Section,
   --#                                SectionDepths;
   is
      Status : SPARK_XML.SchemaStatus;
      XML : SPARK_XML.XMLString;

   begin

      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          SectionDepths (Section),
                          XML,
                          Status);
      HandleError (Status);

      EStrings.PutString (Report,
                                 XML);

   end EndSection;




   -- Simple tags, ones whose opening and closing tags are generated
   -- at the same time, with simple text contents
   -- E.g. <filename>file.txt</filename>

   procedure SimpleTag (Tag      : in SPARK_XML.TagID;
                        Contents : in VString;
                        Report   : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                Tag &
   --#         SPARK_IO.File_Sys from *,
   --#                                Contents,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Tag;
   is
      AccXML, TmpXML : SPARK_XML.XMLString;  -- XML accumulator and temp variable.
      Depth : SPARK_XML.TagDepth;
      Status : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    Tag,
                                    Status);
      HandleError (Status);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  AccXML,
                                  Depth, -- Record the section depth.
                                  Status);

      HandleError (Status);


      EStrings.AppendExaminerString (AccXML, FilterString (Contents));

      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          Depth,
                          TmpXML,
                          Status);

      EStrings.AppendExaminerString (AccXML, TmpXML);

      HandleError (Status);

      EStrings.PutString (Report,
                                 AccXML);

   end SimpleTag;




   procedure Filename (File   : in VString;
                       Report : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                CommandLineData.Content,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                File,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      if CommandLineData.Content.PlainOutput then
         SimpleTag (TagIDs (MTFilename),
                    EStrings.LowerCase (FileSystem.JustFile (File, True)),
                    Report);
      else
         SimpleTag (TagIDs (MTFilename),
                    File,
                    Report);
      end if;

   end Filename;



   procedure Index (Idx    : in VString;
                    Report : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                CommandLineData.Content,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                Idx,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      if CommandLineData.Content.PlainOutput then
         SimpleTag (TagIDs (MTIndex),
                    EStrings.LowerCase (FileSystem.JustFile (Idx, True)),
                    Report);
      else
         SimpleTag (TagIDs (MTIndex),
                    Idx,
                    Report);
      end if;

   end Index;



   procedure Option (Opt    : in VString;
                     Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Opt,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      SimpleTag (TagIDs (MTOption),
                 Opt,
                 Report);
   end Option;


   procedure Suppressed (Item   : in VString;
                         Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      SimpleTag (TagIDs (MTSuppressed),
                 Item,
                 Report);
   end Suppressed;

   procedure SuppressedPragma (Item   : in VString;
                               Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
   begin
      SimpleTag (TagIDs (MTSuppressedPragma),
                 Item,
                 Report);
   end SuppressedPragma;




   ---------------------------------------------
   -- More complex tags that have attributes. --
   ---------------------------------------------

   procedure OpenCompilerItem (Item   : in     VString;
                               Depth  :    out SPARK_XML.TagDepth;
                               Report : in     SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives Depth,
   --#         SchemaState       from Item,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;

  is
      AccXML : SPARK_XML.XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (MTCompiler_Item),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("item"),
                                 Item,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  AccXML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 AccXML);  -- Output the opening tag
   end OpenCompilerItem;


   procedure CloseCompilerItem (Depth  : in     SPARK_XML.TagDepth;
                                Report : in     SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Depth &
   --#         SPARK_IO.File_Sys from *,
   --#                                Depth,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState;
   is
      AccXML : SPARK_XML.XMLString;  -- XML accumulator and temp variable.
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          Depth,
                          AccXML,
                          Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 AccXML);

   end CloseCompilerItem;

   procedure CompilerItem (Item   : in VString;
                           Val    : in VString;
                           Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Item,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Val;
   is
      Depth : SPARK_XML.TagDepth;
   begin

      OpenCompilerItem (Item,
                        Depth,
                        Report);

      EStrings.PutString (Report,
                                 FilterString (Val));  -- Output the value


      CloseCompilerItem (Depth,
                         Report);
   end CompilerItem;


   procedure LongCompilerItem (Item   : in VString;
                               Val    : in VLongString;
                               Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Item,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Item,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Val;
   is
      Depth : SPARK_XML.TagDepth;
   begin

      OpenCompilerItem (Item,
                        Depth,
                        Report);

      ELStrings.PutString (Report,
                                     FilterLongString (Val));  -- Output the value


      CloseCompilerItem (Depth,
                         Report);

   end LongCompilerItem;



   procedure Unit (Name   : in VString;
                   Typ    : in VString;
                   Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Name,
   --#                                Schema,
   --#                                TagIDs,
   --#                                Typ &
   --#         SPARK_IO.File_Sys from *,
   --#                                Name,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Typ;
   is
      AccXML, TmpXML : SPARK_XML.XMLString;  -- XML accumulator and temp variable.
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (MTUnit),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("name"),
                                 Name,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("type"),
                                 Typ,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  AccXML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);


      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          Depth,
                          TmpXML,
                          Stat);

      EStrings.AppendExaminerString (AccXML, TmpXML);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 AccXML);

   end Unit;



   procedure AdaUnit (Name       : in VString;
                      Typ        : in VString;
                      UnitStatus : in VString;
                      Report     : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Name,
   --#                                Schema,
   --#                                TagIDs,
   --#                                Typ,
   --#                                UnitStatus &
   --#         SPARK_IO.File_Sys from *,
   --#                                Name,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Typ,
   --#                                UnitStatus;
   is
      AccXML : SPARK_XML.XMLString;  -- XML accumulator.
      Depth : SPARK_XML.TagDepth;    -- Dummy variable.
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (MTUnit),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("name"),
                                 Name,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("type"),
                                 Typ,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("status"),
                                 UnitStatus,
                                 Stat);
      HandleError (Stat);


      SPARK_XML.OutputOpeningTag (Schema,  -- Expect inneffective assignment to Depth
                                  SchemaState,
                                  AccXML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 AccXML);

      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          Depth,
                          AccXML,
                          Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 AccXML);

   end AdaUnit;  -- Depth is neither referenced or exported.





   ---------------------------
   -- Message construction. --
   ---------------------------

   procedure StartMessage (Class  : in VString;
                           Code   : in Integer;
                           Line   : in Integer;
                           Offset : in Integer;
                           Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Class,
   --#                                Code,
   --#                                Line,
   --#                                Offset,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Class,
   --#                                Code,
   --#                                Line,
   --#                                Offset,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;

   begin

      SPARK_XML.InitOpeningTagNOCHECK (SchemaState,
                                       TagIDs (MTMessage),
                                       Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("class"),
                                 Class,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("code"),
                                 Code,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("line"),
                                 Line,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("offset"),
                                 Offset,
                                 Stat);
      HandleError (Stat);

      --# accept Flow, 10, Depth, "Expected ineffective assignment to Depth";
      SPARK_XML.OutputOpeningTag (Schema, -- Expect inneffective assignment
                                  SchemaState,
                                  XML,
                                  Depth,
                                  Stat);
      --# end accept;
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

      --# accept Flow, 33, Depth, "Expected Depth to be neither referenced nor exported";
   end StartMessage;  -- Expect "Depth not referenced or exported"



   procedure Symbol (Sym    : in VString;
                     SymNum : in Integer;
                     Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Schema,
   --#                                SymNum,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                Sym,
   --#                                SymNum,
   --#                                TagIDs;
   is
      AccXML, TmpXML : SPARK_XML.XMLString;  -- XML accumulator and temp variable.
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (MTSymbol),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("id"),
                                 SymNum,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  AccXML,
                                  Depth, -- Record the section depth.
                                  Stat);
      HandleError (Stat);

      -- Add the symbol contents.
      EStrings.AppendExaminerString (AccXML, FilterString (Sym));

      -- Close the tag.
      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          Depth,
                          TmpXML,
                          Stat);
      HandleError (Stat);
      EStrings.AppendExaminerString (AccXML, TmpXML);

      EStrings.PutString (Report,
                                 AccXML);

   end Symbol;


   -- EndMessage will close the open message (of which there will be one
   -- in the hierarchy since only a <symbol> can be placed in a <message>
   procedure EndMessage (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin
      SPARK_XML.CloseTagByID (Schema,
                              SchemaState,
                              TagIDs (MTMessage),
                              XML,
                              Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

   end EndMessage;


   --------------------------
   -- Metafile contruction --
   --------------------------

   procedure StartMetaFile (Name   : in     VString;
                            Report : in     SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Name,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Name,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
      Id : SPARK_XML.TagDepth;
   begin

      -- Generate the XML

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (MTMetafile),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("name"),
                                 Name,
                                 Stat);
      HandleError (Stat);

      --# accept Flow, 10, Id, "Expected ineffective assignment to Id";
      SPARK_XML.OutputOpeningTag (Schema, --Expect inneffective assignment to Id
                                  SchemaState,
                                  XML,
                                  Id,
                                  Stat);
      --# end accept;
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

      --# accept Flow, 33, Id, "Expected Id to be neither referenced nor exported";
   end StartMetaFile;  -- Expect "id neither referenced nor exported"



   procedure EndMetaFile (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin

      SPARK_XML.CloseTopTagByID (Schema,
                                 SchemaState,
                                 TagIDs (MTMetafile),
                                 XML,
                                 Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

   end EndMetaFile;


   ------------------------------
   -- Results Section contents --
   ------------------------------

   procedure StartFile (FName    : in VString;
                        Report   : in SPARK_IO.File_Type)
   --# global in     CommandLineData.Content;
   --#        in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --#           out FileDepth;
   --# derives FileDepth,
   --#         SchemaState       from CommandLineData.Content,
   --#                                FName,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                CommandLineData.Content,
   --#                                FName,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin
      SPARK_XML.InitOpeningTagNOCHECK (SchemaState,
                                       TagIDs (MTFile),
                                       Stat);
      HandleError (Stat);

      -- Add the attributes

      if CommandLineData.Content.PlainOutput then
         SPARK_XML.AddAttributeStr (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("filename"),
                                    EStrings.LowerCase
                                      (FileSystem.JustFile (FName, True)),
                                    Stat);
      else
         SPARK_XML.AddAttributeStr (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("filename"),
                                    FName,
                                    Stat);
      end if;
      HandleError (Stat);

      SPARK_XML.OutputOpeningTag (Schema,
                                  SchemaState,
                                  XML,
                                  FileDepth, -- Record the depth.
                                  Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

   end StartFile;



   procedure EndFile (Report  : in SPARK_IO.File_Type)
   --# global in     FileDepth;
   --#        in     Schema;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                FileDepth &
   --#         SPARK_IO.File_Sys from *,
   --#                                FileDepth,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;

   begin
      -- Close the tag.
      SPARK_XML.CloseTag (Schema,
                          SchemaState,
                          FileDepth,
                          XML,
                          Stat);
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

   end EndFile;

   --------------------------
   -- Justification output --
   --------------------------

   procedure BriefJustifications (Matched   : in Natural;
                                  Unmatched : in Natural;
                                  Report    : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Matched,
   --#                                Schema,
   --#                                TagIDs,
   --#                                Unmatched &
   --#         SPARK_IO.File_Sys from *,
   --#                                Matched,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs,
   --#                                Unmatched;
   is
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin
      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (MTBrief_Justifications),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("matched"),
                                 Matched,
                                 Stat);
      HandleError (Stat);

      if Unmatched > 0 then
         SPARK_XML.AddAttributeInt (Schema,
                                    SchemaState,
                                    SPARK_XML.XStr ("unmatched"),
                                    Unmatched,
                                    Stat);

         HandleError (Stat);
      end if;

      --# accept Flow, 10, Depth, "Expected ineffective assignment to Depth";
      SPARK_XML.OutputOpeningTag (Schema, -- Expect inneffective assignment
                                  SchemaState,
                                  XML,
                                  Depth,
                                  Stat);
      --# end accept;
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

      SPARK_XML.CloseTagByID (Schema,
                              SchemaState,
                              TagIDs (MTBrief_Justifications),
                              XML,
                              Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

      --# accept Flow, 33, Depth, "Expected Depth to be neither referenced nor exported";
   end BriefJustifications;

   procedure StartFullJustification (Class     : in VString;
                                     Code      : in Integer;
                                     LineFrom  : in Integer;
                                     LineTo    : in VString;
                                     MatchNo   : in Integer;
                                     MatchLine : in Integer;
                                     Report    : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                Class,
   --#                                Code,
   --#                                LineFrom,
   --#                                LineTo,
   --#                                MatchLine,
   --#                                MatchNo,
   --#                                Schema,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Class,
   --#                                Code,
   --#                                LineFrom,
   --#                                LineTo,
   --#                                MatchLine,
   --#                                MatchNo,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Depth : SPARK_XML.TagDepth;
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;

   begin

      SPARK_XML.InitOpeningTagByID (Schema,
                                    SchemaState,
                                    TagIDs (MTFull_Justification),
                                    Stat);
      HandleError (Stat);

      -- Add the attributes

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("class"),
                                 Class,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("code"),
                                 Code,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("line_from"),
                                 LineFrom,
                                 Stat);
      HandleError (Stat);

      SPARK_XML.AddAttributeStr (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("line_to"),
                                 LineTo,
                                 Stat);

      HandleError (Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("match_no"),
                                 MatchNo,
                                 Stat);

      HandleError (Stat);

      SPARK_XML.AddAttributeInt (Schema,
                                 SchemaState,
                                 SPARK_XML.XStr ("match_line"),
                                 MatchLine,
                                 Stat);

      HandleError (Stat);

      --# accept Flow, 10, Depth, "Expected ineffective assignment to Depth";
      SPARK_XML.OutputOpeningTag (Schema, -- Expect inneffective assignment
                                  SchemaState,
                                  XML,
                                  Depth,
                                  Stat);
      --# end accept;
      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);

      --# accept Flow, 33, Depth, "Expected Depth to be neither referenced nor exported";
   end StartFullJustification;

   procedure EndFullJustification (Report : in SPARK_IO.File_Type)
   --# global in     Schema;
   --#        in     TagIDs;
   --#        in out SchemaState;
   --#        in out SPARK_IO.File_Sys;
   --# derives SchemaState       from *,
   --#                                TagIDs &
   --#         SPARK_IO.File_Sys from *,
   --#                                Report,
   --#                                Schema,
   --#                                SchemaState,
   --#                                TagIDs;
   is
      Stat : SPARK_XML.SchemaStatus;
      XML : EStrings.T;
   begin
      SPARK_XML.CloseTagByID (Schema,
                              SchemaState,
                              TagIDs (MTFull_Justification),
                              XML,
                              Stat);

      HandleError (Stat);

      EStrings.PutString (Report,
                                 XML);
   end EndFullJustification;

end XMLReport;
